/* 
   ipmi-lan-interface-udm.c - IPMI UDM LAN Interface

   Copyright (C) 2003, 2004, 2005 FreeIPMI Core Team

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.

*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/types.h>

#include <stdio.h>
#include <stdlib.h>
#ifdef STDC_HEADERS
#include <string.h>
#endif /* STDC_HEADERS */
#if HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */
#include <assert.h>
#include <errno.h>

#include "freeipmi/udm/ipmi-messaging-support-cmds-udm.h"
#include "freeipmi/ipmi-authentication-type-spec.h"

#include "ipmi-udm-device.h"
#include "ipmi-lan-session-util.h"

#include "freeipmi-portability.h"
#include "udm-err-wrappers.h"
#include "udm-fiid-wrappers.h"

fiid_template_t tmpl_lan_raw =
  {
    {8192, "raw_data", FIID_FIELD_OPTIONAL | FIID_FIELD_LENGTH_VARIABLE},
    {0, "", 0}
  };

int8_t 
ipmi_lan_cmd (ipmi_device_t dev, 
	      fiid_obj_t obj_cmd_rq,
	      fiid_obj_t obj_cmd_rs)
{
  uint8_t authentication_type;
  uint32_t internal_workaround_flags = 0;

  UDM_ERR_DEV_CHECK (dev && dev->magic == IPMI_UDM_DEVICE_MAGIC);

  UDM_ERR_DEVICE_NOT_OPEN(dev->type == IPMI_DEVICE_LAN);
  UDM_ERR_DEVICE_NOT_OPEN(dev->io.outofband.sockfd); 

  UDM_ERR_PARAMETERS (fiid_obj_valid(obj_cmd_rq)
                      && fiid_obj_valid(obj_cmd_rs));
  
  UDM_FIID_OBJ_PACKET_VALID(obj_cmd_rq);
  
  if (dev->io.outofband.per_msg_auth_disabled)
    {
      authentication_type = IPMI_AUTHENTICATION_TYPE_NONE;
      if (dev->workaround_flags & IPMI_WORKAROUND_FLAGS_CHECK_UNEXPECTED_AUTHCODE)
        internal_workaround_flags |= IPMI_LAN_INTERNAL_WORKAROUND_FLAGS_CHECK_UNEXPECTED_AUTHCODE;
    }
  else
    authentication_type = dev->io.outofband.authentication_type;

  /* if auth type NONE, still pass password.  Needed for
   * check_unexpected_authcode workaround 
   */
  return ipmi_lan_cmd_wrapper (dev, 
                               internal_workaround_flags,
                               dev->lun,
                               dev->net_fn,
                               authentication_type,
                               &(dev->io.outofband.session_sequence_number),
                               dev->io.outofband.session_id,
                               &(dev->io.outofband.rq_seq),
                               dev->io.outofband.password,
                               IPMI_1_5_MAX_PASSWORD_LENGTH,
                               obj_cmd_rq,
                               obj_cmd_rs);
}

int32_t 
ipmi_lan_cmd_raw (ipmi_device_t dev, 
		  uint8_t *buf_rq, 
		  size_t buf_rq_len, 
		  uint8_t *buf_rs, 
		  size_t buf_rs_len)
{
  fiid_obj_t obj_cmd_rq = NULL;
  fiid_obj_t obj_cmd_rs = NULL;
  int retval = -1;
  int32_t len;
  uint8_t authentication_type;
  uint32_t internal_workaround_flags = 0;

  UDM_ERR_DEV_CHECK (dev && dev->magic == IPMI_UDM_DEVICE_MAGIC);

  UDM_ERR_DEVICE_NOT_OPEN(dev->type == IPMI_DEVICE_LAN);
  UDM_ERR_DEVICE_NOT_OPEN(dev->io.outofband.sockfd); 

  UDM_ERR_PARAMETERS (buf_rq
                      && buf_rq_len > 0
                      && buf_rs
                      && buf_rs_len > 0);

  UDM_FIID_OBJ_CREATE_CLEANUP(obj_cmd_rq, tmpl_lan_raw);
  UDM_FIID_OBJ_CREATE_CLEANUP(obj_cmd_rs, tmpl_lan_raw);

  UDM_FIID_OBJ_SET_ALL_CLEANUP (obj_cmd_rq, buf_rq, buf_rq_len);

  if (dev->io.outofband.per_msg_auth_disabled)
    {
      authentication_type = IPMI_AUTHENTICATION_TYPE_NONE;
      if (dev->workaround_flags & IPMI_WORKAROUND_FLAGS_CHECK_UNEXPECTED_AUTHCODE)
        internal_workaround_flags |= IPMI_LAN_INTERNAL_WORKAROUND_FLAGS_CHECK_UNEXPECTED_AUTHCODE;
    }
  else
    authentication_type = dev->io.outofband.authentication_type;

  if (ipmi_lan_cmd_wrapper (dev, 
                            internal_workaround_flags,
                            dev->lun,
                            dev->net_fn,
                            authentication_type,
                            &(dev->io.outofband.session_sequence_number),
                            dev->io.outofband.session_id,
                            &(dev->io.outofband.rq_seq),
                            dev->io.outofband.password,
                            IPMI_1_5_MAX_PASSWORD_LENGTH,
                            obj_cmd_rq,
                            obj_cmd_rs) < 0)
    goto cleanup;

  UDM_FIID_OBJ_GET_ALL_LEN_CLEANUP (len, obj_cmd_rs, buf_rs, buf_rs_len);
  retval = len;

 cleanup:
  UDM_FIID_OBJ_DESTROY (obj_cmd_rq);
  UDM_FIID_OBJ_DESTROY (obj_cmd_rs);
  return (retval);
}

int8_t 
ipmi_lan_2_0_cmd (ipmi_device_t dev, 
                  fiid_obj_t obj_cmd_rq,
                  fiid_obj_t obj_cmd_rs)
{
  uint8_t payload_authenticated;
  uint8_t payload_encrypted;

  UDM_ERR_DEV_CHECK (dev && dev->magic == IPMI_UDM_DEVICE_MAGIC);

  UDM_ERR_DEVICE_NOT_OPEN(dev->type == IPMI_DEVICE_LAN_2_0);
  UDM_ERR_DEVICE_NOT_OPEN(dev->io.outofband.sockfd); 

  UDM_ERR_PARAMETERS (fiid_obj_valid(obj_cmd_rq)
                      && fiid_obj_valid(obj_cmd_rs));
  
  UDM_FIID_OBJ_PACKET_VALID(obj_cmd_rq);
  
  if (dev->io.outofband.integrity_algorithm == IPMI_INTEGRITY_ALGORITHM_NONE)
    payload_authenticated = IPMI_PAYLOAD_FLAG_UNAUTHENTICATED;
  else
    payload_authenticated = IPMI_PAYLOAD_FLAG_AUTHENTICATED;

  if (dev->io.outofband.confidentiality_algorithm == IPMI_CONFIDENTIALITY_ALGORITHM_NONE)
    payload_encrypted = IPMI_PAYLOAD_FLAG_UNENCRYPTED;
  else
    payload_encrypted = IPMI_PAYLOAD_FLAG_ENCRYPTED;

  return ipmi_lan_2_0_cmd_wrapper (dev, 
                                   dev->lun,
                                   dev->net_fn,
                                   IPMI_PAYLOAD_TYPE_IPMI,
                                   payload_authenticated,
                                   payload_encrypted,
                                   NULL,
                                   &(dev->io.outofband.session_sequence_number),
                                   dev->io.outofband.managed_system_session_id,
                                   &(dev->io.outofband.rq_seq),
                                   dev->io.outofband.authentication_algorithm,
                                   dev->io.outofband.integrity_algorithm,
                                   dev->io.outofband.confidentiality_algorithm,
                                   dev->io.outofband.integrity_key_ptr,
                                   dev->io.outofband.integrity_key_len,
                                   dev->io.outofband.confidentiality_key_ptr,
                                   dev->io.outofband.confidentiality_key_len,
                                   strlen(dev->io.outofband.password) ? dev->io.outofband.password : NULL,
                                   strlen(dev->io.outofband.password),
                                   obj_cmd_rq,
                                   obj_cmd_rs);
}

int32_t 
ipmi_lan_2_0_cmd_raw (ipmi_device_t dev, 
                      uint8_t *buf_rq, 
                      size_t buf_rq_len, 
                      uint8_t *buf_rs, 
                      size_t buf_rs_len)
{
  uint8_t payload_authenticated;
  uint8_t payload_encrypted;
  fiid_obj_t obj_cmd_rq = NULL;
  fiid_obj_t obj_cmd_rs = NULL;
  int retval = -1;
  int32_t len;

  UDM_ERR_DEV_CHECK (dev && dev->magic == IPMI_UDM_DEVICE_MAGIC);

  UDM_ERR_DEVICE_NOT_OPEN(dev->type == IPMI_DEVICE_LAN_2_0);
  UDM_ERR_DEVICE_NOT_OPEN(dev->io.outofband.sockfd); 

  UDM_ERR_PARAMETERS (buf_rq
                      && buf_rq_len > 0
                      && buf_rs
                      && buf_rs_len > 0);

  UDM_FIID_OBJ_CREATE_CLEANUP(obj_cmd_rq, tmpl_lan_raw);
  UDM_FIID_OBJ_CREATE_CLEANUP(obj_cmd_rs, tmpl_lan_raw);

  UDM_FIID_OBJ_SET_ALL_CLEANUP (obj_cmd_rq, buf_rq, buf_rq_len);

  if (dev->io.outofband.integrity_algorithm == IPMI_INTEGRITY_ALGORITHM_NONE)
    payload_authenticated = IPMI_PAYLOAD_FLAG_UNAUTHENTICATED;
  else
    payload_authenticated = IPMI_PAYLOAD_FLAG_AUTHENTICATED;

  if (dev->io.outofband.confidentiality_algorithm == IPMI_CONFIDENTIALITY_ALGORITHM_NONE)
    payload_encrypted = IPMI_PAYLOAD_FLAG_UNENCRYPTED;
  else
    payload_encrypted = IPMI_PAYLOAD_FLAG_ENCRYPTED;

  if (ipmi_lan_2_0_cmd_wrapper (dev, 
                                dev->lun,
                                dev->net_fn,
                                IPMI_PAYLOAD_TYPE_IPMI,
                                payload_authenticated,
                                payload_encrypted,
                                NULL,
                                &(dev->io.outofband.session_sequence_number),
                                dev->io.outofband.managed_system_session_id,
                                &(dev->io.outofband.rq_seq),
                                dev->io.outofband.authentication_algorithm,
                                dev->io.outofband.integrity_algorithm,
                                dev->io.outofband.confidentiality_algorithm,
                                dev->io.outofband.integrity_key_ptr,
                                dev->io.outofband.integrity_key_len,
                                dev->io.outofband.confidentiality_key_ptr,
                                dev->io.outofband.confidentiality_key_len,
                                strlen(dev->io.outofband.password) ? dev->io.outofband.password : NULL,
                                strlen(dev->io.outofband.password),
                                obj_cmd_rq,
                                obj_cmd_rs) < 0)
    goto cleanup;

  UDM_FIID_OBJ_GET_ALL_LEN_CLEANUP (len, obj_cmd_rs, buf_rs, buf_rs_len);
  retval = len;

 cleanup:
  UDM_FIID_OBJ_DESTROY (obj_cmd_rq);
  UDM_FIID_OBJ_DESTROY (obj_cmd_rs);
  return (retval);
}
