/* 
   ipmi-lan-interface-udm.c - IPMI UDM LAN Interface

   Copyright (C) 2003, 2004, 2005 FreeIPMI Core Team

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.

*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/types.h>

#include <stdio.h>
#include <stdlib.h>
#ifdef STDC_HEADERS
#include <string.h>
#endif /* STDC_HEADERS */
#include <sys/socket.h>
#include <netinet/in.h>
#if TIME_WITH_SYS_TIME
#include <sys/time.h>
#include <time.h>
#else /* !TIME_WITH_SYS_TIME */
#if HAVE_SYS_TIME_H
#include <sys/time.h>
#else /* !HAVE_SYS_TIME_H */
#include <time.h>
#endif /* !HAVE_SYS_TIME_H */
#endif  /* !TIME_WITH_SYS_TIME */
#if HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */
#include <assert.h>
#include <errno.h>

#include "freeipmi/udm/ipmi-messaging-support-cmds-udm.h"
#include "freeipmi/ipmi-authentication-type-spec.h"
#include "freeipmi/ipmi-channel-spec.h"
#include "freeipmi/ipmi-cipher-suite-spec.h"
#include "freeipmi/ipmi-debug.h"
#include "freeipmi/ipmi-netfn-spec.h"
#include "freeipmi/ipmi-ipmb-interface.h"
#include "freeipmi/ipmi-lan.h"
#include "freeipmi/ipmi-lan-interface.h"
#include "freeipmi/ipmi-lan-utils.h"
#include "freeipmi/ipmi-privilege-level-spec.h"
#include "freeipmi/ipmi-rmcpplus.h"
#include "freeipmi/ipmi-rmcpplus-interface.h"
#include "freeipmi/ipmi-rmcpplus-status-spec.h"
#include "freeipmi/ipmi-rmcpplus-utils.h"
#include "freeipmi/rmcp.h"

#include "ipmi-udm-device.h"
#include "ipmi-lan-session-util.h"

#include "freeipmi-portability.h"
#include "udm-err-wrappers.h"
#include "udm-fiid-wrappers.h"

#define IPMI_PKT_LEN                   1024
#define IPMI_LAN_BACKOFF_COUNT         2

#define IPMI_SEQUENCE_NUMBER_WINDOW    8
#define IPMI_MAX_SEQUENCE_NUMBER       0xFFFFFFFF

struct socket_to_close {
  int fd;
  struct socket_to_close *next;
};

static int
_session_timed_out(ipmi_device_t dev)
{
  struct timeval current;
  struct timeval session_timeout;
  struct timeval session_timeout_len;

  assert(dev 
         && dev->magic == IPMI_UDM_DEVICE_MAGIC
         && (dev->type == IPMI_DEVICE_LAN
             || dev->type == IPMI_DEVICE_LAN_2_0));
  
  session_timeout_len.tv_sec = dev->io.outofband.session_timeout / 1000;
  session_timeout_len.tv_usec = (dev->io.outofband.session_timeout - (session_timeout_len.tv_sec * 1000)) * 1000;
  timeradd(&(dev->io.outofband.last_received), &session_timeout_len, &session_timeout);

  UDM_ERR (!(gettimeofday(&current, NULL) < 0));

  return timercmp(&current, &session_timeout, >);
}

static int
_calculate_timeout(ipmi_device_t dev, 
                   unsigned int retransmission_count,
                   struct timeval *timeout)
{
  struct timeval current;
  struct timeval session_timeout;
  struct timeval session_timeout_len;
  struct timeval session_timeout_val;
  struct timeval retransmission_timeout;
  struct timeval retransmission_timeout_len;
  struct timeval retransmission_timeout_val;
  unsigned int retransmission_timeout_multiplier;

  assert(dev 
	 && dev->magic == IPMI_UDM_DEVICE_MAGIC
         && (dev->type == IPMI_DEVICE_LAN
             || dev->type == IPMI_DEVICE_LAN_2_0)
	 && timeout);

  UDM_ERR (!(gettimeofday(&current, NULL) < 0));

  session_timeout_len.tv_sec = dev->io.outofband.session_timeout / 1000;
  session_timeout_len.tv_usec = (dev->io.outofband.session_timeout - (session_timeout_len.tv_sec * 1000)) * 1000;

  timeradd(&current, &session_timeout_len, &session_timeout);
  timersub(&session_timeout, &current, &session_timeout_val);

  retransmission_timeout_multiplier = (retransmission_count / IPMI_LAN_BACKOFF_COUNT) + 1;
  
  retransmission_timeout_len.tv_sec = (retransmission_timeout_multiplier * dev->io.outofband.retransmission_timeout) / 1000;
  retransmission_timeout_len.tv_usec = ((retransmission_timeout_multiplier * dev->io.outofband.retransmission_timeout) - (retransmission_timeout_len.tv_sec * 1000)) * 1000;

  timeradd(&dev->io.outofband.last_send, &retransmission_timeout_len, &retransmission_timeout);
  timersub(&retransmission_timeout, &current, &retransmission_timeout_val);

  if (timercmp(&retransmission_timeout_val, &session_timeout_val, <))
    {
      timeout->tv_sec = retransmission_timeout_val.tv_sec;
      timeout->tv_usec = retransmission_timeout_val.tv_usec;
    }
  else
    {
      timeout->tv_sec = session_timeout_val.tv_sec;
      timeout->tv_usec = session_timeout_val.tv_usec;
    }

  return 0;
}

static void
_ipmi_lan_dump_rq (ipmi_device_t dev, 
                   uint8_t *pkt,
                   uint32_t pkt_len,
                   fiid_obj_t obj_cmd_rq)
{
  char *hdr =
    "================================================\n"
    "IPMI 1.5 Request\n"
    "================================================";
  fiid_field_t *tmpl_cmd = NULL;

  assert (dev
	  && dev->magic == IPMI_UDM_DEVICE_MAGIC
	  && (dev->flags & IPMI_FLAGS_DEBUG_DUMP)
          && pkt
          && pkt_len
	  && fiid_obj_valid(obj_cmd_rq));

  /* Don't cleanup/return an error here.  It's just debug code. */

  if ((tmpl_cmd = fiid_obj_template(obj_cmd_rq)))
    {
      ipmi_dump_lan_packet (STDERR_FILENO,
                            NULL,
                            hdr,
                            pkt,
                            pkt_len,
                            tmpl_lan_msg_hdr_rq,
                            tmpl_cmd);
      fiid_template_free (tmpl_cmd);
    }
}

static void
_ipmi_lan_dump_rs (ipmi_device_t dev, 
                   uint8_t *pkt,
                   uint32_t pkt_len,
                   fiid_obj_t obj_cmd_rs)
{
  char *hdr =
    "================================================\n"
    "IPMI 1.5 Response\n"
    "================================================";
  fiid_field_t *tmpl_cmd = NULL;

  assert (dev
	  && dev->magic == IPMI_UDM_DEVICE_MAGIC
	  && (dev->flags & IPMI_FLAGS_DEBUG_DUMP)
          && pkt
          && pkt_len
	  && fiid_obj_valid(obj_cmd_rs));

  /* Don't cleanup/return an error here.  It's just debug code. */

  if ((tmpl_cmd = fiid_obj_template(obj_cmd_rs)))
    {
      ipmi_dump_lan_packet (STDERR_FILENO,
                            NULL,
                            hdr,
                            pkt,
                            pkt_len,
                            tmpl_lan_msg_hdr_rs,
                            tmpl_cmd);
      fiid_template_free (tmpl_cmd);
    }
}

static int8_t
_ipmi_check_session_sequence_number(ipmi_device_t dev,
                                    uint32_t session_sequence_number)
{
  uint32_t shift_num, wrap_val;
  int8_t rv = 0;

  /* achu: This algorithm is more or less from Appendix A of the IPMI
   * spec.  It may not be entirely necessary, since the requester
   * sequence number puts packets into lock-step mode.  Oh well.
   *
   * I know that technically I could remove a lot of code here if I
   * just let unsigned ints be unsigned ints (i.e. 0x00 - 0xff = 1).
   * I dunno, I like to see all of the code actually written out b/c
   * it makes more sense to the casual code reviewer.  Maybe I'll
   * change it later.
   */

  assert(dev 
         && dev->magic == IPMI_UDM_DEVICE_MAGIC
         && (dev->type == IPMI_DEVICE_LAN
             || dev->type == IPMI_DEVICE_LAN_2_0));

  /* IPMI Workaround (achu)
   *
   * Discovered on Sun Fire 4100.
   *
   * The session sequence numbers for IPMI 1.5 are the wrong endian.
   * So we have to flip the bits to workaround it.
   */
  if (dev->workaround_flags & IPMI_WORKAROUND_FLAGS_BIG_ENDIAN_SEQUENCE_NUMBER)
    {
      uint32_t tmp_session_sequence_number = session_sequence_number;

      session_sequence_number =
        ((tmp_session_sequence_number & 0xFF000000) >> 24)
        | ((tmp_session_sequence_number & 0x00FF0000) >> 8)
        | ((tmp_session_sequence_number & 0x0000FF00) << 8)
        | ((tmp_session_sequence_number & 0x000000FF) << 24);
    }

  /* Drop duplicate packet */
  if (session_sequence_number == dev->io.outofband.highest_received_sequence_number)
    goto out;
  
  /* In IPMI 2.0, sequence number 0 is special, and shouldn't happen */
  if (dev->type == IPMI_DEVICE_LAN_2_0 && session_sequence_number == 0)
    goto out;

  /* Check if sequence number is greater than highest received and is
   * within range
   */
  if (dev->io.outofband.highest_received_sequence_number > (IPMI_MAX_SEQUENCE_NUMBER - IPMI_SEQUENCE_NUMBER_WINDOW))
    {
      wrap_val = IPMI_SEQUENCE_NUMBER_WINDOW - (IPMI_MAX_SEQUENCE_NUMBER - dev->io.outofband.highest_received_sequence_number) - 1;

      /* In IPMI 2.0, sequence number 0 isn't possible, so adjust wrap_val */
      if (dev->type == IPMI_DEVICE_LAN_2_0)
        wrap_val++;
      
      if (session_sequence_number > dev->io.outofband.highest_received_sequence_number || session_sequence_number <= wrap_val)
        {
          if (session_sequence_number > dev->io.outofband.highest_received_sequence_number && session_sequence_number <= IPMI_MAX_SEQUENCE_NUMBER)
            shift_num = session_sequence_number - dev->io.outofband.highest_received_sequence_number;
          else
            {
              if (dev->type == IPMI_DEVICE_LAN)
                shift_num = session_sequence_number + (IPMI_MAX_SEQUENCE_NUMBER - dev->io.outofband.highest_received_sequence_number) + 1;
              else
                /* IPMI 2.0 Special Case b/c 0 isn't a legit sequence number */
                shift_num = session_sequence_number + (IPMI_MAX_SEQUENCE_NUMBER - dev->io.outofband.highest_received_sequence_number);
            }

          dev->io.outofband.highest_received_sequence_number = session_sequence_number;
          dev->io.outofband.previously_received_list <<= shift_num;
          dev->io.outofband.previously_received_list |= (0x1 << (shift_num - 1));
          rv++;
        }
    }
  else
    {
      if (session_sequence_number > dev->io.outofband.highest_received_sequence_number
          && (session_sequence_number - dev->io.outofband.highest_received_sequence_number) <= IPMI_SEQUENCE_NUMBER_WINDOW)
        {
          shift_num = (session_sequence_number - dev->io.outofband.highest_received_sequence_number);
          dev->io.outofband.highest_received_sequence_number = session_sequence_number;
          dev->io.outofband.previously_received_list <<= shift_num;
          dev->io.outofband.previously_received_list |= (0x1 << (shift_num - 1));
          rv++;
        }
    }

  /* Check if sequence number is lower than highest received, is
   * within range, and hasn't been seen yet
   */
  if (dev->io.outofband.highest_received_sequence_number < IPMI_SEQUENCE_NUMBER_WINDOW)
    {
      uint32_t wrap_val = IPMI_MAX_SEQUENCE_NUMBER - (IPMI_SEQUENCE_NUMBER_WINDOW - dev->io.outofband.highest_received_sequence_number) + 1;
      
      /* In IPMI 2.0, sequence number 0 isn't possible, so adjust wrap_val */
      if (dev->type == IPMI_DEVICE_LAN_2_0)
        wrap_val--;

      if (session_sequence_number < dev->io.outofband.highest_received_sequence_number || session_sequence_number >= wrap_val)
        {
          if (session_sequence_number > dev->io.outofband.highest_received_sequence_number && session_sequence_number <= IPMI_MAX_SEQUENCE_NUMBER)
            {
              if (dev->type == IPMI_DEVICE_LAN)
                shift_num = dev->io.outofband.highest_received_sequence_number + (IPMI_MAX_SEQUENCE_NUMBER - session_sequence_number) + 1;
              else
                /* IPMI 2.0 Special Case b/c 0 isn't a legit sequence number */
                shift_num = dev->io.outofband.highest_received_sequence_number + (IPMI_MAX_SEQUENCE_NUMBER - session_sequence_number);
            }
          else
            shift_num = dev->io.outofband.highest_received_sequence_number - session_sequence_number;
          
          /* Duplicate packet check*/
          if (dev->io.outofband.previously_received_list & (0x1 << (shift_num - 1)))
            goto out;
          
          dev->io.outofband.previously_received_list |= (0x1 << (shift_num - 1));
          rv++;
        }
    }
  else
    {
      if (session_sequence_number < dev->io.outofband.highest_received_sequence_number
          && session_sequence_number >= (dev->io.outofband.highest_received_sequence_number - IPMI_SEQUENCE_NUMBER_WINDOW))
        {
          shift_num = dev->io.outofband.highest_received_sequence_number - session_sequence_number;
          
          /* Duplicate packet check*/
          if (dev->io.outofband.previously_received_list & (0x1 << (shift_num - 1)))
            goto out;

          dev->io.outofband.previously_received_list |= (0x1 << (shift_num - 1));
          rv++;
        }
    }

 out:
  return rv;
}
                                        

static int8_t
_ipmi_lan_cmd_send (ipmi_device_t dev,
                    uint8_t lun,
                    uint8_t net_fn,
                    uint8_t authentication_type,
                    uint32_t session_sequence_number,
                    uint32_t session_id,
                    uint8_t rq_seq,
                    char *password,
                    uint32_t password_len,
                    fiid_obj_t obj_cmd_rq)
{
  uint8_t *pkt;
  int32_t pkt_len = 1024;
  int32_t send_len = 0;

  assert(dev
	 && dev->magic == IPMI_UDM_DEVICE_MAGIC
         && (dev->type == IPMI_DEVICE_LAN
             || dev->type == IPMI_DEVICE_LAN_2_0)
	 && dev->io.outofband.sockfd 
         && IPMI_BMC_LUN_VALID(lun)
         && IPMI_NET_FN_VALID(net_fn)
         && IPMI_1_5_AUTHENTICATION_TYPE_VALID(authentication_type)
         && !(password && password_len > IPMI_1_5_MAX_PASSWORD_LENGTH)
	 && fiid_obj_valid(obj_cmd_rq)
	 && fiid_obj_packet_valid(obj_cmd_rq));

  UDM_FIID_OBJ_CLEAR(dev->io.outofband.rq.obj_rmcp_hdr);
  UDM_FIID_OBJ_CLEAR(dev->io.outofband.rq.obj_lan_session_hdr);
  UDM_FIID_OBJ_CLEAR(dev->io.outofband.rq.obj_lan_msg_hdr);
    
  pkt = alloca (pkt_len);
  UDM_ERR (pkt);
  memset (pkt, 0, pkt_len);
    
  UDM_ERR (fill_rmcp_hdr_ipmi (dev->io.outofband.rq.obj_rmcp_hdr) != -1);

  UDM_ERR (fill_lan_msg_hdr (net_fn,
                             lun,
			     rq_seq,
			     dev->io.outofband.rq.obj_lan_msg_hdr) != -1);

  UDM_ERR (fill_lan_session_hdr (authentication_type,
				 session_sequence_number,
                                 session_id,
				 dev->io.outofband.rq.obj_lan_session_hdr) != -1);
  UDM_ERR ((send_len = assemble_ipmi_lan_pkt (dev->io.outofband.rq.obj_rmcp_hdr,
					      dev->io.outofband.rq.obj_lan_session_hdr,
					      dev->io.outofband.rq.obj_lan_msg_hdr,
					      obj_cmd_rq,
                                              (uint8_t *)password,
                                              password_len,
					      pkt,
					      pkt_len)) != -1);

  if (dev->flags & IPMI_FLAGS_DEBUG_DUMP && send_len)
    _ipmi_lan_dump_rq (dev, pkt, send_len, obj_cmd_rq);

  UDM_ERR (!(ipmi_lan_sendto (dev->io.outofband.sockfd, 
			      pkt, 
			      send_len, 
			      0, 
			      (struct sockaddr *)&(dev->io.outofband.remote_host), 
			      sizeof(struct sockaddr_in)) < 0));

  UDM_ERR (!(gettimeofday(&dev->io.outofband.last_send, NULL) < 0));
  
  return (0);
}

static int8_t
_ipmi_lan_cmd_recv (ipmi_device_t dev, 
                    uint8_t *pkt,
                    uint32_t pkt_len,
                    unsigned int retransmission_count,
                    fiid_obj_t obj_cmd_rs)
{
  struct sockaddr_in from;
  socklen_t fromlen = 0;
  struct timeval timeout;
  fd_set read_set;
  int status = 0;
  int32_t recv_len;

  assert(dev
	 && dev->magic == IPMI_UDM_DEVICE_MAGIC
         && (dev->type == IPMI_DEVICE_LAN
             || dev->type == IPMI_DEVICE_LAN_2_0)
	 && dev->io.outofband.sockfd 
         && pkt
         && pkt_len
	 && fiid_obj_valid(obj_cmd_rs));  
  
  UDM_FIID_OBJ_CLEAR(dev->io.outofband.rs.obj_rmcp_hdr); 
  UDM_FIID_OBJ_CLEAR(dev->io.outofband.rs.obj_lan_session_hdr); 
  UDM_FIID_OBJ_CLEAR(dev->io.outofband.rs.obj_lan_msg_hdr); 
  UDM_FIID_OBJ_CLEAR(dev->io.outofband.rs.obj_lan_msg_trlr); 

  if (dev->io.outofband.retransmission_timeout != 0)
    {
      FD_ZERO (&read_set);
      FD_SET (dev->io.outofband.sockfd, &read_set);
      
      UDM_ERR (!(_calculate_timeout(dev, retransmission_count, &timeout) < 0));
      
      UDM_ERR (!((status = select ((dev->io.outofband.sockfd + 1), 
				   &read_set,
				   NULL,
				   NULL, 
				   &timeout)) < 0));
      if (status == 0)
        return (0); // resend the request
    }

  UDM_ERR (!((recv_len = ipmi_lan_recvfrom (dev->io.outofband.sockfd, 
					    pkt, 
					    pkt_len, 
					    0, 
					    (struct sockaddr *) &from, 
					    &fromlen)) < 0));
  
  if (dev->flags & IPMI_FLAGS_DEBUG_DUMP && recv_len)
    _ipmi_lan_dump_rs (dev, 
                       pkt,
                       recv_len,
                       obj_cmd_rs);

  UDM_ERR (unassemble_ipmi_lan_pkt(pkt,
				   recv_len,
 				   dev->io.outofband.rs.obj_rmcp_hdr,
				   dev->io.outofband.rs.obj_lan_session_hdr,
				   dev->io.outofband.rs.obj_lan_msg_hdr,
				   obj_cmd_rs,
				   dev->io.outofband.rs.obj_lan_msg_trlr) != -1);

  return (recv_len);
}

int8_t 
ipmi_lan_cmd_wrapper (ipmi_device_t dev, 
                      uint32_t internal_workaround_flags,
                      uint8_t lun,
                      uint8_t net_fn,
                      uint8_t authentication_type,
                      uint32_t *session_sequence_number,
                      uint32_t session_id,
                      uint8_t *rq_seq,
                      char *password,
                      uint32_t password_len,
                      fiid_obj_t obj_cmd_rq,
                      fiid_obj_t obj_cmd_rs)
{
  int retval = -1;
  int ret;
  unsigned int retransmission_count = 0;
  uint8_t pkt[IPMI_PKT_LEN];
  int32_t recv_len;
  struct socket_to_close *sockets = NULL;
  uint64_t rs_session_id;
  uint64_t rs_session_sequence_number;
  
  assert(dev
	 && dev->magic == IPMI_UDM_DEVICE_MAGIC
         && (dev->type == IPMI_DEVICE_LAN
             || dev->type == IPMI_DEVICE_LAN_2_0)
	 && dev->io.outofband.sockfd 
         && IPMI_BMC_LUN_VALID(lun)
         && IPMI_NET_FN_VALID(net_fn)
         && IPMI_1_5_AUTHENTICATION_TYPE_VALID(authentication_type)
         && !(password && password_len > IPMI_1_5_MAX_PASSWORD_LENGTH)
	 && fiid_obj_valid(obj_cmd_rq)
	 && fiid_obj_packet_valid(obj_cmd_rq)
         && fiid_obj_valid(obj_cmd_rs));

  UDM_ERR_DEV_CHECK (dev && dev->magic == IPMI_UDM_DEVICE_MAGIC);
 
  if (!dev->io.outofband.last_received.tv_sec
      && !dev->io.outofband.last_received.tv_usec)
    UDM_ERR (!(gettimeofday(&dev->io.outofband.last_received, NULL) < 0));

  if (_ipmi_lan_cmd_send (dev, 
                          lun, 
                          net_fn,
                          authentication_type,
                          (session_sequence_number) ? *session_sequence_number : 0,
                          session_id,
                          (rq_seq) ? *rq_seq : 0,
                          password,
                          password_len,
                          obj_cmd_rq) < 0)
    goto cleanup;

  while (1)
    {
      if (_session_timed_out(dev))
        {
	  UDM_ERR_SET_ERRNUM (IPMI_ERR_SESSION_TIMEOUT);
          retval = -1;
          break;
        }
     
      memset(pkt, '\0', IPMI_PKT_LEN);
      if ((recv_len = _ipmi_lan_cmd_recv (dev, 
                                          pkt,
                                          IPMI_PKT_LEN,
                                          retransmission_count,
                                          obj_cmd_rs)) < 0)
        {
          retval = -1;
          break;
        }
      
      if (!recv_len)
        {
          if (session_sequence_number)
            (*session_sequence_number)++;
          if (rq_seq)
            *rq_seq = ((*rq_seq) + 1) % (IPMI_LAN_REQUESTER_SEQUENCE_NUMBER_MAX + 1);
          
          retransmission_count++;
          
          /* IPMI Workaround (achu)
           *
           * Discovered on Intel Tiger4 (SR870BN4)
           *
           * If the reply from a previous Get Session Challenge request is
           * lost on the network, the following retransmission will make
           * the BMC confused and it will not respond to future packets.
           *
           * The problem seems to exist only when the retransmitted packet
           * is transmitted from the same source port.  Therefore, the fix
           * is to send the retransmission from a different source port.
           * So we'll create a new socket, re-bind to an ephemereal port
           * (guaranteeing us a brand new port), and store this new
           * socket.
           *
           * In the event we need to resend this packet multiple times, we
           * do not want the chance that old ports will be used again.  We
           * store the old file descriptrs (which are bound to the old
           * ports) on a list, and close all of them after we have gotten
           * past the Get Session Challenge phase of the protocol.
           */
          if (internal_workaround_flags & IPMI_LAN_INTERNAL_WORKAROUND_FLAGS_GET_SESSION_CHALLENGE)
            {
              struct socket_to_close *s;
              struct sockaddr_in addr;

              UDM_ERR_CLEANUP ((s = (struct socket_to_close *)malloc(sizeof(struct socket_to_close))));
              s->fd = dev->io.outofband.sockfd;
              s->next = sockets;
              sockets = s;
              
              UDM_ERR_SYSTEM_ERROR_CLEANUP (!((dev->io.outofband.sockfd = socket (AF_INET, SOCK_DGRAM, 0)) < 0));
              
              memset (&addr, 0, sizeof (struct sockaddr_in));
              addr.sin_family = AF_INET;
              addr.sin_port = htons (0);
              addr.sin_addr.s_addr = htonl (INADDR_ANY);

              UDM_ERR_SYSTEM_ERROR_CLEANUP (!(bind(dev->io.outofband.sockfd,
                                                   (struct sockaddr *)&addr,
                                                   sizeof(struct sockaddr_in)) < 0));
            }

          if (_ipmi_lan_cmd_send (dev, 
                                  lun,
                                  net_fn,
                                  authentication_type,
                                  (session_sequence_number) ? *session_sequence_number : 0,
                                  session_id,
                                  (rq_seq) ? *rq_seq : 0,
                                  password,
                                  password_len,
                                  obj_cmd_rq) < 0)
            goto cleanup;
          
          continue;
        }

      /* else received a packet */

      UDM_FIID_OBJ_GET_CLEANUP (dev->io.outofband.rs.obj_lan_session_hdr, "session_id", &rs_session_id);

      if (session_id != rs_session_id)
        {
          /* IPMI Workaround (achu)
           *
           * Discovered on Tyan S2882 w/ m3289 BMC
           *
           * The remote BMC returns zeroes for the session id instead of the
           * actual session id.  To work around this problem, we'll assume the
           * session id is correct if it is equal to zero.
           */
          if ((dev->workaround_flags & IPMI_WORKAROUND_FLAGS_ACCEPT_SESSION_ID_ZERO)
              && !rs_session_id)
            /* you get a second chance */
            ;
          else
            continue;
        }

      UDM_ERR_CLEANUP (!((ret = ipmi_lan_check_checksum (dev->io.outofband.rs.obj_lan_msg_hdr,
                                                         obj_cmd_rs,
                                                         dev->io.outofband.rs.obj_lan_msg_trlr)) < 0));
      
      if (!ret)
        continue;

      /* IPMI Workaround (achu)
       *
       * Discovered on Dell PowerEdge 2850
       *
       * When per-message authentication is disabled, and we send a
       * message to a remote machine with auth-type none, the Dell
       * motherboard will respond with a message with the auth-type used
       * in the activate session stage and the appropriate authcode. So
       * here is our second session-authcode check attempt under these
       * circumstances.
       */

      UDM_ERR_CLEANUP (!((ret = ipmi_lan_check_session_authentication_code (dev->io.outofband.rs.obj_lan_session_hdr,
                                                                            dev->io.outofband.rs.obj_lan_msg_hdr,
                                                                            obj_cmd_rs,
                                                                            dev->io.outofband.rs.obj_lan_msg_trlr,
                                                                            authentication_type,
                                                                            (uint8_t *)password,
                                                                            password_len)) < 0));

      if ((internal_workaround_flags & IPMI_LAN_INTERNAL_WORKAROUND_FLAGS_CHECK_UNEXPECTED_AUTHCODE)
          && !ret)
        {
          UDM_ERR_CLEANUP (!((ret = ipmi_lan_check_session_authentication_code (dev->io.outofband.rs.obj_lan_session_hdr,
                                                                                dev->io.outofband.rs.obj_lan_msg_hdr,
                                                                                obj_cmd_rs,
                                                                                dev->io.outofband.rs.obj_lan_msg_trlr,
                                                                                dev->io.outofband.authentication_type,
                                                                                (uint8_t *)password,
                                                                                password_len)) < 0));
        }

      if (!ret)
        continue;
  
      if (session_sequence_number)
        {
          UDM_FIID_OBJ_GET_CLEANUP (dev->io.outofband.rs.obj_lan_session_hdr,
                                    "session_sequence_number",
                                    &rs_session_sequence_number);

          UDM_ERR_CLEANUP (!((ret = _ipmi_check_session_sequence_number(dev, 
                                                                        (uint32_t)rs_session_sequence_number)) < 0));

          if (!ret)
            continue;         
        }
          
      UDM_ERR_CLEANUP (!((ret = ipmi_lan_check_rq_seq(dev->io.outofband.rs.obj_lan_msg_hdr, 
                                                      (rq_seq) ? *rq_seq : 0)) < 0));
      
      if (!ret)
        continue;

      UDM_ERR_CLEANUP (!(gettimeofday(&(dev->io.outofband.last_received), NULL) < 0));
      retval = recv_len;
      break;
    }
  
 cleanup:
  if (session_sequence_number)
    (*session_sequence_number)++;
  if (rq_seq)
    *rq_seq = ((*rq_seq) + 1) % (IPMI_LAN_REQUESTER_SEQUENCE_NUMBER_MAX + 1);
  while (sockets)
    {
      close(sockets->fd);
      sockets = sockets->next;
    }
  return (retval);
}

int8_t 
ipmi_lan_open_session (ipmi_device_t dev)
{
  fiid_obj_t obj_cmd_rq = NULL;
  fiid_obj_t obj_cmd_rs = NULL;
  uint64_t authentication_status_anonymous_login;
  uint64_t authentication_status_null_username;
  uint64_t authentication_status_non_null_username;
  uint64_t supported_authentication_type = 0;
  uint64_t temp_session_id = 0;
  uint64_t session_id = 0;
  uint8_t challenge_string[IPMI_CHALLENGE_STRING_LENGTH];
  uint64_t session_sequence_number = 0;
  uint32_t initial_outbound_sequence_number = 0;
  unsigned int seedp;
  int8_t rv = -1;
  int8_t ret;
  uint64_t val;

  assert (dev
          && dev->magic == IPMI_UDM_DEVICE_MAGIC
          && dev->io.outofband.sockfd
          && dev->type == IPMI_DEVICE_LAN
          && strlen(dev->io.outofband.username) <= IPMI_MAX_USER_NAME_LENGTH
          && strlen(dev->io.outofband.password) <= IPMI_1_5_MAX_PASSWORD_LENGTH
          && IPMI_1_5_AUTHENTICATION_TYPE_VALID (dev->io.outofband.authentication_type)
          && IPMI_PRIVILEGE_LEVEL_VALID (dev->io.outofband.privilege_level));
  
  /* Random number generation */
  seedp = (unsigned int) clock () + (unsigned int) time (NULL);
  srand (seedp);

  dev->io.outofband.rq_seq = (double)(IPMI_LAN_REQUESTER_SEQUENCE_NUMBER_MAX) * (rand()/(RAND_MAX + 1.0));
  
  UDM_FIID_OBJ_CREATE_CLEANUP(obj_cmd_rq, tmpl_cmd_get_channel_authentication_capabilities_rq);
  UDM_FIID_OBJ_CREATE_CLEANUP(obj_cmd_rs, tmpl_cmd_get_channel_authentication_capabilities_rs);
  
  UDM_ERR_CLEANUP (!(fill_cmd_get_channel_authentication_capabilities (IPMI_CHANNEL_NUMBER_CURRENT_CHANNEL,
                                                                       dev->io.outofband.privilege_level,
                                                                       obj_cmd_rq) < 0));

  if (ipmi_lan_cmd_wrapper (dev,
                            0,
                            IPMI_BMC_IPMB_LUN_BMC,
                            IPMI_NET_FN_APP_RQ,
                            IPMI_AUTHENTICATION_TYPE_NONE,
                            NULL,
                            0,
                            &(dev->io.outofband.rq_seq),
                            NULL,
                            0,
                            obj_cmd_rq,
                            obj_cmd_rs) < 0)
    goto cleanup;

  /* IPMI Workaround (achu)
   *
   * Discovered on an ASUS P5M2 motherboard.
   *
   * The ASUS motherboard reports incorrect settings of anonymous
   * vs. null vs non-null username capabilities.  The workaround is to
   * skip these checks.
   */

  if (!(dev->workaround_flags & IPMI_WORKAROUND_FLAGS_AUTHENTICATION_CAPABILITIES))
    {
      UDM_FIID_OBJ_GET_CLEANUP (obj_cmd_rs,
                                "authentication_status.anonymous_login",
                                &authentication_status_anonymous_login);
      UDM_FIID_OBJ_GET_CLEANUP (obj_cmd_rs, 
                                "authentication_status.null_username",
                                &authentication_status_null_username);
      UDM_FIID_OBJ_GET_CLEANUP (obj_cmd_rs, 
                                "authentication_status.non_null_username",
                                &authentication_status_non_null_username);
      
      if ((!strlen(dev->io.outofband.username) && !strlen(dev->io.outofband.password)
           && !authentication_status_anonymous_login)
          || (!strlen(dev->io.outofband.username)
              && !authentication_status_anonymous_login
              && !authentication_status_null_username)
          || (strlen(dev->io.outofband.username)
              && !authentication_status_non_null_username))
        {
          dev->errnum = IPMI_ERR_USERNAME_INVALID;
          goto cleanup;
        }
    }

  /* IPMI Workaround (achu)
   *
   * Discovered on IBM eServer 325
   *
   * The remote BMC ignores if permsg authentiction is enabled
   * or disabled.  So we need to force it no matter what.
   */
  if (!(dev->workaround_flags & IPMI_WORKAROUND_FLAGS_FORCE_PERMSG_AUTHENTICATION))
    {
      UDM_FIID_OBJ_GET_CLEANUP (obj_cmd_rs,
                                "authentication_status.per_message_authentication",
                                &val);
      dev->io.outofband.per_msg_auth_disabled = val;
    }
  else
    dev->io.outofband.per_msg_auth_disabled = 0;

  switch (dev->io.outofband.authentication_type)
    {
    case IPMI_AUTHENTICATION_TYPE_NONE:
      UDM_FIID_OBJ_GET_CLEANUP (obj_cmd_rs, 
				"authentication_type.none", 
				&supported_authentication_type);
      break;
    case IPMI_AUTHENTICATION_TYPE_MD2:
      UDM_FIID_OBJ_GET_CLEANUP (obj_cmd_rs, 
				"authentication_type.md2", 
				&supported_authentication_type);
      break;
    case IPMI_AUTHENTICATION_TYPE_MD5:
      UDM_FIID_OBJ_GET_CLEANUP (obj_cmd_rs, 
				"authentication_type.md5", 
				&supported_authentication_type);
      break;
    case IPMI_AUTHENTICATION_TYPE_STRAIGHT_PASSWORD_KEY:
      UDM_FIID_OBJ_GET_CLEANUP (obj_cmd_rs, 
				"authentication_type.straight_password_key", 
				&supported_authentication_type);
      break;
    case IPMI_AUTHENTICATION_TYPE_OEM_PROP:
      UDM_FIID_OBJ_GET_CLEANUP (obj_cmd_rs, 
				"authentication_type.oem_prop", 
				&supported_authentication_type);
      break;
    }

  if (!supported_authentication_type)
    UDM_ERR_SET_ERRNUM_CLEANUP(IPMI_ERR_AUTHENTICATION_TYPE_UNAVAILABLE);

  UDM_FIID_OBJ_DESTROY(obj_cmd_rq);
  UDM_FIID_OBJ_DESTROY(obj_cmd_rs);

  UDM_FIID_OBJ_CREATE_CLEANUP (obj_cmd_rq, tmpl_cmd_get_session_challenge_rq);
  UDM_FIID_OBJ_CREATE_CLEANUP (obj_cmd_rs, tmpl_cmd_get_session_challenge_rs);

  UDM_ERR_CLEANUP (!(fill_cmd_get_session_challenge (dev->io.outofband.authentication_type,
                                                     dev->io.outofband.username,
                                                     IPMI_MAX_USER_NAME_LENGTH,
                                                     obj_cmd_rq) < 0));

  if (ipmi_lan_cmd_wrapper (dev,
                            IPMI_LAN_INTERNAL_WORKAROUND_FLAGS_GET_SESSION_CHALLENGE,
                            IPMI_BMC_IPMB_LUN_BMC,
                            IPMI_NET_FN_APP_RQ,
                            IPMI_AUTHENTICATION_TYPE_NONE,
                            NULL,
                            0,
                            &(dev->io.outofband.rq_seq),
                            NULL,
                            0,
                            obj_cmd_rq,
                            obj_cmd_rs) < 0)
    goto cleanup;

  UDM_ERR_CLEANUP (!((ret = ipmi_check_completion_code_success (obj_cmd_rs)) < 0));

  if (!ret)
    {
      if (ipmi_check_completion_code(obj_cmd_rs, IPMI_COMP_CODE_INVALID_USERNAME) == 1
          || ipmi_check_completion_code(obj_cmd_rs, IPMI_COMP_CODE_NULL_USERNAME_NOT_ENABLED) == 1)
        UDM_ERR_SET_ERRNUM_CLEANUP(IPMI_ERR_USERNAME_INVALID);
      else
        UDM_BAD_COMPLETION_CODE_TO_UDM_ERRNUM (dev, obj_cmd_rs);
      UDM_ERR_LOG_CLEANUP(0);
    }
  
  UDM_FIID_OBJ_GET_CLEANUP (obj_cmd_rs, 
			    "temp_session_id", 
			    &temp_session_id);
  
  UDM_FIID_OBJ_GET_DATA_CLEANUP (obj_cmd_rs, 
				 "challenge_string", 
				 challenge_string,
				 IPMI_CHALLENGE_STRING_LENGTH);

  UDM_FIID_OBJ_DESTROY(obj_cmd_rq);
  UDM_FIID_OBJ_DESTROY(obj_cmd_rs);

  UDM_FIID_OBJ_CREATE_CLEANUP (obj_cmd_rq, tmpl_cmd_activate_session_rq);
  UDM_FIID_OBJ_CREATE_CLEANUP (obj_cmd_rs, tmpl_cmd_activate_session_rs);

  initial_outbound_sequence_number = rand ();

  UDM_ERR_CLEANUP (!(fill_cmd_activate_session (dev->io.outofband.authentication_type,
                                                dev->io.outofband.privilege_level,
                                                challenge_string,
                                                IPMI_CHALLENGE_STRING_LENGTH,
                                                initial_outbound_sequence_number,
                                                obj_cmd_rq) < 0));

  if (ipmi_lan_cmd_wrapper (dev,
                            0,
                            IPMI_BMC_IPMB_LUN_BMC,
                            IPMI_NET_FN_APP_RQ,
                            dev->io.outofband.authentication_type,
                            NULL,
                            (uint32_t)temp_session_id,
                            &(dev->io.outofband.rq_seq),
                            dev->io.outofband.password,
                            IPMI_1_5_MAX_PASSWORD_LENGTH,
                            obj_cmd_rq,
                            obj_cmd_rs) < 0)
    {
      if (dev->errnum == IPMI_ERR_SESSION_TIMEOUT)
        UDM_ERR_SET_ERRNUM_CLEANUP(IPMI_ERR_PASSWORD_VERIFICATION_TIMEOUT);
      goto cleanup;
    }

  UDM_ERR_CLEANUP (!((ret = ipmi_check_completion_code_success (obj_cmd_rs)) < 0));

  if (!ret)
    {
      if (ipmi_check_completion_code(obj_cmd_rs, IPMI_COMP_CODE_NO_SESSION_SLOT_AVAILABLE) == 1
          || ipmi_check_completion_code(obj_cmd_rs, IPMI_COMP_CODE_NO_SLOT_AVAILABLE_FOR_GIVEN_USER) == 1
          || ipmi_check_completion_code(obj_cmd_rs, IPMI_COMP_CODE_NO_SLOT_AVAILABLE_TO_SUPPORT_USER) == 1)
        UDM_ERR_SET_ERRNUM_CLEANUP(IPMI_ERR_BMC_BUSY);
      else if (ipmi_check_completion_code(obj_cmd_rs, IPMI_COMP_CODE_EXCEEDS_PRIVILEGE_LEVEL) == 1)
        UDM_ERR_SET_ERRNUM_CLEANUP(IPMI_ERR_PRIVILEGE_LEVEL_CANNOT_BE_OBTAINED);
      else
        UDM_BAD_COMPLETION_CODE_TO_UDM_ERRNUM (dev, obj_cmd_rs);
      UDM_ERR_LOG_CLEANUP(0);
    }

  UDM_FIID_OBJ_GET_CLEANUP (obj_cmd_rs, 
			    "session_id", 
			    &session_id);

  dev->io.outofband.session_id = session_id;

  /* achu: On some buggy BMCs the initial outbound sequence number on
   * the activate session response is off by one.  So we just accept
   * whatever sequence number they give us even if it isn't the
   * initial outbound sequence number.
   */
  UDM_FIID_OBJ_GET_CLEANUP (dev->io.outofband.rs.obj_lan_session_hdr,
                            "session_sequence_number",
                            &val);
  dev->io.outofband.highest_received_sequence_number = val;
  
  /* IPMI Workaround (achu)
   *
   * Discovered on Sun Fire 4100.
   *
   * The session sequence numbers for IPMI 1.5 are the wrong endian.
   * So we have to flip the bits to workaround it.
   */
  if (dev->workaround_flags & IPMI_WORKAROUND_FLAGS_BIG_ENDIAN_SEQUENCE_NUMBER)
    {
      uint32_t tmp_session_sequence_number = dev->io.outofband.highest_received_sequence_number;

      dev->io.outofband.highest_received_sequence_number =
        ((tmp_session_sequence_number & 0xFF000000) >> 24)
        | ((tmp_session_sequence_number & 0x00FF0000) >> 8)
        | ((tmp_session_sequence_number & 0x0000FF00) << 8)
        | ((tmp_session_sequence_number & 0x000000FF) << 24);
    }

  UDM_FIID_OBJ_GET_CLEANUP (obj_cmd_rs, 
			    "initial_inbound_sequence_number", 
			    &session_sequence_number);

  dev->io.outofband.session_sequence_number = session_sequence_number;
  
  UDM_FIID_OBJ_GET_CLEANUP (obj_cmd_rs,
			    "authentication_type",
			    &val);

  /* IPMI Workaround (achu)
   *
   * Discovered on Supermicro H8QME with SIMSO daughter card.
   *
   * (Note: This could work for "IBM eServer 325" per msg auth
   * problem.  But I don't have hardware to test it :-()
   *
   * The remote BMC ignores if permsg authentiction is disabled.
   * Handle it appropriately by just not doing permsg authentication.
   */
  if (dev->io.outofband.per_msg_auth_disabled
      && val != IPMI_AUTHENTICATION_TYPE_NONE)
    dev->io.outofband.per_msg_auth_disabled = 0;

  UDM_FIID_OBJ_DESTROY(obj_cmd_rq);
  UDM_FIID_OBJ_DESTROY(obj_cmd_rs);

  /* achu: At this point in time, the session is actually setup
   * legitimately, so we can use the actual set session privilege
   * level UDM function.
   */
  UDM_FIID_OBJ_CREATE_CLEANUP (obj_cmd_rs, tmpl_cmd_set_session_privilege_level_rs);

  if (ipmi_cmd_set_session_privilege_level (dev, 
                                            dev->io.outofband.privilege_level,
                                            obj_cmd_rs) < 0)
    {
      if (dev->errnum == IPMI_ERR_BAD_COMPLETION_CODE)
	{
	  if (ipmi_check_completion_code(obj_cmd_rs, IPMI_COMP_CODE_RQ_LEVEL_NOT_AVAILABLE_FOR_USER) == 1
	      || ipmi_check_completion_code(obj_cmd_rs, IPMI_COMP_CODE_RQ_LEVEL_EXCEEDS_USER_PRIVILEGE_LIMIT) == 1)
	    UDM_ERR_SET_ERRNUM_CLEANUP(IPMI_ERR_PRIVILEGE_LEVEL_CANNOT_BE_OBTAINED);
	}
      UDM_ERR_LOG_CLEANUP(0);
    }

  rv = 0;
 cleanup:
  UDM_FIID_OBJ_DESTROY(obj_cmd_rq);
  UDM_FIID_OBJ_DESTROY(obj_cmd_rs);
  return (rv);
}

int8_t 
ipmi_lan_close_session (ipmi_device_t dev)
{
  fiid_obj_t obj_cmd_rs = NULL;
  int8_t rv = -1;

  assert (dev 
          && dev->magic == IPMI_UDM_DEVICE_MAGIC
          && (dev->type == IPMI_DEVICE_LAN
              || dev->type == IPMI_DEVICE_LAN_2_0)
          && dev->io.outofband.sockfd);

  UDM_FIID_OBJ_CREATE (obj_cmd_rs, tmpl_cmd_close_session_rs);

  if (dev->type == IPMI_DEVICE_LAN)
    {
      if (ipmi_cmd_close_session(dev,
                                 dev->io.outofband.session_id,
                                 obj_cmd_rs) < 0)
        goto cleanup;
    }
  else
    {
      if (ipmi_cmd_close_session(dev,
                                 dev->io.outofband.managed_system_session_id,
                                 obj_cmd_rs) < 0)
        goto cleanup;
    }
  
  rv = 0;
 cleanup:
  UDM_FIID_OBJ_DESTROY(obj_cmd_rs);
  return (rv);
}

static void
_ipmi_lan_2_0_dump_rq (ipmi_device_t dev, 
                       uint8_t authentication_algorithm,
                       uint8_t integrity_algorithm,
                       uint8_t confidentiality_algorithm,
                       uint8_t *integrity_key,
                       uint32_t integrity_key_len,
                       uint8_t *confidentiality_key,
                       uint32_t confidentiality_key_len,
                       uint8_t *pkt,
                       uint32_t pkt_len,
                       fiid_obj_t obj_cmd_rq)
{
  char *hdr =
    "================================================\n"
    "IPMI 2.0 Request\n"
    "================================================";
  fiid_field_t *tmpl_cmd = NULL;

  assert (dev
	  && dev->magic == IPMI_UDM_DEVICE_MAGIC
	  && (dev->flags & IPMI_FLAGS_DEBUG_DUMP)
          && IPMI_AUTHENTICATION_ALGORITHM_VALID(authentication_algorithm)
          && IPMI_INTEGRITY_ALGORITHM_VALID(integrity_algorithm)
          && IPMI_CONFIDENTIALITY_ALGORITHM_VALID(confidentiality_algorithm)
          && pkt
          && pkt_len
	  && fiid_obj_valid(obj_cmd_rq));

  /* Don't cleanup/return an error here.  It's just debug code. */

  if ((tmpl_cmd = fiid_obj_template(obj_cmd_rq)))
    {
      ipmi_dump_rmcpplus_packet (STDERR_FILENO,
                                 NULL,
                                 hdr,
                                 authentication_algorithm,
                                 integrity_algorithm,
                                 confidentiality_algorithm,
                                 integrity_key,
                                 integrity_key_len,
                                 confidentiality_key,
                                 confidentiality_key_len,
                                 pkt,
                                 pkt_len,
                                 tmpl_lan_msg_hdr_rq,
                                 tmpl_cmd);
      fiid_template_free (tmpl_cmd);
    }
}

static void
_ipmi_lan_2_0_dump_rs (ipmi_device_t dev, 
                       uint8_t authentication_algorithm,
                       uint8_t integrity_algorithm,
                       uint8_t confidentiality_algorithm,
                       uint8_t *integrity_key,
                       uint32_t integrity_key_len,
                       uint8_t *confidentiality_key,
                       uint32_t confidentiality_key_len,
                       uint8_t *pkt,
                       uint32_t pkt_len,
                       fiid_obj_t obj_cmd_rs)
{
  char *hdr =
    "================================================\n"
    "IPMI 2.0 Response\n"
    "================================================";
  fiid_field_t *tmpl_cmd = NULL;

  assert (dev
	  && dev->magic == IPMI_UDM_DEVICE_MAGIC
	  && (dev->flags & IPMI_FLAGS_DEBUG_DUMP)
          && IPMI_AUTHENTICATION_ALGORITHM_VALID(authentication_algorithm)
          && IPMI_INTEGRITY_ALGORITHM_VALID(integrity_algorithm)
          && IPMI_CONFIDENTIALITY_ALGORITHM_VALID(confidentiality_algorithm)
          && pkt
          && pkt_len
	  && fiid_obj_valid(obj_cmd_rs));

  /* Don't cleanup/return an error here.  It's just debug code. */

  if ((tmpl_cmd = fiid_obj_template(obj_cmd_rs)))
    {
      ipmi_dump_rmcpplus_packet (STDERR_FILENO,
                                 NULL,
                                 hdr,
                                 authentication_algorithm,
                                 integrity_algorithm,
                                 confidentiality_algorithm,
                                 integrity_key,
                                 integrity_key_len,
                                 confidentiality_key,
                                 confidentiality_key_len,
                                 pkt,
                                 pkt_len,
                                 tmpl_lan_msg_hdr_rs,
                                 tmpl_cmd);
      fiid_template_free (tmpl_cmd);
    }
}

static int8_t
_ipmi_lan_2_0_cmd_send (ipmi_device_t dev,
                        uint8_t lun,
                        uint8_t net_fn,
                        uint8_t payload_type,
                        uint8_t payload_authenticated,
                        uint8_t payload_encrypted,
                        uint32_t session_sequence_number,
                        uint32_t session_id,
                        uint8_t rq_seq,
                        uint8_t authentication_algorithm,
                        uint8_t integrity_algorithm,
                        uint8_t confidentiality_algorithm,
                        uint8_t *integrity_key,
                        uint32_t integrity_key_len,
                        uint8_t *confidentiality_key,
                        uint32_t confidentiality_key_len,
                        char *password,
                        uint32_t password_len,
                        fiid_obj_t obj_cmd_rq)
{
  uint8_t *pkt;
  int32_t pkt_len = 1024;
  int32_t send_len = 0;

  assert(dev
	 && dev->magic == IPMI_UDM_DEVICE_MAGIC
	 && dev->io.outofband.sockfd 
         && IPMI_BMC_LUN_VALID(lun)
         && IPMI_NET_FN_VALID(net_fn)
         && (payload_type == IPMI_PAYLOAD_TYPE_IPMI
             || payload_type == IPMI_PAYLOAD_TYPE_RMCPPLUS_OPEN_SESSION_REQUEST
             || payload_type == IPMI_PAYLOAD_TYPE_RAKP_MESSAGE_1
             || payload_type == IPMI_PAYLOAD_TYPE_RAKP_MESSAGE_3)
         && IPMI_PAYLOAD_AUTHENTICATED_FLAG_VALID(payload_authenticated)
         && IPMI_PAYLOAD_ENCRYPTED_FLAG_VALID(payload_encrypted)
         && IPMI_AUTHENTICATION_ALGORITHM_VALID(authentication_algorithm)
         && IPMI_INTEGRITY_ALGORITHM_VALID(integrity_algorithm)
         && IPMI_CONFIDENTIALITY_ALGORITHM_VALID(confidentiality_algorithm)
         && !(password && password_len > IPMI_2_0_MAX_PASSWORD_LENGTH)
	 && fiid_obj_valid(obj_cmd_rq)
	 && fiid_obj_packet_valid(obj_cmd_rq));

  UDM_FIID_OBJ_CLEAR(dev->io.outofband.rq.obj_rmcp_hdr);
  UDM_FIID_OBJ_CLEAR(dev->io.outofband.rq.obj_rmcpplus_session_hdr);
  UDM_FIID_OBJ_CLEAR(dev->io.outofband.rq.obj_lan_msg_hdr);
  UDM_FIID_OBJ_CLEAR(dev->io.outofband.rq.obj_rmcpplus_session_trlr);
    
  pkt = alloca (pkt_len);
  UDM_ERR (pkt);
  memset (pkt, 0, pkt_len);
    
  UDM_ERR (fill_rmcp_hdr_ipmi (dev->io.outofband.rq.obj_rmcp_hdr) != -1);

  UDM_ERR (fill_rmcpplus_session_hdr (payload_type,
                                      payload_authenticated,
                                      payload_encrypted,
                                      0, /* oem_iana */
                                      0, /* oem_payload_id */
                                      session_id,
                                      session_sequence_number,
                                      dev->io.outofband.rq.obj_rmcpplus_session_hdr) != -1);

  UDM_ERR (fill_lan_msg_hdr (net_fn,
                             lun,
			     rq_seq,
			     dev->io.outofband.rq.obj_lan_msg_hdr) != -1);

  UDM_ERR (fill_rmcpplus_session_trlr(dev->io.outofband.rq.obj_rmcpplus_session_trlr) != -1);

  UDM_ERR ((send_len = assemble_ipmi_rmcpplus_pkt(authentication_algorithm,
                                                  integrity_algorithm,
                                                  confidentiality_algorithm,
                                                  integrity_key,
                                                  integrity_key_len,
                                                  confidentiality_key,
                                                  confidentiality_key_len,
                                                  (uint8_t *)password,
                                                  password_len,
                                                  dev->io.outofband.rq.obj_rmcp_hdr,
                                                  dev->io.outofband.rq.obj_rmcpplus_session_hdr,
                                                  dev->io.outofband.rq.obj_lan_msg_hdr,
                                                  obj_cmd_rq,
                                                  dev->io.outofband.rq.obj_rmcpplus_session_trlr,
                                                  pkt,
                                                  pkt_len)) != -1);

  if (dev->flags & IPMI_FLAGS_DEBUG_DUMP && send_len)
    _ipmi_lan_2_0_dump_rq (dev, 
                           authentication_algorithm,
                           integrity_algorithm,
                           confidentiality_algorithm,
                           integrity_key,
                           integrity_key_len,
                           confidentiality_key,
                           confidentiality_key_len,
                           pkt,
                           send_len,
                           obj_cmd_rq);

  UDM_ERR (!(ipmi_lan_sendto (dev->io.outofband.sockfd, 
			      pkt, 
			      send_len, 
			      0, 
			      (struct sockaddr *)&(dev->io.outofband.remote_host), 
			      sizeof(struct sockaddr_in)) < 0));

  UDM_ERR (!(gettimeofday(&dev->io.outofband.last_send, NULL) < 0));
  
  return (0);
}

static int8_t
_ipmi_lan_2_0_cmd_recv (ipmi_device_t dev, 
                        uint8_t authentication_algorithm,
                        uint8_t integrity_algorithm,
                        uint8_t confidentiality_algorithm,
                        uint8_t *integrity_key,
                        uint32_t integrity_key_len,
                        uint8_t *confidentiality_key,
                        uint32_t confidentiality_key_len,
                        uint8_t *pkt,
                        uint32_t pkt_len,
                        unsigned int retransmission_count,
                        fiid_obj_t obj_cmd_rs)
{
  struct sockaddr_in from;
  socklen_t fromlen = 0;
  struct timeval timeout;
  int32_t recv_len = 0;
  
  fd_set read_set;
  int status = 0;
  
  assert(dev
	 && dev->magic == IPMI_UDM_DEVICE_MAGIC
	 && dev->io.outofband.sockfd 
         && IPMI_AUTHENTICATION_ALGORITHM_VALID(authentication_algorithm)
         && IPMI_INTEGRITY_ALGORITHM_VALID(integrity_algorithm)
         && IPMI_CONFIDENTIALITY_ALGORITHM_VALID(confidentiality_algorithm)
         && pkt
         && pkt_len
	 && fiid_obj_valid(obj_cmd_rs));
  
    
  UDM_FIID_OBJ_CLEAR(dev->io.outofband.rs.obj_rmcp_hdr); 
  UDM_FIID_OBJ_CLEAR(dev->io.outofband.rs.obj_rmcpplus_session_hdr); 
  UDM_FIID_OBJ_CLEAR(dev->io.outofband.rs.obj_lan_msg_hdr); 
  UDM_FIID_OBJ_CLEAR(dev->io.outofband.rs.obj_rmcpplus_payload); 
  UDM_FIID_OBJ_CLEAR(dev->io.outofband.rs.obj_lan_msg_trlr); 
  UDM_FIID_OBJ_CLEAR(dev->io.outofband.rs.obj_rmcpplus_session_trlr); 
  
  if (dev->io.outofband.retransmission_timeout != 0)
    {
      FD_ZERO (&read_set);
      FD_SET (dev->io.outofband.sockfd, &read_set);
      
      UDM_ERR (!(_calculate_timeout(dev, retransmission_count, &timeout) < 0));
      
      UDM_ERR (!((status = select ((dev->io.outofband.sockfd + 1), 
				   &read_set,
				   NULL,
				   NULL, 
				   &timeout)) < 0));
      if (status == 0)
        return (0); // resend the request
    }

  UDM_ERR (!((recv_len = ipmi_lan_recvfrom (dev->io.outofband.sockfd, 
					    pkt, 
					    pkt_len, 
					    0, 
					    (struct sockaddr *) &from, 
					    &fromlen)) < 0));
  
  if (dev->flags & IPMI_FLAGS_DEBUG_DUMP && recv_len)
    _ipmi_lan_2_0_dump_rs (dev, 
                           authentication_algorithm,
                           integrity_algorithm,
                           confidentiality_algorithm,
                           integrity_key,
                           integrity_key_len,
                           confidentiality_key,
                           confidentiality_key_len,
                           pkt,
                           recv_len,
                           obj_cmd_rs);

  UDM_ERR (unassemble_ipmi_rmcpplus_pkt (authentication_algorithm,
                                         integrity_algorithm,
                                         confidentiality_algorithm,
                                         integrity_key,
                                         integrity_key_len,
                                         confidentiality_key,
                                         confidentiality_key_len,
                                         pkt,
                                         recv_len,
                                         dev->io.outofband.rs.obj_rmcp_hdr,
                                         dev->io.outofband.rs.obj_rmcpplus_session_hdr,
                                         dev->io.outofband.rs.obj_rmcpplus_payload,
                                         dev->io.outofband.rs.obj_lan_msg_hdr,
                                         obj_cmd_rs,
                                         dev->io.outofband.rs.obj_lan_msg_trlr,
                                         dev->io.outofband.rs.obj_rmcpplus_session_trlr) != -1);

  return (recv_len);
}

int8_t 
ipmi_lan_2_0_cmd_wrapper (ipmi_device_t dev, 
                          uint8_t lun,
                          uint8_t net_fn,
                          uint8_t payload_type,
                          uint8_t payload_authenticated,
                          uint8_t payload_encrypted,
                          uint8_t *message_tag,
                          uint32_t *session_sequence_number,
                          uint32_t session_id,
                          uint8_t *rq_seq,
                          uint8_t authentication_algorithm,
                          uint8_t integrity_algorithm,
                          uint8_t confidentiality_algorithm,
                          uint8_t *integrity_key,
                          uint32_t integrity_key_len,
                          uint8_t *confidentiality_key,
                          uint32_t confidentiality_key_len,
                          char *password,
                          uint32_t password_len,
                          fiid_obj_t obj_cmd_rq,
                          fiid_obj_t obj_cmd_rs)
{
  int retval = -1;
  int ret;
  unsigned int retransmission_count = 0;
  uint8_t pkt[IPMI_PKT_LEN];
  int32_t recv_len;
  uint64_t val;
  uint64_t rs_session_sequence_number;

  assert(dev
	 && dev->magic == IPMI_UDM_DEVICE_MAGIC
         && dev->type == IPMI_DEVICE_LAN_2_0
	 && dev->io.outofband.sockfd 
         && IPMI_BMC_LUN_VALID(lun)
         && IPMI_NET_FN_VALID(net_fn)
         && (payload_type == IPMI_PAYLOAD_TYPE_IPMI
             || payload_type == IPMI_PAYLOAD_TYPE_RMCPPLUS_OPEN_SESSION_REQUEST
             || payload_type == IPMI_PAYLOAD_TYPE_RAKP_MESSAGE_1
             || payload_type == IPMI_PAYLOAD_TYPE_RAKP_MESSAGE_3)
         && IPMI_PAYLOAD_AUTHENTICATED_FLAG_VALID(payload_authenticated)
         && IPMI_PAYLOAD_ENCRYPTED_FLAG_VALID(payload_encrypted)
         && IPMI_AUTHENTICATION_ALGORITHM_VALID(authentication_algorithm)
         && IPMI_INTEGRITY_ALGORITHM_VALID(integrity_algorithm)
         && IPMI_CONFIDENTIALITY_ALGORITHM_VALID(confidentiality_algorithm)
         && !(password && password_len > IPMI_2_0_MAX_PASSWORD_LENGTH)
	 && fiid_obj_valid(obj_cmd_rq)
	 && fiid_obj_packet_valid(obj_cmd_rq)
         && fiid_obj_valid(obj_cmd_rs));

  UDM_ERR_DEV_CHECK (dev && dev->magic == IPMI_UDM_DEVICE_MAGIC);
 
  if (!dev->io.outofband.last_received.tv_sec
      && !dev->io.outofband.last_received.tv_usec)
    UDM_ERR (!(gettimeofday(&dev->io.outofband.last_received, NULL) < 0));

  if (_ipmi_lan_2_0_cmd_send (dev, 
                              lun, 
                              net_fn,
                              payload_type,
                              payload_authenticated,
                              payload_encrypted,
                              (session_sequence_number) ? *session_sequence_number : 0,
                              session_id,
                              (rq_seq) ? *rq_seq : 0,
                              authentication_algorithm,
                              integrity_algorithm,
                              confidentiality_algorithm,
                              integrity_key,
                              integrity_key_len,
                              confidentiality_key,
                              confidentiality_key_len,
                              password,
                              password_len,
                              obj_cmd_rq) < 0)
    goto cleanup;

  while (1)
    {
      if (_session_timed_out(dev))
        {
	  UDM_ERR_SET_ERRNUM (IPMI_ERR_SESSION_TIMEOUT);
          retval = -1;
          break;
        }
     
      memset(pkt, '\0', IPMI_PKT_LEN);
      if ((recv_len = _ipmi_lan_2_0_cmd_recv (dev, 
                                              authentication_algorithm,
                                              integrity_algorithm,
                                              confidentiality_algorithm,
                                              integrity_key,
                                              integrity_key_len,
                                              confidentiality_key,
                                              confidentiality_key_len,
                                              pkt,
                                              IPMI_PKT_LEN,
                                              retransmission_count,
                                              obj_cmd_rs)) < 0)
        {
          retval = -1;
          break;
        }

      if (!recv_len)
        {
          if (message_tag)
            (*message_tag)++;
          /* In IPMI 2.0, session sequence numbers of 0 are special */
          if (session_sequence_number)
            {
              (*session_sequence_number)++;
              if (!(*session_sequence_number))
                (*session_sequence_number)++;
            }
          if (rq_seq)
            *rq_seq = ((*rq_seq) + 1) % (IPMI_LAN_REQUESTER_SEQUENCE_NUMBER_MAX + 1);

          retransmission_count++;

          if (payload_type == IPMI_PAYLOAD_TYPE_RMCPPLUS_OPEN_SESSION_REQUEST
              || payload_type == IPMI_PAYLOAD_TYPE_RAKP_MESSAGE_1
              || payload_type == IPMI_PAYLOAD_TYPE_RAKP_MESSAGE_3)
            {
              /* Unlike most packets, the open session request, rakp 1
               * and rakp 3 messages have the message tags in a
               * non-header field.  So this is a special case.
               */
              UDM_FIID_OBJ_SET_CLEANUP(obj_cmd_rq, "message_tag", (*message_tag));
            }

          if (_ipmi_lan_2_0_cmd_send (dev, 
                                      lun, 
                                      net_fn,
                                      payload_type,
                                      payload_authenticated,
                                      payload_encrypted,
                                      (session_sequence_number) ? *session_sequence_number : 0,
                                      session_id,
                                      (rq_seq) ? *rq_seq : 0,
                                      authentication_algorithm,
                                      integrity_algorithm,
                                      confidentiality_algorithm,
                                      integrity_key,
                                      integrity_key_len,
                                      confidentiality_key,
                                      confidentiality_key_len,
                                      password,
                                      password_len,
                                      obj_cmd_rq) < 0)
            goto cleanup;
          
          continue;
        }

      /* else received a packet */

      if (payload_type == IPMI_PAYLOAD_TYPE_IPMI)
        {
          uint64_t payload_type;

          UDM_FIID_OBJ_GET_CLEANUP (dev->io.outofband.rs.obj_rmcpplus_session_hdr,
                                    "payload_type",
                                    &payload_type);
          if (payload_type != IPMI_PAYLOAD_TYPE_IPMI)
            continue;

          UDM_FIID_OBJ_GET_CLEANUP (dev->io.outofband.rs.obj_rmcpplus_session_hdr,
                                    "session_id",
                                    &val);
          
          if (val != dev->io.outofband.remote_console_session_id)
            continue;

          UDM_ERR_CLEANUP (!((ret = ipmi_lan_check_checksum (dev->io.outofband.rs.obj_lan_msg_hdr,
                                                             obj_cmd_rs,
                                                             dev->io.outofband.rs.obj_lan_msg_trlr)) < 0));
          
          if (!ret)
            continue;

          UDM_ERR_CLEANUP (!((ret = ipmi_rmcpplus_check_packet_session_authentication_code (integrity_algorithm,
                                                                                            pkt,
                                                                                            recv_len,
                                                                                            integrity_key,
                                                                                            integrity_key_len,
                                                                                            (uint8_t *)password,
                                                                                            password_len,
                                                                                            dev->io.outofband.rs.obj_rmcpplus_session_trlr)) < 0));
          
          if (!ret)
            continue;

          if (session_sequence_number)
            {
              UDM_FIID_OBJ_GET_CLEANUP (dev->io.outofband.rs.obj_rmcpplus_session_hdr,
                                        "session_sequence_number",
                                        &rs_session_sequence_number);
              
              UDM_ERR_CLEANUP (!((ret = _ipmi_check_session_sequence_number(dev, 
                                                                            (uint32_t)rs_session_sequence_number)) < 0));
              
              if (!ret)
                continue;
            }

          UDM_ERR_CLEANUP (!((ret = ipmi_lan_check_rq_seq(dev->io.outofband.rs.obj_lan_msg_hdr, 
                                                          (rq_seq) ? *rq_seq : 0)) < 0));
          
          if (!ret)
            continue;
        }
      else if (payload_type == IPMI_PAYLOAD_TYPE_RMCPPLUS_OPEN_SESSION_REQUEST)
        {
          uint64_t val;

          UDM_FIID_OBJ_GET_CLEANUP (dev->io.outofband.rs.obj_rmcpplus_session_hdr,
                                    "payload_type",
                                    &val);
          if (val != IPMI_PAYLOAD_TYPE_RMCPPLUS_OPEN_SESSION_RESPONSE)
            continue;

          UDM_FIID_OBJ_GET_CLEANUP (obj_cmd_rs,
                                    "remote_console_session_id",
                                    &val);
          
          if (val != dev->io.outofband.remote_console_session_id)
            continue;

          if (message_tag)
            {
              UDM_FIID_OBJ_GET_CLEANUP (obj_cmd_rs,
                                        "message_tag",
                                        &val);
              if (val != *message_tag)
                continue;
            }
        }
      else if (payload_type == IPMI_PAYLOAD_TYPE_RAKP_MESSAGE_1)
        {
          uint64_t val;

          UDM_FIID_OBJ_GET_CLEANUP (dev->io.outofband.rs.obj_rmcpplus_session_hdr,
                                    "payload_type",
                                    &val);
          if (val != IPMI_PAYLOAD_TYPE_RAKP_MESSAGE_2)
            continue;

          UDM_FIID_OBJ_GET_CLEANUP (obj_cmd_rs,
                                    "remote_console_session_id",
                                    &val);
          
          if (val != dev->io.outofband.remote_console_session_id)
            continue;

          if (message_tag)
            {
              UDM_FIID_OBJ_GET_CLEANUP (obj_cmd_rs,
                                        "message_tag",
                                        &val);
              if (val != *message_tag)
                continue;
            }
        }
      else if (payload_type == IPMI_PAYLOAD_TYPE_RAKP_MESSAGE_3)
        {
          uint64_t val;

          UDM_FIID_OBJ_GET_CLEANUP (dev->io.outofband.rs.obj_rmcpplus_session_hdr,
                                    "payload_type",
                                    &val);
          if (val != IPMI_PAYLOAD_TYPE_RAKP_MESSAGE_4)
            continue;

          UDM_FIID_OBJ_GET_CLEANUP (obj_cmd_rs,
                                    "remote_console_session_id",
                                    &val);
          
          if (val != dev->io.outofband.remote_console_session_id)
            continue;

          if (message_tag)
            {
              UDM_FIID_OBJ_GET_CLEANUP (obj_cmd_rs,
                                        "message_tag",
                                        &val);
              if (val != *message_tag)
                continue;
            }
        }
     
      UDM_ERR_CLEANUP (!(gettimeofday(&dev->io.outofband.last_received, NULL) < 0));
      retval = recv_len;
      break;
    }

 cleanup:
  if (message_tag)
    (*message_tag)++;
  /* In IPMI 2.0, session sequence numbers of 0 are special */
  if (session_sequence_number)
    {
      (*session_sequence_number)++;
      if (!(*session_sequence_number))
        (*session_sequence_number)++;
    }
  if (rq_seq)
    *rq_seq = ((*rq_seq) + 1) % (IPMI_LAN_REQUESTER_SEQUENCE_NUMBER_MAX + 1);
  return (retval);
}

int8_t 
ipmi_lan_2_0_open_session (ipmi_device_t dev)
{
  fiid_obj_t obj_cmd_rq = NULL;  
  fiid_obj_t obj_cmd_rs = NULL;
  uint64_t authentication_status_anonymous_login;
  uint64_t authentication_status_null_username;
  uint64_t authentication_status_non_null_username;
  uint64_t authentication_status_k_g;
  uint64_t ipmi_v20_extended_capabilities_available;
  uint64_t channel_supports_ipmi_v20_connections;
  uint8_t maximum_privilege_level;
  uint8_t rmcpplus_status_code;
  uint8_t remote_console_random_number[IPMI_REMOTE_CONSOLE_RANDOM_NUMBER_LENGTH];
  uint8_t managed_system_random_number[IPMI_MANAGED_SYSTEM_RANDOM_NUMBER_LENGTH];
  int32_t managed_system_random_number_len;
  uint8_t managed_system_guid[IPMI_MANAGED_SYSTEM_GUID_LENGTH];
  int32_t managed_system_guid_len;
  uint8_t key_exchange_authentication_code[IPMI_MAX_KEY_EXCHANGE_AUTHENTICATION_CODE_LENGTH];
  int32_t key_exchange_authentication_code_len;
  uint8_t message_tag;
  char *username;
  char username_buf[IPMI_MAX_USER_NAME_LENGTH+1];
  uint32_t username_len;
  char *password;
  uint32_t password_len;
  uint8_t authentication_algorithm;
  uint8_t requested_maximum_privilege;
  uint8_t assume_rakp_4_success = 0;
  uint8_t name_only_lookup;
  int8_t rv = -1;
  unsigned int seedp;
  int ret;
  uint64_t val;

  assert (dev
          && dev->magic == IPMI_UDM_DEVICE_MAGIC
          && dev->io.outofband.sockfd
          && dev->type == IPMI_DEVICE_LAN_2_0
          && strlen(dev->io.outofband.username) <= IPMI_MAX_USER_NAME_LENGTH
          && strlen(dev->io.outofband.password) <= IPMI_2_0_MAX_PASSWORD_LENGTH
          && IPMI_PRIVILEGE_LEVEL_VALID (dev->io.outofband.privilege_level)
          && IPMI_CIPHER_SUITE_ID_SUPPORTED (dev->io.outofband.cipher_suite_id)
          && dev->io.outofband.sik_key_ptr == dev->io.outofband.sik_key
          && dev->io.outofband.sik_key_len == IPMI_MAX_SIK_KEY_LENGTH
          && dev->io.outofband.integrity_key_ptr == dev->io.outofband.integrity_key
          && dev->io.outofband.integrity_key_len == IPMI_MAX_INTEGRITY_KEY_LENGTH
          && dev->io.outofband.confidentiality_key_ptr == dev->io.outofband.confidentiality_key
          && dev->io.outofband.confidentiality_key_len == IPMI_MAX_CONFIDENTIALITY_KEY_LENGTH);

  /* Random number generation */
  seedp = (unsigned int) clock () + (unsigned int) time (NULL);
  srand (seedp);

  /* Unlike IPMI 1.5, there is no initial sequence number negotiation, so we don't 
   * start at a random sequence number.
   */
  dev->io.outofband.session_sequence_number = 1;
    
  dev->io.outofband.rq_seq = (uint8_t)((double)(IPMI_LAN_REQUESTER_SEQUENCE_NUMBER_MAX) * (rand()/(RAND_MAX + 1.0)));
  
  UDM_FIID_OBJ_CREATE_CLEANUP(obj_cmd_rq, tmpl_cmd_get_channel_authentication_capabilities_v20_rq);
  UDM_FIID_OBJ_CREATE_CLEANUP(obj_cmd_rs, tmpl_cmd_get_channel_authentication_capabilities_v20_rs);
  
  UDM_ERR_CLEANUP (!(fill_cmd_get_channel_authentication_capabilities_v20 (IPMI_CHANNEL_NUMBER_CURRENT_CHANNEL,
                                                                           dev->io.outofband.privilege_level,
                                                                           IPMI_GET_IPMI_V20_EXTENDED_DATA,
                                                                           obj_cmd_rq) < 0));

  /* This portion of the protocol is sent via IPMI 1.5 */
  if (ipmi_lan_cmd_wrapper (dev,
                            0,
                            IPMI_BMC_IPMB_LUN_BMC,
                            IPMI_NET_FN_APP_RQ,
                            IPMI_AUTHENTICATION_TYPE_NONE,
                            NULL,
                            0,
                            &(dev->io.outofband.rq_seq),
                            NULL,
                            0,
                            obj_cmd_rq,
                            obj_cmd_rs) < 0)
    goto cleanup;

  UDM_FIID_OBJ_GET_CLEANUP (obj_cmd_rs,
                            "authentication_status.anonymous_login",
                            &authentication_status_anonymous_login);
  UDM_FIID_OBJ_GET_CLEANUP (obj_cmd_rs, 
                            "authentication_status.null_username",
                            &authentication_status_null_username);
  UDM_FIID_OBJ_GET_CLEANUP (obj_cmd_rs, 
                            "authentication_status.non_null_username",
                            &authentication_status_non_null_username);
  UDM_FIID_OBJ_GET_CLEANUP (obj_cmd_rs, 
                            "authentication_status.k_g",
                            &authentication_status_k_g);
  UDM_FIID_OBJ_GET_CLEANUP (obj_cmd_rs, 
                            "authentication_type.ipmi_v2.0_extended_capabilities_available",
                            &ipmi_v20_extended_capabilities_available);
  UDM_FIID_OBJ_GET_CLEANUP (obj_cmd_rs, 
                            "channel_supports_ipmi_v2.0_connections",
                            &channel_supports_ipmi_v20_connections);

  if (!ipmi_v20_extended_capabilities_available
      || !channel_supports_ipmi_v20_connections)
    {
      dev->errnum = IPMI_ERR_IPMI_2_0_UNAVAILABLE;
      goto cleanup;
    }

  /* IPMI Workaround
   *
   * Discovered on an ASUS P5M2 motherboard.
   *
   * The ASUS motherboard reports incorrect settings of anonymous
   * vs. null vs non-null username capabilities.  The workaround is to
   * skip all these checks.
   *
   * Discovered on an ASUS P5MT-R motherboard
   *
   * K_g status is reported incorrectly too.  Again, skip the checks.
   */
  if (!(dev->workaround_flags & IPMI_WORKAROUND_FLAGS_AUTHENTICATION_CAPABILITIES))
    {
      if ((!strlen(dev->io.outofband.username) && !strlen(dev->io.outofband.password)
           && !authentication_status_anonymous_login)
          || (!strlen(dev->io.outofband.username)
              && !authentication_status_anonymous_login
              && !authentication_status_null_username)
          || (strlen(dev->io.outofband.username)
              && !authentication_status_non_null_username))
        {
          dev->errnum = IPMI_ERR_USERNAME_INVALID;
          goto cleanup;
        }
      
      if ((!dev->io.outofband.k_g_configured && authentication_status_k_g)
          || (dev->io.outofband.k_g_configured && !authentication_status_k_g))
        {
          UDM_ERR_SET_ERRNUM_CLEANUP(IPMI_ERR_K_G_INVALID);
          UDM_ERR_LOG_CLEANUP(0);
        }
    }

  UDM_FIID_OBJ_DESTROY(obj_cmd_rq);
  UDM_FIID_OBJ_DESTROY(obj_cmd_rs);

  UDM_FIID_OBJ_CREATE_CLEANUP (obj_cmd_rq, tmpl_rmcpplus_open_session_request);
  UDM_FIID_OBJ_CREATE_CLEANUP (obj_cmd_rs, tmpl_rmcpplus_open_session_response);

  message_tag = (uint8_t)rand();

  /* In IPMI 2.0, session_ids of 0 are special */
  do
    {
      UDM_ERR_CLEANUP (!(ipmi_get_random((uint8_t *)&(dev->io.outofband.remote_console_session_id),
                                         sizeof(dev->io.outofband.remote_console_session_id)) < 0));
    } while (!dev->io.outofband.remote_console_session_id);

  UDM_ERR_CLEANUP (!(ipmi_cipher_suite_id_to_algorithms(dev->io.outofband.cipher_suite_id,
                                                        &(dev->io.outofband.authentication_algorithm),
                                                        &(dev->io.outofband.integrity_algorithm),
                                                        &(dev->io.outofband.confidentiality_algorithm)) < 0));

  /* IPMI Workaround (achu)
   *
   * Discovered on SE7520AF2 with Intel Server Management Module
   * (Professional Edition)
   *
   * The Intel's return IPMI_PRIVILEGE_LEVEL_HIGHEST_LEVEL instead
   * of an actual privilege.
   */
  if (dev->workaround_flags & IPMI_WORKAROUND_FLAGS_INTEL_2_0_SESSION)
    requested_maximum_privilege = dev->io.outofband.privilege_level;
  else
    requested_maximum_privilege = IPMI_PRIVILEGE_LEVEL_HIGHEST_LEVEL;

  UDM_ERR_CLEANUP (!(fill_rmcpplus_open_session (message_tag,
                                                 requested_maximum_privilege,
                                                 dev->io.outofband.remote_console_session_id,
                                                 dev->io.outofband.authentication_algorithm,
                                                 dev->io.outofband.integrity_algorithm,
                                                 dev->io.outofband.confidentiality_algorithm,
                                                 obj_cmd_rq)));

  if (ipmi_lan_2_0_cmd_wrapper (dev,
                                IPMI_BMC_IPMB_LUN_BMC, /* doesn't actually matter here */
                                IPMI_NET_FN_APP_RQ, /* doesn't actually matter here */
                                IPMI_PAYLOAD_TYPE_RMCPPLUS_OPEN_SESSION_REQUEST,
                                IPMI_PAYLOAD_FLAG_UNAUTHENTICATED,
                                IPMI_PAYLOAD_FLAG_UNENCRYPTED,
                                &message_tag,
                                NULL,
                                0,
                                NULL,
                                IPMI_AUTHENTICATION_ALGORITHM_RAKP_NONE,
                                IPMI_INTEGRITY_ALGORITHM_NONE,
                                IPMI_CONFIDENTIALITY_ALGORITHM_NONE,
                                NULL,
                                0,
                                NULL,
                                0,
                                NULL,
                                0,
                                obj_cmd_rq,
                                obj_cmd_rs) < 0)
    goto cleanup;
  
  UDM_FIID_OBJ_GET_CLEANUP (obj_cmd_rs,
                            "rmcpplus_status_code",
                            &val);
  rmcpplus_status_code = val;
  
  if (rmcpplus_status_code != RMCPPLUS_STATUS_NO_ERRORS)
    {
      if (rmcpplus_status_code == RMCPPLUS_STATUS_NO_CIPHER_SUITE_MATCH_WITH_PROPOSED_SECURITY_ALGORITHMS)
        UDM_ERR_SET_ERRNUM_CLEANUP(IPMI_ERR_CIPHER_SUITE_ID_UNAVAILABLE);
      else if (rmcpplus_status_code == RMCPPLUS_STATUS_INSUFFICIENT_RESOURCES_TO_CREATE_A_SESSION
               || rmcpplus_status_code == RMCPPLUS_STATUS_INSUFFICIENT_RESOURCES_TO_CREATE_A_SESSION_AT_THE_REQUESTED_TIME)
        UDM_ERR_SET_ERRNUM_CLEANUP(IPMI_ERR_BMC_BUSY);
      else
        UDM_ERR_SET_ERRNUM_CLEANUP(IPMI_ERR_BAD_RMCPPLUS_STATUS_CODE);
      UDM_ERR_LOG_CLEANUP(0);
    }

  /* Check if we can eventually authentication at the privilege we want */
  UDM_FIID_OBJ_GET_CLEANUP (obj_cmd_rs,
                            "maximum_privilege_level",
                            &val);
  maximum_privilege_level = val;
  
  /* IPMI Workaround (achu)
   *
   * Discovered on SE7520AF2 with Intel Server Management Module
   * (Professional Edition)
   *
   * The Intel's return IPMI_PRIVILEGE_LEVEL_HIGHEST_LEVEL instead
   * of an actual privilege.
   */

  if (!((dev->io.outofband.privilege_level == IPMI_PRIVILEGE_LEVEL_USER
         && (maximum_privilege_level == IPMI_PRIVILEGE_LEVEL_USER
             || maximum_privilege_level == IPMI_PRIVILEGE_LEVEL_OPERATOR
             || maximum_privilege_level == IPMI_PRIVILEGE_LEVEL_ADMIN
             || maximum_privilege_level == IPMI_PRIVILEGE_LEVEL_OEM))
        || (dev->io.outofband.privilege_level == IPMI_PRIVILEGE_LEVEL_OPERATOR
            && (maximum_privilege_level == IPMI_PRIVILEGE_LEVEL_OPERATOR
                || maximum_privilege_level == IPMI_PRIVILEGE_LEVEL_ADMIN
                || maximum_privilege_level == IPMI_PRIVILEGE_LEVEL_OEM))
        || (dev->io.outofband.privilege_level == IPMI_PRIVILEGE_LEVEL_ADMIN
            && (maximum_privilege_level == IPMI_PRIVILEGE_LEVEL_ADMIN
                || maximum_privilege_level == IPMI_PRIVILEGE_LEVEL_OEM))
        || ((dev->workaround_flags & IPMI_WORKAROUND_FLAGS_INTEL_2_0_SESSION)
            && (maximum_privilege_level == IPMI_PRIVILEGE_LEVEL_HIGHEST_LEVEL))))
    {
      UDM_ERR_SET_ERRNUM_CLEANUP(IPMI_ERR_PRIVILEGE_LEVEL_CANNOT_BE_OBTAINED);
      UDM_ERR_LOG_CLEANUP(0);
    }

  UDM_FIID_OBJ_GET_CLEANUP (obj_cmd_rs,
                            "managed_system_session_id",
                            &val);
  dev->io.outofband.managed_system_session_id = val;

  UDM_FIID_OBJ_DESTROY(obj_cmd_rq);
  UDM_FIID_OBJ_DESTROY(obj_cmd_rs);

  UDM_FIID_OBJ_CREATE_CLEANUP (obj_cmd_rq, tmpl_rmcpplus_rakp_message_1);
  UDM_FIID_OBJ_CREATE_CLEANUP (obj_cmd_rs, tmpl_rmcpplus_rakp_message_2);

  UDM_ERR_CLEANUP (!(ipmi_get_random(remote_console_random_number,
                                     IPMI_REMOTE_CONSOLE_RANDOM_NUMBER_LENGTH) < 0));
  
  if (strlen(dev->io.outofband.username))
    username = dev->io.outofband.username;
  else
    username = NULL;

  /* IPMI Workaround (achu)
   *
   * Discovered on SE7520AF2 with Intel Server Management Module
   * (Professional Edition)
   *
   * The username must be padded despite explicitly not being
   * allowed.  "No Null characters (00h) are allowed in the name".
   * Table 13-11 in the IPMI 2.0 spec.
   */
  if (dev->workaround_flags & IPMI_WORKAROUND_FLAGS_INTEL_2_0_SESSION)
    {
      memset(username_buf, '\0', IPMI_MAX_USER_NAME_LENGTH+1);
      if (username)
        strcpy(username_buf, username);
      username = username_buf;
      username_len = IPMI_MAX_USER_NAME_LENGTH;
    }
  else
    username_len = (username) ? strlen(username) : 0;

  /* achu: Unlike IPMI 1.5, the length of the username must be actual
   * length, it can't be the maximum length.
   */
  UDM_ERR_CLEANUP (!(fill_rmcpplus_rakp_message_1 (message_tag,
                                                   dev->io.outofband.managed_system_session_id,
                                                   remote_console_random_number,
                                                   IPMI_REMOTE_CONSOLE_RANDOM_NUMBER_LENGTH,
                                                   dev->io.outofband.privilege_level,
                                                   IPMI_NAME_ONLY_LOOKUP,
                                                   username,
                                                   username_len,
                                                   obj_cmd_rq)));

  if (ipmi_lan_2_0_cmd_wrapper (dev,
                                IPMI_BMC_IPMB_LUN_BMC, /* doesn't actually matter here */
                                IPMI_NET_FN_APP_RQ, /* doesn't actually matter here */
                                IPMI_PAYLOAD_TYPE_RAKP_MESSAGE_1,
                                IPMI_PAYLOAD_FLAG_UNAUTHENTICATED,
                                IPMI_PAYLOAD_FLAG_UNENCRYPTED,
                                &message_tag,
                                NULL,
                                0,
                                NULL,
                                IPMI_AUTHENTICATION_ALGORITHM_RAKP_NONE,
                                IPMI_INTEGRITY_ALGORITHM_NONE,
                                IPMI_CONFIDENTIALITY_ALGORITHM_NONE,
                                NULL,
                                0,
                                NULL,
                                0,
                                NULL,
                                0,
                                obj_cmd_rq,
                                obj_cmd_rs) < 0)
    goto cleanup;

  UDM_FIID_OBJ_GET_CLEANUP (obj_cmd_rs,
                            "rmcpplus_status_code",
                            &val);
  rmcpplus_status_code = val;

  if (rmcpplus_status_code != RMCPPLUS_STATUS_NO_ERRORS)
    {
      if (rmcpplus_status_code == RMCPPLUS_STATUS_UNAUTHORIZED_NAME)
        UDM_ERR_SET_ERRNUM_CLEANUP(IPMI_ERR_USERNAME_INVALID);
      else if (rmcpplus_status_code == RMCPPLUS_STATUS_UNAUTHORIZED_ROLE_OR_PRIVILEGE_LEVEL_REQUESTED)
        UDM_ERR_SET_ERRNUM_CLEANUP(IPMI_ERR_PRIVILEGE_LEVEL_CANNOT_BE_OBTAINED);
      else if (rmcpplus_status_code == RMCPPLUS_STATUS_INSUFFICIENT_RESOURCES_TO_CREATE_A_SESSION
               || rmcpplus_status_code == RMCPPLUS_STATUS_INSUFFICIENT_RESOURCES_TO_CREATE_A_SESSION_AT_THE_REQUESTED_TIME)
        UDM_ERR_SET_ERRNUM_CLEANUP(IPMI_ERR_BMC_BUSY);
      else
        UDM_ERR_SET_ERRNUM_CLEANUP(IPMI_ERR_BAD_RMCPPLUS_STATUS_CODE);
      UDM_ERR_LOG_CLEANUP(0);
    }

  UDM_FIID_OBJ_GET_DATA_LEN_CLEANUP (managed_system_random_number_len,
                                     obj_cmd_rs,
                                     "managed_system_random_number",
                                     managed_system_random_number,
                                     IPMI_MANAGED_SYSTEM_RANDOM_NUMBER_LENGTH);

  UDM_FIID_OBJ_GET_DATA_LEN_CLEANUP (managed_system_guid_len,
                                     obj_cmd_rs,
                                     "managed_system_guid",
                                     managed_system_guid,
                                     IPMI_MANAGED_SYSTEM_GUID_LENGTH);

  if (managed_system_random_number_len != IPMI_MANAGED_SYSTEM_RANDOM_NUMBER_LENGTH
      || managed_system_guid_len != IPMI_MANAGED_SYSTEM_GUID_LENGTH)
    {
      UDM_ERR_SET_ERRNUM_CLEANUP(IPMI_ERR_IPMI_ERROR);
      UDM_ERR_LOG_CLEANUP(0);
    }


  /* IPMI Workaround (achu)
   *
   * Discovered on SE7520AF2 with Intel Server Management Module
   * (Professional Edition)
   *
   * The username must be padded despite explicitly not being
   * allowed.  "No Null characters (00h) are allowed in the name".
   * Table 13-11 in the IPMI 2.0 spec.
   */

  if (dev->workaround_flags & IPMI_WORKAROUND_FLAGS_INTEL_2_0_SESSION)
    {
      memset(username_buf, '\0', IPMI_MAX_USER_NAME_LENGTH+1);
      if (strlen(dev->io.outofband.username))
        strcpy(username_buf, dev->io.outofband.username);
      username = username_buf;
      username_len = IPMI_MAX_USER_NAME_LENGTH;
    }
  else
    {
      if (strlen(dev->io.outofband.username))
        username = dev->io.outofband.username;
      else
        username = NULL;
      username_len = (username) ? strlen(username) : 0;
    }

  if (strlen(dev->io.outofband.password))
    password = dev->io.outofband.password;
  else
    password = NULL;
  password_len = (password) ? strlen(password) : 0;

  /* IPMI Workaround (achu)
   *
   * Discovered on SE7520AF2 with Intel Server Management Module
   * (Professional Edition)
   *
   * When the authentication algorithm is HMAC-MD5-128 and the
   * password is greater than 16 bytes, the Intel BMC truncates the
   * password to 16 bytes when generating keys, hashes, etc.  So we
   * have to do the same when generating keys, hashes, etc.
   */
  if (dev->workaround_flags & IPMI_WORKAROUND_FLAGS_INTEL_2_0_SESSION
      && dev->io.outofband.authentication_algorithm == IPMI_AUTHENTICATION_ALGORITHM_RAKP_HMAC_MD5
      && password_len > IPMI_1_5_MAX_PASSWORD_LENGTH)
    password_len = IPMI_1_5_MAX_PASSWORD_LENGTH;

  if (dev->workaround_flags & IPMI_WORKAROUND_FLAGS_SUPERMICRO_2_0_SESSION)
    {
      uint8_t keybuf[IPMI_PKT_LEN];
      int32_t keybuf_len;

      /* IPMI Workaround (achu)
       *
       * Discovered on Supermicro H8QME with SIMSO daughter card.
       *
       * The IPMI 2.0 packet responses for RAKP 2 have payload lengths
       * that are off by 1 (i.e. if the payload length should be X,
       * the payload length returned in the packet is X + 1)
       *
       * We fix/adjust for the situation here.
       */

      UDM_FIID_OBJ_GET_DATA_LEN_CLEANUP (keybuf_len,
                                         obj_cmd_rs,
                                         "key_exchange_authentication_code",
                                         keybuf,
                                         IPMI_PKT_LEN);

      if (dev->io.outofband.authentication_algorithm == IPMI_AUTHENTICATION_ALGORITHM_RAKP_NONE
          && keybuf_len == 1)
        UDM_FIID_OBJ_CLEAR_FIELD_CLEANUP (obj_cmd_rs, "key_exchange_authentication_code");
      else if (dev->io.outofband.authentication_algorithm == IPMI_AUTHENTICATION_ALGORITHM_RAKP_HMAC_SHA1
               && keybuf_len == (IPMI_HMAC_SHA1_DIGEST_LENGTH + 1))
        UDM_FIID_OBJ_SET_DATA_CLEANUP (obj_cmd_rs,
                                       "key_exchange_authentication_code",
                                       keybuf,
                                       IPMI_HMAC_SHA1_DIGEST_LENGTH);
      else if (dev->io.outofband.authentication_algorithm == IPMI_AUTHENTICATION_ALGORITHM_RAKP_HMAC_MD5
               && keybuf_len == (IPMI_HMAC_MD5_DIGEST_LENGTH + 1))
        UDM_FIID_OBJ_SET_DATA_CLEANUP (obj_cmd_rs,
                                       "key_exchange_authentication_code",
                                       keybuf,
                                       IPMI_HMAC_MD5_DIGEST_LENGTH);
    }

  /* IPMI Workaround (achu)
   *
   * Discovered on Sun Fire 4100.
   *
   * The key exchange authentication code is the wrong length.  We
   * need to shorten it.
   *
   * Notes: Cipher suite 1,2,3 are the ones that use HMAC-SHA1 and
   * have the problem.
   */
  if (dev->workaround_flags & IPMI_WORKAROUND_FLAGS_SUN_2_0_SESSION
      && (dev->io.outofband.authentication_algorithm == IPMI_AUTHENTICATION_ALGORITHM_RAKP_HMAC_SHA1))
    {
      uint8_t buf[IPMI_MAX_KEY_EXCHANGE_AUTHENTICATION_CODE_LENGTH];
      int32_t buf_len;

      UDM_FIID_OBJ_GET_DATA_LEN_CLEANUP (buf_len,
                                         obj_cmd_rs,
                                         "key_exchange_authentication_code",
                                         buf,
                                         IPMI_MAX_KEY_EXCHANGE_AUTHENTICATION_CODE_LENGTH);

      if (buf_len == (IPMI_HMAC_SHA1_DIGEST_LENGTH + 1))
        {
          UDM_FIID_OBJ_CLEAR_FIELD_CLEANUP (obj_cmd_rs,
                                            "key_exchange_authentication_code");
          UDM_FIID_OBJ_SET_DATA_CLEANUP (obj_cmd_rs,
                                         "key_exchange_authentication_code",
                                         buf,
                                         IPMI_HMAC_SHA1_DIGEST_LENGTH);
        }
    }

  UDM_ERR_CLEANUP (!((ret = ipmi_rmcpplus_check_rakp_2_key_exchange_authentication_code (dev->io.outofband.authentication_algorithm,
                                                                                         (uint8_t *)password,
                                                                                         password_len,
                                                                                         dev->io.outofband.remote_console_session_id,
                                                                                         dev->io.outofband.managed_system_session_id,
                                                                                         remote_console_random_number,
                                                                                         IPMI_REMOTE_CONSOLE_RANDOM_NUMBER_LENGTH,
                                                                                         managed_system_random_number,
                                                                                         managed_system_random_number_len,
                                                                                         managed_system_guid,
                                                                                         managed_system_guid_len,
                                                                                         IPMI_NAME_ONLY_LOOKUP,
                                                                                         dev->io.outofband.privilege_level,
                                                                                         username,
                                                                                         username_len,
                                                                                         obj_cmd_rs)) < 0));
  
  if (!ret)
    {
      UDM_ERR_SET_ERRNUM_CLEANUP(IPMI_ERR_PASSWORD_INVALID);
      UDM_ERR_LOG_CLEANUP(0);
    }

  if (strlen(dev->io.outofband.username))
    username = dev->io.outofband.username;
  else
    username = NULL;
  username_len = (username) ? strlen(username) : 0;

  if (strlen(dev->io.outofband.password))
    password = dev->io.outofband.password;
  else
    password = NULL;
  password_len = (password) ? strlen(password) : 0;

  /* IPMI Workaround (achu)
   *
   * Discovered on SE7520AF2 with Intel Server Management Module
   * (Professional Edition)
   *
   * When the authentication algorithm is HMAC-MD5-128 and the
   * password is greater than 16 bytes, the Intel BMC truncates the
   * password to 16 bytes when generating keys, hashes, etc.  So we
   * have to do the same when generating keys, hashes, etc.
   */
  if ((dev->workaround_flags & IPMI_WORKAROUND_FLAGS_INTEL_2_0_SESSION)
      && dev->io.outofband.authentication_algorithm == IPMI_AUTHENTICATION_ALGORITHM_RAKP_HMAC_MD5
      && password_len > IPMI_1_5_MAX_PASSWORD_LENGTH)
    password_len = IPMI_1_5_MAX_PASSWORD_LENGTH;

  /* IPMI Workaround (achu)
   *
   * Discovered on SE7520AF2 with Intel Server Management Module
   * (Professional Edition)
   *
   * For some reason we have to create this key with the name only
   * lookup turned off.  I was skeptical about this actually being
   * a bug until I saw that the ipmitool folks implemented the
   * same workaround.
   */
  if (dev->workaround_flags & IPMI_WORKAROUND_FLAGS_INTEL_2_0_SESSION)
    name_only_lookup = IPMI_USER_NAME_PRIVILEGE_LOOKUP;
  else
    name_only_lookup = IPMI_NAME_ONLY_LOOKUP;

  UDM_ERR_CLEANUP (!((key_exchange_authentication_code_len = ipmi_calculate_rakp_3_key_exchange_authentication_code(dev->io.outofband.authentication_algorithm,
                                                                                                                    (uint8_t *)password,
                                                                                                                    password_len,
                                                                                                                    managed_system_random_number,
                                                                                                                    managed_system_random_number_len,
                                                                                                                    dev->io.outofband.remote_console_session_id,
                                                                                                                    name_only_lookup,
                                                                                                                    dev->io.outofband.privilege_level,
                                                                                                                    username,
                                                                                                                    username_len,
                                                                                                                    key_exchange_authentication_code,
                                                                                                                    IPMI_MAX_KEY_EXCHANGE_AUTHENTICATION_CODE_LENGTH)) < 0));

  UDM_FIID_OBJ_DESTROY(obj_cmd_rq);
  UDM_FIID_OBJ_DESTROY(obj_cmd_rs);

  UDM_FIID_OBJ_CREATE_CLEANUP (obj_cmd_rq, tmpl_rmcpplus_rakp_message_3);
  UDM_FIID_OBJ_CREATE_CLEANUP (obj_cmd_rs, tmpl_rmcpplus_rakp_message_4);
  
  UDM_ERR_CLEANUP (!(fill_rmcpplus_rakp_message_3(message_tag,
                                                  RMCPPLUS_STATUS_NO_ERRORS,
                                                  dev->io.outofband.managed_system_session_id,
                                                  key_exchange_authentication_code,
                                                  key_exchange_authentication_code_len,
                                                  obj_cmd_rq) < 0));

  if (ipmi_lan_2_0_cmd_wrapper (dev,
                                IPMI_BMC_IPMB_LUN_BMC, /* doesn't actually matter here */
                                IPMI_NET_FN_APP_RQ, /* doesn't actually matter here */
                                IPMI_PAYLOAD_TYPE_RAKP_MESSAGE_3,
                                IPMI_PAYLOAD_FLAG_UNAUTHENTICATED,
                                IPMI_PAYLOAD_FLAG_UNENCRYPTED,
                                &message_tag,
                                NULL,
                                0,
                                NULL,
                                IPMI_AUTHENTICATION_ALGORITHM_RAKP_NONE,
                                IPMI_INTEGRITY_ALGORITHM_NONE,
                                IPMI_CONFIDENTIALITY_ALGORITHM_NONE,
                                NULL,
                                0,
                                NULL,
                                0,
                                NULL,
                                0,
                                obj_cmd_rq,
                                obj_cmd_rs) < 0)
    goto cleanup;

  UDM_FIID_OBJ_GET_CLEANUP (obj_cmd_rs,
                            "rmcpplus_status_code",
                            &val);
  rmcpplus_status_code = val;
  
  if (rmcpplus_status_code != RMCPPLUS_STATUS_NO_ERRORS)
    {
      if (rmcpplus_status_code == RMCPPLUS_STATUS_INSUFFICIENT_RESOURCES_TO_CREATE_A_SESSION
          || rmcpplus_status_code == RMCPPLUS_STATUS_INSUFFICIENT_RESOURCES_TO_CREATE_A_SESSION_AT_THE_REQUESTED_TIME)
        UDM_ERR_SET_ERRNUM_CLEANUP(IPMI_ERR_BMC_BUSY);
      else
        UDM_ERR_SET_ERRNUM_CLEANUP(IPMI_ERR_BAD_RMCPPLUS_STATUS_CODE);
      UDM_ERR_LOG_CLEANUP(0);
    }
  
  UDM_ERR_CLEANUP (!(ipmi_calculate_rmcpplus_session_keys(dev->io.outofband.authentication_algorithm,
                                                          dev->io.outofband.integrity_algorithm,
                                                          dev->io.outofband.confidentiality_algorithm,
                                                          strlen(dev->io.outofband.password) ? (uint8_t *)dev->io.outofband.password : NULL,
                                                          strlen(dev->io.outofband.password),
                                                          (dev->io.outofband.k_g_configured) ? dev->io.outofband.k_g : NULL,
                                                          (dev->io.outofband.k_g_configured) ? IPMI_MAX_K_G_LENGTH : 0,
                                                          remote_console_random_number,
                                                          IPMI_REMOTE_CONSOLE_RANDOM_NUMBER_LENGTH,
                                                          managed_system_random_number,
                                                          IPMI_MANAGED_SYSTEM_RANDOM_NUMBER_LENGTH,
                                                          IPMI_NAME_ONLY_LOOKUP,
                                                          dev->io.outofband.privilege_level,
                                                          strlen(dev->io.outofband.username) ? dev->io.outofband.username : NULL,
                                                          strlen(dev->io.outofband.username),
                                                          &(dev->io.outofband.sik_key_ptr),
                                                          &(dev->io.outofband.sik_key_len),
                                                          &(dev->io.outofband.integrity_key_ptr),
                                                          &(dev->io.outofband.integrity_key_len),
                                                          &(dev->io.outofband.confidentiality_key_ptr),
                                                          &(dev->io.outofband.confidentiality_key_len)) < 0));

  /* IPMI Workaround (achu)
   *
   * Discovered on SE7520AF2 with Intel Server Management Module
   * (Professional Edition)
   *
   * For some reason, the intel ipmi 2.0 responds with the integrity
   * check value based on the integrity algorithm instead of the
   * authentication algorithm.
   *
   * Thanks to the ipmitool folks (ipmitool.sourceforge.net) for this
   * one.  Would have taken me awhile to figure this one out :-)
   */

  if (dev->workaround_flags & IPMI_WORKAROUND_FLAGS_INTEL_2_0_SESSION)
    {
      if (dev->io.outofband.integrity_algorithm == IPMI_INTEGRITY_ALGORITHM_NONE)
        authentication_algorithm = IPMI_AUTHENTICATION_ALGORITHM_RAKP_NONE;
      else if (dev->io.outofband.integrity_algorithm == IPMI_INTEGRITY_ALGORITHM_HMAC_SHA1_96)
        authentication_algorithm = IPMI_AUTHENTICATION_ALGORITHM_RAKP_HMAC_SHA1;
      else if (dev->io.outofband.integrity_algorithm == IPMI_INTEGRITY_ALGORITHM_HMAC_MD5_128)
        authentication_algorithm = IPMI_AUTHENTICATION_ALGORITHM_RAKP_HMAC_MD5;
      else if (dev->io.outofband.integrity_algorithm == IPMI_INTEGRITY_ALGORITHM_MD5_128)
        /* achu: I have thus far been unable to reverse engineer this
         * corner case.  So we're just going to accept whatever the
         * remote BMC gives us.  This has been documented in the
         * manpage.
         */
        assume_rakp_4_success++;
    }
  else
    authentication_algorithm = dev->io.outofband.authentication_algorithm;

  if (!assume_rakp_4_success)
    {
      UDM_ERR_CLEANUP (!((ret = ipmi_rmcpplus_check_rakp_4_integrity_check_value(authentication_algorithm,
                                                                                 dev->io.outofband.sik_key_ptr,
                                                                                 dev->io.outofband.sik_key_len,
                                                                                 remote_console_random_number,
                                                                                 IPMI_REMOTE_CONSOLE_RANDOM_NUMBER_LENGTH,
                                                                                 dev->io.outofband.managed_system_session_id,
                                                                                 managed_system_guid,
                                                                                 managed_system_guid_len,
                                                                                 obj_cmd_rs)) < 0));
      
      if (!ret)
        {
          UDM_ERR_SET_ERRNUM_CLEANUP(IPMI_ERR_K_G_INVALID);
          UDM_ERR_LOG_CLEANUP(0);
        }
    }
  
  UDM_FIID_OBJ_DESTROY(obj_cmd_rq);
  UDM_FIID_OBJ_DESTROY(obj_cmd_rs);

  /* achu: At this point in time, the session is actually setup
   * legitimately, so we can use the actual set session privilege
   * level UDM function.
   */

  UDM_FIID_OBJ_CREATE_CLEANUP (obj_cmd_rs, tmpl_cmd_set_session_privilege_level_rs);

  if (ipmi_cmd_set_session_privilege_level (dev, 
                                            dev->io.outofband.privilege_level,
                                            obj_cmd_rs) < 0)
    {
      if (dev->errnum == IPMI_ERR_BAD_COMPLETION_CODE)
	{
	  if (ipmi_check_completion_code(obj_cmd_rs, IPMI_COMP_CODE_RQ_LEVEL_NOT_AVAILABLE_FOR_USER) == 1
	      || ipmi_check_completion_code(obj_cmd_rs, IPMI_COMP_CODE_RQ_LEVEL_EXCEEDS_USER_PRIVILEGE_LIMIT) == 1)
	    UDM_ERR_SET_ERRNUM_CLEANUP(IPMI_ERR_PRIVILEGE_LEVEL_CANNOT_BE_OBTAINED);
	}
      UDM_ERR_LOG_CLEANUP(0);
    }

  rv = 0;
 cleanup:
  UDM_FIID_OBJ_DESTROY(obj_cmd_rq);
  UDM_FIID_OBJ_DESTROY(obj_cmd_rs);
  return (rv);
}

int8_t 
ipmi_lan_2_0_close_session (ipmi_device_t dev)
{
  fiid_obj_t obj_cmd_rs = NULL;
  int8_t rv = -1;

  assert (dev 
          && dev->magic == IPMI_UDM_DEVICE_MAGIC
          && dev->type == IPMI_DEVICE_LAN_2_0
          && dev->io.outofband.sockfd);

  UDM_FIID_OBJ_CREATE (obj_cmd_rs, tmpl_cmd_close_session_rs);

  if (ipmi_cmd_close_session(dev,
                             dev->io.outofband.managed_system_session_id,
                             obj_cmd_rs) < 0)
    goto cleanup;
  
  rv = 0;
 cleanup:
  UDM_FIID_OBJ_DESTROY(obj_cmd_rs);
  return (rv);
}
