/*
 * alarms.c
 *
 * This tool reads and sets the alarms panel on an Intel Telco chassis.
 * Note that the Intel Server Management software will set these alarms
 * based on firmware-detected thresholds and events.
 * 
 * Author:  Andy Cress  arcress@users.sourceforge.net
 * Copyright (c) 2003-2006 Intel Corporation. 
 *
 * 02/25/03 Andy Cress - created
 * 04/08/03 Andy Cress - added -i for ChassisIdentify
 * 04/30/03 Andy Cress - only try to set ID on/off if option specified
 * 01/20/04 Andy Cress - mods for mBMC w Chesnee platform
 * 05/05/04 Andy Cress - call ipmi_close before exit
 * 10/11/04 Andy Cress 1.4 - if -o set relays too (fsetall)
 * 11/01/04 Andy Cress 1.5 - add -N / -R for remote nodes
 * 03/07/05 Andy Cress 1.6 - add bus for Intel TIGI2U
 * 03/28/05 Andy Cress 1.7 - add check for BMC TAM if setting alarms
 * 06/22/05 Andy Cress 1.8 - adding fpicmg for ATCA alarm LEDs
 * 03/17/06 Andy Cress 1.9 - adding BUS_ID7 for Harbison 
 * 04/20/07 Andy Cress 1.25 - adding disk Enclosure HSC LEDs
 */
/*M*
Copyright (c) 2003-2006, Intel Corporation
All rights reserved.

Redistribution and use in source and binary forms, with or without 
modification, are permitted provided that the following conditions are met:

  a.. Redistributions of source code must retain the above copyright notice, 
      this list of conditions and the following disclaimer. 
  b.. Redistributions in binary form must reproduce the above copyright notice,
      this list of conditions and the following disclaimer in the documentation 
      and/or other materials provided with the distribution. 
  c.. Neither the name of Intel Corporation nor the names of its contributors 
      may be used to endorse or promote products derived from this software 
      without specific prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR 
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *M*/
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#ifdef WIN32
#include "getopt.h"
#else
#include <getopt.h>
#endif
#include "ipmicmd.h"
 
/*
 * Global variables 
 */
static char * progver   = "2.12";
static char * progname  = "alarms";
#ifdef WIN32
static char * tamcmd = "bmcTamActive";
#else
static char * tamcmd = "/usr/local/tam/bin/bmcTamActive";
#endif
static char   fdebug    = 0;
static char   fbmctam   = 0;
static char   fpicmg    = 0;
static char   fHasAlarms = 1;
static char   fChesnee   = 0;
static char   fHasEnc    = 0;     /* Has disk Enclosure HSC? */
static uchar ipmi_maj, ipmi_min;

#define PRIVATE_BUS_ID      0x03 // w Sahalee,  the 8574 is on Private Bus 1
#define PRIVATE_BUS_ID5     0x05 // for Intel TIGI2U
#define PRIVATE_BUS_ID7     0x07 // for Intel Harbison
#define PERIPHERAL_BUS_ID   0x24 // w mBMC, the 8574 is on the Peripheral Bus
#define ALARMS_PANEL_WRITE  0x40 
#define ALARMS_PANEL_READ   0x41
#define DISK_LED_WRITE      0x44 // only used for Chesnee mBMC
#define DISK_LED_READ       0x45 // only used for Chesnee mBMC
#define ENC_LED_WRITE       0x21 // only used for Ballenger-CT HSC
#define ENC_LED_READ        0x20 // only used for Ballenger-CT HSC

#define HSC_SA                    0xC0
#define NETFN_ENC                 0x30
#define NETFN_PICMG               0x2c
#define PICMG_GET_LED_PROPERTIES  0x05
#define PICMG_SET_LED_STATE       0x07
#define PICMG_GET_LED_STATE       0x08

static uchar busid = PRIVATE_BUS_ID;

static uchar get_diskleds(void)
{
	uchar inputData[4];
	uchar responseData[16];
	int responseLength = 4;
	uchar completionCode;
	int ret;

	inputData[0] = busid;
	inputData[1] = DISK_LED_READ;
	inputData[2] = 0x1;   // return one byte of LED data
	inputData[3] = 0x00;  // init data to zero
        ret = ipmi_cmd(MASTER_WRITE_READ, inputData, 3, responseData,
                        &responseLength, &completionCode, fdebug);
	if (ret != 0) {
           printf("get_diskleds: ret = %d, ccode %02x, leds = %02x\n",
                    ret, completionCode, responseData[0]);
	   return(0);
	   }
	return(responseData[0]);
}  /*end get_diskleds()*/

static int set_diskleds(uchar val)
{
	uchar inputData[4];
	uchar responseData[16];
	int responseLength = 4;
	uchar completionCode;
	int ret;

	inputData[0] = busid;
	inputData[1] = DISK_LED_WRITE;
	inputData[2] = 0x01;   // len = one byte of LED data
	inputData[3] = val;    
        ret = ipmi_cmd(MASTER_WRITE_READ, inputData, 4, responseData,
                        &responseLength, &completionCode, fdebug);
	if (ret != 0) {
           printf("set_diskleds: ret = %d, ccode %02x, leds = %02x\n",
                    ret, completionCode, val);
	   return(0);
	   }
	return(ret);
}  /*end set_diskleds()*/

static void show_diskleds(uchar val)
{
	if (fdebug) printf("diskled = %02x\n",val);
	printf("disk A: ");
	if ((val & 0x20) == 0) printf("present");
	else printf("not present");
	if ((val & 0x02) == 0) printf("/faulted ");
	printf("\ndisk B: ");
	if ((val & 0x10) == 0) printf("present");
	else printf("not present");
	if ((val & 0x01) == 0) printf("/faulted ");
	printf("\n");
}

static int get_enc_leds(uchar *val)
{
    uchar idata[16];
    uchar rdata[16];
    int rlen;
    int rv = 0;
    uchar cc;
    rlen = sizeof(rdata);
    rv = ipmi_cmdraw( ENC_LED_READ, NETFN_ENC, HSC_SA, PUBLIC_BUS,BMC_LUN,
                      idata,0, rdata, &rlen, &cc, fdebug);
    if (fdebug) printf("get_enc_leds() rv=%d cc=%x val=%02x\n",rv,cc,rdata[0]);
    if (rv == 0 && cc != 0) rv = cc;
    if (rv == 0) { /*success*/
       *val = rdata[0];
    }
    return(rv);
}

static int set_enc_leds(uchar val)
{
    uchar idata[16];
    uchar rdata[16];
    int rlen;
    int rv = 0;
    uchar cc;

    idata[0] = val;
    rlen = sizeof(rdata);
    rv = ipmi_cmdraw( ENC_LED_WRITE, NETFN_ENC, HSC_SA, PUBLIC_BUS,BMC_LUN,
                      idata,1, rdata, &rlen, &cc, fdebug);
    if (fdebug) printf("set_enc_leds(%02x) rv = %d, cc = %x\n",val,rv,cc);
    if (rv == 0 && cc != 0) rv = cc;
    return(rv);
}

static void show_enc_leds(uchar val)
{
    char *enc_pattn = "disk slot %d LED:   %s\n";
    if (fdebug) printf("val = %02x\n",val);
    if (val & 0x01) printf(enc_pattn,0,"ON");
    else printf(enc_pattn,0,"off");
    if (val & 0x02) printf(enc_pattn,1,"ON");
    else printf(enc_pattn,1,"off");
    if (val & 0x04) printf(enc_pattn,2,"ON");
    else printf(enc_pattn,2,"off");
    if (val & 0x08) printf(enc_pattn,3,"ON");
    else printf(enc_pattn,3,"off");
    if (val & 0x10) printf(enc_pattn,4,"ON");
    else printf(enc_pattn,4,"off");
    if (val & 0x20) printf(enc_pattn,5,"ON");
    else printf(enc_pattn,5,"off");
#ifdef NEW
    /* Ballenger HSC only supports 6 slots */
    if (val & 0x40) printf(enc_pattn,6,"ON");
    else printf(enc_pattn,6,"off");
    if (val & 0x80) printf(enc_pattn,7,"ON");
    else printf(enc_pattn,7,"off");
#endif
}

static int set_chassis_id(uchar val)
{
	uchar inputData[4];
	uchar responseData[16];
	int responseLength = 4;
	uchar completionCode;
	int ret;
        uchar ilen;

        ilen = 1;
	inputData[0] = val;   /* #seconds to turn on id, 0=turn off */
        /* IPMI 2.0 has an optional 2nd byte, 
         *       if 01, turn ID on indefinitely.  */
        if (val == 255 && ipmi_maj >= 2) {
           inputData[1] = 1;
           ilen = 2;
        }
        /* CHASSIS_IDENTIFY=0x04, using NETFN_CHAS (=00) */
        ret = ipmi_cmd(CHASSIS_IDENTIFY, inputData, ilen, responseData,
                        &responseLength, &completionCode, fdebug);
	if (ret == 0 && completionCode != 0) ret = completionCode;
	if (ret != 0) {
           printf("set_chassis_id: ret = %d, ccode %02x, value = %02x\n",
                    ret, completionCode, val);
	   }
	return(ret);
} /*end set_chassis_id*/


static uchar get_alarms_picmg(uchar picmg_id, uchar fruid)
{
	uchar inputData[4];
	uchar responseData[16];
	int responseLength;
	uchar completionCode;
	int ret, i;

	inputData[0] = picmg_id;
	inputData[1] = fruid;
	inputData[2] = 0;   // blue led
	responseLength = sizeof(responseData);
        ret = ipmi_cmdraw( PICMG_GET_LED_STATE, NETFN_PICMG,
			BMC_SA, PUBLIC_BUS,BMC_LUN,
                        inputData,3, responseData, &responseLength, 
			&completionCode, fdebug);
	if ((ret != 0) || (completionCode != 0)) { 
           printf("get_alarms_picmg: ret = %d, ccode %02x\n",
                    ret, completionCode);
	   return(0);
	   }
	if (fdebug) {
	   printf("get_alarms_picmg: ");
           for (i = 0; i < responseLength; i++)
		printf("%02x ",responseData[i]);
           printf("\n");
	   /* Note: This logic hasn't been verified yet. */
	   if (responseData[2] & 0x04) printf("lamp test on\n");
	   if (responseData[2] & 0x02) { /*if overwrite ok*/
		if (responseData[6] == 0) printf("overwrite off\n");
	   	else printf("overwrite on\n");
	   }
	}
	return(responseData[5]);
}

static int set_alarms_picmg(uchar val, uchar picmg_id, uchar fruid)
{
	uchar inputData[6];
	uchar responseData[16];
	int responseLength;
	uchar completionCode;
	int ret, i;

	inputData[0] = picmg_id;
	inputData[1] = fruid;
	inputData[2] = 0;   // blue led
	inputData[3] = val; 
	inputData[4] = 0;  
	inputData[5] = 1;   // blue 
	responseLength = sizeof(responseData);
        ret = ipmi_cmdraw( PICMG_SET_LED_STATE, NETFN_PICMG,
			BMC_SA, PUBLIC_BUS,BMC_LUN,
                        inputData,6, responseData, &responseLength, 
			&completionCode, fdebug);
	if (ret != 0) {
           printf("set_alarms_picmg: ret = %d, ccode %02x\n",
                    ret, completionCode);
	   return(ret);
	   }
        printf("set_alarms_picmg: ");
        for (i = 0; i < responseLength; i++)
		printf("%02x ",responseData[i]);
        printf("\n");
	return(0);  /* success */
}

static void show_alarms_picmg(uchar val)
{
    if (fdebug) printf("alarm LED state is %02x\n",val);

    if (val == 0) printf("alarm LED is off\n");
    else if (val == 0xFF) printf("alarm LED is ON\n");
    else printf("alarm LED is unknown (%02x)\n",val);
}

static uchar get_alarms(void)
{
	uchar inputData[4];
	uchar responseData[16];
	int responseLength = 4;
	uchar completionCode;
	int ret;

	inputData[0] = busid;
	inputData[1] = ALARMS_PANEL_READ;
	inputData[2] = 0x1;   // return one byte of alarms data
	inputData[3] = 0x00;  // init data to zero
        ret = ipmi_cmd(MASTER_WRITE_READ, inputData, 3, responseData,
                        &responseLength, &completionCode, fdebug);
	if (ret != 0 || completionCode != 0) {
           printf("get_alarms: ret = %d, ccode %02x, alarms = %02x\n",
                    ret, completionCode, responseData[0]);
	   return(0);
	   }
	return(responseData[0]);
}  /*end get_alarms()*/

static int set_alarms(uchar val)
{
	uchar inputData[4];
	uchar responseData[16];
	int responseLength = 4;
	uchar completionCode;
	int ret;

	inputData[0] = busid; 
	inputData[1] = ALARMS_PANEL_WRITE;
	inputData[2] = 0x1;   // one byte of alarms data
	inputData[3] = val;  
        ret = ipmi_cmd(MASTER_WRITE_READ, inputData, 4, responseData,
                        &responseLength, &completionCode, fdebug);
	if (ret != 0) {
           printf("set_alarms: ret = %d, ccode %02x, value = %02x\n",
                    ret, completionCode, val);
	   return(ret);
	   }
	if (completionCode != 0) ret = completionCode;
	return(ret);
}  /*end set_alarms()*/

/* 
 * show_alarms
 *
 * The alarm control/status byte is decoded as follows:
 * bit
 * 7 = reserved, always write 1
 * 6 = LED colors, 1 = amber (default), 0 = red
 *     Colors were added in some later firmware versions, but
 *     not for all platforms.
 * 5 = Minor Relay bit, 0 = on, 1 = off, always write 1
 * 4 = Major Relay bit, 0 = on, 1 = off, always write 1
 * 3 = Minor LED bit, 0 = on, 1 = off
 * 2 = Major LED bit, 0 = on, 1 = off
 * 1 = Critical LED bit, 0 = on, 1 = off
 * 0 = Power LED bit, 0 = on, 1 = off
 *
 * Note that the Power LED is also wired to the System Fault LED 
 * in the back of the system, so this state may be off for Power,
 * but the LED could be lit for a System Fault reason instead.
 */
void show_alarms(uchar val)
{
   char *scrit = "ON ";
   char *smaj = "ON ";
   char *smin = "ON ";
   char *spow = "ON ";
   char *rmaj = "ON";
   char *rmin = "ON";
   if (fdebug) printf("alarms = %02x\n",val);
    
   if (val & 0x01) spow = "off";
   if (val & 0x02) scrit = "off";
   if (val & 0x04) smaj = "off";
   if (val & 0x08) smin = "off";
   printf("Alarm LEDs:   critical = %s major = %s minor = %s power = %s\n",
           scrit,smaj,smin,spow);
   if (val & 0x10) rmaj = "off ";
   if (val & 0x20) rmin = "off ";
   printf("Alarm Relays: major = %s minor = %s\n", rmaj, rmin);
}


#ifdef METACOMMAND
int i_alarms(int argc, char **argv)
#else
#ifdef WIN32
int __cdecl
#else
int
#endif
main(int argc, char **argv)
#endif
{
   int ret = 0;
   char c;
   uchar fsetled = 0;
   uchar fsetall = 0;
   uchar fsetdisk = 0;
   uchar fsetid = 0;
   uchar fcrit = 2;
   uchar fmaj = 2;
   uchar fmin = 2;
   uchar fpow = 2;
   uchar fdiska = 2;
   uchar fdiskb = 2;
   uchar fdiskn = 2;
   uchar ndisk  = 0;
   uchar fid = 0;
   uchar alarms = 0;
   uchar newvalue = 0xff;
   uchar diskled = 0xff;
   uchar encled = 0;
   uchar devrec[16];

   printf("%s ver %s\n", progname,progver);

   while ( (c = getopt( argc, argv,"rxa:b:c:d:m:n:p:i:oEF:P:N:R:U:T:V:J:Y?")) != EOF ) 
      switch(c) {
          case 'r': fsetled=0; fsetid=0; fsetdisk = 0; break; /* read only */
          case 'a': fdiska = atoi(optarg);     /* set disk A LED value */
		    fsetdisk = 1; break; 
          case 'b': fdiskb = atoi(optarg);     /* set disk B LED value */
		    fsetdisk = 2; break; 
          case 'c': fcrit = atoi(optarg);      /* set critical alarm value */
		    fsetled = 1; break; 
          case 'd': ndisk  = optarg[0] & 0x0f; /* set disk N LED on or off */
                    fdiskn = optarg[1] & 0x0f;
		    fsetdisk = 3; break; 
          case 'm': fmaj = atoi(optarg);      /* set major alarm value */
		    fsetled = 1; break; 
          case 'n': fmin = atoi(optarg);      /* set minor alarm value */
		    fsetled = 1; break; 
          case 'p': fpow = atoi(optarg);      /* set power alarm value */
		    fsetled = 1; break; 
          case 'i': fid = atoi(optarg);       /* set chassis id on/off */
		    fsetid=1; break; 
          case 'o': fcrit=0; fmaj=0; fmin=0; fpow=0; /* set all alarms off */
		    fsetdisk = 1; fdiska = 0; fdiskb = 0; fsetall=1;
		    fsetled = 1; fsetid=1; fid=0; break; 
          case 'x': fdebug = 1;     break;  /* debug messages */
          case 'N':    /* nodename */
	  case 'U':    /* remote username */
 	  case 'P':    /* remote password */
 	  case 'R':    /* remote password */ 
          case 'E':    /* get password from IPMI_PASSWORD environment var */
          case 'F':    /* force driver type */
          case 'T':    /* auth type */
          case 'J':    /* cipher suite */
          case 'V':    /* priv level */
 	  case 'Y':    /* prompt for remote password */
                parse_lan_options(c,optarg,fdebug);
                break;
	  default:
                printf("Usage: %s [-abcdimnoprx -NUPRETVFY]\n", progname);
                printf(" where -r   means read-only\n");
                printf("       -i5  sets Chassis ID on for 5 sec\n");
                printf("       -i0  sets Chassis ID off\n");
                printf("       -a1  sets Disk A Fault on\n");
                printf("       -a0  sets Disk A Fault off\n");
                printf("       -b1  sets Disk B Fault on\n");
                printf("       -b0  sets Disk B Fault off\n");
                printf("       -c1  sets Critical Alarm on\n");
                printf("       -c0  sets Critical Alarm off\n");
                printf("       -d31 sets Disk 3 Fault on (disks 0-6)\n");
                printf("       -d30 sets Disk 3 Fault off\n");
                printf("       -m1  sets Major Alarm on\n");
                printf("       -m0  sets Major Alarm off\n");
                printf("       -n1  sets Minor Alarm on\n");
                printf("       -n0  sets Minor Alarm off\n");
                printf("       -p1  sets Power Alarm on\n");
                printf("       -p0  sets Power Alarm off\n");
                printf("       -o   sets all Alarms off\n");
                printf("       -x   show eXtra debug messages\n");
		print_lan_opt_usage();
                exit(1);
      }

   /* 
    * Check the Device ID to determine which bus id to use.
    */
   ret = ipmi_getdeviceid(devrec,16,fdebug);
   if (ret != 0) {
        show_outcome(progname,ret); 
	ipmi_close_();
        exit(1);
   } else {
      int prod_id, vend_id;
      char *pstr;
      int j;

      if (fdebug) {
	printf("devid: ");
	for (j = 0; j < 16; j++) printf("%02x ",devrec[j]);
	printf("\n");
      }
      ipmi_maj = devrec[4] & 0x0f;
      ipmi_min = devrec[4] >> 4;
      prod_id = devrec[9] + (devrec[10] << 8);
      vend_id = devrec[6] + (devrec[7] << 8) + (devrec[8] << 16);
      if (vend_id == VENDOR_NSC)  { /*NSC mBMC, Chesnee*/
	  pstr = "mBMC";
	  busid = PERIPHERAL_BUS_ID;
          fHasAlarms = 1;
          fChesnee = 1;
      } else if (vend_id == VENDOR_INTEL) { /*Intel BMC*/
	  pstr = "BMC";
          switch (prod_id) {
             case 0x0022:   
		busid = PRIVATE_BUS_ID5;  /* Intel TIGI2U */
		fbmctam = 1;  /* Intel TIGI2U may have bmc tam */
                fHasAlarms = 1;
                break;
             case 0x001B:   
	        busid = PRIVATE_BUS_ID;
		fbmctam = 1;  /* Intel TIGPR2U may have bmc tam */
                fHasAlarms = 1;
                break;
             case 0x0808:   
	        fpicmg = 1;   /* Intel ATCA platform, supports PICMG */
                fHasAlarms = 1;
                break;
             case 0x4311:  
	        busid = PERIPHERAL_BUS_ID; /* SJR2 */
	        pstr = "mBMC";
                break;
             case 0x0026:   
             case 0x0028:
             case 0x0811:   
		busid = PRIVATE_BUS_ID7;  /* Intel Harbision */
		fbmctam = 1;  /* Intel TIGW1U may have bmc tam */
                fHasAlarms = 1;
                fHasEnc = 1;
                break;
             default:
	        busid = PRIVATE_BUS_ID;
                break;
          }
      } else {  /* Not NSC or Intel, use defaults */
	  pstr = "BMC";
	  busid = PRIVATE_BUS_ID;
      }
      printf("-- %s version %x.%x, IPMI version %d.%d \n",
             pstr, devrec[2],  devrec[3], ipmi_maj, ipmi_min);
   }

   if (fpicmg) {
     alarms = get_alarms_picmg(0,0);
     show_alarms_picmg(alarms);
   } else {
     alarms = get_alarms();
     if (alarms == 0) {  /* failed to get alarm panel data */
	if (fHasAlarms) {  /* baseboard which may have Telco alarm panel*/
	   printf("Could not obtain Telco LED states, Telco alarm panel "
		"may not be present.\n\n");
        }
        fHasAlarms = 0;
     } else {   /* have Telco alarm panel data */
	show_alarms(alarms);
     }
   }
	
   if (fsetid) {
	printf("Setting id to %d ...\n\n",fid);
	ret = set_chassis_id(fid);
   }
   if (fsetled && fHasAlarms) {
      if (fpicmg) {
         if ((fcrit == 1) || (fmaj == 1) || (fmin == 1) ||
	     (fpow == 1)) newvalue = 0xFF;   /*on*/
         else newvalue = 0;  /*off */
         printf("\nSetting alarm LED to %02x ...\n",newvalue);
         ret = set_alarms_picmg(newvalue,0,0);
         alarms = get_alarms_picmg(0,0);
         show_alarms_picmg(alarms);
      } else { 
         if (fbmctam) {    /* Platform supports BMC Telco Alarms Manager */
#ifdef WIN32
	   /* TODO: Locate TAM command on Windows */
#else
	   /* Check if SDRs have BMC TAM enabled */
	   printf("Checking for BMC TAM ...\n");
	   ret = system(tamcmd);
	   if (fdebug) printf("%s ret = %d\n",tamcmd,ret);
	   if (ret == 256) { 
	      /*If so, print warning, use Intel TAM API instead.*/
	      printf("Warning: BMC TAM is active and managing the LEDs.\n"
		  "Use tamutil or the Intel TAM API to set alarms instead.\n");
	      exit(ret);
	   }
#endif
         }
         if (fsetall) newvalue = 0xFF;   /* alarms and relays */
         else {
           newvalue = alarms;
           if      (fcrit == 1) newvalue &= 0xFD; /*bit1 = 0*/
           else if (fcrit == 0) newvalue |= 0xF2; 
           if      (fmaj == 1)  newvalue &= 0xFB; /*bit2 = 0*/
           else if (fmaj == 0)  newvalue |= 0xF4; 
           if      (fmin == 1)  newvalue &= 0xF7; /*bit3 = 0*/
           else if (fmin == 0)  newvalue |= 0xF8; 
           if      (fpow == 1)  newvalue &= 0xFE; /*bit0 = 0*/
           else if (fpow == 0)  newvalue |= 0xF1; 
         }
         printf("\nSetting alarms to %02x ...\n",newvalue);
         ret = set_alarms(newvalue);
         alarms = get_alarms();
         show_alarms(alarms);
      }  /*end else*/
   } /*end fsetled*/

   if (fHasEnc & !fipmi_lan) {   /* Ballenger disk Enclosure HSC */
      ret = get_enc_leds(&encled);
      if (ret == 0) {
	 show_enc_leds(encled);
         if (fsetdisk) {
	    /* Set fault if user param, and disk is present. */
            if (fsetall) newvalue = 0x00;   /* all LEDs off */
            else {
               newvalue = encled;
	       if (fdiskb == 1)  newvalue |= 0x02;
	       else if (fdiskb == 0) newvalue &= 0xFD;
	       if (fdiska == 1)  newvalue |= 0x01; 
	       else if (fdiska == 0) newvalue &= 0xFE;
               if (fdiskn == 1) newvalue |= (0x01 << ndisk);
               else if (fdiskn == 0) newvalue &= ~(0x01 << ndisk);
            }
	    printf("\nSetting Enclosure LEDs to %02x ...\n",newvalue);
	    ret = set_enc_leds(newvalue);
            ret = get_enc_leds(&encled);
	    show_enc_leds(encled);
	 }
      }
   }
   else if (fChesnee) { /* Chesnee platform has special disk LEDs */
        diskled = get_diskleds();
	show_diskleds(diskled);
	if (fsetdisk) {
	   newvalue = diskled;
	   // newvalue |= 0xFC;  /*leave upper bits high (off) */
	   /* Set fault if user param, and disk is present. */
	   if (fdiskb == 1)  newvalue &= 0xFE;               /*bit0=0*/
	   else if (fdiskb == 0) newvalue |= 0x01;
	   if (fdiska == 1) newvalue &= 0xFD;               /*bit1=0*/
	   else if (fdiska == 0) newvalue |= 0x02;
	   printf("\nSetting Disk LEDs to %02x ...\n",newvalue);
	   ret = set_diskleds(newvalue);
           diskled = get_diskleds();
	   show_diskleds(diskled);
	}
   }
   ipmi_close_();
   show_outcome(progname,ret);
   exit (ret);
}  /* end main()*/

/* end alarms.c */
