/*
 * bmchealth.c
 *
 * This tool checks the health of the BMC via IPMI.
 * 
 * Author:  Andy Cress  arcress@users.sourceforge.net
 * Copyright (c) 2006 Intel Corporation. 
 *
 * 03/22/06 Andy Cress - created
 * 06/20/06 Andy Cress 0.6 - more vendor strings, add ping_node() stub for now
 * 10/20/06 Andy Cress 1.1 - added -g for guid
 * 01/10/07 Andy Cress 1.4 - added product strings
 * 02/25/07 Andy Cress 2.8 - added more Chassis Status decoding
 */
/*M*
Copyright (c) 2006, Intel Corporation
All rights reserved.

Redistribution and use in source and binary forms, with or without 
modification, are permitted provided that the following conditions are met:

  a.. Redistributions of source code must retain the above copyright notice, 
      this list of conditions and the following disclaimer. 
  b.. Redistributions in binary form must reproduce the above copyright notice,
      this list of conditions and the following disclaimer in the documentation 
      and/or other materials provided with the distribution. 
  c.. Neither the name of Intel Corporation nor the names of its contributors 
      may be used to endorse or promote products derived from this software 
      without specific prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR 
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *M*/
#ifdef WIN32
#include <windows.h>
#include <stdio.h>
#include "getopt.h"
#else
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <getopt.h>
#endif
#include <string.h>
#include "ipmicmd.h"
 
#define  SELFTEST_STATUS   0x04
#define  GET_POWER_STATE   0x07
/*
 * Global variables 
 */
static char * progname  = "bmchealth";
static char * progver   = "2.12";
static char   fdebug    = 0;
static char   do_hsc    = 0;
static char   do_guid   = 0;

static int get_selftest_status(uchar *rdata, int rlen)
{
	uchar idata[4];
	uchar ccode;
	int ret;

        ret = ipmi_cmdraw( SELFTEST_STATUS, NETFN_APP,
                        BMC_SA,PUBLIC_BUS,BMC_LUN,
                        idata,0, rdata,&rlen,&ccode, fdebug);
	if (ret == 0 && ccode != 0) ret = ccode;
	return(ret);
}  /*end get_selftest_status()*/

static int get_chassis_status(uchar *rdata, int rlen)
{
	uchar idata[4];
	uchar ccode;
	int ret;

        ret = ipmi_cmdraw( CHASSIS_STATUS, NETFN_CHAS,
                        BMC_SA,PUBLIC_BUS,BMC_LUN,
                        idata,0, rdata,&rlen,&ccode, fdebug);
	if (ret == 0 && ccode != 0) ret = ccode;
	return(ret);
}  /*end chassis_status()*/

static char *chs_string(uchar state)
{
   static char chs_strbuf[80];
   char *pstr;
   pstr = &chs_strbuf[0];
   if (state & 0x01) strcpy(pstr,"on");
   else strcpy(pstr,"off");
   if (state & 0x02) strcat(pstr,", overload");
   if (state & 0x04) strcat(pstr,", interlock");
   if (state & 0x08) strcat(pstr,", fault");
   if (state & 0x10) strcat(pstr,", control error");
   if (state & 0x20) strcat(pstr,", restore_policy=last_state");
   else if (state & 0x40) strcat(pstr,", restore_policy=turn_on");
   else strcat(pstr,", restore_policy=stay_off");
   pstr = &chs_strbuf[0];
   return(pstr);
}

static int get_power_state(uchar *rdata, int rlen)
{
	uchar idata[4];
	uchar ccode;
	int ret;

        ret = ipmi_cmdraw( GET_POWER_STATE, NETFN_APP,
                        BMC_SA,PUBLIC_BUS,BMC_LUN,
                        idata,0, rdata,&rlen,&ccode, fdebug);
	if (ret == 0 && ccode != 0) ret = ccode;
	return(ret);
}  /*end get_power_state()*/

static char *pwr_string(uchar pstate)
{
   char *pstr;
   switch(pstate) {
      case 0x00: pstr = "S0: working"; break; 
      case 0x01: pstr = "S1: clock stopped, context ok"; break; 
      case 0x02: pstr = "S2: clock stopped, context lost"; break; 
      case 0x03: pstr = "S3: suspend-to-RAM"; break; 
      case 0x04: pstr = "S4: suspend-to-Disk"; break; 
      case 0x05: pstr = "S5: soft off"; break; 
      case 0x06: pstr = "S4/S5: soft off, either S4 or S5"; break; 
      case 0x07: pstr = "G3: mechanical off"; break; 
      case 0x08: pstr = "S1-S3: sleeping"; break; 
      case 0x09: pstr = "S1-S4: sleeping"; break; 
      case 0x0A: pstr = "S5/o: soft off by override"; break; 
      case 0x20: pstr = "legacy on"; break; 
      case 0x21: pstr = "legacy soft-off"; break; 
      default:   pstr = "unknown"; break;
   }
   return(pstr);
}

#ifdef PING_OK
extern int ping_bmc(char *node, char fdebug);

static int ping_node(char *node)
{
   int rv = 0;
   /* verify that the BMC LAN channel is configured & active */
   /* send rmcp_ping to node's BMC */
   rv = ping_bmc(node,fdebug);
   return(rv);
}
#endif

static int get_hsc_devid(uchar *rdata, int rlen)
{
	uchar ccode;
	int ret;

        ret = ipmi_cmdraw( 0x01, /*(GET_DEVICEID & 0x00ff)*/ 
			NETFN_APP, 0xC0,PUBLIC_BUS,BMC_LUN,
                        NULL,0, rdata,&rlen,&ccode, fdebug);
	if (ret == 0 && ccode != 0) ret = ccode;
	return(ret);
}  /*end get_hsc_devid*/

static int get_chan_auth(uchar *rdata, int rlen)
{
	uchar idata[4];
	uchar ccode;
	int ret;

        idata[0] = 0x0e; /*this channel*/
        idata[1] = 0x02; /*user level*/
        ret = ipmi_cmdraw( 0x38, /*(GET_CHAN_AUTH & 0x00ff)*/ 
			NETFN_APP, BMC_SA, PUBLIC_BUS,BMC_LUN,
                        idata,2, rdata,&rlen,&ccode, fdebug);
	if (ret == 0 && ccode != 0) ret = ccode;
	return(ret);
}  /*end get_chan_auth*/

void show_chan_auth(char *tag, uchar *rec, int srec)
{
   char pstr[40];
   pstr[0] = 0;
   if (rec[1] & 0x01) strcat(pstr,"None ");
   if (rec[1] & 0x02) strcat(pstr,"MD2 ");
   if (rec[1] & 0x04) strcat(pstr,"MD5 ");
   if (rec[1] & 0x10) strcat(pstr,"Straight_Passwd ");
   if (rec[1] & 0x20) strcat(pstr,"OEM ");
   printf("Channel %d Auth Types: %s\n",rec[0],pstr);
   printf("          Status = %02x, OEM ID %02x%02x%02x OEM Aux %02x\n",
          rec[2],rec[4],rec[5],rec[6],rec[7]);
}

#define BMC  1
#define HSC  2

void show_devid(int dtype, uchar *devrec, int sdevrec)
{
   uchar ipmi_maj = 0;
   uchar ipmi_min = 0;
   char *tag; 
   int mfg, prod;
   char *mfgstr = "";
   char *prodstr = "";

      ipmi_maj = devrec[4] & 0x0f;
      ipmi_min = devrec[4] >> 4;
      mfg  = devrec[6] + (devrec[7] << 8) + (devrec[8] << 16);
      prod = devrec[9] + (devrec[10] << 8);
      if (dtype == HSC) tag = "HSC";
      else tag = "BMC";
      /* For a list of all IANA enterprise mfg vendor numbers, 
       * see http://www.iana.org/assignments/enterprise-numbers 
       * Product numbers are different for each mfg vendor. 
       */
      switch(mfg) {
         case VENDOR_NSC: mfgstr = "(NSC)";    /*=0x000322*/
			if (dtype == BMC) tag="mBMC";
			if (prod == 0x4311) prodstr = "(TIGPT1U)";
			break;
         case 0x000002: mfgstr = "(IBM)";   break;
         case 0x00000B: mfgstr = "(HP)";    break;
         case 0x00002A: mfgstr = "(Sun)";   break;
         case 0x002880: mfgstr = "(Fujitsu-Siemens)"; break;
         case 0x002A7C: mfgstr = "(SuperMicro)";   break;
         case 0x000175: mfgstr = "(Tatung)";  break;
         case 0x0002A2: mfgstr = "(Dell)";    break;
         case 0x0051EE: mfgstr = "(AMI)";     break;
         case 0x0035AE: mfgstr = "(Raritan)";   break;
         case 0x0028C5: mfgstr = "(Peppercon)"; break;
         case 0x0028B2: mfgstr = "(Avocent)";   break;
         case 0x002B5E: mfgstr = "(OSA)";     break;
         case 0x0000BA: mfgstr = "(Toshiba)"; break;
         case 0x000F85: mfgstr = "(Aelita Software)"; break; /*3973. HP DL140*/
         case       94: mfgstr = "(Nokia)"; break;
         case     4337: mfgstr = "(Radisys)"; break;
         case     5593: mfgstr = "(Magnum Technologies)"; break;
         case     9237: mfgstr = "(Newisys)"; break;
         case    11129: mfgstr = "(Google)"; break;
         case    12634: mfgstr = "(PICMG)"; break;
         case    15000: mfgstr = "(Kontron)"; break;
         case 0x000074: 
         case 0x00018F: mfgstr = "(Hitachi)"; break;
         case VENDOR_NEC:  mfgstr = "(NEC)";  break;
         case VENDOR_LMC:  mfgstr = "(LMC)";  break; /*=0x0878 with SuperMicro*/
         case VENDOR_TYAN: mfgstr = "(Tyan)"; break;
         case VENDOR_INTEL: mfgstr = "(Intel)";    /*=0x000157*/
             switch(prod) {     /* show product names for some */
		 case 0x000C:   prodstr = "(TSRLT2)"; break;   /*SCB2*/
		 case 0x001B:   prodstr = "(TIGPR2U)"; break;  /*SWV2*/
		 case 0x0022:   prodstr = "(TIGI2U)"; break;   /*SJR2*/
		 case 0x0026:   prodstr = "(Bridgeport)"; break;
		 case 0x0028:   prodstr = "(S5000PAL)"; break;
		 case 0x0029:   prodstr = "(S5000PSL)"; break;
		 case 0x0100:   prodstr = "(Tiger4)"; break;
		 case 0x0103:   prodstr = "(McCarran)"; break;
		 case 0x0800:   prodstr = "(ZT5504)";   break;
		 case 0x0808:   prodstr = "(MPCBL0001)"; break; /*ATCA*/
		 case 0x0811:   prodstr = "(TIGW1U)";   break; /*S5000PHB*/
		 case 0x4311:   prodstr = "(NSI2U)"; break;
                 default:       prodstr = ""; break;
             }
             break;
         default:       mfgstr = ""; break;
      }
      printf("%s version %x.%x, IPMI version %d.%d \n",
             tag, devrec[2],  devrec[3], ipmi_maj, ipmi_min);
      printf("%s manufacturer = %06x %s, product = %04x %s\n",
		tag, mfg,mfgstr,prod,prodstr);
      /* could show product rev, if available (sdevrec > 14) */
      return;
}


#ifdef METACOMMAND
int i_health(int argc, char **argv)
#else
#ifdef WIN32
int __cdecl
#else
int
#endif
main(int argc, char **argv)
#endif
{
   int ret = 0;
   char c;
   uchar freadonly = 1;
   uchar freset = 0;
   uchar fdisable = 0;
   uchar selfbuf[16];
   uchar devrec[16];
   uchar cc;
   int t = 0;
   int selfstatus;
   uchar pwr_state;
   uchar chs_status;
   char *selfstr = "";
   char *s;
   int i, sresp;

   printf("%s ver %s\n", progname,progver);

   while ( (c = getopt( argc, argv,"ghT:V:J:YEF:P:N:R:U:x?")) != EOF ) 
      switch(c) {
          case 'x': fdebug = 1;     break;  /* debug messages */
          case 'h': do_hsc = 1;     break;  /* check the HSC too */
          case 'g': do_guid = 1;    break;  /* get the System GUID also */
          case 'N':    /* nodename */
          case 'U':    /* remote username */
          case 'P':    /* remote password */
          case 'R':    /* remote password */
          case 'E':    /* get password from IPMI_PASSWORD environment var */
          case 'F':    /* force driver type */
          case 'T':    /* auth type */
          case 'J':    /* cipher suite */ 
          case 'V':    /* priv level */
          case 'Y':    /* prompt for remote password */
                parse_lan_options(c,optarg,fdebug);
                break;
	  default:
                printf("Usage: %s [-ghx -N node -U user -P/-R pswd -EFTVY]\n", progname);
                printf(" where -x      show eXtra debug messages\n");
                printf("       -h      check the HotSwap Controller also\n");
                printf("       -g      get the System GUID also\n");
		print_lan_opt_usage();
                exit(1);
      }

   ret = ipmi_getdeviceid(devrec,16,fdebug);
   if (ret != 0) {
        show_outcome(progname,ret); 
	ipmi_close_();
	exit(1);
   } else {
       show_devid(BMC,devrec,12);
   }

   ret = get_chassis_status(selfbuf,4);
   if (ret != 0) {
	printf("Cannot do ipmi_getpowerstate, ret = %d\n",ret);
	ipmi_close_();
	exit(1);
   } else {
        chs_status = selfbuf[0] & 0x7f;
	printf("Chassis Status   = %02x   (%s)\n",
		chs_status,chs_string(chs_status));
   }

   ret = get_power_state(selfbuf,4);
   if (ret != 0) {
	printf("Cannot do ipmi_getpowerstate, ret = %d\n",ret);
	ipmi_close_();
	exit(1);
   } else {
        pwr_state = selfbuf[0] & 0x7f;
	printf("Power State      = %02x   (%s)\n",
		pwr_state,pwr_string(pwr_state));
   }

   ret = get_selftest_status(&selfbuf[0],sizeof(selfbuf));
   if (ret != 0) {
	printf("get_selftest_status error: ret = %x\n",ret);
	ipmi_close_();
	exit(1);
   } else {
        selfstatus = selfbuf[0] + (selfbuf[1] << 8);
        switch(selfstatus) {
           case 0x0055: selfstr = "(OK)";   break;
           default:     selfstr = "";   break;
        }
	printf("Selftest status  = %04x %s\n",selfstatus,selfstr);
   }

   ret = get_chan_auth(&devrec[0],sizeof(devrec));
   if (ret == 0) {  
      show_chan_auth("Channel Auth Cap",devrec,8);
   } else 
      printf("get_chan_auth error: ret = %x\n",ret);

   if (do_guid) {
      sresp = sizeof(devrec);
      ret = ipmi_cmd(GET_SYSTEM_GUID,NULL,0,devrec,&sresp,&cc,fdebug);
      if (fdebug) printf("system_guid: ret = %d, cc = %x\n",ret,cc);
      if (ret == 0 && cc == 0) {
         printf("System GUID      = ");
         for (i=0; i<16; i++) {
            if ((i == 4) || (i == 6) || (i == 8) || (i == 10)) s = "-";
            else s = "";
            printf("%s%02x",s,devrec[i]);
         }
         printf("\n");
      }
   }

   if (do_hsc) {
      /* Get HSC status */
      ret = get_hsc_devid(&devrec[0],sizeof(devrec));
      if (ret == 0) {  /* only if HSC is detected */
         show_devid(HSC,devrec,14);
      }
   }
   ipmi_close_();

#ifdef PING_OK
   /* Currently some problems with this:
    * works first time, but locks up BMC LAN on subsequent attempts.
    */
   ret = ping_node(gnode);
   printf("ping_node(%s): ret = %d\n",gnode,ret);
#endif

   show_outcome(progname,ret);  
   exit (ret);
}  /* end main()*/

/* end bmchealth.c */
