/*
 * events.c
 *
 * This file decodes the IPMI event into a readable string.
 * It is used by showsel.c and getevent.c.
 *
 * Author:  Andy Cress  arcress@users.sourceforge.net
 *
 * Copyright (c) 2006 Intel Corporation. 
 *
 * 05/26/05 Andy Cress - created from showsel.c (see there for history)
 * 06/20/05 Andy Cress - changed PowerSupply present/not to Inserted/Removed
 * 08/01/05 Andy Cress - added more Power Unit & Battery descriptions
 * 03/02/06 Andy Cress - more decoding for Power Unit 0b vs. 6f
 * 04/11/07 Andy Cress - added events -p decoding for PET data
 * 10/03/07 Andy Cress - added file_grep for -p in Windows
 * 03/03/08 Andy Cress - added -f to interpret raw SEL file
 */
/*M*
Copyright (c) 2006, Intel Corporation
All rights reserved.

Redistribution and use in source and binary forms, with or without 
modification, are permitted provided that the following conditions are met:

  a.. Redistributions of source code must retain the above copyright notice, 
      this list of conditions and the following disclaimer. 
  b.. Redistributions in binary form must reproduce the above copyright notice,
      this list of conditions and the following disclaimer in the documentation 
      and/or other materials provided with the distribution. 
  c.. Neither the name of Intel Corporation nor the names of its contributors 
      may be used to endorse or promote products derived from this software 
      without specific prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR 
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *M*/
#include <stdio.h>
#ifdef WIN32
#include <windows.h>
#else
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <syslog.h>
#endif
#include <time.h>

#include "ipmicmd.h" 
 
#define  SELprintf  printf  
static char fdebug = 0;   

#pragma pack(1)
typedef struct
{
        short              record_id;
        uchar              record_type;
        uint               timestamp;
        short              generator_id;    /*slave_addr/channel*/
        uchar              evm_rev;         //event message revision
        uchar              sensor_type;
        uchar              sensor_number;
        uchar              event_trigger;
        uchar              event_data1;
        uchar              event_data2;
        uchar              event_data3;
}       SEL_RECORD;
#pragma pack()

#ifdef WIN32
   static char sensfil[64] = "sensor_out.txt";
   static char outfil[64] = "stype.tmp";
#else
   static char sensfil[64] = "/usr/share/ipmiutil/sensor_out.txt";
   static char outfil[] = "/tmp/stype.tmp";
#endif
   static char rawfil[64] = "";

/* sensor_types: See IPMI 1.5 Table 36-3, IPMI 2.0 Table 42-3 */
#define NSTYPES   0x2E
static const char *sensor_types[NSTYPES] = {  
/* 00h */ "reserved",
/* 01h */ "Temperature",
/* 02h */ "Voltage",
/* 03h */ "Current",
/* 04h */ "Fan",
/* 05h */ "Platform Security",
/* 06h */ "Security Violation",
/* 07h */ "Processor",
/* 08h */ "Power Supply",
/* 09h */ "Power Unit",
/* 0Ah */ "Cooling Device",
/* 0Bh */ "FRU Sensor",
/* 0Ch */ "Memory",
/* 0Dh */ "Drive Slot",
/* 0Eh */ "POST Memory Resize",
/* 0Fh */ "System Firmware",    /*incl POST code errors*/
/* 10h */ "SEL Disabled",
/* 11h */ "Watchdog_1",
/* 12h */ "System Event",          /* offset 0,1,2 */
/* 13h */ "Critical Interrupt",        /* offset 0,1,2 */
/* 14h */ "Button",                /* offset 0,1,2 */
/* 15h */ "Board",
/* 16h */ "Microcontroller",
/* 17h */ "Add-in Card",
/* 18h */ "Chassis",
/* 19h */ "Chip Set",
/* 1Ah */ "Other FRU",
/* 1Bh */ "Cable/Interconnect",
/* 1Ch */ "Terminator",
/* 1Dh */ "System Boot Initiated",
/* 1Eh */ "Boot Error",
/* 1Fh */ "OS Boot",
/* 20h */ "OS Critical Stop",
/* 21h */ "Slot/Connector",
/* 22h */ "ACPI Power State",
/* 23h */ "Watchdog_2",
/* 24h */ "Platform Alert",
/* 25h */ "Entity Presence",
/* 26h */ "Monitor ASIC",
/* 27h */ "LAN",
/* 28h */ "Management Subsystem Health",
/* 29h */ "Battery",
/* 2Ah */ "Session Audit",
/* 2Bh */ "Version Change",
/* 2Ch */ "FRU State",
/* 2Dh */ "SMI Timeout"  /* 0xF3 == OEM SMI Timeout */
};

#define NFWERRS  15
static struct {    /* See Table 36-3, type 0Fh, offset 00h */
  int code; char *msg; 
  } fwerrs[NFWERRS] = {
 { 0x00, "Unspecified"},
 { 0x01, "No system memory"},
 { 0x02, "No usable memory"},
 { 0x03, "Unrecovered Hard Disk"},
 { 0x04, "Unrecovered System Board"},
 { 0x05, "Unrecovered Diskette"},
 { 0x06, "Unrecovered Hard Disk Ctlr"},
 { 0x07, "Unrecovered PS2 USB"},
 { 0x08, "Boot media not found"},
 { 0x09, "Unrecovered video controller"},
 { 0x0A, "No video device"},
 { 0x0B, "Firmware ROM corruption"},
 { 0x0C, "CPU voltage mismatch"},
 { 0x0D, "CPU speed mismatch"},
 { 0x0E, "Reserved" }
};

#define NFWSTAT  27
static struct {    /* See Table 36-3, type 0Fh, offset 01h & 02h */
  int code; char *msg; 
  } fwstat[NFWSTAT] = {
 { 0x00, "Unspecified"},
 { 0x01, "Memory init"},
 { 0x02, "Hard disk init"},
 { 0x03, "Secondary processor"},
 { 0x04, "User authentication"},
 { 0x05, "User-init sys setup"},
 { 0x06, "USB configuration"},
 { 0x07, "PCI configuration"},
 { 0x08, "Option ROM init"},
 { 0x09, "Video init"},
 { 0x0a, "Cache init"},
 { 0x0b, "SM Bus init"},
 { 0x0c, "Keyboard init"},
 { 0x0d, "Mgt controller"},
 { 0x0e, "Docking attach"},
 { 0x0f, "Enabling docking"},
 { 0x10, "Docking eject"},
 { 0x11, "Disabling docking"},
 { 0x12, "OS wake-up"},
 { 0x13, "Starting OS boot"},
 { 0x14, "Baseboard init"},
 { 0x15, "reserved"},
 { 0x16, "Floppy init"},
 { 0x17, "Keyboard test"},
 { 0x18, "Mouse test"},
 { 0x19, "Primary processor"},
 { 0x1A, "Reserved"}
};

#define NGDESC   7
static struct {
 ushort g_id;
 const char desc[10];
} gen_desc[NGDESC] = {  /*empirical, format defined in IPMI 1.5 Table 23-5 */
 { 0x0000, "IPMB"},
 { 0x0001, "EFI "},
 { 0x0003, "BIOS"},
 { 0x0020, "BMC "},
 { 0x0021, "SMI "},
 { 0x0028, "CHAS"},    /* Chassis Bridge Controller */
/*{0x0031, "SMI/IPMB"}, * ia64/mBMC POST errors   (rqSa/rqLun)  */
/*{0x0033, "MemCtlr"},  * ia64 memory controller? BIOS/SMI? (rqSa/rqLun) */
 { 0x00c0, "HSC "}     /* SAF-TE Hot-Swap Controller*/
};

#define NCRITS  10
char * crit_int_str[NCRITS] = {  /* Critical Interrupt descriptions */
 /*00*/ "FP NMI  ",  /* Front Panel NMI */
 /*01*/ "Bus Timout",
 /*02*/ "IOch NMI ", /* IO channel check NMI */
 /*03*/ "Soft NMI ",
 /*04*/ "PCI PERR ",
 /*05*/ "PCI SERR ",
 /*06*/ "EISA Timout",
 /*07*/ "Bus Warn ",  /* Bus Correctable Error */
 /*08*/ "Bus Error",  /* Bus Uncorrectable Error */
 /*09*/ "Fatal NMI" };

#define NSLOTC  9
char * slot_str[NSLOTC] = {  /* Slot/Connector descriptions */
 /*00*/ "Fault   ", 
 /*01*/ "Identify",
 /*02*/ "Inserted",
 /*03*/ "InsReady",
 /*04*/ "RemReady",
 /*05*/ "PowerOff",
 /*06*/ "RemRequest",
 /*07*/ "Interlock",
 /*08*/ "Disabled" }; 

#define NBATT  3
char * batt_str[NBATT] = {  /* Battery descriptions */
 /*00*/ "Low",
 /*01*/ "Failed",
 /*02*/ "Present" };

#define NPROC  11
char * proc_str[NPROC] = {  /* Processor descriptions */
 /*00*/ "IERR",
 /*01*/ "Thermal Trip",
 /*02*/ "FRB1/BIST failure",
 /*03*/ "FRB2 timeout",
 /*04*/ "FRB3 Proc error",
 /*05*/ "Config error",
 /*06*/ "SMBIOS CPU error",
 /*07*/ "Present",
 /*08*/ "Disabled",
 /*09*/ "Terminator",
 /*0A*/ "Throttled" };

#define NACPIP  15
char * acpip_str[NACPIP] = {  /* ACPI Power State descriptions */
 /*00*/ "S0/G0 Working",
 /*01*/ "S1 sleeping, proc/hw context maintained",
 /*02*/ "S2 sleeping, proc context lost",
 /*03*/ "S3 sleeping, proc/hw context lost, mem ok",
 /*04*/ "S4 non-volatile sleep/suspend",
 /*05*/ "S5/G2 soft-off",
 /*06*/ "S4/S5 soft-off, no specific state",
 /*07*/ "G3/Mechanical off",
 /*08*/ "Sleeping in an S1/S2/S3 state",
 /*09*/ "G1 sleeping",
 /*0A*/ "S5 entered by override",
 /*0B*/ "Legacy ON state",
 /*0C*/ "Legacy OFF state",
 /*0D*/ "Unknown",
 /*0E*/ "Unknown" }; 

#define NMEM  6
char * mem_str[NMEM] = {  /* Memory descriptions */
 /*00*/ "Correctable ECC",
 /*01*/ "Uncorrectable ECC",
 /*02*/ "Parity",
 /*03*/ "Memory Scrub Failed",
 /*04*/ "Memory Device Disabled",
 /*05*/ "ECC limit reached" }; 

#define NAUDIT  2
char * audit_str[NAUDIT] = {  /* Session Audit descriptions */
 /*00*/ "Activated",
 /*01*/ "Deactivated"};

#define NSDESC   72
struct {
 ushort genid;  /*generator id: BIOS, BMC, etc. (slave_addr/channel)*/
 uchar s_typ;   /*1=temp,2=voltage,4=fan,... */
 uchar s_num;
 uchar evtrg;   /*event trigger/type, see IPMI 1.5 table 36.1 & 36.2 */
 uchar data1;
 uchar data2;
 uchar data3;
 const char desc[40];
} sens_desc[NSDESC] = {
{0xffff,0x05, 0xff, 0x6f, 0x40, 0xff, 0xff, "Chassis intrusion"}, /*chassis*/
{0xffff,0x05, 0xff, 0x6f, 0x44, 0xff, 0xff, "LAN unplugged"},   /*chassis*/
{0xffff,0x05, 0xff, 0xef, 0x44, 0xff, 0xff, "LAN restored "},   /*chassis*/
{0xffff,0x06, 0xff, 0xff, 0x45, 0xff, 0xff, "Password"},        /*security*/
{0xffff,0x07, 0xff, 0xff, 0x41, 0xff, 0xff, "Thermal trip"},    /*processor*/
{0xffff,0x08, 0xff, 0x6f, 0x40, 0xff, 0xff, "Inserted"},         /*PowerSupply*/
{0xffff,0x08, 0xff, 0x6f, 0x41, 0xff, 0xff, "Failure detected"}, /*PowerSupply*/
{0xffff,0x08, 0xff, 0x6f, 0x42, 0xff, 0xff, "Predictive failure"},
{0xffff,0x08, 0xff, 0x6f, 0x43, 0xff, 0xff, "AC Lost"},
{0xffff,0x08, 0xff, 0xef, 0x40, 0xff, 0xff, "Removed"},          /*PowerSupply*/
{0xffff,0x08, 0xff, 0xef, 0x41, 0xff, 0xff, "is OK  "},          /*PowerSupply*/
{0xffff,0x08, 0xff, 0xef, 0x42, 0xff, 0xff, "Predictive OK"},
{0xffff,0x08, 0xff, 0xef, 0x43, 0xff, 0xff, "AC Regained"},      /*PowerSupply*/
{0xffff,0x09, 0xff, 0x0B, 0x40, 0xff, 0xff, "Redundancy OK  "},  /*power unit*/
{0xffff,0x09, 0xff, 0x0B, 0x41, 0xff, 0xff, "Redundancy Lost"},  /*power unit*/
{0xffff,0x09, 0xff, 0x0B, 0x42, 0xff, 0xff, "Redundancy Degraded"},
{0xffff,0x09, 0xff, 0x0B, 0x43, 0xff, 0xff, "Not Redundant"},
{0xffff,0x09, 0xff, 0x0B, 0x44, 0xff, 0xff, "Sufficient Resources"},
{0xffff,0x09, 0xff, 0x0B, 0x45, 0xff, 0xff, "Insufficient Resources"},
{0xffff,0x09, 0xff, 0x8B, 0x40, 0xff, 0xff, "Redundancy NOT ok"},
{0xffff,0x09, 0xff, 0x8B, 0x41, 0xff, 0xff, "Redundancy Regained"},
{0xffff,0x09, 0xff, 0x8B, 0x42, 0xff, 0xff, "Redundancy Restored"},
{0xffff,0x09, 0xff, 0x8B, 0x43, 0xff, 0xff, "Redundant"},        /*power unit*/
{0xffff,0x09, 0xff, 0x6f, 0x40, 0xff, 0xff, "Power Off    "},    /*power unit*/
{0xffff,0x09, 0xff, 0x6f, 0x41, 0xff, 0xff, "Power Cycle  "},    /*power unit*/
{0xffff,0x09, 0xff, 0x6f, 0x42, 0xff, 0xff, "240VA power down"},
{0xffff,0x09, 0xff, 0x6f, 0x43, 0xff, 0xff, "Interlock power down"},
{0xffff,0x09, 0xff, 0x6f, 0x44, 0xff, 0xff, "AC Lost"},
{0xffff,0x09, 0xff, 0x6f, 0x45, 0xff, 0xff, "Soft Powerup failure"},
{0xffff,0x09, 0xff, 0x6f, 0x46, 0xff, 0xff, "Failure detected"},
{0xffff,0x09, 0xff, 0x6f, 0x47, 0xff, 0xff, "Predictive failure"},/*power unit*/
{0xffff,0x09, 0xff, 0xef, 0x40, 0xff, 0xff, "Power Restored"},
{0xffff,0x09, 0xff, 0xef, 0x44, 0xff, 0xff, "AC Regained"},
{0xffff,0x0c, 0xff, 0xff, 0x00, 0xff, 0xff, "Correctable ECC"},  /*memory*/
{0xffff,0x0f, 0x06, 0xff, 0xff, 0xff, 0xff, "POST Code"},
{0xffff,0x10, 0x09, 0xff, 0x42, 0x0f, 0xff, "Log Cleared"},
{0xffff,0x10, 0x09, 0xff, 0xc0, 0x03, 0xff, "ECC Memory Errors"},
     /*  Often see these 3 Boot records with reboot:
      * 12 83 6f 05 00 ff = System/SEL ClockSync_1 (going down)
      * 12 83 6f 05 80 ff = System/SEL ClockSync_2 (coming up)
      * 12 83 6f 01 ff ff = OEM System Boot Event  (Completed POST), or
      * 12 83 6f 41 0f ff = OEM System Boot Event  (same w 01 & 41) */
{0x0003,0x12, 0x83, 0xff, 0x05, 0x00, 0xff, "Boot: ClockSync_1"},   
{0x0003,0x12, 0x83, 0xff, 0x05, 0x80, 0xff, "Boot: ClockSync_2"},   
{0x0003,0x12, 0x83, 0xff, 0x01, 0xff, 0xff, "OEM System Booted"},   
{0x0001,0x12, 0x08, 0xff, 0x01, 0xff, 0xff, "OEM System Booted"},    /*ia64*/
{0x0033,0x12, 0x01, 0x6f, 0x01, 0xff, 0xff, "OEM System Booted"},    /*S5000*/
{0x0031,0x12, 0x00, 0x6f, 0xc3, 0xff, 0xff, "PEF Action"},           /*ia64*/
{0x0020,0x12, 0x83, 0x6f, 0x80, 0xff, 0xff, "System Reconfigured"},  /*BMC*/
{0x0020,0x12, 0x83, 0xff, 0x41, 0xff, 0xff, "OEM System Boot"},      /*BMC*/
{0x0020,0x12, 0x83, 0x6f, 0x42, 0xff, 0xff, "System HW failure"},    /*BMC*/
{0x0020,0x12, 0x83, 0x6f, 0x04, 0xff, 0xff, "PEF Action"},           /*BMC*/
{0x00c0,0x0d, 0xff, 0x08, 0x00, 0xff, 0xff, "Device Removed"},    /*HSC*/
{0x00c0,0x0d, 0xff, 0x08, 0x01, 0xff, 0xff, "Device Inserted"},   /*HSC*/
{0xffff,0x14, 0xff, 0xff, 0x42, 0xff, 0xff, "Reset Button pressed"},
{0xffff,0x14, 0xff, 0xff, 0x40, 0xff, 0xff, "Power Button pressed"},
{0xffff,0x14, 0xff, 0xff, 0x01, 0xff, 0xff, "ID Button pressed"},
{0xffff,0x23, 0xff, 0xff, 0x40, 0xff, 0xff, "Expired, no action"},/*watchdog2*/
{0xffff,0x23, 0xff, 0xff, 0x41, 0xff, 0xff, "Hard Reset action"}, /*watchdog2*/
{0xffff,0x23, 0xff, 0xff, 0x42, 0xff, 0xff, "Power down action"}, /*watchdog2*/
{0xffff,0x23, 0xff, 0xff, 0x43, 0xff, 0xff, "Power cycle action"},/*watchdog2*/
{0xffff,0x23, 0xff, 0xff, 0x48, 0xff, 0xff, "Timer interrupt"},   /*watchdog2*/
{0xffff,0xf3, 0x85, 0x83, 0x41, 0xff, 0xff, "SMI de-asserted"},
{0xffff,0xf3, 0x85, 0x03, 0x41, 0xff, 0xff, "SMI asserted"},
{0xffff,0x20, 0x00, 0xff, 0xff, 0xff, 0xff, "OS Kernel Panic"},
{0xffff,0xff, 0xff, 0x01, 0x57, 0xff, 0xff, "Hi Noncrit thresh"},
{0xffff,0xff, 0xff, 0x01, 0x59, 0xff, 0xff, "Hi Crit thresh"},
{0xffff,0xff, 0xff, 0x01, 0x5B, 0xff, 0xff, "Hi NoRec thresh"},
{0xffff,0xff, 0xff, 0x01, 0x50, 0xff, 0xff, "Lo Noncrit thresh"},
{0xffff,0xff, 0xff, 0x01, 0x52, 0xff, 0xff, "Lo Crit thresh"},
{0xffff,0xff, 0xff, 0x01, 0x54, 0xff, 0xff, "Lo NoRec thresh"},
{0xffff,0xff, 0xff, 0x81, 0x57, 0xff, 0xff, "HiN thresh OK now"},
{0xffff,0xff, 0xff, 0x81, 0x59, 0xff, 0xff, "HiC thresh OK now"},
{0xffff,0xff, 0xff, 0x81, 0x5B, 0xff, 0xff, "HiR thresh OK now"},
{0xffff,0xff, 0xff, 0x81, 0x50, 0xff, 0xff, "LoN thresh OK now"},
{0xffff,0xff, 0xff, 0x81, 0x52, 0xff, 0xff, "LoC thresh OK now"},
{0xffff,0xff, 0xff, 0x81, 0x54, 0xff, 0xff, "LoR thresh OK now"}
	/*Thresholds apply to sensor types 1=temp, 2=voltage, 4=fan */
	/*Note that last 2 bytes usu show actual & threshold values*/
	/*evtrg: 0x01=thresh, 0x81=restored */
};

/*------------------------------------------------------------------------ 
 * get_misc_desc
 * Uses the sens_desc array to decode misc entries not otherwise handled.
 * Called by decode_sel_entry
 *------------------------------------------------------------------------*/
char *
get_misc_desc(uchar type, uchar num, ushort genid, uchar trig,
		 uchar data1, uchar data2, uchar data3)
{
	int i;
	char *pstr = NULL; 

	/* Use sens_desc array for other misc descriptions */
	data1 &= 0x0f;  /*ignore top half of sensor offset for matching */
	for (i = 0; i < NSDESC; i++) {
           if ((sens_desc[i].s_typ == 0xff) || 
               (sens_desc[i].s_typ == type)) {
	      if (sens_desc[i].s_num != 0xff &&
	         sens_desc[i].s_num != num)
			    continue;
	      if (sens_desc[i].genid != 0xffff &&
	         sens_desc[i].genid != genid)
			    continue;
	      if (sens_desc[i].evtrg != 0xff &&
	         sens_desc[i].evtrg != trig)
				    continue;
	      if (sens_desc[i].data1 != 0xff && 
      	          (sens_desc[i].data1 & 0x0f) != (data1 & 0x0f))
				    continue;
	      if (sens_desc[i].data2 != 0xff && 
      	          sens_desc[i].data2 != data2)
				    continue;
	      if (sens_desc[i].data3 != 0xff && 
      	          sens_desc[i].data3 != data3)
				    continue;
	      /* have a match, use description */
	      pstr = (char *)sens_desc[i].desc; 
	      break;
           }
	} /*end for*/
#if 0
        if (i >= NSDESC))  fthresh = 0;          /*not found*/
        else if (i >= (NSDESC - 8)) fthresh = 1; /*threshold*/
	else fthresh = 0;  
#endif
	return(pstr);
}  /* end get_misc_desc() */

static void fmt_time(time_t etime, char *buf, int bufsz)
{
	strcpy(buf,"00/00/00 00:00:00");
	strftime(buf,bufsz, "%x %H:%M:%S",
#ifdef WIN32
                             gmtime(&etime));
#else
                             localtime(&etime));
#endif
	return;
}

/* 
 * findmatch 
 * Finds a matching pattern within a string buffer.
 * returns offset of the match if found, or -1 if not found.  
 */
static int
findmatch(char *buffer, int sbuf, char *pattern, int spattern, char figncase)
{
    int c, i, j, imatch;

    j = 0;
    imatch = 0;
    for (j = 0; j < sbuf; j++) {
        if ((sbuf - j) < spattern && imatch == 0) return(-1);
        c = buffer[j];
        if (c == pattern[imatch]) {
            imatch++;
        } else if ((figncase == 1) &&
                   ((c & 0x5f) == (pattern[imatch] & 0x5f))) {
            imatch++;
        } else if (pattern[imatch] == '?') {  /*wildcard char*/
            imatch++;
        } else {
            if (imatch > 0) {
               if (j > 0) j--; /* try again with the first match char */
               imatch = 0;
	    }
        }
        if (imatch == spattern) break;
    }
    if (imatch == spattern) {
        i = (j+1) - imatch;  /*buffer[i] is the match */
        return(i);
    } else return (-1);  /*not found*/
}				/*end findmatch */

/* 
 * file_grep
 * Search (grep) for a pattern within a file.
 * Inputs:  fname  = file name to search
 *          pattn  = pattern string to search for
 *          line   = line buffer 
 *          sline  = size of line buffer
 *          bmode  = 0 to use last match, 1 to use first match, 
 *                   2 to specify starting line number & use first match.
 *          nret   = starting line number
 * Outputs: line   = resulting line (stringified) that matches pattn
 *          nret   = resulting line number within file (0-based)
 *          returns 0 if match, -1 otherwise
 */
static int file_grep(char *fname, char *pattn, char *line, int sline, 
		     char bmode, int *nret)
{
    FILE *fp;
    char buff[1024];
    int ret = -1;
    int i, plen, blen;
    int n = 0;
    int nstart = 0;

    if (bmode == 2) nstart = *nret;
    
    fp = fopen(fname,"r");
    if (fp == NULL) {
          fprintf(stderr,"file_grep: Cannot open %s\n",fname);
	  ret = -1;
    } else {
	 plen = strlen(pattn);
         while (fgets(buff, 1023, fp)) {
           if (n < nstart) {
		  n++;
		  continue;  /*skip until line# >= nstart*/
	   }
	   blen = strlen(buff);
	   /* check for pattern in this line */
	   i = findmatch(buff,blen,pattn,plen,0);
  	   if (i >= 0) {
		ret = 0;  /* found it, success */
		if (blen >= sline) blen = sline - 1;
		strncpy(line,buff,blen);
		line[blen] = 0;  /*stringify*/
                *nret = n;
		if (bmode > 0) break;  
	 	/* else keep looking, use last one if multiples */
	   }
           n++;  /*line number*/
         } /*end while*/
         fclose(fp);
    }  /*end else file opened*/
    return(ret); 
}  /*end file_grep*/

/*------------------------------------------------------------------------ 
 * decode_sel_entry
 * Parse and decode the SEL record into readable format.
 * This routine is constructed so that it could be used as a library
 * function.
 * Note that this routine outputs 14 of the 16 bytes in either text or
 * raw hex form.  For the two bytes not shown:
 *   . record type:   usually = 02, is shown otherwise (e.g. OEM type)
 *   . event msg revision: =03 if IPMI 1.0, =04 if IPMI 1.5 or IPMI 2.0
 *
 * Input:   psel, a pointer to the IPMI SEL record
 * Output:  outbuf, a description of the event, max 80 chars.
 * Called by: ReadSEL()
 * Calls:     fmt_time() get_misc_desc()
 *------------------------------------------------------------------------*/
void decode_sel_entry( uchar *pevt, char *outbuf)
{
	uchar dtype;
	char mystr[26] = "(123)";  /*for panic string*/
	char *pstr;
	char genstr[10] = "03  ";
	char poststr[24] = "OEM Post Code = %x%x";
	char *gstr;
	int i, j;
	time_t eventTime;
	uchar *evtime;
	char timebuf[40];
	uchar *pc;
	SEL_RECORD *psel;

	if (outbuf == NULL) return;
	if (pevt == NULL) {
		outbuf[0] = 0;
		return;
	}
	psel = (SEL_RECORD *)pevt;

		if (psel->record_type >= 0xe0) { /*OEM Record 26.3*/
		   /* 3 bytes header, 13 bytes data, no timestamp */
		   pc = (uchar *)&psel->timestamp;  /*bytes 4:16*/
		   sprintf(outbuf,"%04x OEM Event %02x %02x ", 
			(ushort)psel->record_id, pc[0], pc[1]);
		   j = strlen(outbuf);
		   for (i = 2; i < 13; i++) {  /* 4:16 = 13 bytes data */
		      if (psel->record_type == 0xf0) { 
		        /* panic string will be type 0xf0 */
			if (pc[i] == 0) break;
			outbuf[j++] = pc[i];
		      } else {
			sprintf(&outbuf[j],"%02x ",pc[i]);
			j += 3;
		      }
		   }
		   outbuf[j++] = '\n';
		   outbuf[j++] = 0;
		} else if (psel->record_type >= 0xc0) { /*OEM Record 26.3*/
		   /* 10 bytes header, 6 bytes data, has timestamp */
		   evtime = (uchar *)&psel->timestamp;
		   eventTime = evtime[0] + (evtime[1] << 8) + 
		   		(evtime[2] << 16) + (evtime[3] << 24);
		   fmt_time(eventTime, timebuf, sizeof(timebuf));
		   pc = (uchar *)&psel->generator_id;  /* byte 8 */
		   sprintf(outbuf,"%04x %s %02x%02x OEM Event ", 
			   psel->record_id, timebuf, pc[0],pc[1]);
		   j = strlen(outbuf);
		   for (i = 2; i < 8; i++) {  /* 16-(8+2) = 6 bytes data */
			sprintf(&outbuf[j],"%02x ",pc[i]);
			j += 3;
		   }
		   outbuf[j++] = '\n';
		   outbuf[j++] = 0;
		} else if (psel->record_type == 0x02) {
		  /* most records are record type 2 */
		    /* set dtype, used as array index below */
		    if (psel->sensor_type == 0xF3) dtype = 0x2D; /*OEM SMI*/
		    else if (psel->sensor_type >= NSTYPES) dtype = 0;
		    else dtype = psel->sensor_type;
		    switch(psel->sensor_type) {
		     case 0x20:   /*OS Crit Stop*/
		        /* Show first 3 chars of panic string */
		        mystr[0] = '(';
		        mystr[1] = psel->sensor_number & 0x7f;
		        mystr[2] = psel->event_data2 & 0x7f;
		        mystr[3] = psel->event_data3 & 0x7f;
			mystr[4] = ')';
			mystr[5] = 0;
			pstr = mystr;
			if (psel->sensor_number & 0x80) 
				strcat(mystr,"Oops!");
			if (psel->event_data2 & 0x80) 
				strcat(mystr,"Int!");
			if (psel->event_data3 & 0x80) 
				strcat(mystr,"NullPtr!");
		        break;
		     case 0x07:   /*Processor (CPU)*/
			i = psel->event_data1 & 0x0f;
			if (psel->event_trigger == 0x6f) {
			  /* Processor status sensor */
			  if (i >= NPROC) i = NPROC - 1;
			  sprintf(mystr,"%s",proc_str[i]);
			  pstr = mystr;
			} else if (psel->event_trigger == 0x03) {
			  if (i) pstr = "Proc Config Error";
			  else   pstr = "Proc Config OK";
                        } else {
			  /* else other processor sensor */
			  if (i) pstr = "ProcErr Asserted";
			  else   pstr = "ProcErr Deasserted";
			}
		        break;
		     case 0x0C:   /*Memory*/
			i = psel->event_data1 & 0x0f;
			if (i >= NMEM) i = NMEM - 1;
			sprintf(mystr,"%s, DIMM%d",mem_str[i],
					psel->event_data3);
			pstr = mystr;
		        break;
		     case 0x0f:    /*Post Errs*/
			switch (psel->event_data1)
			{ 
			   case 0x00:  /* System firmware errors */
				i = psel->event_data2;
				if (i > NFWERRS) i = NFWERRS;
        		        pstr = fwerrs[i].msg;
				break;
			   case 0x01:  /* System firmware hang  */
				i = psel->event_data2;
				if (i > NFWSTAT) i = NFWSTAT;
				sprintf(poststr,"hang, %s",fwstat[i].msg);
				pstr = poststr;
				break;
			   case 0x02:  /* System firmware progress */
				i = psel->event_data2;
				if (i > NFWSTAT) i = NFWSTAT;
				sprintf(poststr,"prog, %s",fwstat[i].msg);
				pstr = poststr;
				break;
			   case 0xa0:  /* OEM post codes */
				/* OEM post codes in bytes 2 & 3 (lo-hi) */
				sprintf(poststr,"POST Code %02x%02x",
					psel->event_data3,
					psel->event_data2);
				pstr = poststr;
				break;
			   default:  
			        pstr = get_misc_desc( psel->sensor_type,
						psel->sensor_number,
						psel->generator_id,
						psel->event_trigger,
						psel->event_data1,
						psel->event_data2,
						psel->event_data3);
				if (pstr == NULL) 
				   pstr = "POST Error";  /*default string*/
			}  /*end switch(data1)*/
		        break;
		     case 0x13:   /*Crit Int*/
			i = psel->event_data1 & 0x0f;
			if (i >= NCRITS) i = NCRITS - 1;
			pstr = crit_int_str[i];
		        break;
		     case 0x21:   /*Slot/Con*/
			i = psel->event_data1 & 0x0f;
			if (i >= NSLOTC) i = NSLOTC - 1;
			sprintf(mystr,"%s",slot_str[i]);
			/* could also decode data2/data3 here if valid */
			pstr = mystr;
		        break;
		     case 0x22:   /*ACPI Power state*/
			i = psel->event_data1 & 0x0f;
			if (i >= NACPIP) i = NACPIP - 1;
			sprintf(mystr,"%s",acpip_str[i]);
			pstr = mystr;
		        break;
		     case 0x29:   /*Battery*/
			i = psel->event_data1 & 0x0f;
			if (i >= NBATT) i = NBATT - 1;
			sprintf(mystr,"%s",batt_str[i]);
			pstr = mystr;
		        break;
		     case 0x2A:   /*Session Audit, new for IPMI 2.0*/
			i = psel->event_data1 & 0x0f;
			if (i >= NAUDIT) i = NAUDIT - 1;
			sprintf(mystr,"%s User %d",audit_str[i],
					psel->event_data2);
			    /* see also psel->event_data3 for cause/channel*/
			pstr = mystr;
		        break;
		     default:   /* all other sensor types, see sens_desc */
			pstr = get_misc_desc( psel->sensor_type,
			     psel->sensor_number,
			     psel->generator_id,
			     psel->event_trigger,
      			     psel->event_data1,
      			     psel->event_data2,
      			     psel->event_data3);
			if (pstr == NULL) {  /* none found, unknown */
			   mystr[0] = '-'; mystr[1] = 0;
			   pstr = mystr;
			}
		    }  /*end switch(sensor_type)*/

		    /* show the Generator ID */
		    sprintf(genstr,"%04x",psel->generator_id);
		    gstr = genstr;  /* default */
		    for (i = 0; i < NGDESC; i++)
		    {
			if (gen_desc[i].g_id == psel->generator_id)
			   gstr = (char *)gen_desc[i].desc;  
		    }

		    /*firmware timestamp is #seconds since 1/1/1970 localtime*/
		    evtime = (uchar *)&psel->timestamp;
		    eventTime = evtime[0] + (evtime[1] << 8) + 
		   		(evtime[2] << 16) + (evtime[3] << 24);
		    if (fdebug) {
			uchar tbuf[40];
			char *tz;  char *lctime;
			strftime(timebuf,sizeof(timebuf), "%x %H:%M:%S %Z",
				localtime(&eventTime));
			strftime(tbuf,sizeof(tbuf), "%x %H:%M:%S %Z",
				gmtime(&eventTime));
			tz = getenv("TZ");
			if (tz == NULL) tz = "";
			lctime = getenv("LC_TIME");
			if (lctime == NULL) lctime = "";
			SELprintf("%s\nTZ=%s, LC_TIME=%s, gmtime=%s\n",
				timebuf,tz,lctime,tbuf);
		    }
		    fmt_time(eventTime, timebuf, sizeof(timebuf));

		    if ((psel->generator_id == 0x0020) &&  /*from BMC*/
		        ((psel->event_trigger == 0x01) ||  /*threshold*/
		         (psel->event_trigger == 0x81))) { /*threshold ok*/
		    sprintf(outbuf,
			"%04x %s %s %02x %s #%02x %s act=%02x thr=%02x\n",
			(ushort)psel->record_id,
			timebuf,     /*  psel->timestamp */
			/* record_type,  2 = SEL record */
			/* evm_rev,   event msg rev =3 if 1.0, =4 if 1.5 */
			gstr,   /* psel->generator_id */
			psel->sensor_type,
			sensor_types[dtype] ,
			psel->sensor_number,
			pstr,
			psel->event_data2,    /*actual reading*/
			psel->event_data3 );  /*threshold value*/
		    } else {
		    sprintf(outbuf,
			"%04x %s %s %02x %s #%02x %s %02x [%02x %02x %02x]\n",
			(ushort)psel->record_id,
			timebuf,     /*  psel->timestamp */
			/* record_type,  2 = SEL record */
			/* evm_rev,   event msg rev =3 if 1.0, =4 if 1.5 */
			gstr,   /* psel->generator_id */
			psel->sensor_type,
			sensor_types[dtype] ,
			psel->sensor_number,
			pstr,
			psel->event_trigger,
			psel->event_data1,
			psel->event_data2,
			psel->event_data3 );
		    }
		}  /*endif type 2 */
		else {  /* other misc record type */
		   pc = (uchar *)&psel->record_type; 
		   sprintf(outbuf,"%04x Type%02x ", 
			(ushort)psel->record_id,pc[0]);
		   j = strlen(outbuf);
		   for (i = 1; i < 14; i++) {
			sprintf(genstr,"%02x ",pc[i]);
			strcat(outbuf,genstr);
		   }
		   strcat(outbuf,"\n");
		}  /*endif misc type*/
   return;
}  /*end decode_sel_entry()*/

#ifdef TEST
static char *progver   = "2.12";
static char *progname  = "events";
/*
 * Define TEST flag to "make events" and then feed it
 * raw event data from some log text for decoding.
 * Sample usage:
 * events fb 07 02 e5 1a b8 44 21 00 03 1d 9a 6f 40 8f ff
 * events 14 04 02 BE 35 13 45 33 40 04 0C 08 6F 20 00 04
 */
// char fdebug;
uchar htoi(uchar *inhex)
{
   // char rghex[16] = "0123456789ABCDEF";
   uchar val;
   uchar c;
   c = inhex[0] & 0x5f;  /* force cap */
   if (c > '9') c += 9;  /* c >= 'A' */
   val = (c & 0x0f) << 4;
   c = inhex[1] & 0x5f;  /* force cap */
   if (c > '9') c += 9;  /* c >= 'A' */
   val += (c & 0x0f);
   return(val);
}

/*
 * The events utility interprets standard 16-byte IPMI events into
 * human-readable form by default. 
 *
 * For interpreting the 47-byte hex data from SNMP Platform Event Traps,
 * specify events -p with the 34 data bytes following the 16-byte GUID.
 * 
 * Sample PET Data for events:
 * 0000: 3C A7 56 85 08 C5 11 D7 C3 A2 00 04 23 BC AC 12 
 * 0010: 51 14 11 72 38 58 FF FF 20 20 00 10 83 07 01 41 
 * 0020: 0F FF 00 00 00 00 00 19 00 00 01 57 00 22 C1    
 *
 * Sample events -p command from above data:
 * # events -p 51 14 11 72 38 58 FF FF 20 20 00 10 83 07 01 41 0F FF 00 00 00 00
 * RecId Date/Time_______ Source_ Evt_Type SensNum Evt_detail - Trig [Evt_data]
 * 0014 04/11/07 12:03:20 BMC  12 System Event #83 OEM System Boot 07 [41 0f ff]
 *
 * Offset Len  Meaning
 *   0    16   System GUID 
 *  16     2   Sequence Number/Cookie 
 *  18     4   Local Timestamp 
 *  22     2   UTC Offset 
 *  24     1   Trap Source Type (usu 0x20)
 *  25     1   Event Source Type (usu 0x20)
 *  26     1   Event Severity
 *  27     1   Sensor Device
 *  28     1   Sensor Number
 *  29     1   Entity
 *  30     1   Entity Instance
 *  31     8   Event Data (8 bytes max, 3 bytes used)
 *  39     1   Filler byte (usu 0x19)
 *  40     3   Manufacturer IANA number (Intel=0x000157)
 *  43     2   Product ID
 *  46     1   OEM data, C1="No more fields"
 */
#ifdef METACOMMAND
int ievents(int argc, char **argv)
#else
#ifdef WIN32
int __cdecl
#else
int
#endif
main(int argc, char **argv)
#endif
{
   uchar buf[50];
   uchar msg[132];
   uchar *pmsg;
   int i, len;
   char fPET = 0;
   char frawfile = 0;
   int rv = 0;
   
   printf("%s version %s\n",progname,progver);
   if (argc > 0) { argc--; argv++; }
   while ((argc > 0) && argv[0][0] == '-') 
   {
      if (argv[0][1] == 'x') { fdebug = 1; }
      else if (argv[0][1] == 'p') {
          fPET = 1; /*incoming data is in PET format*/
      }
      else if (argv[0][1] == 'f') { /* interpret raw SEL file from optarg */
	  frawfile = 1;
          if (argc > 1) { /*next argv is filename*/
             len = strlen(argv[1]);
             if (len >= sizeof(rawfil)) 
                len = sizeof(rawfil) - 1;
             strncpy(rawfil,argv[1],len);
             rawfil[len] = 0;  /*stringify*/
             argc--; argv++;
          }
      }
      else if (argv[0][1] == 's') { /* get sensor file from optarg */
          if (argc > 1) { /*next argv is filename*/
             len = strlen(argv[1]);
             if (len >= sizeof(sensfil)) 
                len = sizeof(sensfil) - 1;
             strncpy(sensfil,argv[1],len);
             sensfil[len] = 0;  /*stringify*/
             argc--; argv++;
          }
      }
      argc--; argv++;   
   }  /*end while options*/

   len = argc;  /*number of data bytes*/
   if (!fPET && len > 16) len = 16;
   if (!frawfile && (len < 16)) {
      printf("Need 16 bytes for an IPMI event, got %d bytes input\n",len);
      printf("Usage: events [-psx] 01 02 03 04 05 06 07 08 09 0a 0b 0c 0d 0e 0f 10\n");
      printf("where -p = decode PET event bytes, use 34 PET data bytes,\n");
      printf("           skipping the 1st 8 of the 47-byte PET data.  \n");
      printf("           Default assumes a 16-byte IPMI event.\n");
      printf("      -s = sensor file with output of 'ipmiutil sensor', used\n");
      printf("           to get the PET sensor_type from the sensor_num.\n");
      printf("           The default is %s\n",sensfil);
      printf("      -f = interpret File with raw SEL data captured via ipmiutil sel -r\n");
      printf("      -x = show eXtra debug messages\n");
      exit(1);
   }
   for (i = 0; i < len; i++)
   {
      if (fPET) msg[i] = htoi(argv[i]);
      else buf[i] = htoi(argv[i]);
   }
   if (fPET) {  /*reorder bytes to std event format*/
      /* msg[11] is sensor device */
      /* msg[12] is sensor number */
      /* msg[13] is Entity */
      uchar snum, styp;
      int  timestamp;
      int  yrs, time2; 
      char pattn[20];
      char sensline[100];
      int nline;

      pmsg = &msg[8];  /*skip the GUID*/
      if (fdebug) printf("decoding IPMI PET event bytes\n");
      snum = pmsg[12];
      styp = 0x12;  /*default to System sensor type, not msg[13]*/
      nline = 0;
      sprintf(pattn,"snum %02x",snum);
      /* Use this logic for both Linux and Windows */
      rv = file_grep(sensfil,pattn, sensline, sizeof(sensline), 1, &nline);
      if (rv == 0) {
          // if (fdebug) 
          printf("%s\n",sensline);
          styp = htoi(&sensline[25]);
      }
      else if (fdebug) printf("snum %02x not found, rv = %d\n",snum,rv);

      buf[0] = pmsg[1];   /*record id (sequence num)*/
      buf[1] = 0;        /*  was pmsg[0]; */
      buf[2] = 0x02;     /*event type*/
#ifdef RAW
      buf[3] = pmsg[5];   /*timestamp*/
      buf[4] = pmsg[4];   /*timestamp*/
      buf[5] = pmsg[3];   /*timestamp*/
      buf[6] = pmsg[2];   /*timestamp*/
#else
      timestamp = pmsg[5] + (pmsg[4] << 8) + (pmsg[3] << 16) + (pmsg[2] << 24);
      /* add 28 years, includes 7 leap days, less 1 hour TZ fudge */
      // yrs = ((3600 * 24) * 365 * 28) + (7 * (3600 * 24)) - 3600;
      yrs = 0x34aace70;
      time2 = timestamp + yrs;
      if (fdebug) 
          printf("timestamp: %08x + %08x = %08x\n",timestamp,yrs,time2);
      buf[3] = time2  & 0x000000ff;          /*timestamp*/
      buf[4] = (time2 & 0x0000ff00) >> 8;   /*timestamp*/
      buf[5] = (time2 & 0x00ff0000) >> 16;  /*timestamp*/
      buf[6] = (time2 & 0xff000000) >> 24;  /*timestamp*/
#endif
      buf[7] = pmsg[9];   /*generator_id*/
      buf[8] = 0;
      buf[9] = 0x04;     /*evm_rev*/
      buf[10] = styp;    /*derived sensor type, from above */
      buf[11] = snum;    /*sensor number*/
      /* set the event trigger based on context */
      if (styp == 0x12) buf[12] = 0x6f;  /*event trigger (sensor-specific)*/
      else if (styp == 0x09) buf[12] = 0x0b;  /*event trigger (Power Unit)*/
      else if (styp == 0x01 && pmsg[10] == 0x04) buf[12] = 0x81; /*temp ok*/
      else buf[12] = pmsg[14];  /*event trigger (Entity Instance) */
      memcpy(&buf[13],&pmsg[15],3);  /*event data*/
      decode_sel_entry(buf,msg);
      printf(msg);
   } else if (frawfile) {
      FILE *fp;
      char buff[256];
      char fvalid;
      fp = fopen(rawfil,"r");
      if (fp == NULL) {
	 printf("Cannot open file %s\n",rawfil);
      } else {
         if (fdebug) printf("decoding raw file with IPMI event bytes\n");
         printf("RecId Date/Time_______ Source_ Evt_Type SensNum Evt_detail - Trig [Evt_data]\n");
         while (fgets(buff, 255, fp)) {
	    len = strlen(buff);
	    fvalid = 0;
	    if (buff[0] >= '0' && (buff[0] <= '9')) fvalid = 1;
	    else if (buff[0] >= 'a' && (buff[0] <= 'f')) fvalid = 1;
	    else if (buff[0] >= 'A' && (buff[0] <= 'F')) fvalid = 1;
	    if (fvalid == 0) continue;
	    for (i = 0; i < 16; i++) {
               buf[i] = htoi(&buff[i*3]);
	    }
            decode_sel_entry(buf,msg);
            printf(msg);
         }
      }
   } else {
      if (fdebug) printf("decoding standard IPMI event bytes\n");
      if (fdebug) 
         printf("buf[10]: %02x %02x %02x %02x %02x %02x\n",buf[10],buf[11],
		buf[12],buf[13],buf[14],buf[15]); 
      decode_sel_entry(buf,msg);
      /* show header for the event record */
      printf("RecId Date/Time_______ Source_ Evt_Type SensNum Evt_detail - Trig [Evt_data]\n");
      printf(msg);
   }
   return(rv);
}
#endif

/* end events.c */
