/*
 * fruconfig.c
 *
 * This tool reads the FRU configuration, and optionally sets the asset
 * tag field in the FRU data.  See IPMI v1.5 spec section 28.
 * It can use either the Intel /dev/imb driver or VALinux /dev/ipmikcs.
 *
 * Author:  Andy Cress  arcress@users.sourceforge.net
 *
 * Copyright (c) 2001-2005 Intel Corporation. 
 *
 * 10/28/02 Andy Cress - created
 * 12/11/02 Andy Cress v0.9  - disable write until checksum fixed.
 * 12/13/02 Andy Cress v0.91 - don't abort if cc=c9 in load_fru loop.
 * 01/27/03 Andy Cress v0.92 - more debug, do checksums,
 * 01/28/03 Andy Cress v1.0  do writes in small chunks, tested w SCB2 & STL2
 * 02/19/03 Andy Cress v1.1  also get System GUID
 * 03/10/03 Andy Cress v1.2  do better bounds checking on FRU size
 * 04/30/03 Andy Cress v1.3  Board Part# & Serial# reversed
 * 05/13/03 Andy Cress v1.4  Added Chassis fields
 * 06/19/03 Andy Cress v1.5  added errno.h (email from Travers Carter)
 * 05/03/04 Andy Cress v1.6  BladeCenter has no product area, only board area
 * 05/05/04 Andy Cress v1.7  call ipmi_close before exit,
 *                           added WIN32 compile options.
 * 11/01/04 Andy Cress v1.8  add -N / -R for remote nodes   
 * 12/10/04 Andy Cress v1.9  add gnu freeipmi interface
 * 01/13/05 Andy Cress v1.10 add logic to scan SDRs for all FRU devices,
 *                           and interpret them
 * 01/17/05 Andy Cress v1.11 decode SPD Manufacturer
 * 01/21/05 Andy Cress v1.12 format SystemGUID display
 * 02/03/05 Andy Cress v1.13 fixed fwords bit mask in load_fru,
 *                           decode DIMM size from SPD also.
 * 02/04/05 Andy Cress v1.14 decode FRU Board Mfg DateTime
 * 03/16/05 Andy Cress v1.15 show Asset Tag Length earlier
 * 05/24/05 Andy Cress v1.16 only do write_asset if successful show_fru
 * 06/20/05 Andy Cress v1.17 handle Device SDRs also for ATCA
 * 08/22/05 Andy Cress v1.18 allow setting Product Serial Number also (-s),
 *                           also add -b option to show only baseboard data.
 * 10/31/06 Andy Cress v1.25 handle 1-char asset/serial strings (avoid c1)
 */
/*M*
Copyright (c) 2001-2005, Intel Corporation
All rights reserved.

Redistribution and use in source and binary forms, with or without 
modification, are permitted provided that the following conditions are met:

  a.. Redistributions of source code must retain the above copyright notice, 
      this list of conditions and the following disclaimer. 
  b.. Redistributions in binary form must reproduce the above copyright notice,
      this list of conditions and the following disclaimer in the documentation 
      and/or other materials provided with the distribution. 
  c.. Neither the name of Intel Corporation nor the names of its contributors 
      may be used to endorse or promote products derived from this software 
      without specific prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR 
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *M*/
#ifdef WIN32
#include <windows.h>
#include <stdio.h>
#include "getopt.h"
#else
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#endif
#include <time.h>

#include "ipmicmd.h" 

extern int errno;

static char *progname  = "fruconfig";
static char *progver   = "2.12";
static char fdebug = 0;
static char fpicmg = 0;
static char fonlybase = 0;
static char fwritefru = 0;
static char fdevsdrs = 0;
static char fshowlen = 0;
static char fgetfru  = 0;
static uchar bmc_sa = BMC_SA;  /*defaults to 0x20*/
static uchar guid[17] = "";
static uchar *frubuf = NULL;
#define FIELD_LEN   20
static int    sfru = 0;
static int    asset_offset = -1;
static int    asset_len = 0;
static int    sernum_offset = -1;
static int    sernum_len = 0;
static char   asset_tag[FIELD_LEN]  = {0};
static char   serial_num[FIELD_LEN] = {0};
static int    maxprod = 0;
static int ctype;
static uchar g_bus = PUBLIC_BUS;
static uchar g_sa  = 0;
static uchar g_lun = BMC_LUN;
static uchar g_addrtype = ADDR_SMI;
static uchar g_fruid = 0;  /* default to fruid 0 */

#define STRING_DATA_TYPE_BINARY         0x00
#define STRING_DATA_TYPE_BCD_PLUS       0x01
#define STRING_DATA_TYPE_SIX_BIT_ASCII  0x02
#define STRING_DATA_TYPE_LANG_DEPENDANT 0x03

#define FRUCHUNK_SZ   16
#define FRU_END         0xC1
#define FRU_EMPTY_FIELD 0xC0
#define FRU_TYPE_MASK   0xC0
#define FRU_LEN_MASK    0x3F

#define NUM_BOARD_FIELDS    6
#define NUM_PRODUCT_FIELDS  8
#define NUM_CHASSIS_FIELDS  3
#define MAX_CTYPE   26
char *ctypes[MAX_CTYPE] = { "", "Other", "Unknown", "Desktop", 
	"Low Profile Desktop", "Pizza Box", "Mini-Tower", "Tower", 
	"Portable", "Laptop", "Notebook", "Handheld", "Docking Station", 
	"All-in-One", "Sub-Notebook", "Space-saving", "Lunch Box", 
	"Main Server Chassis", "Expansion Chassis", "SubChassis", 
	"Bus Expansion Chassis", "Peripheral Chassis", 
	"RAID Chassis", /*0x17=23.*/ "Rack-Mount Chassis", 
	"New24" , "New25"};
	/* what about bladed chassies? */
char *ctype_hdr = 
"Chassis Type        :"; 
char *chassis[NUM_CHASSIS_FIELDS] = {
"Chassis Part Number :", 
"Chassis Serial Num  :",
"Chassis OEM Field   :" };
char *board[NUM_BOARD_FIELDS] = {
"Board Manufacturer  :",
"Board Product Name  :",
"Board Serial Number :",
"Board Part Number   :",
"Board FRU File ID   :",
"Board OEM Field     :" };
char *product[NUM_PRODUCT_FIELDS] = {
"Product Manufacturer:",
"Product Name        :",
"Product Part Number :",
"Product Version     :",
"Product Serial Num  :",  
"Product Asset Tag   :",
"Product FRU File ID :",
"Product OEM Field   :" };
char *asset_hdr = 
"   Asset Tag Length :";

#if 0
typedef struct {
	uchar len :6;
	uchar type:2;
	} TYPE_LEN;  /*old, assumes lo-hi byte order*/
#else
typedef struct {
	uchar len;
	uchar type;
	} TYPE_LEN;
#endif

void
free_fru(void)
{
   if (frubuf != NULL) {
      free(frubuf);
      frubuf = NULL;
   }
   return;
}

int
load_fru(uchar sa, uchar frudev)
{
   int ret = 0;
   uchar indata[16];
   uchar resp[18];
   int sresp;
   uchar cc;
   int sz;
   char fwords;
   ushort fruoff = 0;
   int i;
   int chunk;

   indata[0] = frudev;
   sresp = sizeof(resp);
   if (fdebug) printf("load_fru: sa = %02x, frudev = %02x\n",sa,frudev);
   ret = ipmi_cmd_mc(GET_FRU_INV_AREA,indata,1,resp,&sresp,&cc,fdebug);
   if (fdebug) printf("load_fru: inv ret = %d, cc = %x\n",ret,cc);
   if (ret != 0) return(ret);
   if (cc != 0) { ret = (cc & 0x00ff); return(ret); }

   sz = resp[0] + (resp[1] << 8);
   if (resp[2] & 0x01) { fwords = 1; sz = sz * 2; }
   else fwords = 0;

   frubuf = malloc(sz);
   if (frubuf == NULL) return(errno);
   sfru = sz;
      
   /* Loop on READ_FRU_DATA */
   for (i = 0, chunk=FRUCHUNK_SZ; i < sz; i+=chunk)
   {
	if ((i+chunk) >= sz) chunk = sz - i;
	indata[0] = frudev;  /* FRU Device ID */
	if (fwords) {
	   indata[3] = chunk / 2;
	   fruoff = (i/2);
	} else {
	   indata[3] = chunk;
	   fruoff = i;
	}
        indata[1] = fruoff & 0x00FF;
        indata[2] = (fruoff & 0xFF00) >> 8;
	sresp = sizeof(resp);
        ret = ipmi_cmd_mc(READ_FRU_DATA,indata,4,resp,&sresp,&cc,fdebug);
	if (ret != 0) break;
	else if (cc != 0) {
		if (i == 0) ret = cc & 0x00ff; 
		if (fdebug) printf("read_fru[%d]: ret = %d cc = %x\n",i,ret,cc);
		break; 
	}
        memcpy(&frubuf[i],&resp[1],chunk);
   }
   if ((frudev == 0) && (sa == bmc_sa)) { /*main system fru*/
     sresp = sizeof(resp);
     ret = ipmi_cmd_mc(GET_SYSTEM_GUID,indata,0,resp,&sresp,&cc,fdebug);
     if (fdebug) printf("system_guid: ret = %d, cc = %x\n",ret,cc);
     if (ret == 0 && cc == 0) {
	if (fdebug) {
	   printf("system guid (%d): ",sresp);
	   for (i=0; i<17; i++) printf("%02x ",resp[i]);
	   printf("\n");
	}
	memcpy(&guid,&resp,16);
   	guid[16] = 0;
     }
   } /*endif*/

   return(ret);
}

static void decode_string(unsigned char type,
                          unsigned char language_code,
                          unsigned char *source,
                          char          *target,
                          int           size)
{
  unsigned char *s = &source[0];
  unsigned char *d = &target[0];
 
  if (size == 0 || size == 1) return;
  memset(target, 0, size);
  if (type == STRING_DATA_TYPE_BCD_PLUS) {
    while(size)
      {
      if ((*s <= (unsigned char) 0x09))
        *d++ = (unsigned char) 0x30 + (unsigned char) *s++;
      else
        {
        if (*s == (unsigned char) 0x0A)
          *d++ = (unsigned char) ' ';
        else if (*s == (unsigned char) 0x0B)
          *d++ = (unsigned char) '-';
        else if (*s == (unsigned char) 0x0C)
          *d++ = (unsigned char) '.';
        else if ((*s <= (unsigned char) 0x0D) && (*s <= (unsigned char) 0x0F))
          {
          *d++ = (unsigned char) '*';
          }
        s++;
        }
      size --;
      }
      target[size] = 0x0;
  } else if (type == STRING_DATA_TYPE_SIX_BIT_ASCII) {
    printf("Six bit ASCII decode not supported\n");
  } else if (type == STRING_DATA_TYPE_LANG_DEPENDANT) {
    if ((language_code == 0x00) || (language_code == 0x25) || (language_code == 0x19))
      {
      strncpy(target, source, size);
      target[size] = 0x0;
      }
    else
      {
      printf("Language 0x%x dependant decode not supported\n",
            language_code);
      return;
      }
  } else if (type == 0) {  /*invalid, must be empty*/
      return;
  } else {
    printf("Unable to decode type 0x%.2x\n",type);
    return;
    }
}


uchar calc_cksum(uchar *pbuf,int len)
{
   int i; 
   uchar cksum;
   uchar sum = 0;
  
   for (i = 0; i < len; i++) sum += pbuf[i];
   cksum = 0 - sum; 
   return(cksum);
}

static void dumpbuf(uchar *pbuf,int sz)
{
   uchar line[9];
   uchar a;
   int i, j;

   line[0] = 0; line[8] = 0;
   j = 0;
   for (i = 0; i < sz; i++) {
	   if (i % 8 == 0) { j = 0; printf("%s\n  %04d: ",line,i); }
	   a = pbuf[i];
	   if (a < 0x20 || a > 0x7f) a = '.';
	   line[j++] = a;
	   printf("%02x ",pbuf[i]);
   }
   printf("\n");
   return;
} 

#define NSPDMFG 7
static struct { 
   uchar id;  char *str;
} spd_mfg[NSPDMFG] = {  /* see JEDEC JEP106 doc */
{ 0x2c, "Micron" },
{ 0x15, "Philips Semi" },
{ 0x1c, "Mitsubishi" },
{ 0xce, "Samsung" },
{ 0xc1, "Infineon" },
{ 0x98, "Kingston" },
{ 0x89, "Intel" }
};

static void 
show_spd(uchar *spd, int lenspd)
{
   int sz;
   char *pstr;
   uchar mrev;
   int i;

   sz = spd[0];  /* sz should == lenspd */
   printf("Memory SPD Size     : %d\n",lenspd);
   if (spd[2] == 0x07) pstr = "DDR";
   else /* usu 0x04 */ pstr = "SDRAM";
   printf("Memory Type         : %s\n",pstr);
   printf("Module Density      : %d MB per bank\n", (spd[31] * 4));
   printf("Module Banks        : %d banks\n", spd[5]);
   printf("Module Rows, Cols   : %d rows, %d cols\n", spd[3], spd[4]);
   if (spd[11] == 0x00) pstr = "Non-parity";
   else /* usu 0x02 */  pstr = "ECC";
   printf("DIMM Config Type    : %s\n",pstr);
   for (i = 0; i < NSPDMFG; i++) 
	if (spd_mfg[i].id == spd[64]) break;
   if (i == NSPDMFG) pstr = "";  /* not found, use null string */
   else pstr = spd_mfg[i].str;
   printf("Manufacturer ID     : %s (0x%02x)\n", pstr, spd[64]);
   mrev = spd[91];  /* save this byte for later */
   spd[91] = 0;     /*stringify part number */
   printf("Manufacturer Part#  : %s\n",&spd[73]);
   printf("Manufacturer Rev    : %02x %02x\n",mrev,spd[92]);
   printf("Manufacturer Date   : year=%02d week=%02d\n", spd[93],spd[94]);
   printf("Assembly Serial Num : %02x%02x%02x%02x\n",
		spd[95], spd[96], spd[97], spd[98]);
   spd[91] = mrev;  /* restore byte, so ok to repeat later */
   return;
}

int
show_fru(uchar frudev, uchar sa)
{
   int ret = 0;
   int i, sz;
   uchar *pfru;
   uchar lang;
   TYPE_LEN tl;
   char newstr[64];
   int iaoff, ialen, bdoff, bdlen; 
   int proff, prlen, choff, chlen;

   if (frubuf[0] == 0x80) {    /* 0x80 = type for DIMMs (SPD) */
      /* FRU Header: 80 08 07 0c 0a 01 48 00 (DIMM) */
      sz = frubuf[0];
      if (fdebug) {
	printf("DIMM SPD Body (size=%d/%d): ",sz,sfru);
	dumpbuf(frubuf,sfru);
      }
      show_spd(frubuf,sfru);
      return(ret);
   }

   /*
    * FRU header:
    *  0 = format_ver (01 is std, usu 0x80 if DIMM)
    *  1 = internal_use offset
    *  2 = chassis_info offset
    *  3 = board_info offset   (usu 6 fields)
    *  4 = product_info offset (usu 8 fields)
    *  5 = multirecord offset
    *  6 = pad (00)
    *  7 = header checksum (zero checksum)
    * FRU Header: 01 01 02 09 13 00 00 e0 (BMC)
    * FRU Header: 01 00 00 00 01 07 00 f7 (Power Cage)
    */
   if ((frudev == 0) && (sa == bmc_sa))
      printf("Mainboard FRU Size  : %d\n",sfru);
   else 
      printf("Component FRU Size  : %d\n",sfru);

   sz = 8;  /*minimum for common header*/
   for (i = 1; i < 6; i++) 	/* walk thru offsets */
	if (frubuf[i] != 0) sz = frubuf[i] * 8;
   if (sz > 8)   		/* if have at least one section */
      sz += frubuf[sz+1] * 8;   /* add length of last section */
   /* Now, sz = size used, sfru = total available size */
   if (sz > sfru) {
      if (fdebug) {
        printf("FRU Header: ");
        for (i = 0; i < 8; i++) printf("%02x ",frubuf[i]);
      }
      printf("FRU size out of bounds: available=%d used=%d\n",sfru,sz);
      printf("Please apply the correct FRU/SDR diskette\n");
      return(-1);
   }
   /* internal area offset, length */
   iaoff = frubuf[1] * 8;  
   ialen = frubuf[iaoff + 1] * 8;
   /* chassis area offset, length */
   choff = frubuf[2] * 8;  
   chlen = frubuf[choff + 1] * 8;
   /* board area offset, length */
   bdoff = frubuf[3] * 8;  
   bdlen = frubuf[bdoff + 1] * 8;
   /* product area offset, length */
   proff = frubuf[4] * 8;  
   prlen = frubuf[proff + 1] * 8;

   if (fdebug) {
      int moff, mlen;
      printf("FRU Header: ");
      for (i = 0; i < 8; i++) printf("%02x ",frubuf[i]);
      printf("\n");
      printf("FRU Body (size=%d/%d): ",sz,sfru);
      dumpbuf(frubuf,sfru);
      printf("internal off=%d, len=%d, cksum = %02x\n",
		iaoff,ialen,calc_cksum(&frubuf[iaoff],ialen-1));
      printf("chassis off=%d, len=%d, cksum = %02x\n",
		choff,chlen,calc_cksum(&frubuf[choff],chlen-1));
      printf("board off=%d, len=%d, cksum = %02x\n",
		bdoff,bdlen,calc_cksum(&frubuf[bdoff],bdlen-1));
      printf("prod  off=%d, len=%d, cksum = %02x\n",
		proff,prlen,calc_cksum(&frubuf[proff],prlen-1));
      /* Multi-record area (may not be present) */
      moff = bdoff + bdlen + prlen;
      mlen = frubuf[moff + 1] * 8; 
      printf("multi off=%d, len=%d, fru sz=%d\n",
		moff,mlen,sz);  // calc_cksum(&frubuf[moff],mlen-1));
      pfru = &frubuf[moff];
      tl.type = (pfru[0] & FRU_TYPE_MASK) >> 6;
      tl.len  = pfru[0] & FRU_LEN_MASK;
      for (i = 0; i < 10; i++)
      {
	 if (moff >= sz) break;  
         if (pfru[0] == FRU_END) break;  /*0xC1 = end of FRU area*/
         tl.type = (pfru[0] & FRU_TYPE_MASK) >> 6;
         tl.len  = pfru[0] & FRU_LEN_MASK;
         pfru++;
         {
	   newstr[0] = 0;
	   decode_string(tl.type,25,pfru,newstr,tl.len);
	   printf("Multi[%d] %s\n",i,newstr);
         }
         pfru += tl.len;
         moff += (tl.len + 1);
      }
   }  /*endif fdebug, show header*/

   if (choff != 0) {
      /* show Chassis area fields */
      pfru = &frubuf[choff];
      lang = 25;  /* English */
      ctype = pfru[2];  /*chassis type*/
      if (fdebug) printf("ctype=%x\n",ctype);
      if (ctype >= MAX_CTYPE) ctype = MAX_CTYPE - 1;
      printf("%s %s\n",ctype_hdr,ctypes[ctype]);
      pfru += 3;  /* skip chassis header */
      for (i = 0; i < NUM_CHASSIS_FIELDS; i++)
      {
         if (pfru[0] == FRU_END) break;  /*0xC1 = end of FRU area*/
         tl.type = (pfru[0] & FRU_TYPE_MASK) >> 6;
         tl.len  = pfru[0] & FRU_LEN_MASK;
         pfru++;
         {
	   newstr[0] = 0;
	   decode_string(tl.type,lang,pfru,newstr,tl.len);
	   printf("%s %s\n",chassis[i],newstr);
         }
         pfru += tl.len;
      }
      if (fdebug) printf("num Chassis fields = %d\n",i);
   }

   if (bdoff != 0) {
      long nMin, nSec;
      /* show Board area fields */
      pfru = &frubuf[bdoff];
      lang = pfru[2];
      /* Decode board mfg date-time (num minutes since 1/1/96) */
      nMin = pfru[3] + (pfru[4] << 8) + (pfru[5] << 16);
		  /* 13674540 min from 1/1/70 to 1/1/96 */
      nSec = (nMin + 13674540) * 60;
      printf("Board Mfg DateTime  : %s",ctime(&nSec));
      pfru += 6;  /* skip board header */
      for (i = 0; i < NUM_BOARD_FIELDS; i++)
      {
         if (pfru[0] == FRU_END) break;  /*0xC1 = end*/
         tl.type = (pfru[0] & FRU_TYPE_MASK) >> 6;
         tl.len  = pfru[0] & FRU_LEN_MASK;
         pfru++;
         {
	   newstr[0] = 0;
	   decode_string(tl.type,lang,pfru,newstr,tl.len);
	   printf("%s %s\n",board[i],newstr);
         }
         pfru += tl.len;
      }
      if (fdebug) printf("num Board fields = %d\n",i);
   }

   if (proff != 0) {
      /* show Product area fields */
      pfru = &frubuf[proff];
      maxprod = pfru[1] * 8;
      lang = pfru[2];
      pfru += 3;  /* skip product header */
      for (i = 0; i < NUM_PRODUCT_FIELDS; i++)
      {
         if (*pfru == FRU_END) break;  /*0xC1 = end*/
         if (*pfru == 0) *pfru = FRU_EMPTY_FIELD; /* fix a broken table */
         tl.type = (pfru[0] & FRU_TYPE_MASK) >> 6;
         tl.len  = pfru[0] & FRU_LEN_MASK;
         if (i == 5) {  /* asset tag */
	   asset_offset = (pfru - frubuf);
	   asset_len    = tl.len;
	   if (fdebug) printf("asset dtype=%d lang=%d len=%d\n",
				tl.type,lang,tl.len);
   	   if (fshowlen && (frudev == 0) && (sa == bmc_sa)) {
		/* show asset tag length for main board */
		printf("%s %d\n",asset_hdr,asset_len);
	   }
         } else if (i == 4) {  /* serial number */
	   sernum_offset = (pfru - frubuf);
	   sernum_len    = tl.len;
	   if (fdebug) printf("sernum dtype=%d lang=%d len=%d\n",
				tl.type, lang, tl.len);
	 } 
         pfru++;
         {
	   newstr[0] = 0;
	   decode_string(tl.type,lang,pfru,newstr,tl.len);
	   printf("%s %s\n",product[i],newstr);
         }
         pfru += tl.len;
      }
      if (fdebug) 
	   printf("num Product fields = %d, last=%x, max = %d\n", 
		   i,*pfru,maxprod );
   }
   if (*pfru == 0x00) *pfru = 0xC1;  /* insert end char if broken */

   if ((frudev == 0) && (sa == bmc_sa)) {
	char *s;
	printf("System GUID         : ");
	for (i=0; i<16; i++) {
	   if ((i == 4) || (i == 6) || (i == 8) || (i == 10)) s = "-";
	   else s = "";
	   printf("%s%02x",s,guid[i]);
	}
	printf("\n");

   }
   return(ret);
}


int
write_asset(char *tag, char *sernum, int flag)
{
   int ret = -1;
   uchar newdata[259];  /* max 4*64 + 3, product area up to serial */
   uchar req[25];
   uchar resp[16];
   int sresp;
   uchar cc;
   ushort fruoff;
   int alen;
   int snlen;
   uchar *pfru;
   uchar *pnew;
   int i, j, plen; 
   int prod_offset; 
   uchar chksum;
   int chunk;
   char fdoasset  = 0;
   char fdosernum = 0;

   if ((flag & 0x01) != 0) {
      fdoasset = 1;
      alen = strlen(tag);  /*new*/
   } else {
      fdoasset = 0;
      alen = asset_len;  /*old*/
   }
   if ((flag & 0x02) != 0) {
      fdosernum = 1;
      snlen = strlen(sernum); /*new*/
   } else {
      fdosernum = 0;
      snlen = sernum_len;  /*old*/
   }
   /* 
    * Abort if offset is negative or if it would clobber
    * fru header (8 bytes). 
    * Abort if asset tag is too big for fru buffer.
    */
   if (fdebug) {
       printf("write_asset: asset_off=%d asset_len=%d alen=%d  sFRU=%d\n",
		   asset_offset,asset_len,alen,sfru);
       printf("            sernum_off=%d sernm_len=%d snlen=%d maxprod=%d\n",
		   sernum_offset,sernum_len,snlen,maxprod);
   }
   if (asset_offset < 8) return -1;
   if (asset_offset + alen > sfru) return -1;
   if (sernum_offset < 8) return -1;
   if (sernum_offset + snlen > sfru) return -1;
   if (alen <= 1 || snlen <= 1) return -1;
   
   prod_offset = frubuf[4] * 8;           // offset of product data
   plen = frubuf[prod_offset+1] * 8;      // length of product data 
   /* Check if asset tag will fit in product data area of FRU. */
   if (fdebug) 
	printf("write_asset: fru[4,p]=[%02x,%02x] prod_off=%d, plen=%d\n",
		frubuf[4],frubuf[prod_offset+1],prod_offset,plen);
   /* asset comes after sernum, so this check works for both */
   if (plen < asset_len) return -2;       // asset bigger than product data
   if (plen > sizeof(newdata)) return -2; // product data bigger than buffer

   memset(newdata,0,plen);
   j = sernum_offset - prod_offset;
   pfru = &frubuf[prod_offset]; 
   pnew = &newdata[0];
   /* Copy fru data before serial number */
   memcpy(pnew,pfru,j);
   pfru += j; 
   pnew += j;

   /* Copy new or old serial number */
   if (fdosernum) {
      pnew[0] = (snlen | FRU_TYPE_MASK);  /*add type=3 to snlen*/
      memcpy(++pnew,sernum,snlen);
   } else {
      pnew[0] = (snlen | FRU_TYPE_MASK);  /*type/len byte*/
      memcpy(++pnew,&frubuf[sernum_offset+1],snlen);
   }
   if (fdebug) printf("fru[%d]: %02x %02x %02x, j=%d, snlen=%d\n",
		   (pfru-frubuf),pfru[0],pfru[1],pfru[2],j,snlen);
   /* increment past serial number, to asset tag */
   j += sernum_len + 1;
   pfru += sernum_len + 1;
   pnew += snlen;  /*already ++pnew above*/

   /* Copy new or old asset tag */
   if (fdoasset) {
      pnew[0] = (alen | FRU_TYPE_MASK);  /*add type=3 to len*/
      memcpy(++pnew,tag,alen);
   } else {
      pnew[0] = (alen | FRU_TYPE_MASK);  /*type/len byte*/
      memcpy(++pnew,&frubuf[asset_offset+1],alen);
   }

   if (fdebug) printf("fru[%d]: %02x %02x %02x, j=%d, alen=%d\n",
		   (pfru-frubuf),pfru[0],pfru[1],pfru[2],j,alen);
   /* copy trailing fru data from saved copy */
   j += asset_len + 1;
   pfru += asset_len + 1;
   pnew += alen;  
   if (fdebug) printf("fru[%d]: %02x %02x %02x, j=%d, remainder\n",
		   (pfru-frubuf),pfru[0],pfru[1],pfru[2],j);
   for (i = 0; i < (plen-j); i++) {
     pnew[i] = pfru[i];
     if (pfru[i] == 0xC1) break;
   }

   /* include new checksum (calc over Product area) */
   chksum = calc_cksum(&newdata[0],plen-1);
   newdata[plen-1] = chksum;

   if (fdebug) {
	printf("new buffer (%d):",plen);
        for (i = 0; i < plen; i++) {
		if (i % 8 == 0) printf("\n %04d: ",i);
		printf("%02x ",newdata[i]);
	}
	printf("\n");
   }

   /* Write the buffer in small 16-byte chunks */
   req[0] = 0x00;  /* FRU Device ID (fruid) */
   fruoff = prod_offset;
   chunk = FRUCHUNK_SZ;
   for (i = 0; i < plen; i += chunk) {
	req[1] = fruoff & 0x00ff;
	req[2] = (fruoff & 0xff00) >> 8;
	if ((i + chunk) > plen) chunk = plen - i;
	memcpy(&req[3],&newdata[i],chunk);
	if (fdebug) {
	   printf("write_fru_data[%d] (len=%d): ",i,chunk+3);
	   for (j = 0; j < chunk+3; j++) printf("%02x ",req[j]);
	   printf("\n");
	}
	sresp = sizeof(resp);
	ret = ipmi_cmd_mc(WRITE_FRU_DATA,req,(uchar)(chunk+3),resp,&sresp,
			&cc,fdebug);
	if ((ret == 0) && (cc != 0)) ret = cc & 0x00ff; 
	if (fdebug && ret == 0) 
		printf("write_fru_data[%d]: %d bytes written\n",i,resp[0]);
	if (ret != 0) break;
	fruoff += chunk;
   }
   return(ret);
}

#define CHUNKSZ   16
#define LAST_REC   0xffff
#define STR_OFF   16

int get_sdr(ushort recid, ushort resid, ushort *recnext, 
		uchar *sdr, int *slen, uchar *pcc)
{
	uchar idata[6];
	uchar rdata[64];
	int sresp;
	ushort cmd;
	uchar cc = 0;
        int len = 0;
	int rc;
	
	idata[0] = resid & 0x00ff;
	idata[1] = (resid & 0xff00) >> 8;
	idata[2] = recid & 0x00ff;
	idata[3] = (recid & 0xff00) >> 8;
	idata[4] = 0;  /*offset*/
	idata[5] = CHUNKSZ;  /*bytes to read*/
	sresp = sizeof(rdata);
        if (fdevsdrs) cmd = GET_DEVICE_SDR;
        else cmd = GET_SDR;
	rc = ipmi_cmd_mc(cmd, idata, 6, rdata, &sresp,&cc, fdebug);
	if (fdebug) printf("get_sdr[%x] ret = %d cc = %x sresp = %d\n",
				recid,rc,cc,sresp);
	*pcc = cc;
	if (rc == 0 && cc == 0) {
	   *recnext = rdata[0] + (rdata[1] << 8);
	   memcpy(sdr,&rdata[2],sresp-2);
	   len = sresp-2;
           /* if an SDR locator record, get the rest of it. */
	   if (sdr [3] == 0x11 || sdr[3] == 0x12)
	      if (*slen > CHUNKSZ) {
		idata[0] = resid & 0x00ff;
		idata[1] = (resid & 0xff00) >> 8;
		idata[2] = recid & 0x00ff;
		idata[3] = (recid & 0xff00) >> 8;
		idata[4] = CHUNKSZ;  /*offset*/
		idata[5] = *slen - CHUNKSZ;  /*bytes to read*/
		sresp = sizeof(rdata);
		rc = ipmi_cmd_mc(cmd, idata, 6, rdata, &sresp,&cc, fdebug);
		if (fdebug) printf("get_sdr[%x] 2nd ret=%d cc=%x sresp=%d\n",
				recid,rc,cc,sresp);
		if (rc == 0 && cc == 0) {
		   sresp -= 2;
	   	   memcpy(&sdr[len],&rdata[2],sresp);
		   len += sresp;
		}
	   }
	   *slen = len;
	}
	return(rc);
}


#ifdef METACOMMAND
int i_fru(int argc, char **argv)
#else
#ifdef WIN32
int __cdecl
#else
int
#endif
main(int argc, char **argv)
#endif
{
   int ret;
   char c;
   char DevRecord[16];
   ushort recid;
   ushort nextid;
   ushort rsvid;
   uchar sdr[40];
   uchar cc;
   uchar sa;
   uchar fruid;
   int prod_id, vend_id;
   int len; //, i;

   printf("%s: version %s\n",progname,progver);
   while ( (c = getopt( argc, argv,"a:bm:s:T:V:J:EYF:P:N:R:U:x?")) != EOF )
      switch(c) {
          case 'x': fdebug = 1;      break;
          case 'b': fonlybase = 1;   break;
          case 'a':
		fwritefru |= 0x01;
		if (optarg) {
                   len = strlen(optarg);
                   if (len >= FIELD_LEN) len = FIELD_LEN - 1;
                   strncpy(asset_tag,optarg,len);
                   if (len == 1) {  /* add a space */
                       asset_tag[1] = ' ';
                       asset_tag[2] = 0;
                   }
                }
		break;
          case 's':
		fwritefru |= 0x02; 
		if (optarg) { 
                   len = strlen(optarg);
                   if (len >= FIELD_LEN) len = FIELD_LEN - 1;
                   strncpy(serial_num,optarg,len);
                   if (len == 1) {  /* add a space */
                       serial_num[1] = ' ';
                       serial_num[2] = 0;
                   }
                }
		break;
          case 'm': /* specific MC, 3-byte address, e.g. "409600" */
                    g_bus = htoi(&optarg[0]);  /*bus/channel*/
                    g_sa  = htoi(&optarg[2]);  /*device slave address*/
                    g_lun = htoi(&optarg[4]);  /*LUN*/
                    g_addrtype = ADDR_IPMB;
                    printf("IPMB FRU for MC at bus=%x sa=%x lun=%x\n",
		            g_bus,g_sa,g_lun);
                    break;
          case 'N':    /* nodename */
          case 'U':    /* remote username */
          case 'P':    /* remote password */
          case 'R':    /* remote password */
          case 'E':    /* get password from IPMI_PASSWORD environment var */
          case 'F':    /* force driver type */
          case 'T':    /* auth type */
          case 'J':    /* cipher suite */ 
          case 'V':    /* priv level */
          case 'Y':    /* prompt for remote password */
                parse_lan_options(c,optarg,fdebug);
                break;
          default:
                printf("Usage: %s [-bmx -a asset_tag -s ser_num -NUPREFTVY]\n",
			 progname);
		printf("   -a tag   Sets the Product Asset Tag\n");
		printf("   -b       Only show Baseboard FRU data\n");
                printf("   -m002000 specific MC (bus 00,sa 20,lun 00)\n");
		printf("   -s snum  Sets the Product Serial Number\n");
		printf("   -x       Display extra debug messages\n");
		print_lan_opt_usage();
                exit(1);
      }

   ret = ipmi_getdeviceid( DevRecord, sizeof(DevRecord),fdebug);
   if (ret == 0) {
      uchar ipmi_maj, ipmi_min;
      ipmi_maj = DevRecord[4] & 0x0f;
      ipmi_min = DevRecord[4] >> 4;
      vend_id = DevRecord[6] + (DevRecord[7] << 8) + (DevRecord[8] << 16);
      prod_id = DevRecord[9] + (DevRecord[10] << 8);
      printf("-- BMC version %x.%x, IPMI version %d.%d \n",
             DevRecord[2],  DevRecord[3],
             ipmi_maj, ipmi_min);
      if ((DevRecord[1] & 0x80) == 0x80) fdevsdrs = 1;
      if (vend_id == VENDOR_NEC) fdevsdrs = 0;
   } else { 
        show_outcome(progname,ret);  
	ipmi_close_(); 
	exit(1); 
   }
 
   ret = ipmi_getpicmg( DevRecord, sizeof(DevRecord),fdebug);
   if (ret == 0)  fpicmg = 1;
   if ((vend_id == VENDOR_INTEL) && (!fpicmg))
       fdevsdrs = 0;   /* override, use SDR repository*/
   if (fdebug) printf("bmc_sa = %02x  fdevsdrs = %d\n",bmc_sa,fdevsdrs);

   if (g_addrtype == ADDR_IPMB) {
       /* target a specific MC via IPMB (usu a picmg blade) */
       if (fdebug) printf("set_mc: %02x:%02x:%02x type=%d\n",
		          g_bus,g_sa,g_lun,g_addrtype);
       ipmi_set_mc(g_bus,g_sa,g_lun,g_addrtype);
       fonlybase = 1;   /*only show this MC*/
   } else {
       g_sa  = bmc_sa;  /* BMC_SA = 0x20 */
   }

   if (!fonlybase) {
     /* loop thru SDRs to find FRU devices */
     {  /* reserve the SDR repository */
	uchar resp[16];
	int sresp;
	uchar cc;
	ushort cmd;
        sresp = sizeof(resp);
        if (fdevsdrs) cmd = RESERVE_DEVSDR_REP;
        else cmd = RESERVE_SDR_REP;
        ret = ipmi_cmd_mc(cmd, NULL, 0, resp, &sresp, &cc, 0);
        if (fdebug) printf("ipmi_cmd RESERVE status = %d, cc = %x\n",ret,cc);
        rsvid = resp[0] + (resp[1] << 8);
     }
     recid = 0;
     while (recid != LAST_REC)
     {
	char idstr[17];
	int ilen;

	len = sizeof(sdr); /*sizeof(sdr); get 32 sdr bytes*/
	ret = get_sdr(recid,rsvid,&nextid,sdr,&len,&cc);
	if ((ret != 0) || (cc != 0)) {
	    printf("SDR[%04x] error %d ccode = %x\n",recid,ret,cc);
	    break;
	}
        fgetfru = 0;
	if ((sdr[3] == 0x11) || (sdr[3] == 0x12)) /* SDR FRU or IPMB type */
	{
	    if (len > STR_OFF) {
		ilen = len - STR_OFF;
		memcpy(idstr,&sdr[STR_OFF],ilen);
		idstr[ilen] = 0;
	    } else idstr[0] = 0;
	    sa = sdr[5];  /* usu 0x20 for bmc_sa */
	    /* Get its FRU data */
	    if ((sdr[3] == 0x11) && (sdr[7] & 0x80)) {  /* FRU SDRs */
	      /* It is a logical FRU device */
	      printf("SDR[%04x] FRU  %02x %02x %02x %02x %s\n", recid, 
			sdr[5],sdr[6],sdr[12],sdr[13],idstr);
	      fruid = sdr[6];
              fruid = sdr[6];
              if (sa == bmc_sa && fruid == 0) /*do this below*/;
              else
                switch(sdr[12])   /*FRU entity id*/
                {
                  case 0x20:   /*DIMM*/
                  case 0x15:   /*Power Cage*/
                  case 0x0a:   /*Power Supply*/
                  default:
                        fgetfru = 1;
                        break;
                }
            } else if (sdr[3] == 0x12) {  /* IPMB SDRs (DLRs for MCs) */
                printf("SDR[%04x] IPMB %02x %02x %02x %02x %s\n", recid,
                        sdr[5],sdr[6],sdr[12],sdr[13],idstr);
                fruid = 0;  /*every MC must have fruid 0*/
                if (sa == bmc_sa && fruid == 0) {  /*do bmc_sa,0 below*/
                   if (fdebug) printf("do bmc_sa %02x below\n",sa);
                } else if (sa == 0x28) { /*do nothing for Bridge Ctlr sa=0x28*/
                   if (fdebug) printf("skipping IPMB sa %02x, %02x\n",sa,fruid);
                } else if (sa == 0xC0) { /* HotSwap Backplane (sa=0xC0) */
                   /* Note: Loading sa 0xC0 over ipmi lan gives a timeout
                    * error, but it works locally. */
                   fgetfru = 1;
                } else  { /* other misc sa,fruid */
                   fgetfru = 1;
                }
            }
            if (fgetfru) {
                uchar adrtype;
                adrtype = g_addrtype;
                if (fdebug) printf("set_mc %02x:%02x:%02x type=%d fruid=%02x\n",
				g_bus,sa,g_lun,adrtype,fruid);
                ipmi_set_mc(g_bus, sa, g_lun,adrtype);
                ret = load_fru(sa,fruid);
                if (ret != 0) {
                   if (ret == 0xC3) printf("\tFRU not present\n");
                   else printf("load_fru(%x,%x) error = %d (0x%x)\n",
                                sa,fruid,ret,ret);
                } else {
                   ret = show_fru(fruid,sa);
                   if (ret != 0) printf("show_fru error = %d\n",ret);
                   free_fru();
                }
                ipmi_restore_mc();
            }
        }  /*endif FRU/IPMB SDR */
        recid = nextid;
     } /*end while sdrs*/
   } /*endif not fonlybase*/
 
   /* load the FRU data for Baseboard (address 0x20) */
   printf("\n");
   sa = g_sa;  /* bmc_sa = BMC_SA = 0x20 */
   if (g_addrtype == ADDR_IPMB)
       ipmi_set_mc(g_bus,sa,g_lun,g_addrtype);
   ret = load_fru(sa,g_fruid);
   if (ret != 0) {
	printf("load_fru(%x,%x) error %d\n",sa,g_fruid,ret);
	free_fru();
	ipmi_close_();
	exit(1);
	}

   /* display the FRU data */
   ret = show_fru(g_fruid,sa);
   if (ret != 0) printf("show_fru error = %d\n",ret);

   if ((fwritefru != 0) && ret == 0) {
	printf("\nWriting new asset tag (%s) ...\n",asset_tag);
        ret = write_asset(asset_tag,serial_num,fwritefru);
        if (ret != 0) printf("write_asset error %d\n",ret);
	free_fru();
        ret = load_fru(sa,g_fruid);
	if (ret != 0) printf("load_fru(%x,%x) error %d\n",sa,g_fruid,ret);
        else ret = show_fru(g_fruid,sa);
	// free(asset_tag);
	// free(serial_num);
   }

   if (ret == 0) free_fru();
   ipmi_close_();
   show_outcome(progname,ret);  
   exit(ret);
}

/* end fruconfig.c */
