/*
 * getevent.c
 *
 * This utility waits for IPMI Event Messages.
 * Some server management functions want to trigger custom actions or 
 * alerts when IPMI hardware-related events occur, but do not want to
 * track all events, just newly occurring events.
 * This utility waits a specified timeout period for any events, and 
 * returns interpreted output for each event.  It is designed as a
 * scriptable command-line utility, but if the timeout is infinite
 * (-t 0), then this code could be used for a sample service as well.
 *
 * There are several methods to do this which are implemented here.
 * The SEL method:     
 *    This method polls the SEL once a second, keeps track of the last 
 *    SEL event read, and only new events are processed.  This ensures 
 *    that in a series of rapid events, all events are received in order, 
 *    however, some transition-to-OK events may not be configured to 
 *    write to the SEL on certain platforms.
 *    This method is used if getevent -s is specified.
 * The ReadEventMessageBuffer method: 
 *    This uses an IPMI Message Buffer in the BMC firmware to read
 *    each new event.  This receives any event, but if two events 
 *    occur nearly simultaneously, only the most recent of the two 
 *    will be returned with this method.  An example of simultaneous 
 *    events might be, if a fan stops/fails, both the non-critical 
 *    and critical fan threshold events would occur at that time.
 *    This is the default method for getevent.
 * The OpenIPMI custom method:  
 *    Different IPMI drivers may have varying behavior.  For instance,
 *    the OpenIPMI driver uses the IPMI GetMessage commands internally 
 *    and does not allow client programs to use those commands.  It has 
 *    its own custom mechanism, see getevent_mv().
 *    This method is used if the OpenIPMI driver is detected.
 * The IMB Async Event method:
 *    This only gets certain IMB Async events, like a shutdown 
 *    request from the BMC to an SMS OS service, and get_software_id.
 *    This method is disabled by default and only turned on if 
 *    getevent option -a is used.
 *    (see DO_ASYNC compile flag comments)
 *
 * Author:  Andy Cress  arcress@users.sourceforge.net
 * Copyright (c) 2005-2006 Intel Corporation. 
 *
 * 02/11/05 Andy Cress - created
 * 05/18/05 Andy Cress - modified bmc_enable bits
 * 05/26/05 Andy Cress - added call to decode_sel_entry
 * 09/09/05 Andy Cress - added sensor_type filtering & return type.
 * 03/16/05 Andy Cress - added loop, and -o for frunOnce
 * 06/27/06 Andy Cress 1.1 - specific message for cc=0x80 (no data)
 * 07/18/06 Andy Cress 1.1 - added getevent_mv, etc.
 * 07/26/06 Andy Cress 1.1 - added msgout() routine for fflush
 * 08/08/06 Andy Cress 1.2 - added -s for SEL method
 * 08/08/06 Andy Cress 1.2 - added -s for SEL method
 * 08/22/06 Andy Cress 1.3 - direct IOs added with ipmiutil-1.7.5
 * 09/13/06 Andy Cress 1.4 - handle empty SEL (0xCB), 
 *                           call syncevent_sel after every new event.
 * 09/21/07 Andy Cress 1.21 - implemented IMB Async method for remote
 *                            OS shutdown via SMS requests.
 */
/*M*
Copyright (c) 2006-2007, Intel Corporation
All rights reserved.

Redistribution and use in source and binary forms, with or without 
modification, are permitted provided that the following conditions are met:

  a.. Redistributions of source code must retain the above copyright notice, 
      this list of conditions and the following disclaimer. 
  b.. Redistributions in binary form must reproduce the above copyright notice,
      this list of conditions and the following disclaimer in the documentation 
      and/or other materials provided with the distribution. 
  c.. Neither the name of Intel Corporation nor the names of its contributors 
      may be used to endorse or promote products derived from this software 
      without specific prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR 
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *M*/
#ifdef WIN32
#include <windows.h>
#include <stdio.h>
#include <stdlib.h>
#include "getopt.h"
#else
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <getopt.h>
#include <unistd.h>
#include <pthread.h>
#include <sys/utsname.h>
#endif
#include <string.h>
#include "imb_api.h"
#include "ipmicmd.h"
 
// #define DO_ASYNC  1   
#define THREADS_OK 1  
#define IMBPTIMEOUT  200   /*200 ms*/
// #define IPMB_CHANNEL  0x00
// #define LAN_CHANNEL   0x02
#define ulong  unsigned long
#define uint   unsigned int
#define ushort unsigned short
#define uchar  unsigned char

extern void decode_sel_entry(uchar *evt, char *obuf);  /* see events.c */
/*
 * Global variables 
 */
static char * progname  = "getevent";
static char * progver   = "2.12";
static char   fdebug    = 0;
static char   frunonce  = 0;
static char   fAsync    = 0; 
static char   fbackground = 0; 
static int    timeout     = 120;   /* 120 seconds default timeout */
static int    wait_interval = 1; /* 1 second between calls */
static FILE  *fdout = NULL;
static uchar  ipmi_maj    = 0;
static uchar  ipmi_min    = 0;
static uint   imb_handle  = 0;
static int drvtype = 0;  /* driver_type from ipmicmd.h: 1=Intel_imb, 3=MV_OpenIPMI */
static int vend_id = 0;
static int prod_id = 0;
static char fselevts = 0;
static ushort sel_recid = 0;
static uint   sel_time  = 0;
#define LAST_REC  0xFFFF
#ifdef WIN32
static char idxfile[80] = "ipmi_evt.idx";
static char outfile[80] = "ipmiutil_evt.log";
#else
static char idxfile[80] = "/usr/share/ipmiutil/evt.idx";
static char outfile[80] = "/var/log/ipmiutil_evt.log";
#endif

static int do_wait(int nsec)
{
   int rv = 0;
#ifdef WIN32
   Sleep(nsec * 1000);   /*Windows Sleep(millisec)*/
#else
   sleep(nsec);    /* Linux sleep(sec) */
#endif
   return(rv);
}

static int get_event_receiver(uchar *sa, uchar *lun)
{
	uchar rdata[30];
	int rlen;
	uchar ccode;
	int ret;

	rlen = 2;
#if 0
        ret = ipmi_cmdraw( 0x01,NETFN_SEVT,BMC_SA,PUBLIC_BUS,BMC_LUN,
			idata,0, rdata,&rlen,&ccode, fdebug);
#endif
        ret = ipmi_cmd(GET_EVENT_RECEIVER,NULL,0, rdata,&rlen,&ccode, 0);
	if (ret == 0 && ccode != 0) ret = ccode;
	if (ret == 0) {
	   *sa  = rdata[0];
	   *lun = rdata[1];
	}
	return(ret);
}  

static int set_bmc_enables(uchar enab)
{
	uchar idata[8];
	uchar rdata[30];
	int rlen;
	uchar ccode;
	int ret;

	idata[0] = enab;
	rlen = 1;
        ret = ipmi_cmdraw( 0x2E,NETFN_APP,BMC_SA,PUBLIC_BUS,BMC_LUN,
			idata,1, rdata,&rlen,&ccode, fdebug);
	if (ret == 0 && ccode != 0) ret = ccode;
	return(ret);
}

static int get_bmc_enables(uchar *enab)
{
	uchar rdata[30];
	int rlen;
	uchar ccode;
	int ret;

	rlen = 1;
        ret = ipmi_cmdraw( 0x2F,NETFN_APP,BMC_SA,PUBLIC_BUS,BMC_LUN,
			NULL,0, rdata,&rlen,&ccode, fdebug);
	if (ret == 0 && ccode != 0) ret = ccode;

	if (ret == 0) *enab  = rdata[0];
	return(ret);
}

/* 
 * msgout
 * wrapper for printf() to include fflush
 */
void msgout(char *pattn, ...)
{
    va_list arglist;

    if (fdout == NULL) return;
    va_start( arglist, pattn );
    vfprintf( fdout, pattn, arglist );
    va_end( arglist );
    fflush( fdout );
}

// #ifdef DO_ASYNC
/* The DO_ASYNC flag enables the IMB Async Message method via get_imb_event.
 * This requires the Intel IMB driver, and is used only for remote shutdown
 * and software ID events.  */

/* The LANDesk library has the same function names as the imbapi.c */
#ifdef LINK_LANDESK
#define StartAsyncMesgPoll ia_StartAsyncMesgPoll
#define SendAsyncImbpRequest ia_SendAsyncImbpRequest
#define GetAsyncImbpMessage ia_GetAsyncImbpMessage
#define GetAsyncImbpMessage_Ex ia_GetAsyncImbpMessage_Ex
#define IsAsyncMessageAvailable ia_IsAsyncMessageAvailable
#define RegisterForImbAsyncMessageNotification ia_RegisterForImbAsyncMessageNotification
#define UnRegisterForImbAsyncMessageNotification ia_UnRegisterForImbAsyncMessageNotification
#endif  /*endif LINK_LANDESK*/

typedef struct {
        uchar rsSa;
        uchar nfLn;
        uchar cSum1;
        uchar rqSa;
        uchar seqLn;
        uchar cmd;
        uchar data[1];
} AsyImbPacket;

#ifdef THREADS_OK
   char message[32];
#ifdef WIN32
   HANDLE    threadid;
#else
   pthread_t threadid;
#endif
#endif

#ifdef WIN32
DWORD WINAPI pollThread( LPVOID p)
#else
void *pollThread(void *p)
#endif
{
   int i;
   int ret, limit;
#ifdef THREADS_OK
   limit = 0;
#else
   limit = 30;
#endif
   for (i = 0; (limit == 0) || (i < limit); i++)
   {
      ret = StartAsyncMesgPoll();
      if (fdebug && i < 5) 
          msgout("StartAsyncMesgPoll [%d] ret = %d\n",i,ret);
      // os_usleep(0,5000);  /* poll interval 5 msec */
      os_usleep(1,0);  /* poll interval 1 sec */
   }
   return(p);
}

int GetBmcLanChannel(uchar *chan)
{
    int ret = 0;
    int j;
    int rlen;
    uchar iData[2];
    uchar rData[10];
    uchar cc;
    uchar mtype;
    uchar chn = 1;

    if (vend_id == 0x000157) {
        if (prod_id == 0x000C || prod_id == 0x001B) {
           *chan = 7;
           return(ret);
        }
    }
    for (j = 1; j < 12; j++) {
        rlen = sizeof(rData);
        iData[0] = j; /*channel #*/
        memset(rData,0,9); /*initialize recv data*/
        ret = ipmi_cmd(GET_CHANNEL_INFO, iData, 1, rData, &rlen, &cc, fdebug);
        if (ret == 0xcc || cc == 0xcc) /* special case for ipmi_lan */
           continue;
        if (ret != 0) {
           if (fdebug) printf("get_chan_info rc = %x\n",ret);
           break;
        }
        mtype = rData[1];  /* channel medium type */
        if (mtype == 4) {  /* 802.3 LAN type*/
           if (fdebug) printf("chan[%d] = lan\n",j);
           chn = j;
           break;
        }
    }
    *chan = chn;
    return(ret);
}

int SoftwareIdResponse(uchar *buf, int blen, uchar hnd, uchar chan)
{
   int rv = 0;
   uchar resp[12] = {0,0xa6,0,0,0,1,0,0x00,0x01,0x57,0x00,0x01};
#ifdef WIN32
   /* check OS version & arch (32/64) */
#else
   struct utsname uts;
   rv = uname(&uts);
   // uts.release=`uname -r`  uts.machine=x86_64,ia64,i586,i386 
   // kver <= 24 bytes, mach/arch <= 6 bytes
#endif
        
   //rv = SendTimedEmpMessageResponse( buf, (char *)(&resp), 12, IMBPTIMEOUT);
   rv = SendTimedLanMessageResponse_Ex( (ImbPacket *)buf, (char *)(&resp), 12, 
				IMBPTIMEOUT, hnd, chan);
   if (fdebug) msgout("SoftwareIdResponse(%d) ret = %d\n",chan,rv);
   return(rv);
}

int SmsOsResponse(uchar *buf, int blen, uchar hnd, uchar chan)
{
   int rv = 0;
   char cc = 0;
   switch(buf[6])   /*data byte has function*/
   {
      case 0x01:    /*shutdown & power down*/
        rv = SendTimedEmpMessageResponse_Ex((ImbPacket *)buf, &cc,1, 
			IMBPTIMEOUT, hnd,chan);
        if (fdebug) msgout("OsResponse(%d) ret = %d\n",chan,rv);
#ifdef WIN32
        rv = system("shutdown -s -d p:01:01 -t 10");
#else
        rv = system("init 0");
#endif
        if (fdebug) msgout("shutdown ret = %d\n",rv);
	break;
      case 0x02:    /*shutdown & reset*/
        rv = SendTimedEmpMessageResponse_Ex((ImbPacket *)buf, &cc,1, 
			IMBPTIMEOUT, hnd,chan);
        if (fdebug) msgout("OsResponse(%d) ret = %d\n",chan,rv);
#ifdef WIN32
        rv = system("shutdown -r -d p:01:01 -t 10");
#else
        rv = system("init 6");
#endif
        if (fdebug) msgout("reboot ret = %d\n",rv);
	break;
      default:     
	rv = 1;
	break;
   }
   return(rv);
}

/*
 * get_imb_event
 * This only gets certain IMB events, like
 * OS requests (e.g. shutdown), and get_software_id 
 */
static int get_imb_event(uchar etype, int timeout, uchar *evt)
{
   	int ret = -1;
	int i;
        int done = 0;
	ulong mlen;
	uchar buffer[512];
	static uint asyseqnum = 0;
        uchar chan;
        uchar sessHandle = 0;
        uchar privilege = 0;

        ret = GetBmcLanChannel(&chan);

	/* clean out pre-existing async messages */
	while(1) {
	   mlen = sizeof(buffer);
	   if (GetAsyncImbpMessage((ImbPacket *)buffer,&mlen, IMBPTIMEOUT, 
				&asyseqnum, IPMB_CHANNEL) != 0) 
		break;
           if (fdebug) msgout("cleaned out an IPMB message seq=%d\n",asyseqnum);
	}
	while(1) {
	   mlen = sizeof(buffer);
	   if (GetAsyncImbpMessage((ImbPacket *)buffer,&mlen, IMBPTIMEOUT, 
				&asyseqnum, LAN_CHANNEL) != 0) 
		break;
           if (fdebug) msgout("cleaned out a LAN message seq=%d\n",asyseqnum);
	}
	ret = RegisterForImbAsyncMessageNotification(&imb_handle);
	if (fdebug) 
            msgout("RegisterForImbAsync ret=%d, handle=%x\n",ret,imb_handle);
	if (ret != 0) { 
		msgout("RegisterAsync error %d\n",ret);
		return(ret);
	}

	for (i = 0; (timeout == 0) || (i < timeout); i++)
	{  /*get one imb event*/
             if (fdebug) msgout("IsAsyncMessageAvailable ...\n");
	     if (IsAsyncMessageAvailable(imb_handle) == 0) 
             {
                if (fdebug) msgout("Async Message is Available\n");
		mlen = sizeof(buffer);
		ret = GetAsyncImbpMessage_Ex ((ImbPacket *)buffer, &mlen,
                                IMBPTIMEOUT, &asyseqnum, ANY_CHANNEL,
                                 &sessHandle, &privilege);
		/* Hack: buffer contains an extra byte to return channel */
                if (fdebug) 
		   msgout("GetAsync(%d,%d) ret = %d, newchan=%x\n",
			asyseqnum,chan,ret,buffer[mlen]);
		if (ret == 0) {
		   /* get the async message command */
                   msgout("got async msg: cmd=%02x len=%d\n",buffer[5],mlen);
                   if (fdebug) dump_buf("async msg",buffer,mlen,0);
		   chan = buffer[mlen];
                   if (mlen > 16) mlen = 16;  
                   memcpy(evt,buffer,mlen);

		   switch(buffer[5]) {
		     case 0x00:  /*Get Software ID*/
			ret = SoftwareIdResponse(buffer,mlen,sessHandle,chan);
			break;
		     case 0x10:  /*SMS OS Request*/
			ret = SmsOsResponse(buffer,mlen,sessHandle,chan);
			if (ret == 0) done = 1;
			break;
		     default:
			ret = -1;
		   }
		   if (done == 1) { 
			if (buffer[6] == 0x01) msgout("shutting down\n");
			else msgout("rebooting\n");
			ret = 0x81; 
			break; 
		   }
	        } 
	     }   /*endif have an event*/
	     else ret = 0x80;  /* no event yet */
	}   /*loop for one event*/
        if (fdebug) msgout("Unregister for imb events\n");
	UnRegisterForImbAsyncMessageNotification (imb_handle,0);
	return(ret);
}
// #endif   /*endif DO_ASYNC*/

static int get_sel_entry(ushort recid, ushort *nextid, uchar *rec)
{
   uchar ibuf[6];
   uchar rbuf[32];
   int rlen;
   ushort id, xid;
   uchar cc;
   int rv; 

   ibuf[0] = 0;
   ibuf[1] = 0;
   ibuf[2] = (recid & 0x00ff);
   ibuf[3] = (recid & 0xff00) >> 8;
   ibuf[4] = 0;
   ibuf[5] = 0xFF;  /*get entire record*/
   rlen = sizeof(rbuf);
   rv = ipmi_cmd(GET_SEL_ENTRY, ibuf, 6, rbuf, &rlen, &cc, fdebug);
   if (rv == 0) {
       if (cc != 0) rv = cc;
       else {  /*success*/
          xid = rbuf[0] + (rbuf[1] << 8);  /*next rec id*/
          memcpy(rec,&rbuf[2],16);
          *nextid = xid;
          id = rbuf[2] + (rbuf[3] << 8);  /*curr rec id*/
	  /* recid (requested) should match newid (received) */
          if (fdebug) {
            if ((recid != id) && (recid != LAST_REC) && (recid != 0)) {
              /* the OpenIPMI driver does this sometimes */
              msgout("get_sel MISMATCH: recid=%x newid=%x next=%x\n",
	   		      recid,id,xid);
              dump_buf("get_sel cmd",ibuf,6,0);
              dump_buf("get_sel rsp",rbuf,rlen,0);
            }
          }
       }
   }
   if (fdebug) msgout("get_sel(%x) rv=%d cc=%x id=%x next=%x\n",
			recid,rv,cc,id,*nextid);
   return(rv);
}

static void startevent_sel(ushort *precid, uint *ptime) 
{
    FILE *fd;
    uchar rec[24];
    uint t = 0;
    ushort r = 0;
    ushort r2 = 0;
    int rv = -1;

    fd = fopen(idxfile,"r");
    if (fdebug) msgout("start: idxfile=%s fd=%p\n",idxfile,fd);
    if (fd != NULL) {
        // Read the file, get savtime & savid
        rv = fscanf(fd,"%x %x",&t,&r);
        fclose(fd);
        if (r == LAST_REC) r = 0;
    } else {  /* treat as first time */
        r = LAST_REC;
        rv = get_sel_entry(r,&r2,rec);
        if (rv == 0) {
            memcpy(&t,&rec[2],4);
            r = rec[0] + (rec[1] << 8); /*use current rec id*/
        } else r = 0;
    }
    if (fdebug) msgout("start: recid=%x time=%x\n",r,t);
    *ptime  = t;
    *precid = r;
}

static void syncevent_sel(ushort recid, uint itime)
{
    FILE *fd;
    // Rewrite the saved time & record id
    if (fdebug) msgout("sync: recid=%x time=%x\n",recid,itime);
    fd = fopen(idxfile,"w");
    if (fd != NULL) {
        fprintf(fd,"%x %x\n",itime,recid);
        fclose(fd);
    }
}

static int getevent_sel(uchar *rdata, int *rlen, uchar *ccode)
{
    uchar rec[24];
    int rv = 0;
    ushort newid;
    ushort nextid;
    ushort recid;
    
    /* get current last record */
    recid = sel_recid;
    rv = get_sel_entry(recid,&nextid,rec);
    if (rv == 0xCB && recid == 0) {  /* SEL is empty */
        *ccode = rv;  /* save the real ccode */
        rv = 0x80;    /* this is ok, just keep waiting  */
    }
    if (rv == 0) {
       if (fdebug) msgout("sel ok, id=%x next=%x\n",recid,nextid);
       if ((nextid == LAST_REC) || (recid == nextid)) { 
           *ccode = 0x80;  /*nothing new*/
       } else {
         recid = nextid;  /* else get new one */
         rv = get_sel_entry(recid,&nextid,rec);
         if (rv == 0) {  /* new event */ 
            newid = rec[0] + (rec[1] << 8);
            if (drvtype == DRV_MV && recid != newid) {  
               /* handle MV driver bug, try to get next one. */
               if (fdebug) msgout("DRV_MV bug, mismatch\n");
            }
            if (fdebug) msgout("recid=%x newid=%x next=%x\n",
	   		       recid,newid,nextid);
            memcpy(rdata,rec,16);
            *rlen = 16;
            *ccode = 0;
            sel_recid = recid;  /*or newid*/
         }
       }
    }
    else {  /* Error reading last recid saved */
       if (fdebug) msgout("sel recid %x error, rv = %d\n",recid,rv);
       /* May also want to set sel_recid = 0 here for some errors. */
    }
    return(rv); 
}

static int get_event(uchar etype, int timeout, uchar *evt, uchar *stype)
{
   	int ret = 0;
   	uchar rdata[64];
   	int rlen;
	uchar ccode;
	int fretry;
	int i;

	for (i = 0; (timeout == 0) || (i < timeout); i++)
	{
	   rlen = sizeof(rdata);
	   fretry = 0;  ccode = 0;
           if (fselevts) {
             ret = getevent_sel(rdata,&rlen,&ccode);
           } else 
#ifndef WIN32
           if (drvtype == DRV_MV) { /* if MV OpenIPMI driver (Linux only) */
             /* use special MV API instead (see ipmimv.c) */
             ret = getevent_mv(rdata,&rlen,&ccode);
           } else
#endif
           ret = ipmi_cmd(READ_EVENT_MSGBUF,NULL,0,rdata,&rlen,&ccode,fdebug);
           /* IPMI 1.5 spec, section 18.8 says cc 0x80 means 
            * "data not available (queue/buffer empty)" */
           if (ret == 0 && ccode != 0) { ret = ccode; }
           if (ret == 0x80) { fretry = 1; do_wait(1);  /*wait 1 sec*/ 
           } else {
        	if (ret == 0) {
	    	    /* parse event types for a specified type */
		    /* etype param == 0xff is a mask, meaning get any event */
            	    if ((etype & rdata[11]) != 0) {
	       	        memcpy(evt,rdata,rlen);
			*stype = rdata[11];  /* return sensor type */
            	    } else {       /* keep looking */
			do_wait(wait_interval);
			continue; 
		    }
        	}
		/* if here, either got one, or need to return error */
		break;  
	    }
	}  /*end for loop*/
	return(ret);
}


void show_event(uchar *evt)
{
   int i;
   char obuf[132];

   msgout("event data: ");
   for (i=0; i<16; i++) msgout("%02x ",evt[i]);
   msgout("\n");

   decode_sel_entry(evt,obuf);
   msgout(obuf); 
}

#ifndef WIN32
static int mkdaemon(int fchdir, int fclose);
static int mkdaemon(int fchdir, int fclose)
{
    int fdlimit = sysconf(_SC_OPEN_MAX); /*fdlimit usu = 1024.*/
    int fd = 0;
 
    switch (fork()) {
        case 0:  break;
        case -1: return -1;
        default: _exit(0);          /* exit the original process */
    }
    if (setsid() < 0) return -1;    /* shouldn't fail */
    switch (fork()) {
        case 0:  break;
        case -1: return -1;
        default: _exit(0);
    }
    if (fchdir) chdir("/");
    if (fclose) {
        /* Close stdin,stdout,stderr and replace them with /dev/null */
        while (fd < fdlimit) close(fd++);
        open("/dev/null",O_RDWR);
        dup(0); dup(0);
    }
    return 0;
}
#endif

#ifdef METACOMMAND
int i_getevt(int argc, char **argv)
#else
#ifdef WIN32
int __cdecl
#else
int
#endif
main(int argc, char **argv)
#endif
{
   int ret = 0;
   char c;
   uchar devrec[16];
   uchar event[16];
   uchar sa, lun;
   uchar enables;
   uchar fevmsgok;
   uchar sensor_type;
   // int i;

   fdout = stdout;
   msgout("%s ver %s\n", progname,progver);

   while ( (c = getopt( argc, argv,"absot:T:V:J:YEF:P:N:R:U:x?")) != EOF ) 
      switch(c) {
          case 'a': fAsync = 1;     break;  /* imb async message method */
          case 'b': fbackground = 1; break; /* background */
          case 'o': frunonce = 1;   break;  /* only run once for first event */
          case 's': fselevts = 1;   break;  /* use SEL event method*/
          case 't': timeout = atoi(optarg); break;  /*timeout*/
          case 'x': fdebug = 1;     break;  /* debug messages */
          case 'N':    /* nodename */
          case 'U':    /* remote username */
          case 'P':    /* remote password */
          case 'R':    /* remote password */
          case 'E':    /* get password from IPMI_PASSWORD environment var */
          case 'F':    /* force driver type */
          case 'T':    /* auth type */
          case 'J':    /* cipher suite */ 
          case 'V':    /* priv level */
          case 'Y':    /* prompt for remote password */
                parse_lan_options(c,optarg,fdebug);
                break;
	  default:
                printf("Usage: %s [-absox -t sec -NPRUEFTVY]\n", progname);
                printf(" where -a      use IMB Async method\n");
                printf("       -b      run in Background\n");
                printf("       -o      run Once for the first event\n");
                printf("       -s      use SEL event method\n");
                printf("       -t N    set timeout to N seconds\n"); 
                printf("       -x      show eXtra debug messages\n");
		print_lan_opt_usage();
                exit(1);
      }

   ret = ipmi_getdeviceid(devrec,16,fdebug);
   if (ret != 0) {
        show_outcome(progname,ret);  
	ipmi_close_();
	exit(1);
   } else {
      ipmi_maj = devrec[4] & 0x0f;
      ipmi_min = devrec[4] >> 4;
      msgout("-- BMC version %x.%x, IPMI version %d.%d \n",
             devrec[2],  devrec[3], ipmi_maj, ipmi_min);
      vend_id = devrec[6] + (devrec[7] << 8) + (devrec[8] << 16);
      prod_id = devrec[9] + (devrec[10] << 8);
   }

   /* get event receiver */
   ret = get_event_receiver(&sa ,&lun);
   if (ret != 0)
        msgout("event receiver error %d\n",ret);
   else msgout("event receiver sa = %02x lun = %02x\n",sa,lun);

   ret = get_bmc_enables(&enables);
   if (ret != 0) msgout("bmc enables error %d\n",ret);
   else {
        msgout("bmc enables = %02x\n",enables);
	if ((enables & 0x02) == 2) fevmsgok = 1;
	else fevmsgok = 0;
   }
   if (fevmsgok == 0 && !fipmi_lan) { 
      msgout("Event Message Buffers not enabled.\n");
      enables |= 0x0f;  /* 0x08=SEL, 0x02=EvtMsgBuf, rest is gravy */
      ret = set_bmc_enables(enables);
      if (ret != 0) {
	 msgout("set_bmc_enables error 0x%x\n",ret);
      }
      else msgout("set_bmc_enables success\n");
   }
   if (fipmi_lan && !fselevts) {
      msgout("Only the SEL method (-s) is supported over IPMI LAN\n");
      exit(1);
   }
   if (fselevts) {
       if (fipmi_lan) {
          strcat(idxfile,"-");
          strcat(idxfile,gnode);
          strcat(outfile,"-");
          strcat(outfile,gnode);
       }
       startevent_sel(&sel_recid,&sel_time);
   }

   drvtype = get_driver_type();
   if (fdebug) msgout("driver_type = %d\n",drvtype);

#ifdef TEST_SEL
   /* This is used to verify that the interface returns valid next ids,
    * and that the get_sel_entry is ok. */
   {
      int i;
      ushort r, r1, r2;
      uchar rec[40];
      int rv;
      r = 0;
      for (i = 0; i < 4; i++) 
      {
          rv = get_sel_entry(r,&r2,rec);
          if (rv == 0) {
             r1 = rec[0] + (rec[1] << 8); /*get current rec id*/
             if (fdebug) msgout("get_sel: r=%x r1=%x rnext=%x\n",r,r1,r2);
	     show_event(&rec[0]);
             r = r2;
          } else break;
      }
   }
#endif
   if (fAsync && (drvtype != DRV_IMB)) {
      msgout("Cannot open IMB driver, required for -a\n");
      exit(1);
   }

#ifdef WIN32
   if (fbackground) {
      msgout("Background not implemented for Windows\n");
      exit(1);
   }
#else
   /* convert to a daemon if background */
   if (fbackground) {
      ret = mkdaemon(1,1);
      if (ret != 0) {
         msgout("%s: Cannot become daemon, ret = %d\n", progname,ret);
         exit(1);
      }
      /* open a log file for messages, set fdout */
      fdout = fopen(outfile,"a");
   }
#endif

// #ifdef DO_ASYNC
   if (fAsync && (drvtype == DRV_IMB)) {  /*use imb async messages*/
     msgout("Wait for an imb event\n");  /* no timeout */
#ifdef THREADS_OK
#ifdef WIN32
     /* Windows threads */
     threadid = CreateThread(NULL, 0, &pollThread, NULL, 0, NULL);
     if (threadid == NULL) pollThread(NULL);
#else
     /* Linux threads */
     ret = pthread_create( &threadid, NULL, pollThread, (void*) message);
     // if (ret == 0) pthread_join( threadid, NULL);
     // if (ret == 0) pthread_detach( threadid);
#endif
     if (fdebug) msgout("pollThread create ret=%d handle=%x\n",ret,threadid);
#else
     /* no threads */
     pollThread(NULL);
#endif
     while (ret == 0) 
     {            /*wait for imb message events*/
        msgout("Waiting %d seconds for an imb event ...\n",timeout);
        ret = get_imb_event(0xff,timeout,event);
	if (ret == 0x81) {  /*ok, shutting down OS*/
	    ret = 0; 
	    break; 
        }
        if (frunonce) break;
        if (timeout == 0 && ret == 0x80) {  /*0x80 = no data yet */
            if (fdebug) msgout("get_event timeout, no event yet.\n");
            do_wait(1);
            ret = 0;  /*ok, keep going*/
        }
     }
   } else
// #endif
      /* loop on events here, like a daemon would. */
      while (ret == 0) 
      {            /*wait for bmc message events*/
         msgout("Waiting %d seconds for an event ...\n",timeout);
         /* note: could also get message flags here */
         /* 0xff is a mask, meaning get any events */
         ret = get_event(0xff,timeout,event,&sensor_type);
         if (fdebug) msgout("get_event ret = %d\n",ret);
         if (ret == 0) { /* got an event successfully */
            msgout("got event, sensor_type = %02x\n",sensor_type);
	    show_event(event);
            if (fselevts) syncevent_sel(sel_recid,sel_time);
         } else {
            if (ret == 0x80) msgout("get_event timeout\n");
            else msgout("get_event error: ret = 0x%x\n",ret);
         }
         if (frunonce) break;
         if (timeout == 0 && ret == 0x80) {  /*0x80 = no data yet */
            if (fdebug) msgout("get_event timeout, no data yet.\n");
            do_wait(1);
            ret = 0;  /*ok, keep going*/
         }
      }  /*end while loop*/
   
   if (fselevts) syncevent_sel(sel_recid,sel_time);
   if (ret == 0x80) ret = 0;
#ifdef THREADS_OK
#ifdef WIN32
   CloseHandle(threadid);
#else
   /* close not needed in Linux (?)  */
#endif
#endif
   ipmi_close_();
   show_outcome(progname,ret);  /*inert if background*/
   if (fbackground && fdout != NULL) fclose(fdout);
   exit(ret);
}  /* end main()*/

/* end getevent.c */
