/*
 * hwreset.c
 *
 * This tool power cycles (or powers off) the IPMI system.
 *    
 * Author:  Andy Cress  arcress@users.sourceforge.net
 * Copyright (c) 2002-2007 Intel Corporation. 
 *
 * 01/08/02 Andy Cress - created
 * 01/31/02 Andy Cress - converted to use ipmi_cmd_ia, 
 *                       added more user options
 * 02/06/02 Andy Cress - added ipmi_cmd_va
 * 02/22/02 Andy Cress - added -s to reboot to service partition
 * 07/02/02 Andy Cress v1.3 added more Usage text
 * 08/02/02 Andy Cress v1.4 moved common ipmi_cmd() code to ipmicmd.c
 * 09/24/02 Andy Cress - stubbed in OS shutdown option
 * 01/29/03 Andy Cress v1.5 added MV OpenIPMI support
 * 04/08/03 Andy Cress v1.6 added OS shutdown option (-o)
 * 05/02/03 Andy Cress v1.7 leave console redir alone in SET_BOOT_OPTIONS
 * 05/05/04 Andy Cress v1.8 call ipmi_close before exit, did WIN32 port.
 * 08/09/04 Andy Cress v1.9 make sure to show error if ccode != 0, and
 *                          detect Langley platforms to do special 
 *                          watchdog method for -o option.
 * 11/01/04 Andy Cress 1.10 add -N / -R for remote nodes   
 * 11/16/04 Andy Cress 1.11 add -U for remote username
 * 11/30/04 Andy Cress 1.12 fix bug 1075550 with -o -N, skip -o if not local.
 * 03/28/05 Andy Cress 1.13 add netapp_reset commands for platforms that
 *                          use this instead of chassis_reset.
 * 05/16/05 Andy Cress 1.14 add -u option for power up
 * 03/31/06 Andy Cress 1.15 add -e -p -m options
 * 09/18/06 Andy Cress 1.20 allow more platforms to do soft reset, and
 *                          if Tyan, ignore set boot options errors.
 * 01/10/07 Andy Cress 1.25 added reset_str(), modify initchar (6) if -o.
 */
/*M*
Copyright (c) 2002-2007, Intel Corporation
All rights reserved.

Redistribution and use in source and binary forms, with or without 
modification, are permitted provided that the following conditions are met:

  a.. Redistributions of source code must retain the above copyright notice, 
      this list of conditions and the following disclaimer. 
  b.. Redistributions in binary form must reproduce the above copyright notice,
      this list of conditions and the following disclaimer in the documentation 
      and/or other materials provided with the distribution. 
  c.. Neither the name of Intel Corporation nor the names of its contributors 
      may be used to endorse or promote products derived from this software 
      without specific prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR 
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *M*/
#ifdef WIN32
#include <windows.h>
#include <stdio.h>
#include "getopt.h"
#else
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <getopt.h>
#endif
#include <string.h>
#include "ipmicmd.h"
 
#define platCG   1    /*Intel TSD servers (Sahalee), use alt soft-shutdown.*/
#define platMBMC  2    /*mini-BMC platforms */
#define  GET_POWER_STATE   0x07 

/*
 * Global variables 
 */
static char * progver   = "2.12";
static char * progname  = "hwreset";
static uchar  ipmi_maj;
static uchar  ipmi_min;
static char   fdebug         = 0;
static char   fignore_opterr = 0;
static char   platform  = 0;   /* platform type: MBMC or TSR */
static int    shuttime  = 60;  /* shutdown timeout in seconds */

static void show_error(char *tag, int rv, uchar cc)
{
   if (rv > 0 && cc == 0) cc = rv;
   printf("%s: error %d ccode = %x %s\n", 
		tag,rv,cc,decode_cc((ushort)0,cc));
   return;
}

static int set_wdt(uchar val, uchar act)
{
        uchar idata[6];
        uchar rdata[16];
        int rlen = 8;
        uchar ccode;
        int ret, t;
 
        t = val * 10;      /* val is in sec, make timeout in 100msec */
        if ((ipmi_maj > 1) ||      /* IPMI 1.5 or greater */
            (ipmi_maj == 1 && ipmi_min >= 5))
             idata[0] = 0x44;    /* DontLog=0, DontStop=1 & use SMS/OS */
        else idata[0] = 0x04;    /* IPMI 1.0 or less */
        idata[1] = act; 
                /* 0x01;    * action: no pretimeout, hard reset action */
		/* 0x02;    * action value for power down instead */
        idata[2] = 0;      /* pretimeout: 30 sec (but disabled) */
        idata[3] = 0x10;   /* clear SMS/OS when done */
        idata[4] = t & 0x00ff;   /*timeout in 100msec: 0x4B0 = 1200. */
        idata[5] = (t & 0xff00) >> 8;
        ret = ipmi_cmd(WATCHDOG_SET, idata, 6, rdata, &rlen, &ccode, fdebug);
	if (fdebug) printf("set_wdt: wd set(%d,%d) rv=%d cc=%x\n",
				val,act,ret,ccode);
        if (ret == 0 && ccode != 0) ret = ccode;

	if (ret == 0) {  /* Start the timer by issuing a watchdog reset */
	   ret = ipmi_cmd(WATCHDOG_RESET,idata,0, rdata,&rlen, &ccode,fdebug);
	   if (fdebug) printf("set_wdt: wd reset rv=%d cc=%x\n",ret,ccode);
	   if (ret == 0 && ccode != 0) ret = ccode;
	}
        return(ret);
}  /*end set_wdt()*/


static int IPMI_Reset(uchar bpower, uchar bootopt)
{
	uchar responseData[MAX_BUFFER_SIZE];
	int responseLength = MAX_BUFFER_SIZE;
	uchar completionCode;
	uchar inputData[6];
	int status = 0;
	uchar cmd;

        /* May need to GetSystemBootOptions first to fill these in. */
	inputData[0] = 0x05;  // param, 5 = boot flags
	inputData[1] = 0x80;  // valid flags, next boot only
        if (bootopt == 2) inputData[1] = 0xA0;  // valid, boot to EFI
        if (bootopt == 1)      inputData[2] = 0x10;   // boot to svc partition
        else if (bootopt == 3) inputData[2] = 0x04;   // boot to PXE
        else if (bootopt == 4) inputData[2] = 0x14;   // boot to CDROM
        else if (bootopt == 5) inputData[2] = 0x08;   // boot to Hard Disk
        else if (bootopt == 6) inputData[2] = 0x18;   // boot to BIOS Setup
        else if (bootopt == 7) inputData[2] = 0x3C;   // boot to Floppy/Remov
        else if (bootopt == 8) inputData[2] = 0x0C;   // boot to HardDisk/Safe
	else inputData[2] = 0x00;  // normal boot
	inputData[3] = 0x48;  // verbose disply, bypass pswd, same console redir
	inputData[4] = 0x00;  // no overrides
	inputData[5] = 0x00;  // 
	responseLength = MAX_BUFFER_SIZE;
        status = ipmi_cmd(SET_BOOT_OPTIONS, inputData, 6, responseData,
                        &responseLength, &completionCode, fdebug);
	if (status != ACCESS_OK || completionCode != 0) {
            printf("set_boot_options ccode %x, resp[0] = %x, resp[1] =  %x\n",
                   completionCode, responseData[0], responseData[1]);
            if (!fignore_opterr) 
               return(-1);  /* abort if the boot options can't be set */
	}
        /* TODO:
         * if (bootopt != 0) then set param 3 to not clear boot valid flag 
         */

#ifdef EXPERIMENTAL
	if ((bpower >= 5) && (platform == platCG)) 
#else
	if ((bpower >= 5) && (platform == platCG) && (!fipmi_lan)) 
#endif
        {     /*os shutdown requested*/
           uchar action;
           char initcmd[16];
           char initchar, shutchar;

	   /*
	    * Special OS shutdown method for CG Servers 
	    * Set up a watchdog event to do reset after timeout. 
	    * Valid on other platforms too if they support watchdog.
	    * Note that the "init 0" only makes sense if local. 
	    */
           if (bpower == 6) { action = 0x02;  /*do power_down*/
               initchar = '0';
               shutchar = 's';
           } else { action = 0x01;  /*do hard_reset*/
               initchar = '6';
               shutchar = 'r';
           }
	   status = set_wdt((uchar)shuttime, action);
#ifdef EXPERIMENTAL
	   if (fipmi_lan) {  /*Try remote shutdown via Bridged SMS */
                int rv;
	        inputData[0] = 5;  /* soft shutdown */
	        responseLength = MAX_BUFFER_SIZE;
		rv = ipmi_cmdraw( CHASSIS_CTL,
			NETFN_APP,BMC_SA,PUBLIC_BUS, SMS_LUN,
                        inputData,1,responseData,&responseLength,
			&completionCode, fdebug);
	        printf("Remote soft shutdown initiated (%d,%d).\n",status,rv);
           } else 
#endif
           if (status == 0) 
           {  /*local shutdown */
#ifdef WIN32
                sprintf(initcmd,"shutdown -%c -c %s",shutchar,progname);
		status = system(initcmd); /* do the OS shutdown */
	        printf("Windows soft shutdown initiated (%s).\n",initcmd);
#else
                sprintf(initcmd,"init %c",initchar);
		status = system(initcmd); /* do the OS shutdown */
	        printf("Linux soft shutdown initiated (%s).\n",initcmd);
#endif
	   }
	   /* 
            * Note that this can generate a Watchdog 2 event in the SEL. 
            * If the init 0/6 is successful within the 60 second timeout,
            * BIOS will stop the watchdog.
            */
	   return(status);
	}

        /* 0 = power down, 1 = power up, 2 = power cycle, 3 = hard reset */
        /* 4 = NMI interrupt, 5 = soft shutdown OS via ACPI  */
	if (bpower > 5) bpower = 5;  /* if invalid, try shutdown */
	inputData[0] = bpower;  // hard reset
        responseLength = MAX_BUFFER_SIZE;
        status = ipmi_cmd(CHASSIS_CTL, inputData, 1, responseData,
                        &responseLength, &completionCode, fdebug);
        if (fdebug) {
                printf("Cmd %x code %x, resp[0] = %x, resp[1] =  %x\n",
		       CHASSIS_CTL, completionCode, 
		       responseData[0], responseData[1]);
                }
	if (status == ACCESS_OK && completionCode == 0) {
		printf("chassis_reset ok\n");
		//successful, done
		return(0);
	} else if (fipmi_lan && (status < 0)) {
           /* Remote IPMI LAN reset could not connect, 
            * no point in continuing.  */
	   return(status);
	} else {
	   if (bpower == 5 && completionCode == 0xcc) {
		/* See IPMI spec 22.3 Chassis Control, Table 22-4 */
		printf("Optional soft-shutdown mode not supported\n");
		/* If get here, need to use method like platCG above. */
	   } else {
		show_error("chassis_reset",status,completionCode);
		status = -1;

		/* Try net_app warm/cold reset commands instead */
		if (bpower = 2) cmd = 2; /* cold reset */
		else cmd = 3;            /* warm reset */
                responseLength = MAX_BUFFER_SIZE;
		status = ipmi_cmdraw( cmd,NETFN_APP,BMC_SA,PUBLIC_BUS,BMC_LUN,
                        inputData,0,responseData,&responseLength,
			&completionCode, fdebug);
		if (status == ACCESS_OK && completionCode == 0) {
		   printf("netapp_reset ok\n");
		} else {
		   show_error("netapp_reset",status,completionCode);
		   status = -1;
		}
	   }
	} /*end else*/
	return(status);
}  /*end IPMI_Reset()*/

char *reset_str(uchar breset, uchar bopt)
{
   char *str;
   switch(breset) {
        case 0:  str = "powering down"; break;
        case 1:  str = "powering up"; break;
        case 2:  str = "power cycling"; break;
        case 3:  str = "resetting"; break;
        case 4:  str = "sending NMI"; break;
        case 5:  str = "shutdown/reset"; break;
        case 6:  str = "shutdown/power_off"; break;
        default: str = "resetting"; break;
   }
   if (bopt > 0) 
     switch(bopt) {
        case 1:  str = "resetting to Svc partition"; break;
        case 2:  str = "resetting to EFI"; break;
        case 3:  str = "resetting to PXE"; break;
        case 4:  str = "resetting to CDROM"; break;
        case 5:  str = "resetting to hard disk"; break;
        case 6:  str = "resetting to BIOS Setup"; break;
        case 7:  str = "resetting to floppy"; break;
        default: str = "resetting"; break;
     }
   return(str);
}

#ifdef METACOMMAND
int i_reset(int argc, char **argv)
#else
#ifdef WIN32
int __cdecl
#else
int
#endif
main(int argc, char **argv)
#endif
{
   int ret; //, i;
   char c;
   uchar breset;
   uchar bopt;
   uchar fshutdown = 0;
   uchar devrec[16];
   int rlen;
   uchar cc;

   // progname = argv[0];
   breset = 3;  /* hard reset as default */
   bopt = 0;    /* Boot Options default */
   printf("%s ver %s\n", progname,progver);

   while ((c = getopt(argc,argv,"bcdDefhumnoprswT:V:J:YEF:N:P:R:U:x?")) != EOF)
      switch(c) {
          case 'd': breset = 0;     break;  /* power down */
          case 'u': breset = 1;     break;  /* power up */
          case 'c': breset = 2;     break;  /* power cycle */
          case 'o': fshutdown = 1; breset = 5; break;  /*soft shutdown,reset*/
          case 'D': fshutdown = 1; breset = 6; break;  /*soft shutdown,pwrdown*/
          case 'n': breset = 4;     break;  /* interrupt (NMI) */
          case 'r': breset = 3;     break;  /* hard reset */
          case 's': bopt  = 1;      break;  /* hard reset to svc part */
          case 'e': bopt  = 2;      break;  /* hard reset to EFI */
          case 'p': bopt  = 3;      break;  /* hard reset to PXE */
          case 'm': bopt  = 4;      break;  /* hard reset to CDROM Media */
          case 'h': bopt  = 5;      break;  /* hard reset to Hard Disk */
          case 'b': bopt  = 6;      break;  /* hard reset to BIOS Setup */
          case 'f': bopt  = 7;      break;  /* hard reset to floppy/remov */
          case 'N':    /* nodename */
          case 'U':    /* remote username */
          case 'P':    /* remote password */
          case 'R':    /* remote password */
          case 'E':    /* get password from IPMI_PASSWORD environment var */
          case 'F':    /* force driver type */
          case 'T':    /* auth type */
          case 'J':    /* cipher suite */ 
          case 'V':    /* priv level */
          case 'Y':    /* prompt for remote password */
                parse_lan_options(c,optarg,fdebug);
                break;
          case 'x': fdebug = 1;     break;  /* debug messages */
	  default:
                printf("Usage: %s [-bcdDefhinorsux -N node -U user -P/-R pswd -EFTVY]\n",
			progname);
                printf(" where -c  power Cycles the system\n");
                printf("       -d  powers Down the system\n");
                printf("       -u  powers Up the system\n");
                printf("       -r  hard Resets the system\n");
                printf("       -n  sends NMI to the system\n");
                printf("       -o  soft-shutdown OS and reset\n");
                printf("       -D  soft-shutdown OS and power down\n");
                printf("       -s  reboots to Service Partition\n");
                printf("       -b  reboots to BIOS Setup\n");
                printf("       -e  reboots to EFI\n");
                printf("       -f  reboots to Floppy/Removable\n");
                printf("       -h  reboots to Hard Disk\n");
                printf("       -p  reboots to PXE\n");
                printf("       -m  reboots to CDROM Media\n");
                printf("       -i  sets Init string for this boot\n");
                printf("       -w  Wait for BMC ready after reset\n");
                printf("       -x  show eXtra debug messages\n");
		print_lan_opt_usage();
                exit(1);
      }

   /* 
    * Check the Device ID to determine the platform type.
    */
   ret = ipmi_getdeviceid(devrec,16,fdebug);
   if (ret != 0) {
        show_outcome(progname,ret);  
	ipmi_close_();
        exit(1);
   } else {
      char *pstr;
      int    vendid;
      ushort prodid;
      uchar j;

      if (fdebug) {
	printf("devid: ");
	for (j = 0; j < 16; j++) printf("%02x ",devrec[j]);
	printf("\n");
      }
      ipmi_maj = devrec[4] & 0x0f;
      ipmi_min = devrec[4] >> 4;
      vendid = devrec[6] + (devrec[7] << 8) + (devrec[8] << 16);
      prodid = devrec[9] + (devrec[10] << 8);
      if (fdebug) printf("vendor = %06x, product_id = %04x\n",vendid,prodid);
      if (vendid == VENDOR_NSC) {   /* NSC mBMC */
	  pstr = "mBMC";
	  platform = platMBMC;
      } else if (vendid == VENDOR_TYAN) {   /* Tyan */
	  pstr = "BMC";
          fignore_opterr = 1;  /* ignore boot options errors */
      } else if (vendid == VENDOR_INTEL) {   /* Intel */
	  pstr = "BMC";
	  if (prodid != 0x0100)  /* ia64 Itanium2 is different */
	      platform = platCG;  /* else handle as Intel Sahalee */
      } else {             /* other vendors */
	  pstr = "BMC";
	  platform = platCG;  /* assume it handles watchdog, set like Intel */
      }
      printf("-- %s version %x.%x, IPMI version %d.%d \n",
             pstr, devrec[2],  devrec[3], ipmi_maj, ipmi_min);
   }

   { /* show current power state */
       char *pstr;
       uchar pstate;
       rlen = sizeof(devrec);
       ret = ipmi_cmdraw( GET_POWER_STATE, NETFN_APP,
                        BMC_SA,PUBLIC_BUS,BMC_LUN,
                        NULL,0, devrec,&rlen,&cc, fdebug);
       if (ret == 0) {
          pstate = devrec[0] & 0x7f;
          switch(pstate) {
             case 0x00: pstr = "S0: working"; break;
             case 0x01: pstr = "S1: clock stopped, context ok"; break;
             case 0x02: pstr = "S2: clock stopped, context lost"; break;
             case 0x03: pstr = "S3: suspend-to-RAM"; break;
             case 0x04: pstr = "S4: suspend-to-Disk"; break;
             case 0x05: pstr = "S5: soft off"; break;
             case 0x06: pstr = "S4/S5: soft off, either S4 or S5"; break;
             case 0x07: pstr = "G3: mechanical off"; break;
             case 0x08: pstr = "S1-S3: sleeping"; break;
             case 0x09: pstr = "S1-S4: sleeping"; break;
             case 0x0A: pstr = "S5/o: soft off by override"; break;
             case 0x20: pstr = "legacy on"; break;
             case 0x21: pstr = "legacy soft-off"; break;
             default:   pstr = "unknown"; break;
          }
          if (cc == 0) 
             printf("Power State      = %02x   (%s)\n",pstate,pstr);
       }
   }

   /* 
    * fshutdown:
    * Since we are in user mode, we can't wait for it to fully 
    * shut down and then issue the IPMI Chassis Reset. 
    * IPMI can trigger this by emulating an overtemp event.
    * There is also a watchdog/init0 method used by platCG. 
    * breset was set above to 5 or 6
    */
   if (fshutdown && fipmi_lan) {
      uchar rqdata[10];
      uchar rsdata[32];
      int rslen;
      int fdaemonok = 0;
      /* Special remote soft-shutdown, requires a service to
       * be running on the target node.
       * GET_SOFTWARE_ID == 0x00 
       * SMS_OS_REQUEST  == 0x10 : (down=0, reset=1)
       * BRIDGE_REQUEST  == 0x20 : (down=0, reset=1)
       * SMS_SA  == 0x81
       */
      rslen = sizeof(rsdata);
      ret = ipmi_cmdraw( 0x00, NETFN_APP,0x81,PUBLIC_BUS, SMS_LUN,
                        rqdata,0, rsdata, &rslen, &cc, fdebug);
      if (fdebug) 
	 printf("ipmilan getswid ret=%d cc=%x, rslen=%d\n",ret,cc,rslen);
      if (ret == 0 && cc == 0) {
         ushort v,x;
         v = (rsdata[6] << 16) + (rsdata[7] << 8) + rsdata[8];
         x = (rsdata[9] << 8) + rsdata[10];
         if (fdebug) printf("v: %06x x: %04x\n",v,x);
	 if (v == 0x000157 && x == 0x0001) fdaemonok = 1;
      }
      if (fdaemonok) {
         if (breset == 0 || breset == 6)
              rqdata[0] = 0x01;   /* shutdown & power down */
         else rqdata[0] = 0x02;   /* shutdown & reset */
         if (fdebug) printf("ipmilan shutdown action=%x\n",rqdata[0]);
         rslen = sizeof(rsdata);
         ret = ipmi_cmdraw( SMS_OS_REQUEST, NETFN_APP,0x81,PUBLIC_BUS, SMS_LUN,
                        rqdata,1, rsdata, &rslen, &cc, fdebug);
         printf("ipmilan shutdown request: ret = %d, cc = %x\n", ret,cc);
      } 
      else printf("ipmilan bridge agent not present\n");
   } else 
   {
      printf("%s: %s ...\n",progname,reset_str(breset,bopt));
      ret = IPMI_Reset(breset,bopt);
   }

   if (ret == 0) {   /* if ok */
	printf("%s: IPMI_Reset ok\n",progname);
	/* It starts resetting by this point, so do not close. */
   } else {
	printf("%s: IPMI_Reset error %d\n",progname,ret);
	ipmi_close_(); 
   }
   show_outcome(progname,ret);  
   exit(ret);
}  /* end main()*/

/* end hwreset.c */
