/*
 * icmd.c
 *
 * This tool takes command line input as an IPMI command.
 * 
 * Author:  Andy Cress  arcress@users.sourceforge.net
 * Copyright (c) 2003-2006 Intel Corporation. 
 *
 * 03/30/04 Andy Cress - created
 * 04/08/04 Andy Cress - display response data, changed usage order
 * 05/05/04 Andy Cress - call ipmi_close before exit
 * 11/01/04 Andy Cress - add -N / -R for remote nodes   
 * 12/08/04 Andy Cress v1.5 changed usage order, moved bus to first byte,
 *                       gives better compatibility with ipmicmd.exe.
 * 04/13/06 Andy Cress v1.6 fixed istart for -U -R
 */
/*M*
Copyright (c) 2006, Intel Corporation
All rights reserved.

Redistribution and use in source and binary forms, with or without 
modification, are permitted provided that the following conditions are met:

  a.. Redistributions of source code must retain the above copyright notice, 
      this list of conditions and the following disclaimer. 
  b.. Redistributions in binary form must reproduce the above copyright notice,
      this list of conditions and the following disclaimer in the documentation 
      and/or other materials provided with the distribution. 
  c.. Neither the name of Intel Corporation nor the names of its contributors 
      may be used to endorse or promote products derived from this software 
      without specific prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR 
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *M*/
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#ifdef WIN32
#include "getopt.h"
#else
#include <getopt.h>
#endif
#include "ipmicmd.h"
 
/*
 * Global variables 
 */
static char   fdebug    = 0;
static char * progver   = "2.12";
static char * progname  = "icmd";

#define PRIVATE_BUS_ID      0x03 // w Sahalee,  the 8574 is on Private Bus 1
#define PERIPHERAL_BUS_ID   0x24 // w mBMC, the 8574 is on the Peripheral Bus
#define MAXLEN 64

static char usagemsg[] = "Usage: %s [-EFNPRTUVmsx] bus rsSa netFn/lun cmd [data bytes]\n";
static uchar busid = PRIVATE_BUS_ID;
static uchar g_bus = PUBLIC_BUS;
static uchar g_sa  = BMC_SA;    /*0x20*/
static uchar g_lun = BMC_LUN;
static uchar g_addrtype = ADDR_SMI;

static int send_icmd(uchar *cmd, int len)
{
	uchar responseData[64];
	int responseLength;
	uchar completionCode;
	int ret, i;
	uchar netfn, lun;
        ushort icmd;

        /* icmd format: 0=bus, 1=rsSa, 2=netFn/lun, 3=cmd,  4-n=data */
	lun = cmd[2] & 0x03;
	netfn = cmd[2] >> 2;
        icmd = cmd[3] + (netfn << 8);
	responseLength = sizeof(responseData);
        if (g_addrtype == ADDR_IPMB) {
           /* if -m used, do ipmi_cmd_mc for IPMB commands  */
           ipmi_set_mc(g_bus,g_sa,g_lun,g_addrtype);
           ret = ipmi_cmd_mc(icmd, &cmd[4], (uchar)(len-4), responseData, 
				&responseLength, &completionCode, fdebug);
           ipmi_restore_mc();
        } else {
	   /* ipmi_cmdraw: bus, cmd, netfn, sa, lun, pdata, ... */
           ret = ipmi_cmdraw(cmd[3], netfn, cmd[1], cmd[0], lun, &cmd[4], 
			(uchar)(len-4), responseData, &responseLength, 
			&completionCode, fdebug);
        }
	if (ret < 0) {
           printf("ipmi_cmd: ret = %d %s\n",ret,decode_rv(ret));
	   return(ret);
        } else if ((ret != 0) || (completionCode != 0)) {
           printf("ipmi_cmd: ret = %d, ccode %02x %s\n",
                    ret, completionCode, decode_cc(icmd,completionCode));
	   return(ret);
	} else if (responseLength > 0) {
	   /* show the response data */
	   printf("respData[len=%d]: ",responseLength);
	   for (i = 0; i < responseLength; i++)
	      printf("%02x ",responseData[i]);
	   printf("\n");
	}
	return(ret);
}  /*end send_icmd()*/


#ifdef METACOMMAND
int i_cmd(int argc, char **argv)
#else
#ifdef WIN32
int __cdecl
#else
int
#endif
main(int argc, char **argv)
#endif
{
   int ret = 0;
   char c;
   uchar devrec[17];
   uchar cmdbuf[MAXLEN];
   int maxlen = MAXLEN;
   int i, istart, cmdlen;
   int fskipdevid = 0;

   printf("%s ver %s\n", progname,progver);
   printf("This is a test tool to compose IPMI commands.\n");
   printf("Do not use without knowledge of the IPMI specification.\n");

   istart = 1;
   while ( (c = getopt( argc, argv,"m:T:V:J:YEF:P:N:R:U:sx?")) != EOF ) 
      switch(c) {
          case 'm': /* specific MC, 3-byte address, e.g. "409600" */
                    g_bus = htoi(&optarg[0]);  /*bus/channel*/
                    g_sa  = htoi(&optarg[2]);  /*device slave address*/
                    g_lun = htoi(&optarg[4]);  /*LUN*/
                    g_addrtype = ADDR_IPMB;
                    break;
          case 's': fskipdevid = 1;   break;  /* skip devid */
          case 'x': fdebug = 1;       break;  /* debug messages */
          case 'N':    /* nodename */
          case 'U':    /* remote username */
          case 'P':    /* remote password */
          case 'R':    /* remote password */
          case 'E':    /* get password from IPMI_PASSWORD environment var */
          case 'F':    /* force driver type */
          case 'T':    /* auth type */
          case 'J':    /* cipher suite */ 
          case 'V':    /* priv level */
          case 'Y':    /* prompt for remote password */
                parse_lan_options(c,optarg,fdebug);
                break;
	  default:
                printf(usagemsg, progname);
                printf(" where -x   shows eXtra debug messages\n"); 
                printf("       -m002000 specific MC (bus 00,sa 20,lun 00)\n"); 
                printf("       -s   skips the GetDeviceID command\n"); 
		print_lan_opt_usage();
                exit(1);
      }
   istart = optind;
   if (fdebug) printf("icmd: argc=%d istart=%d\n",argc,istart);

   if (!fskipdevid) {
    /* 
     * Check the Device ID to determine which bus id to use.
     */
    ret = ipmi_getdeviceid(devrec,16,fdebug);
    if (ret != 0) {
        show_outcome(progname,ret);  
	ipmi_close_();
        exit(1);
    } else {
      uchar b, j;
      char *pstr;

      if (fdebug) {
	printf("devid: ");
	for (j = 0; j < 16; j++) printf("%02x ",devrec[j]);
	printf("\n");
      }
      b = devrec[4] & 0x0f;
      j = devrec[4] >> 4;
      if ((devrec[6] == 0x22) && (devrec[7] == 0x03)) { /*NSC mBMC*/
	  pstr = "mBMC";
	  busid = PERIPHERAL_BUS_ID;  /*used by alarms MWR*/
      } else {  /* treat like Intel Sahalee = 57 01 */
	  pstr = "BMC";
	  busid = PRIVATE_BUS_ID;     /*used by alarms MWR*/
      }
      printf("-- %s version %x.%x, IPMI version %d.%d \n",
             pstr, devrec[2],  devrec[3], b, j);
    }
    // ret = ipmi_getpicmg( devrec, sizeof(devrec),fdebug);
    // if (ret == 0) fpicmg = 1;
   } /*endif skip devid*/

   for (i = 0; i < istart; i++) {
      argv++; argc--;
   }
   if (argc < maxlen) maxlen = argc;
   if (fdebug) printf("icmd: len=%d, cmd byte0=%s\n",maxlen,argv[0]);
   for (i = 0; i < maxlen; i++)
   {
      cmdbuf[i] = htoi(argv[i]);
   }
   cmdlen = i;

   if (fdebug) {
        printf("ipmi_cmd: ");
	for (i = 0; i < cmdlen; i++)
	    printf("%02x ",cmdbuf[i]);
        printf("\n");
   }
   if (cmdlen < 4) {
	printf("command length (%d) is too short\n",cmdlen);      
	printf(usagemsg, progname);
   } else {
	ret = send_icmd(cmdbuf,cmdlen); 
	printf("send_icmd ret = %d\n",ret); 
   }
   ipmi_close_();
   show_outcome(progname,ret); 
   exit (ret);
}  /* end main()*/

/* end icmd.c */
