/*
 * idiscover.c
 * Discover all IPMI-LAN-enabled nodes on this network or subnet.
 * This program is not completely reliable yet, not all IPMI-LAN-enabled
 * nodes respond.
 * Currently this utility is compiled with NO_THREADS, but threads can
 * be enabled by commenting out this flag.
 *
 * Author:  Andy Cress  arcress@users.sourceforge.net
 * Copyright (c) 2006 Intel Corporation. 
 *
 * 10/27/06 Andy Cress - created
 * 05/01/07 Andy Cress - added -g for GetChannelAuthCap method,
 *                       added -a logic for broadcast ping,
 *                       updated WIN32 logic
 * 09/20/07 Andy Cress - fixed send/receive thread order
 */
/*M*
Copyright (c) 2006, Intel Corporation
All rights reserved.

Redistribution and use in source and binary forms, with or without 
modification, are permitted provided that the following conditions are met:

  a.. Redistributions of source code must retain the above copyright notice, 
      this list of conditions and the following disclaimer. 
  b.. Redistributions in binary form must reproduce the above copyright notice,
      this list of conditions and the following disclaimer in the documentation 
      and/or other materials provided with the distribution. 
  c.. Neither the name of Intel Corporation nor the names of its contributors 
      may be used to endorse or promote products derived from this software 
      without specific prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR 
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *M*/
#ifdef WIN32
#include <windows.h>
#include <stdio.h>
#include <stdlib.h>
#include <winsock.h>
#include <io.h>
#include <time.h>
#include "getopt.h"
typedef unsigned int socklen_t;
WSADATA lan_ws;  /*global for WSA*/
#define NO_THREADS  1
#else
/* Linux */
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h> 
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/select.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <net/if.h>
#include <netdb.h>
#include <fcntl.h>
#include <getopt.h>
#include <errno.h>
#include <pthread.h>
/* comment out NO_THREADS to use this utility with threads */
#define NO_THREADS  1
#endif

#include <string.h>
#include "ipmicmd.h"
 
#ifndef INET_ADDRSTRLEN
#define INET_ADDRSTRLEN 16
#endif
#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN 64
#endif

#define RMCP_PRI_RMCP_PORT     0x26F
#define SZ_PING    12
#define IPMI_PING_MAX_LEN  50  /* usu 28 */
#define CMD_GET_CHAN_AUTH_CAP      0x38

/*
 * Global variables 
 */
static char * progver   = "1.2";
static char * progname  = "idiscover";
static char   fdebug    = 0;
static char   fping     = 1;
static char   fBroadcastOk = 0;
static int broadcast_pings = 1;
static uchar ipmi_maj = 0;
static uchar ipmi_min = 0;
static uchar netfn;
static ushort g_port   = RMCP_PRI_RMCP_PORT; 
static int    g_sockfd = 0; 
static struct sockaddr_in _srcaddr;
// static struct sockaddr_in _destaddrlist[255];
static struct  in_addr    _startAddr, _endAddr; 
static char g_startDest[MAXHOSTNAMELEN+1] = {'\0'};  
static char g_endDest[MAXHOSTNAMELEN+1] = {'\0'};
static char g_interface[INET_ADDRSTRLEN+1] = {"eth0"};
static int  g_num_packets = 0;
static int  g_npings = 0;
static int  g_npongs = 0;
static int  g_recv_status = 0;
static int  g_wait   = 1; /* num sec to wait */
static int  g_delay  = 0; /* num usec between sends */

void printerr( const char *pattn, ...)
{
   va_list arglist;
   FILE *fderr;

   // fderr = fopen("/tmp/idiscover.log","a+");
   // if (fderr == NULL) return;
   fderr = stderr;

   va_start(arglist, pattn);
   vfprintf(fderr, pattn, arglist);
   va_end(arglist);

   // fclose(fderr);
}

static char *strlasterr(void)
{
   char *str;
#ifdef WIN32
   static char strbuf[80];
   int rv;
   char *desc;
   rv = WSAGetLastError();
   if (rv == WSAECONNRESET) desc = "Connection reset"; /*10054*/
   else if (rv == WSAECONNREFUSED) desc = "Connection refused"; /*10061*/
   else desc = "";
   sprintf(strbuf,"LastError = %d  %s",rv,desc);
   str = strbuf;
#else
   str = strerror(errno);
#endif
   return(str);
}

static void cleanup(void)
{
   if (g_sockfd > 0) {
#ifdef WIN32
      closesocket(g_sockfd);
      WSACleanup();
#else
      close(g_sockfd);
#endif
   }
   g_sockfd = 0;
}

void show_usage(void)
{
   printf("Usage: %s [-abegix] \n",progname);
   printf("       -a      all nodes, enables broadcast ping\n");
   printf("       -b<ip>  beginning IP address, required\n");
   printf("       -e<ip>  ending IP address, default is begin IP\n");
   printf("       -g      use GetChanAuthCap instead of RMCP ping\n");
   printf("       -i      interface name, default is eth0\n");
   // printf("       -s      specific subnet\n");
   printf("       -x      show eXtra debug messages\n");
}

#ifndef METACOMMAND
void dump_buf(char *tag, uchar *pbuf, int sz, char fshowascii)
{
   uchar line[17];
   uchar a;
   int i, j;
   FILE *fpdbg;

   fpdbg = stdout;
   line[0] = 0; line[16] = 0;
   j = 0;
   fprintf(fpdbg,"%s (len=%d): ", tag,sz);
   for (i = 0; i < sz; i++) {
      if (i % 16 == 0) { j = 0; fprintf(fpdbg,"%s\n  %04x: ",line,i); }
      if (fshowascii) {
         a = pbuf[i];
         if (a < 0x20 || a > 0x7f) a = '.';
         line[j++] = a;
      }
      fprintf(fpdbg,"%02x ",pbuf[i]);
   }
   if (j < 16) {
      line[j] = 0;
      for (i = 0; i < (16-j); i++) fprintf(fpdbg,"   ");
   }
   fprintf(fpdbg,"%s\n",line);
   return;
} 
#endif

static int os_sleep(unsigned int s, unsigned int u)
{
#ifdef WIN32
  if (s == 0) {
     if (u >= 1000) Sleep(u/1000);
  } else {
     Sleep(s * 1000);
  }
#else
/*Linux*/
#ifdef SELECT_TIMER
  struct timeval tv;
  tv.tv_sec  = s;
  tv.tv_usec = u;
  if (select(1, NULL, NULL, NULL, &tv) < 0)
     printerr("select: %s\n", strlasterr());
#else
  if (s == 0) {
      usleep(u);
  } else {
      sleep(s);
  }
#endif
#endif
  return 0;
}

void split_ip(uint addr, uchar *ip)
{
   ip[3] = (addr & 0x000000ff);
   ip[2] = (addr & 0x0000ff00) >> 8;
   ip[1] = (addr & 0x00ff0000) >> 16;
   ip[0] = (addr & 0xff000000) >> 24;
}

int ntoi(int addr)
{
   return(addr);
}

void show_ip(int saddr)
{
   uchar ip[4];
   split_ip(saddr,ip);
   printerr("%d.%d.%d.%d\n",ip[0],ip[1],ip[2],ip[3]);
}

static int ipmilan_sendto(int s, const void *msg, size_t len, int flags, 
				const struct sockaddr *to, socklen_t tolen)
{
    int n;
#ifdef NEED_PAD
    int fusepad = 0;
    /* Never need a pad byte for ping/pong packets */
    if (len == 56 || len == 84 || len == 112 || len == 128 || len == 156) {
        fusepad = 1;
        len += 1;
    }
#endif
    n = sendto(s,msg,len,flags,to,tolen);
    // if (fusepad && (n > 0)) n--;
    return(n);
}

static int ipmilan_recvfrom(int s, void *buf, size_t len, int flags, 
		struct sockaddr *from, socklen_t *fromlen)
{
    int rv;
    rv = recvfrom(s,buf,len,flags,from,fromlen);
    /* Sometimes the OS sends an ECONNREFUSED error, but 
     * retrying will catch the BMC's reply packet. */
#ifdef WIN32
    if (rv < 0) {
        int err;
        err = WSAGetLastError();
        if (err == WSAECONNREFUSED) /*10061*/
            rv = recvfrom(s,buf,len,flags,from,fromlen);
    }
#else
    if ((rv < 0) && (errno == ECONNREFUSED))
        rv = recvfrom(s,buf,len,flags,from,fromlen);
#endif
    return(rv);
}

#ifdef WIN32
int inet_aton(const char *cp, struct in_addr *inp)
{
   int rv;
   int adr;
   inp->s_addr = inet_addr(cp);
   adr = (int)inp->s_addr;
   if (adr == INADDR_NONE) rv = 0;
   else rv = 1;  /*success*/
   return(rv);
}
#endif

int sock_init( char *_interface, char *_startIP, char *_endIP)
{
  int rv;
  // struct hostent *hptr;
 
#ifdef WIN32
  DWORD rvl;
  rvl = WSAStartup(0x0101,&lan_ws);
  if (rvl != 0) {
      printerr("init: WSAStartup(1.1) error %ld\n", rvl);
      return((int)rvl);
  }
#endif

  if ((g_sockfd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
    printerr("socket: %s\n", strlasterr());
    return(-1);
  }

  memset(&_srcaddr, 0, sizeof(_srcaddr));
  _srcaddr.sin_family = AF_INET;
  _srcaddr.sin_port = htons(0);
  _srcaddr.sin_addr.s_addr = htonl(INADDR_ANY);
 
#ifndef WIN32
  if (_interface != NULL)
  {
      if (strchr(_interface, '.') != NULL)
        {   /* assume it is an IP address*/
          if ((rv = inet_pton(AF_INET, _interface, &_srcaddr.sin_addr)) < 0)
            printerr("inet_pton: %s\n", strlasterr());
          if (rv == 0) 
            printerr("invalid interface address\n");
          return(rv);
        }
      else
        {   /* assume interface name, like eth0 */
          struct ifreq ifr;
          struct sockaddr_in temp_sockaddr;
 
          strncpy(ifr.ifr_name, _interface, IFNAMSIZ);
          ifr.ifr_addr.sa_family = AF_INET;
          if (ioctl(g_sockfd, SIOCGIFADDR, &ifr) < 0) {
             printerr("ioctl: %s\n", strlasterr());
             return(-1);
          }
 
          temp_sockaddr = *((struct sockaddr_in *)&ifr.ifr_addr);
          memcpy(&_srcaddr.sin_addr.s_addr, &temp_sockaddr.sin_addr.s_addr,
                 sizeof(_srcaddr.sin_addr.s_addr));
        }
   }
#endif
 
   if (bind(g_sockfd, (struct sockaddr *)&_srcaddr, sizeof(_srcaddr)) < 0) {
      printerr("bind: %s\n", strlasterr());
      return(-1);
   }
 
   rv = inet_aton(_startIP, &_startAddr);
   if (rv ) {
        _startAddr.s_addr = ntohl(_startAddr.s_addr);
        if (fdebug) show_ip(_startAddr.s_addr);
        if (*((unsigned char*)&_startAddr.s_addr) <1 )
                printerr("Malformed begin IP: %s\n", _startIP);
        else if (!fBroadcastOk && (*((unsigned char*)&_startAddr.s_addr) >254))
                printerr("Malformed begin IP: %s\n", _startIP);
        else rv = 0;
   } else {
        printerr("Invalid begin IP: %s\n", _startIP);
   }
 
   rv = inet_aton(_endIP, &_endAddr);
   if (rv ) {
        _endAddr.s_addr = ntohl(_endAddr.s_addr);
        if (fdebug) show_ip(_endAddr.s_addr);
        if (*((unsigned char*)&_endAddr.s_addr) <1 )
              printerr("Malformed end IP: %s\n", _endIP);
        else if (!fBroadcastOk && (*((unsigned char*)&_endAddr.s_addr) >254))
              printerr("Malformed end IP: %s\n", _endIP);
        else rv = 0;
   } else {
        printerr("Invalid end IP: %s\n", _endIP);
   }
   if (rv) show_usage();

   /* calculate g_num_packets */
   g_num_packets = ntoi(_endAddr.s_addr) - ntoi(_startAddr.s_addr) + 1;
   if (fdebug) printerr("g_num_packets = %d\n",g_num_packets);
   if (g_num_packets < 1) g_num_packets = 0;

   return(rv);
}  /*end init*/

void *receiveThread(void *p)
{
   uchar buffer[IPMI_PING_MAX_LEN];
   struct timeval tv;
   fd_set rset;
   int rv, len;

   do
   {
      tv.tv_sec  = g_wait;
      tv.tv_usec = 0;
      FD_ZERO(&rset);
      FD_SET(g_sockfd, &rset);
      g_recv_status = 0;

      if (fdebug) printerr("waiting for ping %d response\n",g_npings);
      if ((rv = select(g_sockfd+1, &rset, NULL, NULL, &tv)) < 0) {
          printerr("select: %s\n", strlasterr());
          cleanup();
          exit(-1);
      }
      
      if (fdebug) printerr("select rv = %d\n",rv);
      if (rv > 0)
          {
              struct sockaddr_in from;
              socklen_t fromlen;
 
              g_recv_status = 1;
              fromlen = sizeof(from);
              len = ipmilan_recvfrom(g_sockfd, buffer, IPMI_PING_MAX_LEN, 0,
                                     (struct sockaddr *)&from, &fromlen);
              if (fdebug) printerr("recvfrom rv = %d\n",rv);
              if (len < 0) { 
		printerr("ipmilan_recvfrom: %s\n", strlasterr());
		continue; 
              }
              g_recv_status = 2;
              g_npongs++;
              // parse received pong 
              printf("%d: response from %s\n",g_npongs,
			inet_ntoa(from.sin_addr));
              if (fdebug) dump_buf("recvfrom",buffer,len,0);
          }
   }
#ifdef NO_THREADS
     while(fBroadcastOk && rv > 0);
#else
     while(1);
#endif
   return(p);
} 

/*
 * send_ping_pkt:
 * RMCP Ping buffer, sent as a UDP packet to port 0x026f.
 * rmcp.ver     = 0x06          // RMCP Version 1.0
 * rmcp.resvd   = 0x00          // RESERVED
 * rmcp.seq     = 0xff          // no RMCP ACK 
 * rmcp.class   = 0x06          // RMCP_CLASS_ASF
 * asf.iana     = 0x000011BE    // ASF_RMCP_IANA (hi-lo)
 * asf.type     = 0x80          // ASF_TYPE_PING
 * asf.tag      = 0x00          // ASF sequence number
 * asf.resvd    = 0x00          // RESERVED
 * asf.len      = 0x00
 */
int send_ping_pkt(struct sockaddr_in *_daddr, uchar seq)
{
   uchar pingbuf[SZ_PING] = {06,0,0xFF,06,0x00,0x00,0x11,0xBE,0x80,0,0,0 };
   int rv, len;

   pingbuf[9] = seq;
   len = sizeof(pingbuf);
   if (fdebug) dump_buf("send_ping",pingbuf,len,0);
   rv = ipmilan_sendto(g_sockfd, pingbuf, len, 0, 
	          (struct sockaddr *)_daddr, sizeof(struct sockaddr_in));
   return(rv);
}

static int send_poke1(struct sockaddr_in *_daddr)
{
   int rv;
   uchar asfpkt[16] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x07, 0x20, 0x18, 0xc8, 0xc2, 0x01, 0x01, 0x3c };
   if (fdebug) dump_buf("send_poke1",asfpkt,16,0);
   rv = ipmilan_sendto(g_sockfd, asfpkt, 16, 0, 
	          (struct sockaddr *)_daddr, sizeof(struct sockaddr_in));
   return rv;
}

static uchar cksum(const uchar *buf, register int len)
{
        register uchar csum;
        register int i;
 
        /* 8-bit 2s compliment checksum */
        csum = 0;
        for (i = 0; i < len; i++)
           csum = (csum + buf[i]) % 256;
        csum = -csum;
        return(csum);
}

static int send_getauth(struct sockaddr_in *_daddr, uchar seq)
{
   int rv, len;
   static uchar swseq = 1;
   uchar getauthpkt[23] = { 0x06, 0x00, 0xff, 0x07, 0x00, 0x00, 0x00, 0x00, 
   		            0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x20, 0x18, 
   			    0xc8, 0x81, 0x04, 0x38, 0x0e, 0x04, 0x31 };
   /* 
    *  0  1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 17 18 19 20 21 22
    * 06 00 ff 07 00 00 00 00 00 00 00 00 00 09 20 18 c8 81 04 38 0e 04 31 
    */
   getauthpkt[18] = seq;   // swseq++;
   getauthpkt[19] = cksum(&getauthpkt[17],2);
   getauthpkt[21] = 0x02;  /*priv_level=user*/
   getauthpkt[22] = cksum(&getauthpkt[20],2);
   len = sizeof(getauthpkt);
   if (fdebug) dump_buf("send_getauth",getauthpkt,len,0);
   rv = ipmilan_sendto(g_sockfd, getauthpkt, len, 0, 
   	          (struct sockaddr *)_daddr, sizeof(struct sockaddr_in));
   if (fdebug) 
       printf("getauth: rv = %d\n",rv);
   return rv;
}

int send_probe(struct sockaddr_in *_daddr, uchar seq)
{
    int rv;

    if (fBroadcastOk) {
       rv = setsockopt(g_sockfd, SOL_SOCKET, SO_BROADCAST,
                       (char *)&broadcast_pings, sizeof(broadcast_pings));
       if (fdebug) 
	   printerr("setsockopt(broadcast) rv=%d %s\n",rv,strlasterr());
    }
    if (fping) 
       rv = send_ping_pkt( _daddr, seq);
    else 
       rv = send_getauth( _daddr, seq);
    return(rv);
}

void *sendThread(void *p)
{
   int i, n;
   // char _dest[MAXHOSTNAMELEN+1];
   char _dest_ip[INET_ADDRSTRLEN+1];
   struct sockaddr_in _destaddr;
   uchar o[4];
   uint ip;
   uchar _seq;
   int rv;

   n = g_num_packets;  /*num*/
   ip = _startAddr.s_addr;

   for (i = 0; i < n; i++) 
   {
      split_ip(ip,o);
      if (o[3] == 0) continue;
      if (!fBroadcastOk && (o[3] == 255)) continue;
      sprintf(_dest_ip,"%d.%d.%d.%d",o[0],o[1],o[2],o[3]);

      /* set _destaddr */
      _destaddr.sin_family = AF_INET;
      _destaddr.sin_port = htons(g_port);
      if ( !inet_aton( _dest_ip, &_destaddr.sin_addr)) {
          printerr("inet_aton error %s\n",_dest_ip);
          continue;
      }

      /* send ping buffer */
      _seq = 0;
      rv = send_probe(&_destaddr,_seq);
      g_npings++;
      if (fdebug) printerr("sendto[%d,%d] %s rv = %d\n",
			    g_npings,_seq,_dest_ip,rv);
      if (rv < 0) {  /*try to send again*/
          rv = send_probe(&_destaddr,++_seq);
          if (rv < 0) {
              printerr("sendto[%d,%d] %s error %s\n",
			g_npings,_seq,_dest_ip,strlasterr()); 
              continue;
          }
      } 

#ifdef NO_THREADS
      receiveThread(NULL);
      if (g_recv_status == 0 && !fBroadcastOk) {
          /* nothing returned, try again */
          if (fping) {
             rv = send_poke1(&_destaddr);
             if (fdebug) printerr("sendto[%d,%d] %s poke rv = %d\n",
				   g_npings,_seq,_dest_ip,rv);
          }
          rv = send_probe(&_destaddr,++_seq);
          if (fdebug) printerr("sendto[%d,%d] %s rv = %d\n",
				g_npings,_seq,_dest_ip,rv);
          if (rv >= 0) {
             receiveThread(NULL);
          }
      }
#endif

      /* sleep an interval (g_delay usec) */
      if (g_delay > 0) os_sleep(0,g_delay);

      ip++; /* increment to next IP */
   }
   return(p);
}

#ifdef METACOMMAND
int i_discover(int argc, char **argv)
#else
#ifdef WIN32
int __cdecl
#else
int
#endif
main(int argc, char **argv)
#endif
{
   int rv = 0;
   char c;
#ifndef NO_THREADS
   char message[32];
   pthread_t thread[2];
   int  iret[2];
#endif

   printf("%s ver %s\n", progname,progver);

   while ( (c = getopt( argc, argv,"ab:e:gi:p:s:x?")) != EOF ) 
      switch(c) {
          case 'a': fBroadcastOk = 1; fping = 1;
		break;  /*all (broadcast ping)*/
          case 'g': fping = 0; break;   /*use get chan auth cap method */
          case 'b':   /*begin*/
                strncpy(g_startDest,optarg,MAXHOSTNAMELEN);
		break;
          case 'e':   /*end*/
                strncpy(g_endDest,optarg,MAXHOSTNAMELEN);
		break;
          case 'i':   /*interface*/
                strncpy(g_interface,optarg,sizeof(g_interface));
		break;
          case 'p':   /*port/socket*/
                g_port = atoi(optarg);
		break;
          case 's':   /*subnet*/
                /* copy optarg from 10.243.42.0 or similar, to
                 * begin/end range. */
		break;
          case 'x': fdebug = 1;     break;  /* debug messages */
	  default:
		if (fdebug) printerr("getopt(%c) default\n",c);
                show_usage();
                exit(1);
      }
   if (g_startDest[0] == 0) {
      printerr("A beginning IP is required, using -b\n");
      show_usage();
      exit(1);
   }
   if (g_endDest[0] == 0 || fBroadcastOk)
      strcpy(g_endDest,g_startDest);   /*only one IP address*/
   if (fdebug) 
      printerr("begin=%s end=%s port=%d\n",g_startDest,g_endDest,g_port);

   rv = sock_init(g_interface, g_startDest, g_endDest); 
   if (fdebug) printerr("init rv = %d, sockfd = %d\n",rv,g_sockfd);
   if (rv != 0) exit(rv);

   printerr("Discovering IPMI Devices:\n");
#ifdef NO_THREADS
   sendThread(NULL);
#else
   iret[1] = pthread_create( &thread[1], NULL, receiveThread, (void*) message);
   iret[0] = pthread_create( &thread[0], NULL, sendThread, (void*) message);
   pthread_join( thread[0], NULL);
   pthread_join( thread[1], NULL);
#endif

   // if (fdebug) 
   printerr("%s: %d pings sent, %d responses\n",progname,g_npings,g_npongs);

   cleanup();
   // show_outcome(progname,rv); 
   exit (rv);
}  /* end main()*/

/* end idiscover.c */
