/*M*
//  PVCS:
//      $Workfile:   ipmicmd.c  $
//      $Revision:   1.12  $
//      $Modtime:   23 Feb 2005 11:24:14  $
//      $Author:   arcress@users.sourceforge.net  $  
//
//  Define the ipmi_cmd routine and supporting logic to execute IPMI 
//  commands via one of the supported IPMI drivers:
//     /dev/ipmi0  /dev/ipmi/0    = MontaVista OpenIPMI driver
//     /dev/imb                   = Intel IMB ipmidrvr (comes with ISM)
//     /dev/ipmikcs /dev/ipmi/kcs = valinux driver by San Mehat
//     libfreeipmi.so             = GNU FreeIPMI user-space library
//     ldipmidaemon               = LanDesk IPMI daemon (user-space process) 
// 
//  08/05/02 ARC - created 
//  08/15/02 ARC - added decode_cc
//  10/24/02 ARC - made cmd param ushort to be more unique
//  01/29/03 ARC - added MontaVista OpenIPMI driver support
//  07/25/03 ARC - added serial-over-lan commands
//  07/30/03 ARC - added GetThresholds, fix for ipmi_cmd_raw,
//                 changed some error messages
//  09/04/03 ARC - added debug messages for fDriverTyp first time
//  05/05/04 ARC - leave _mv device open, rely on each app calling ipmi_close,
//		   helps performance.
//  08/10/04 ARC - fix typo in ipmi_cmd_raw/mv: cmd->icmd (thanks Kevin Gao)
//  08/26/04 ARC - fix out-of-bounds error in decode_cc
//  10/27/04 ARC - added gnu FreeIPMI library support
//  11/11/04 ARC - added fdebug to ipmi_getdeviceid & ipmi_open_gnu 
//  02/23/05 ARC - added routines for LanDesk, fDriverTyp=5
//  07/15/05 ARC - test for ldipmi first, since it hangs KCS if another 
//                 driver tries to coexist.
//  07/06/06 ARC - better separate driver implementations, cleaner now
 *M*/
/*----------------------------------------------------------------------*
The BSD License 

Copyright (c) 2002-2006, Intel Corporation
All rights reserved.

Redistribution and use in source and binary forms, with or without 
modification, are permitted provided that the following conditions are met:

  a.. Redistributions of source code must retain the above copyright notice, 
      this list of conditions and the following disclaimer. 
  b.. Redistributions in binary form must reproduce the above copyright notice,
      this list of conditions and the following disclaimer in the documentation 
      and/or other materials provided with the distribution. 
  c.. Neither the name of Intel Corporation nor the names of its contributors 
      may be used to endorse or promote products derived from this software 
      without specific prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR 
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *----------------------------------------------------------------------*/
#ifdef WIN32
#include <windows.h>
#include <stdio.h>
#include <io.h>
#include <fcntl.h>
#elif defined (EFI32) || defined (EFI64) || defined(EFIX64)
#include <bmc.h>
#include <libdbg.h>
#else
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <termios.h>
#include <stdarg.h>
#include <errno.h>
#endif

#include "ipmicmd.h"    /* has NCMDS, ipmi_cmd_t */

ipmi_cmd_t ipmi_cmds[NCMDS] = { 
 /*empty,temp*/ 0, BMC_SA, PUBLIC_BUS, NETFN_APP,   BMC_LUN, 0, 20, 
 GET_SEL_INFO,     BMC_SA, PUBLIC_BUS, NETFN_STOR,  BMC_LUN, 0, 14,
 GET_SEL_ENTRY,    BMC_SA, PUBLIC_BUS, NETFN_STOR,  BMC_LUN, 6, 18,
 RESERVE_SEL,      BMC_SA, PUBLIC_BUS, NETFN_STOR,  BMC_LUN, 0, 2,
 CLEAR_SEL,        BMC_SA, PUBLIC_BUS, NETFN_STOR,  BMC_LUN, 6, 1,
 GET_SEL_TIME,     BMC_SA, PUBLIC_BUS, NETFN_STOR,  BMC_LUN, 6, 4,
 GET_LAN_CONFIG,   BMC_SA, PUBLIC_BUS, NETFN_TRANS, BMC_LUN, 4, 19,
 SET_LAN_CONFIG,   BMC_SA, PUBLIC_BUS, NETFN_TRANS, BMC_LUN, 21, 0,
 GET_SER_CONFIG,   BMC_SA, PUBLIC_BUS, NETFN_TRANS, BMC_LUN, 4, 19,
 SET_SER_CONFIG,   BMC_SA, PUBLIC_BUS, NETFN_TRANS, BMC_LUN, 21, 0,
 SET_SER_MUX,      BMC_SA, PUBLIC_BUS, NETFN_TRANS, BMC_LUN, 2, 0,
 GET_PEF_CONFIG,   BMC_SA, PUBLIC_BUS, NETFN_SEVT,  BMC_LUN, 3, 22,
 SET_PEF_CONFIG,   BMC_SA, PUBLIC_BUS, NETFN_SEVT,  BMC_LUN, 22, 0,
//  SET_PEF_ENABLE,  BMC_SA, PUBLIC_BUS, NETFN_APP, BMC_LUN, 4, 0, /*old*/
 GET_DEVSDR_INFO,  BMC_SA, PUBLIC_BUS, NETFN_SEVT,  BMC_LUN, 0, 6,
 GET_DEVICE_SDR,   BMC_SA, PUBLIC_BUS, NETFN_SEVT,  BMC_LUN, 6, 18,
 RESERVE_DEVSDR_REP,BMC_SA,PUBLIC_BUS, NETFN_SEVT,  BMC_LUN, 0, 2,
 GET_SENSOR_READING,BMC_SA,PUBLIC_BUS, NETFN_SEVT,  BMC_LUN, 1, 4,
 GET_SENSOR_READING_FACTORS,BMC_SA,PUBLIC_BUS, NETFN_SEVT, BMC_LUN, 2, 7,
 GET_SENSOR_TYPE,  BMC_SA, PUBLIC_BUS, NETFN_SEVT,    BMC_LUN, 1, 2,
 GET_SENSOR_THRESHOLD,BMC_SA,PUBLIC_BUS, NETFN_SEVT,  BMC_LUN, 1, 7,
 SET_SENSOR_THRESHOLD,BMC_SA,PUBLIC_BUS, NETFN_SEVT,  BMC_LUN, 8, 0,
 GET_SENSOR_HYSTERESIS,BMC_SA,PUBLIC_BUS, NETFN_SEVT, BMC_LUN, 2, 2,
 SET_SENSOR_HYSTERESIS,BMC_SA,PUBLIC_BUS, NETFN_SEVT, BMC_LUN, 4, 0,
 GET_SDR,          BMC_SA, PUBLIC_BUS, NETFN_STOR,  BMC_LUN, 6, 18, /*full=63*/
 GET_SDR_REPINFO,  BMC_SA, PUBLIC_BUS, NETFN_STOR,  BMC_LUN, 0, 14,
 RESERVE_SDR_REP,  BMC_SA, PUBLIC_BUS, NETFN_STOR,  BMC_LUN, 0, 2,
 GET_FRU_INV_AREA, BMC_SA, PUBLIC_BUS, NETFN_STOR,  BMC_LUN, 1, 3,
 READ_FRU_DATA,    BMC_SA, PUBLIC_BUS, NETFN_STOR,  BMC_LUN, 4, 18,
 WRITE_FRU_DATA,   BMC_SA, PUBLIC_BUS, NETFN_STOR,  BMC_LUN, 20 /*3+N(17)*/, 1,
 GET_DEVICE_ID,    BMC_SA, PUBLIC_BUS, NETFN_APP,   BMC_LUN, 0, 15,
 SET_USER_ACCESS,  BMC_SA, PUBLIC_BUS, NETFN_APP,   BMC_LUN, 4, 0,
 GET_USER_ACCESS,  BMC_SA, PUBLIC_BUS, NETFN_APP,   BMC_LUN, 2, 4,
 GET_USER_NAME,    BMC_SA, PUBLIC_BUS, NETFN_APP,   BMC_LUN, 1, 16,
 SET_USER_NAME,    BMC_SA, PUBLIC_BUS, NETFN_APP,   BMC_LUN, 17, 0,
 SET_USER_PASSWORD,BMC_SA, PUBLIC_BUS, NETFN_APP,   BMC_LUN, 18, 0,
 MASTER_WRITE_READ,BMC_SA, PUBLIC_BUS, NETFN_APP,   BMC_LUN, 4 /*or 3*/, 1,
 GET_SYSTEM_GUID,  BMC_SA, PUBLIC_BUS, NETFN_APP,   BMC_LUN, 0, 16,
 WATCHDOG_GET,     BMC_SA, PUBLIC_BUS, NETFN_APP,   BMC_LUN, 0, 8,
 WATCHDOG_SET,     BMC_SA, PUBLIC_BUS, NETFN_APP,   BMC_LUN, 6, 0,
 WATCHDOG_RESET,   BMC_SA, PUBLIC_BUS, NETFN_APP,   BMC_LUN, 0, 0,
 CHASSIS_STATUS,   BMC_SA, PUBLIC_BUS, NETFN_CHAS,  BMC_LUN, 0, 2,
 CHASSIS_CTL,      BMC_SA, PUBLIC_BUS, NETFN_CHAS,  BMC_LUN, 1, 0,
 CHASSIS_IDENTIFY, BMC_SA, PUBLIC_BUS, NETFN_CHAS,  BMC_LUN, 1, 0,
 GET_POWERON_HOURS,BMC_SA, PUBLIC_BUS, NETFN_CHAS,  BMC_LUN, 0, 0,
 SET_BOOT_OPTIONS, BMC_SA, PUBLIC_BUS, NETFN_CHAS,  BMC_LUN, 19, 0,
 GET_BOOT_OPTIONS, BMC_SA, PUBLIC_BUS, NETFN_CHAS,  BMC_LUN, 3, 18,
 ACTIVATE_SOL1,    BMC_SA, PUBLIC_BUS, NETFN_SOL,   BMC_LUN, 0, 0,
 SET_SOL_CONFIG,   BMC_SA, PUBLIC_BUS, NETFN_SOL,   BMC_LUN, 3, 0,
 GET_SOL_CONFIG,   BMC_SA, PUBLIC_BUS, NETFN_SOL,   BMC_LUN, 4, 2,
 ACTIVATE_SOL2,    BMC_SA, PUBLIC_BUS, NETFN_TRANS, BMC_LUN, 0, 0,
 SET_SOL_CONFIG2,  BMC_SA, PUBLIC_BUS, NETFN_TRANS, BMC_LUN, 3, 0,
 GET_SOL_CONFIG2,  BMC_SA, PUBLIC_BUS, NETFN_TRANS, BMC_LUN, 4, 2,
 GET_SEVT_ENABLE,  BMC_SA, PUBLIC_BUS, NETFN_SEVT,  BMC_LUN, 1, 5,
 SET_SEVT_ENABLE,  BMC_SA, PUBLIC_BUS, NETFN_SEVT,  BMC_LUN, 6, 0,
 REARM_SENSOR,     BMC_SA, PUBLIC_BUS, NETFN_SEVT,  BMC_LUN, 6, 0,
 READ_EVENT_MSGBUF,BMC_SA, PUBLIC_BUS, NETFN_APP,   BMC_LUN, 0, 16,
 GET_EVENT_RECEIVER,BMC_SA,PUBLIC_BUS, NETFN_SEVT,  BMC_LUN, 0, 2,
 GET_CHANNEL_INFO, BMC_SA, PUBLIC_BUS, NETFN_APP,   BMC_LUN, 1, 9,
 SET_CHANNEL_ACC,  BMC_SA, PUBLIC_BUS, NETFN_APP,   BMC_LUN, 3, 0,
 GET_CHANNEL_ACC,  BMC_SA, PUBLIC_BUS, NETFN_APP,   BMC_LUN, 2, 1 };

/* Subroutine definitions for each driver */
#ifdef EFI
int ipmi_open_efi(char fdebug);
int ipmi_cmdraw_efi( uchar cmd, uchar netfn, uchar lun, uchar sa, 
		uchar bus, uchar *pdata, uchar sdata, 
	 	uchar *presp, int *sresp, uchar *pcc, char fdebugcmd);
#else
extern int ipmi_cmdraw_ia( uchar cmd, uchar netfn, uchar lun, uchar sa, 
		uchar bus, uchar *pdata, uchar sdata, 
	 	uchar *presp, int *sresp, uchar *pcc, char fdebugcmd);
extern int ipmi_open_ia(char fdebug);
extern int ipmi_close_ia(void);
#ifndef GNU_LAN
#include "ipmilan2.h"  /*includes ipmilan.h also*/
#endif
#ifndef WIN32
extern int ipmi_cmdraw_mv(uchar cmd, uchar netfn, uchar lun, uchar sa, 
			uchar bus, uchar *pdata, uchar sdata, 
			uchar *presp, int *sresp, uchar *pcc, char fdebugcmd);
extern int ipmi_cmd_mv(ushort cmd, uchar *pdata, uchar sdata, uchar *presp, 
			int *sresp, uchar *pcc, char fdebugcmd);
extern int ipmi_open_mv(char fdebug);
extern int ipmi_close_mv(void);
extern int ipmi_open_ld(char fdebug);
extern int ipmi_close_ld(void);
extern int ipmi_cmdraw_ld(uchar cmd, uchar netfn, uchar lun, uchar sa, 
			uchar bus, uchar *pdata, uchar sdata, 
			uchar *presp, int *sresp, uchar *pcc, char fdebugcmd);
extern int ipmi_cmd_ld(ushort cmd, uchar *pdata, uchar sdata, uchar *presp, 
			int *sresp, uchar *pcc, char fdebugcmd);
extern int ipmi_open_gnu(char *node, char *user, char *pswd, int fdebug);
extern int ipmi_close_gnu(char *node);
extern int ipmi_cmdraw_gnu(char *node, uchar cmd, uchar netfn, uchar lun, 
			uchar sa, uchar bus, uchar *pdata, uchar sdata, 
			uchar *presp, int *sresp, uchar *pcc, char fdebugcmd);
extern int ipmi_cmd_gnu(char *node, ushort cmd, uchar *pdata, uchar sdata, 
	 	uchar *presp, int *sresp, uchar *pcc, char fdebugcmd);
extern int ipmi_open_direct(char fdebug);
extern int ipmi_close_direct(void);
extern int ipmi_cmdraw_direct( uchar cmd, uchar netfn, uchar lun, 
			uchar sa, uchar bus, uchar *pdata, uchar sdata, 
			uchar *presp, int *sresp, uchar *pcc, char fdebugcmd);
extern int ipmi_cmd_direct(ushort cmd, uchar *pdata, uchar sdata, uchar *presp, 
			int *sresp, uchar *pcc, char fdebugcmd);
#endif
extern int fd_wait(int fd, int nsec, int usec);
#endif

/* Global Data */
int fDriverTyp = 0;   /* 1= IMB driver,   2= VA driver,  3= MV open driver,  */
                      /* 4= GNU FreeIPMI, 5= LanDesk,    6= builtin IPMI LAN */
		      /* 7= direct KCS,   8= direct SMB, 9= IPMI LAN v2.0 */
#define DRV_SMB   8
#define DRV_LAN2  9
int fipmi_lan = 0;    
FILE *fperr = NULL;
FILE *fpdbg = NULL;
FILE *fplog = NULL;
char log_name[60] = {'\0'};  /*log_name global*/
char *gnode = "localhost";  /*used for gnu FreeIPMI, util option -N */
char *guser = NULL;
char *gpswd = NULL;
char gnodename[SZGNODE] = {0};  /*the node name returned after a connection */
int gcipher_suite = 3;          /*used in ipmilanplus.c*/
int gshutdown  = 0;
int gauth_type  = IPMI_SESSION_AUTHTYPE_MD5; /*if 0, use any: MD5, MD2, etc.*/
int gpriv_level = IPMI_PRIV_LEVEL_ADMIN;
uchar my_devid[20];  /*saved devid data, only needs 16 bytes*/

typedef struct {
  uchar adrtype;
  uchar sa;
  uchar bus;
  uchar lun;
  uchar capab;
} mc_info;
mc_info  bmc = { ADDR_SMI,  BMC_SA, PUBLIC_BUS, BMC_LUN, 0x8F }; /*BMC via SMI*/
mc_info  mc2 = { ADDR_IPMB, BMC_SA, PUBLIC_BUS, BMC_LUN, 0x4F }; /*IPMB */
static mc_info  *mc  = &bmc;

/* From IPMI v1.5/v2.0 spec, Table 5-2 Completion Codes */
#define NUMCC    32
struct {
	uchar code;
	char *mesg;
	}  cc_mesg[NUMCC] = { 
/* Note: completion codes 0x80-0x9f may vary depending on the command.
 * 0x80 = Invalid Session Handle or Empty Buffer or Unsupported Feature 
 */
0x00, "Command completed successfully",
0x80, "Invalid Session Handle or Empty Buffer",
0x81, "Lost Arbitration",   
0x82, "Bus Error",
0x83, "NAK on Write - busy",
0x84, "Truncated Read",
0x85, "Invalid session ID in request",            /*for ActivateSession*/
0x86, "Requested privilege level exceeds limit",  /*for ActivateSession*/
0xC0, "Node Busy",
0xC1, "Invalid Command",
0xC2, "Command invalid for given LUN",
0xC3, "Timeout while processing command",
0xC4, "Out of space",
0xC5, "Invalid Reservation ID, or cancelled",
0xC6, "Request data truncated",
0xC7, "Request data length invalid",
0xC8, "Request data field length limit exceeded",
0xC9, "Parameter out of range",
0xCA, "Cannot return requested number of data bytes",
0xCB, "Requested sensor, data, or record not present",
0xCC, "Invalid data field in request",
0xCD, "Command illegal for this sensor/record type",
0xCE, "Command response could not be provided",
0xCF, "Cannot execute duplicated request",
0xD0, "SDR Repository in update mode, no response",
0xD1, "Device in firmware update mode, no response",
0xD2, "BMC initialization in progress, no response",
0xD3, "Destination unavailable",
0xD4, "Cannot execute command. Insufficient privilege level",
0xD5, "Cannot execute command. Request parameters not supported",
0xD6, "Cannot execute command. Subfunction unavailable", 
0xFF, "Unspecified error"
};

char * decode_cc(ushort icmd, uchar cc)
{
   char *pmsg;
   int i;
   for (i = 0; i < NUMCC; i++) {
	if (cc == cc_mesg[i].code) break;
   }
   if (i == NUMCC) i--;  /* if not found, use Unspecified */
   if ((icmd == READ_EVENT_MSGBUF) && (cc == 0x80))
        pmsg = "no data available (queue/buffer empty)";
   else pmsg = cc_mesg[i].mesg;
   return(pmsg);
}

int get_cmd_rslen(uchar cmd, uchar netfn)
{			/* used by ipmicmd_gnu */
   int rslen = 0;
   int i;
   ushort cmdkey;
   cmdkey = cmd | (netfn << 8);
   for (i = 0; i < NCMDS; i++) {
      if (ipmi_cmds[i].cmdtyp == cmdkey) {
	rslen = ipmi_cmds[i].rslen;
	break;
	}
   }
   return(rslen);
} /*end get_cmd_rslen()*/

static int ndrivers = NDRIVERS + 1;
static struct {
     int idx;
     char *tag;
   } drv_types[NDRIVERS + 1] = {
  { DRV_IMB,  "imb" },
  { DRV_VA,   "va" },
  { DRV_MV,   "open" },
  { DRV_GNU,  "gnu" },
  { DRV_LD,   "landesk" },
  { DRV_LAN2, "lan2" },
  { DRV_LAN2, "lan2i" },
  { DRV_LAN,  "lan" },
  { DRV_KCS,  "kcs" },
  { DRV_SMB,  "smb" }
};

char *show_driver_type(int idx)
{
   int i;
   char *tag;
   for (i = 0; i < ndrivers; i++)
   {
      if (drv_types[i].idx == idx) {
         tag = drv_types[i].tag;
         break;
      }
   }
   if (i >= ndrivers) {  /*not found*/
      tag = "unknown";
   }
   return(tag);
}

int set_driver_type(char *tag)
{
   int rv = 0;
   int i;
   if (strcmp(tag,"lan2i") == 0) { /*set vendor id to Intel */
        my_devid[6] = 0x57;
        my_devid[7] = 0x01;
        my_devid[8] = 0x00;
   }
   for (i = 0; i < ndrivers; i++)
   {
      if (strcmp(drv_types[i].tag, tag) == 0) {
         fDriverTyp = drv_types[i].idx;
         break;
      }
   }
   if (i >= ndrivers) {  /*not found*/
      fDriverTyp = 0;  /*not set yet, so detect*/
      rv = 1;
      // if (fdebugcmd) 
      {
         printf("Invalid argument, valid driver types are:\n");
         for (i = 0; i < ndrivers; i++)
            printf("\t%s\n",drv_types[i].tag);
      }
   }
   return(rv);
}

int get_driver_type(void)
{
    return(fDriverTyp);
}

int nodeislocal(char *nodename)
{
   if (nodename == NULL) return 1;
   if (nodename[0] == 0) return 1;
   if (strcmp(nodename,"localhost") == 0) return 1;
   return 0;
}

/* 
 * ipmi_open
 * This is called by ipmi_cmd and ipmicmd_raw if a session has not
 * yet been opened (fDriverTyp == DRV_UNKNOWN).
 * The order to try these drivers could be customized for specific
 * environments by modifying this routine.
 */
int ipmi_open(char fdebugcmd)
{
   int rc = 0;
   fperr = stderr;
   fpdbg = stdout;

#ifdef EFI
   rc = ipmi_open_efi(fdebugcmd);
#else
   if (!nodeislocal(gnode)) fipmi_lan = 1;
   if (fdebugcmd) printf("ipmi_open: driver type = %s\n",
			show_driver_type(fDriverTyp));
   /* first time, so try each */
   if (fipmi_lan) {
#ifdef GNU_LAN
#ifdef WIN32
	    return(-1);
#else
	    rc = ipmi_open_gnu(gnode,guser,gpswd,fdebugcmd); 
	    fDriverTyp = DRV_GNU;
#endif
#else   /*not GNU_LAN*/
      /* Try IPMI LAN 1.5 first */
      rc = ipmi_open_lan(gnode,guser,gpswd,fdebugcmd); 
      fDriverTyp = DRV_LAN;
      if (rc == LAN_ERR_V2) { 
         /* Use IPMI LAN 2.0 if indicated */
         rc = ipmi_open_lan2(gnode,guser,gpswd,fdebugcmd); 
         fDriverTyp = DRV_LAN2;
      }
#endif  /* GNU_LAN*/
   } else {  /* local, not lan */
#ifdef WIN32
	rc = ipmi_open_ia(fdebugcmd);
	if (rc == ACCESS_OK) fDriverTyp = DRV_IMB; 
#else
	if ((rc = ipmi_open_ld(fdebugcmd)) == ACCESS_OK) {
		fDriverTyp = DRV_LD;
		ipmi_close_ld();
	} else if ((rc = ipmi_open_mv(fdebugcmd)) == ACCESS_OK) {
		fDriverTyp = DRV_MV;
		/* ipmi_close_mv(); * leave it open until explicit close */
	} else if ((rc = ipmi_open_ia(fdebugcmd)) == ACCESS_OK) {
		fDriverTyp = DRV_IMB;
	} else if ((rc = ipmi_open_va(fdebugcmd)) == ACCESS_OK) {
		fDriverTyp = DRV_VA;
		ipmi_close_va();
	} else if ((rc = ipmi_open_gnu(gnode,guser,gpswd,fdebugcmd)) == ACCESS_OK) {
		fDriverTyp = DRV_GNU;
	} else if ((rc = ipmi_open_direct(fdebugcmd)) == ACCESS_OK) {
		/* set to either DRV_KCS or DRV_SMB */
	} else rc = ERR_NO_DRV;
#endif

   }  /*endelse local, not lan*/
#endif
   if (fdebugcmd) printf("ipmi_open rc = %d type = %s\n",rc,
			show_driver_type(fDriverTyp));
   return (rc);
}

int ipmi_close_(void)
{
   int rc = 0;
#ifndef EFI
   switch (fDriverTyp)
   {
	case DRV_IMB: rc = ipmi_close_ia(); break;
#ifndef WIN32
	case DRV_VA:  rc = ipmi_close_va(); break;
	case DRV_MV:  rc = ipmi_close_mv(); break;
	case DRV_GNU: rc = ipmi_close_gnu(gnode); break;
	case DRV_LD:  rc = ipmi_close_ld(); break;
	case DRV_KCS: rc = ipmi_close_direct(); break;
	case DRV_SMB: rc = ipmi_close_direct(); break;
#endif
#ifndef GNU_LAN
	case DRV_LAN:  rc = ipmi_close_lan(gnode); break;
	case DRV_LAN2: rc = ipmi_close_lan2(gnode); break;
#endif
	default:  break;
   }  /*end switch*/
#endif
   return (rc);
}

#if defined (EFI)
int ipmi_open_efi(int fdebugcmd)
{
    int rc = 0;
    static bool BmcLibInitialized = false;
 
    if (BmcLibInitialized == false ) {
          rc = BmcLibInitialize();
          if (rc == 0) {
                BmcLibInitialized = true;
                fDriverTyp = DRV_LAN2 + 1;
          } 
    }
    return rc;
}

#define TIMEOUT_EFI  (1000*1000)  /*see ipmi_timeout_ia*/
int ipmi_cmdraw_efi(uchar cmd, uchar netfn, uchar sa, uchar bus, uchar lun,
                uchar *pdata, uchar sdata, uchar *presp,
                int *sresp, uchar *pcc, char fdebugcmd)
{
    BMC_MESSAGE ReqMsg;
    BMC_MESSAGE RespMsg;
    int status = 0;
    uchar * pc;
    int sz, i;
 
    ReqMsg.DevAdd   = sa;
    ReqMsg.NetFn    = netfn;
    ReqMsg.LUN      = lun;
    ReqMsg.Cmd      = cmd;
    ReqMsg.Len      = sdata;
    for( sz=0; (sz<sdata && sz< MAX_BUFFER_SIZE); sz++ )
            ReqMsg.Data[sz] = pdata[sz];
    sz = *sresp;  /* note that sresp must be pre-set */
    memset(presp, 0, sz);
    for ( i =0 ; i < BMC_MAX_RETRIES; i++)
    {
        *sresp = sz;   /* retries may need to re-init *sresp */
        if((status =ProcessTimedMessage(&ReqMsg, &RespMsg,TIMEOUT_EFI)) == 0) {
            *sresp = RespMsg.Len;
            for( sz=0 ; sz<RespMsg.Len && sz<DATA_BUF_SIZE ; sz++ )
                            presp[sz] = RespMsg.Data[sz];
            *pcc = RespMsg.CompCode;
            break;
        }
        if (fdebugcmd)   // only gets here if error
            fprintf(fpdbg,"ipmi_cmd_efi: ProcessTimedMessage error status=%x\n",
                    (uint)status);
    }
    return(status);
}
#endif

/* 
 * ipmi_cmdraw()
 *
 * This routine can be used to invoke IPMI commands that are not
 * already pre-defined in the ipmi_cmds array.
 * It invokes whichever driver-specific routine is needed (ia, mv, etc.).
 */
int ipmi_cmdraw(uchar cmd, uchar netfn, uchar sa, uchar bus, uchar lun,
		uchar *pdata, uchar sdata, uchar *presp,
                int *sresp, uchar *pcc, char fdebugcmd)
{
    int rc;
    ushort icmd;

    fperr = stderr;
    fpdbg = stdout;

    if (fDriverTyp == DRV_UNKNOWN) {   /*first time, so find which one */
        rc = ipmi_open(fdebugcmd);
	if (fdebugcmd) 
		fprintf(fpdbg,"Driver type %s, open rc = %d\n",
			show_driver_type(fDriverTyp),rc);
        if (rc == ERR_NO_DRV && !fipmi_lan) fprintf(fpdbg,
			"Cannot open an IPMI driver: /dev/imb, /dev/ipmi0, "
			"/dev/ipmi/0, \n\t/dev/ipmikcs, /dev/ipmi/kcs, "
			"libfreeipmi, or ldipmi.\n");
        else if (rc != 0) fprintf(fpdbg,"ipmi_open error = %d\n", rc);
    }  /*endif first time*/

    icmd = (cmd & 0x00ff) | (netfn << 8);
#ifdef OLD
    /* Set up the temporary ipmi_cmds array structure.  */
    ipmi_cmds[0].cmdtyp = icmd;
    ipmi_cmds[0].sa    = sa;
    ipmi_cmds[0].bus   = bus;
    ipmi_cmds[0].netfn = netfn;
    ipmi_cmds[0].lun   = lun;
    ipmi_cmds[0].len   = sdata;  /*length of request data*/
#endif
    *pcc = 0;

#ifdef EFI
    rc = ipmi_cmdraw_efi(cmd, netfn, lun, sa, bus, pdata,sdata,
                                presp,sresp, pcc, fdebugcmd);
#else
    switch (fDriverTyp)
    {
	case DRV_IMB: 
	   rc = ipmi_cmdraw_ia(cmd, netfn, lun, sa, bus, pdata,sdata, 
				presp,sresp, pcc, fdebugcmd);
	   break;
#ifndef WIN32
	case DRV_VA: 
           rc = ipmi_cmdraw_va(cmd,  netfn, lun, sa, bus, pdata,sdata, 
			 	presp,sresp, pcc, fdebugcmd);
	   break;
	case DRV_MV: 
           rc = ipmi_cmdraw_mv(cmd,  netfn, lun, sa, bus, pdata,sdata, 
			 	presp,sresp, pcc, fdebugcmd);
	   break;
	case DRV_GNU: 
	   rc = ipmi_cmdraw_gnu(gnode, cmd, netfn, lun, sa, bus, 
				pdata,sdata, presp,sresp, pcc, fdebugcmd);
	   break;
	case DRV_LD: 
	   rc = ipmi_cmdraw_ld( cmd, netfn, lun, sa, bus, 
				pdata,sdata, presp,sresp, pcc, fdebugcmd);
	   break;
	case DRV_KCS: 
           rc = ipmi_cmdraw_direct(cmd,  netfn, lun, sa, bus, pdata,sdata, 
				presp,sresp, pcc, fdebugcmd);
	   break;
	case DRV_SMB: 
           rc = ipmi_cmdraw_direct(cmd,  netfn, lun, sa, bus, pdata,sdata, 
				presp,sresp, pcc, fdebugcmd);
	   break;
#endif
#ifndef GNU_LAN
	case DRV_LAN: 
	   rc = ipmi_cmdraw_lan(gnode, cmd, netfn, lun, sa, bus, 
				pdata,sdata, presp,sresp, pcc, fdebugcmd);
	   break;
	case DRV_LAN2: 
	   rc = ipmi_cmdraw_lan2(gnode, cmd, netfn, lun, sa, bus, 
				pdata,sdata, presp,sresp, pcc, fdebugcmd);
	   break;
#endif
	default:    /* no ipmi driver */
	   rc = ERR_NO_DRV;
	   break;
    }  /*end switch*/
#endif

    if ((rc >= 0) && (*pcc != 0) && fdebugcmd) {
          fprintf(fpdbg,"ccode %x: %s\n",*pcc,decode_cc(icmd,*pcc));
    }
    /* clear the temp cmd */
    ipmi_cmds[0].cmdtyp = 0;    
    ipmi_cmds[0].sa = BMC_SA;

    return(rc);
}

/* 
 * ipmi_cmd_mc()
 * This uses the mc pointer to route commands via either the SMI or
 * IPMB method to the designated mc.
 * See also ipmi_set_mc and ipmi_restore_mc.
 */
int ipmi_cmd_mc(ushort icmd, uchar *pdata, uchar sdata, uchar *presp,
                int *sresp, uchar *pcc, char fdebugcmd)
{
   uchar cmd, netfn;
   int rv;
 
   cmd = icmd & CMDMASK;
   netfn = (icmd & 0xFF00) >> 8;
   if (mc->adrtype == ADDR_IPMB) {
      rv = ipmi_cmd_ipmb(cmd, netfn, mc->sa, mc->bus, mc->lun,
                       pdata, sdata, presp, sresp, pcc, fdebugcmd);
   } else {    /* use ADDR_SMI */
      rv = ipmi_cmdraw(cmd, netfn, mc->sa, mc->bus, mc->lun,
                      pdata, sdata, presp, sresp, pcc, fdebugcmd);
   }
   return(rv);
}

/* 
 * ipmi_cmd()
 *
 * This is the externally exposed subroutine for commands that
 * are defined in the ipmi_cmds array above.
 * It calls the ipmi_cmdraw routine for further processing.
 */
int ipmi_cmd(ushort icmd, uchar *pdata, uchar sdata, uchar *presp,
                int *sresp, uchar *pcc, char fdebugcmd)
{
    int rc, i;
    uchar bcmd;
    uchar netfn, sa, bus, lun;

    fperr = stderr;
    fpdbg = stdout;
    if (fDriverTyp == DRV_UNKNOWN) {   /*first time, so find which one */
        rc = ipmi_open(fdebugcmd);
	if (fdebugcmd) 
		fprintf(fpdbg,"Driver type %s, open rc = %d\n",
			show_driver_type(fDriverTyp),rc);
        if (rc != 0) {
           if (rc == ERR_NO_DRV && !fipmi_lan) fprintf(fpdbg,
			"Cannot open an IPMI driver: /dev/imb, /dev/ipmi0, "
			"/dev/ipmi/0, \n\t/dev/ipmikcs, /dev/ipmi/kcs, "
			"libfreeipmi, or ldipmi.\n");
           else fprintf(fpdbg,"ipmi_open error = %d\n", rc);
           return(rc);
        }
    }  /*endif first time*/

    for (i = 0; i < NCMDS; i++) {
       if (ipmi_cmds[i].cmdtyp == icmd) break;
    }
    if (i >= NCMDS) {
        fprintf(fperr, "ipmi_cmd: Unknown command %x\n",icmd);
        return(-1);
        }
    bcmd  = icmd & CMDMASK;  /* unmask it */
    netfn = ipmi_cmds[i].netfn;
    lun   = ipmi_cmds[i].lun;
    sa    = ipmi_cmds[i].sa;
    bus   = ipmi_cmds[i].bus;

    rc = ipmi_cmdraw(bcmd, netfn, sa, bus, lun, 
			pdata,sdata, presp,sresp, pcc, fdebugcmd);
    return(rc);
}

#ifdef MOVED_TO_IPMILAN_C
/* 
 * ipmi_cmd_ipmb()
 *
 * cksum() is defined in ipmilan.c 
 * Also, iseq needs to be the same as ipmi_hdr.swseq if
 * the session is ipmilan, so this routine was moved.
 * However, ipmi_cmd_ipmb() will also work ok via 
 * ipmi_cmdraw() for local commands.
 */
int ipmi_cmd_ipmb(uchar cmd, uchar netfn, uchar sa, uchar bus, uchar lun,
		uchar *pdata, uchar sdata, uchar *presp,
                int *sresp, uchar *pcc, char fdebugcmd)
{
    int rc, i, j;
    uchar idata[MAX_BUFFER_SIZE];
    int ilen, rlen;
    static uchar iseq = 0;

    ++iseq;
    i = 0;
    idata[i++] = bus;
    j = i;
    idata[i++] = sa;
    idata[i++] = (netfn << 2) | (lun & 0x03);
    idata[i++] = cksum(&idata[j],2);
    j = i;
    idata[i++] = BMC_SA;
    idata[i++] = (iseq << 2) | 0x02;  /*Seq num, SMS Message LUN*/
    idata[i++] = cmd;
    memcpy(&idata[i],pdata,sdata);
    i += sdata;
    idata[i] = cksum(&idata[j],(i - j));
    ilen = ++i;
    rc = ipmi_cmdraw(IPMB_SEND_MESSAGE,NETFN_APP, BMC_SA,PUBLIC_BUS,BMC_LUN,
                  idata, ilen, presp, &rlen, pcc, fdebugcmd);
    if (fdebugcmd) {
       if (rc == 0 && *pcc == 0) 
            dump_buf("ipmb sendmsg ok",presp,rlen,0);
       else fprintf(fpdbg,"ipmb sendmsg error %d, cc %x\n",rc,*pcc);
    }
    if (rc != 0 || *pcc != 0) return(rc);
    /* sent ok, now issue a GETMESSAGE_CMD to get the response. */
    for (i = 0; i < 5; i++)
    {
       rc = ipmi_cmdraw(IPMB_GET_MESSAGE,NETFN_APP, BMC_SA,PUBLIC_BUS,BMC_LUN,
                        idata, 0, presp, sresp, pcc, fdebugcmd);
       if (fdebugcmd) printf("ipmb get_message rc=%d cc=%x\n",rc,*pcc);
       if (rc == 0x80 || *pcc == 0x80) /*retry*/;
       else if (rc == 0x83 || *pcc == 0x83) /*retry*/;
       else break;       /* done w success or error */
       fd_wait(0,0,10);  /* wait 1 msec before retry */
    }
    if (fdebugcmd) {
       if (rc == 0 && *pcc == 0) 
            dump_buf("ipmb getmsg ok",presp,*sresp,0);
       else fprintf(fpdbg,"ipmb getmsg error %d, cc %x\n",rc,*pcc);
    }
    return(rc);
}
#endif

int ipmi_getpicmg(uchar *presp, int sresp, char fdebug)
{
   uchar idata[2];
   int rc; uchar cc;

   /* check that sresp is big enough */
   if (sresp < 4) return(-3);
   idata[0] = PICMG_ID;
   rc = ipmi_cmdraw(PICMG_GET_PROPERTIES, NETFN_PICMG,
		    BMC_SA, PUBLIC_BUS, BMC_LUN, 
		    idata, 1, presp,&sresp, &cc, fdebug);
   if (rc != ACCESS_OK) return(rc);
   if (cc != 0) return(cc);
   return(ACCESS_OK);   /* success */
}

int ipmi_getdeviceid(uchar *presp, int sresp, char fdebug)
{
   int rc; uchar cc;
   /* check that sresp is big enough (default is 15 bytes for Langley)*/
   if (sresp < 15) return(-3);
   rc = ipmi_cmd(GET_DEVICE_ID, NULL, 0, presp,&sresp, &cc, fdebug);
   if (rc != ACCESS_OK) return(rc);
   if (cc != 0) return(cc);
   if (sresp > sizeof(my_devid)) sresp = sizeof(my_devid);
   memcpy(my_devid,presp,sresp); /* save device id for later use */
   return(ACCESS_OK);   /* success */
}

void ipmi_set_mc(uchar bus, uchar sa, uchar lun, uchar type) 
{
   mc = &mc2;
   mc->bus  = bus;
   mc->sa   = sa;
   mc->lun  = lun;
   mc->adrtype = type; /* ADDR_SMI or ADDR_IPMB */
   return;
}

void ipmi_restore_mc(void)
{
   mc = &bmc;
   return;
}

/* Need our own copy of strdup(), named strdup_(), since Windows does
 * not have the same subroutine. */
char * strdup_(const char *instr)
{
    char *newstr;
    newstr = malloc(strlen(instr)+1);
    if (newstr != NULL) strcpy(newstr,instr);
    return (newstr);
}

/*
 * htoi
 * Almost all of the utilities use this subroutine 
 * Input:  a 2 character string of hex digits.
 * Output: a hex byte.
 */
uchar htoi(uchar *inhex)
{
   // char rghex[16] = "0123456789ABCDEF";
   uchar val;
   uchar c;
   if (inhex[1] == 0) { /* short string, one char */
      c = inhex[0] & 0x5f;  /* force cap */
      if (c > '9') c += 9;  /* c >= 'A' */
      val = (c & 0x0f);
   } else {
      c = inhex[0] & 0x5f;  /* force cap */
      if (c > '9') c += 9;  /* c >= 'A' */
      val = (c & 0x0f) << 4;
      c = inhex[1] & 0x5f;  /* force cap */
      if (c > '9') c += 9;  /* c >= 'A' */
      val += (c & 0x0f);
   }
   return(val);
}

void dump_buf(char *tag, uchar *pbuf, int sz, char fshowascii)
{
   uchar line[17];
   uchar a;
   int i, j;
   char *stag;

   if (tag == NULL) stag = "dump_buf"; /*safety valve*/
   else stag = tag;
   fprintf(fpdbg,"%s (len=%d): ", stag,sz);
   line[0] = 0; line[16] = 0;
   j = 0;
   if (sz < 0) { fprintf(fpdbg,"\n"); return; } /*safety valve*/
   for (i = 0; i < sz; i++) {
      if (i % 16 == 0) { 
 	 line[j] = 0; 
         j = 0; 
	 fprintf(fpdbg,"%s\n  %04x: ",line,i); 
      }
      if (fshowascii) {
         a = pbuf[i];
         if (a < 0x20 || a > 0x7f) a = '.';
         line[j++] = a;
      }
      fprintf(fpdbg,"%02x ",pbuf[i]);
   }
   if (fshowascii) {
     if ((j > 0) && (j < 16)) {
       /* space over the remaining number of hex bytes */
       for (i = 0; i < (16-j); i++) fprintf(fpdbg,"   ");
     } 
     else j = 16;
     line[j] = 0;
   }
   fprintf(fpdbg,"%s\n",line);
   return;
}

#ifdef WIN32
static HANDLE con_in  = INVALID_HANDLE_VALUE; 
static DWORD cmodein;
static DWORD cmodeold;
void tty_setraw(int mode)
{
    // system("@echo off");
    con_in  = GetStdHandle(STD_INPUT_HANDLE);
    GetConsoleMode(con_in,  &cmodein);
    cmodeold = cmodein;
    if (mode == 2) {
       cmodein &= ~(ENABLE_LINE_INPUT | ENABLE_PROCESSED_INPUT | 
    		   ENABLE_ECHO_INPUT);
    } else {  /* (mode==1) just suppress ECHO */
       cmodein &= ~ENABLE_ECHO_INPUT;
    }
    SetConsoleMode(con_in,  cmodein);
}
void tty_setnormal(int mode)
{
    // system("echo on");
    if (mode == 1) 
       cmodein  |= ENABLE_ECHO_INPUT;
    else 
       cmodein = cmodeold;
    SetConsoleMode(con_in,  cmodein);
}
#else
          /*Linux*/
static struct termios mytty;
static ulong  tty_oldflags;
void tty_setraw(int mode)
{
    int i;
    // system("stty -echo");
    i = tcgetattr(STDIN_FILENO, &mytty);
    if (i == 0) {
        tty_oldflags = mytty.c_lflag;
        if (mode == 2)   /*raw mode*/
           mytty.c_lflag &= ~(ICANON | ISIG | ECHO);
        else   /* (mode==1) just suppress ECHO */
           mytty.c_lflag &= ~ECHO;
        tcsetattr(STDIN_FILENO, TCSANOW, &mytty);
    }
}
void tty_setnormal(int mode)
{
    // system("stty echo");
    if (mode == 1) 
       mytty.c_lflag |= ECHO;
    else  /*(mode==2)*/
       mytty.c_lflag = tty_oldflags;
    tcsetattr(STDIN_FILENO, TCSANOW, &mytty);
}
#endif

static char *getline(char *prompt, char fwipe)
{
   /* getline is the same format as readline, but much simpler, and portable. */
   static char linebuf[128];
   int c, i;

   if (prompt != NULL) printf("%s\n",prompt);
   if (fwipe) tty_setraw(1); 
   for (i = 0; i < (sizeof(linebuf)-1); i++)
   {
       c = getc(stdin);
       if (c == EOF)  break;
       if (c == '\n') break;
       if ((c < 0x20) || (c > 0x7F)) break;  /*out of bounds for ASCII */
       linebuf[i] = c & 0xff;
   }
   linebuf[i] = 0;
   if (fwipe) {
       for (c = 0; c < i; c++) putc('*',stdout);
       putc('\n',stdout);
       tty_setnormal(1);
   }
   if (i == 0) return NULL;
   return(linebuf);
}

static char *getline_wipe(char *prompt)
{
   return(getline(prompt,1));
}

void parse_lan_options(char c, char *optarg, char fdebugcmd)
{
   int i;

#ifndef EFI
   switch(c) 
   {
          case 'F':      /* force driver type */
                i = set_driver_type(optarg);
                break;
          case 'T':      /* auth type */
                i = atoi(optarg);
                if (i >= 0 && i <= 5) gauth_type = i;
                break;
          case 'V':      /* priv level */
                i = atoi(optarg);
                if (i > 0 && i <= 5) gpriv_level = i;
                break;
          case 'J':
                i = atoi(optarg);
                if (i <= 14) gcipher_suite = i;
                break;
          case 'N': gnode  = optarg;   /* nodename */
                    fipmi_lan = 1;
                    break;
          case 'U':
                guser = strdup_(optarg);  /*remote username */
                if (!guser)  perror("strdup_");
                else {  /* Hide password from 'ps' */
                        memset(optarg, ' ', strlen(optarg));
                }
                break;
          case 'R':
          case 'P':
                gpswd = strdup_(optarg);  /*remote password */
                if (!gpswd) perror("strdup_");
                else {  /* Hide password from 'ps' */
                        memset(optarg, ' ', strlen(optarg));
                }
                break;
          case 'E':      /* get password from IPMI_PASSWORD environment var */
                gpswd = getenv("IPMI_PASSWORD");
                if (gpswd == NULL) perror("getenv(IPMI_PASSWORD)");
                else {
                   if (strlen(gpswd) > 16) gpswd[16] = 0;
                   if (fdebugcmd) printf("using IPMI_PASSWORD\n");
                }
                break;
          case 'Y':      /* prompt for remote password */
		gpswd = getline_wipe("Enter IPMI LAN Password: ");
                break;
          default:
                break;
   }
#endif
   return;
}  /*end parse_lan_options*/

void print_lan_opt_usage(void)
{
#ifndef EFI
    printf("       -N node  Nodename or IP address of target system\n");
    printf("       -U user  Username for remote node\n");
    printf("       -P/-R pswd  Remote Password\n");
    printf("       -E   use password from Environment IPMI_PASSWORD\n");
    printf("       -T 1 use auth Type: 1=MD2, 2=MD5(default)\n");
    printf("       -V 2 use priVilege level: 2=user, 4=admin(default)\n");
    printf("       -J 0 use lanplus cipher suite 0: 0 thru 14, 3=default\n");
    printf("       -F   force driver type (e.g. imb, lan2)\n");
    printf("       -Y   prompt for remote password\n");
#endif
}  /*end parse_lan_options*/

void close_log(void)
{
    if ((fplog != NULL) && (fplog != stderr)) {
       fclose(fplog);
       fplog = NULL;
    }
}

FILE *open_log(char *mname)
{
    FILE *fp;
    char *pname;

    /* log_name is global */
    if (log_name[0] == 0) {
      if (mname == NULL) { 
              pname = "ipmiutil";
      }  else pname = mname;
#ifdef WIN32
      sprintf(log_name,"%s.log",pname);
#else
      sprintf(log_name,"/var/log/%s.log",pname);
#endif
    }
    close_log();
    fp = fopen( log_name, "a+" );
    if (fp == NULL) {
       fp = stderr;
       fprintf(fp,"cannot open %s\n",log_name);
    } 
    fplog = fp;
    return fp;
}

void flush_log(void)
{
    if (fplog != NULL) fflush(fplog);
}

void print_log( char *pattn, ... )
{
    va_list arglist;
    if (fplog == NULL) fplog = open_log(NULL);
    /* if error, open_log sets fplog = stderr */
    va_start( arglist, pattn );
    vfprintf( fplog, pattn, arglist );
    va_end( arglist );
}

/* 
 * logmsg
 * This does an open/close if no log is alread open 
 */
void logmsg( char *pname, char *pattn, ... )
{
    va_list arglist;
    FILE *fp = NULL;
    int opened = 0;
 
    if (fplog == NULL) {
        fp = open_log(pname);
        if (fp == NULL) return;
        opened = 1;
    } else fp = fplog;
    va_start( arglist, pattn );
    vfprintf( fp, pattn, arglist );
    va_end( arglist );
    if ((opened) && (fp != stderr))
	{ fclose(fp); }
}

void dump_log(FILE *fp, char *tag, uchar *pbuf, int sz, char fshowascii)
{ 
    FILE *fpsav;
    fpsav = fpdbg;  
    if (fplog != NULL) fpdbg = fplog;
    if (fp != NULL)    fpdbg = fp;
    dump_buf(tag, pbuf, sz, fshowascii);  /*uses fpdbg*/
    fflush(fpdbg);
    fpdbg = fpsav; 
}

extern int lasterr;  /*defined in ipmilan.c */
extern void show_LastError(char *tag, int err);

void show_outcome(char *prog, int ret)
{
   if (prog == NULL) prog = "";
   printf("%s: %s\n",prog,decode_rv(ret));
   if (ret == -1 && lasterr != 0) show_LastError(prog,lasterr);
}

/* end ipmicmd.c */
