/*M*
//  PVCS:
//      $Workfile:   ipmilan.c  $
//      $Revision:   1.0  $
//      $Modtime:   2 Feb 2006 15:31:14  $
//      $Author:   arcress@users.sourceforge.net  $  
//
//  This implements support for the IPMI LAN interface natively.
// 
//  02/02/06 ARC - created.
//  05/16/06 ARC - more added.
//  06/19/06 ARCress - fixed sequence numbers for 1.7.1
//  08/08/06 ARCress - fix AUTHTYPE masks
//  08/11/06 ARCress - added lan command retries
//  10/17/06 ARCress - special case for no MsgAuth headers
//  11/28/06 ARCress - added ipmi_cmd_ipmb routine
//  05/08/07 ARCress - added 1.5 SOL data packet format to _send_lan_cmd,
//                     not working yet.
//  08/21/07 ARCress - handle Dell 1855 blades that return different authcode
//  04/17/08 ARCress - check FD_ISSET in fd_wait
 *M*/
/*----------------------------------------------------------------------*
The BSD License 

Copyright (c) 2005-2006, Intel Corporation
All rights reserved.

Redistribution and use in source and binary forms, with or without 
modification, are permitted provided that the following conditions are met:

  a.. Redistributions of source code must retain the above copyright notice, 
      this list of conditions and the following disclaimer. 
  b.. Redistributions in binary form must reproduce the above copyright notice,
      this list of conditions and the following disclaimer in the documentation 
      and/or other materials provided with the distribution. 
  c.. Neither the name of Intel Corporation nor the names of its contributors 
      may be used to endorse or promote products derived from this software 
      without specific prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR 
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *----------------------------------------------------------------------*/

#ifdef WIN32
#include <windows.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <winsock.h>
#include <io.h>
#include <time.h>

#define INET_ADDRSTRLEN 16
#define MSG_NONE     0x000 
#define MSG_WAITALL  0x100   /* Wait for a full request */ 
#define int32_t     int
#define u_int32_t   unsigned int
#define uint32      unsigned int
#define uchar       unsigned char
typedef unsigned int socklen_t;
#define RECV_MSG_FLAGS  MSG_NONE

#else   /* Linux */
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/select.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <net/if.h>
#include <netdb.h>
#include <sys/time.h>
#define RECV_MSG_FLAGS  MSG_WAITALL
#endif

#include <signal.h>
#include "ipmicmd.h"
#include "ipmilan.h"

// #define TEST_LAN   1      /*++++for DEBUG++++*/
#ifdef NON_GPL
#undef  MD2OK    /* use NON_GPL flag if non-GPL code is needed. */
#else
#define MD2OK     1
#endif

typedef struct {
  int type;
  int len;
  char *data;
  } SOL_RSP_PKT;

extern FILE *fperr;  /*defined in ipmicmd.c, usu stderr */
extern FILE *fpdbg;  /*defined in ipmicmd.c, usu stdout */
extern ipmi_cmd_t ipmi_cmds[NCMDS];


/* 
 * These variables pertain to ipmilan, for the node given at open time.
 * The assumption here is that these are utilities, so no more than one
 * node will be open at a given time.  
 * See also gnode, guser, gpswd in ipmicmd.c
 */
static int fdebuglan = 0;
static int fdopoke1  = 1;
static int fdopoke2  = 0;
static int frequireping = 0; /*=1 if ping is required, =0 ignore ping error */
static int finsession = 0;
static uchar fMsgAuth   = 1;
static char nodename[33] = "";
static int sockfd = 0;
static struct sockaddr_in _destaddr;
static struct sockaddr_in _srcaddr;
static char _dest[MAXHOSTNAMELEN+1];
static char _dest_ip[INET_ADDRSTRLEN+1];
static int  _destaddr_len = 0;
static uchar  auth_type = AUTHTYPE_INIT;  /*initial value, not set*/
static char  *authcode = NULL;
static int    authcode_len = 0;
static uint32 session_id = 0;
static uint32 in_seq = 0x01;    /* inbound sequence num */
static uint32 start_out_seq = 0x01;   /* initial outbound sequence num */
static int ipmi_timeout = 2;       /* timeout: 10 sec -> 2 sec */
static int ipmi_try = 4;           /* retries: 4 */
static int connect_state = 0;   
	/*  0 = init, 1 = socket() complete, 2 = bind/gethost complete,
	 *  3 = ping sent, 4 = pong received, 5 = session activated.  */
static char *conn_state_str[6] = {
	"init state", "socket complete", "bind complete", 
	"ping sent", "pong received", "session activated" };
int lasterr = 0;

#define SOL_DATA   0xFD       /*SOL Data command*/
#define SOL_MSG    0x10000000 /*SOL message type*/
static uchar sol_op       = 0x80;  /* encrypted/not */
static uchar sol_snd_seq  = 0;     /* valid if non-zero*/
static uchar sol_rcv_seq  = 0;
static uchar sol_offset   = 0;
static uchar sol_seed_cnt = 0x01;  /* set after activate response */
static char     sol_Encryption = 0;      /*for SOL 1.5*/
static uint32   g_Seed[ 16 ];          /*for SOL 1.5*/
static uchar    g_Cipher[ 16 ][ 16 ];  /*SeedCount x CipherHash for SOL 1.5*/


#pragma pack(1)
typedef struct {
  uchar rmcp_ver; 
  uchar rmcp_res; 
  uchar rmcp_seq; 
  uchar rmcp_type; 
  uchar auth_type;
  uint32 seq_num;   /*outgoing*/
  uint32 sess_id;
  uchar  auth_code[16];
  uchar  msg_len;  /* size here = 30 bytes = RQ_HDR_LEN */
  uchar  swid;
  uchar  swseq;
  uchar  swlun;
  uchar  priv_level;
  uint32 iseq_num;  /*incoming*/
  uchar  password[16];
  uchar  challenge[16];
  } IPMI_HDR;
static IPMI_HDR ipmi_hdr = { 0x06, 0, 0xFF, 0x07, 0x00,   0,   0, 
     {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0}, 0,
     0x81, 1, 0
  };
static IPMI_HDR *phdr;
#pragma pack()

#ifdef WIN32
WSADATA lan_ws; 

char * strlasterr(int rv)
{
   char *desc;
   /* See http://msdn2.microsoft.com/en-us/library/ms740668.aspx 
    * or doc/winsockerr.txt  */
   switch(rv) {
      case WSAEINTR: /*10004*/ desc = "Interrupted function call"; break;
      case WSAEBADF: /*10009*/ desc = "File handle is not valid"; break;
      case WSAEACCES: /*10013*/ desc = "Permission denied"; break;
      case WSAEFAULT: /*10014*/ desc = "Bad address"; break;
      case WSAEINVAL: /*10022*/ desc = "Invalid argument"; break;
      case WSAEMFILE: /*10024*/ desc = "Too many open files"; break;
      case WSAENOTSOCK: /*10038*/ desc = "Socket operation on nonsocket"; 
			break;
      case WSAEDESTADDRREQ: /*10039*/ desc = "Destination address required"; 
			break;
      case WSAEMSGSIZE: /*10040*/ desc = "Message too long"; break;
      case WSAEOPNOTSUPP: /*10045*/ desc = "Operation not supported"; break;
      case WSAEADDRINUSE: /*10048*/ desc = "Address already in use"; break;
      case WSAEADDRNOTAVAIL: /*10049*/ desc = "Cannot assign requested address";
			break;
      case WSAENETDOWN: /*10050*/ desc = "Network is down"; break;
      case WSAENETUNREACH: /*10051*/ desc = "Network is unreachable"; break;
      case WSAENETRESET: /*10052*/ desc = "Network dropped connection on reset";
			break;
      case WSAECONNABORTED: /*10053*/ desc = "Software caused connection abort";
			break;
      case WSAECONNRESET: /*10054*/ desc = "Connection reset by peer"; break;
      case WSAENOTCONN: /*10057*/ desc = "Socket is not connected"; break;
      case WSAECONNREFUSED: /*10061*/ desc = "Connection refused"; break;
      case WSAEHOSTDOWN: /*10064*/ desc = "Host is down"; break;
      case WSAEHOSTUNREACH: /*10065*/ desc = "No route to host"; break;
      default:  desc = ""; break;
    }
    return(desc);
}
#endif

#ifdef MD2OK
extern void md2_sum(uchar *string, int len, uchar *mda); /*from md2.c*/
#endif
extern void md5_sum(uchar *string, int len, uchar *mda); /*from md5.c*/

int _ipmilan_cmd(int s, struct sockaddr *to, int tolen,
     		uchar cmd, uchar netfn, uchar lun, uchar sa, 
		uchar *sdata, int slen, uchar *rdata, int *rlen, 
		int fdebugcmd);
static int _send_lan_cmd(int s, uchar *pcmd, int scmd, uchar *presp, int *sresp,
			 struct sockaddr *to, int tolen);
static int ipmilan_open_session(int sfd, struct sockaddr *destaddr, 
			int destaddr_len, uchar auth_type, char *username, 
			char *authcode, int authcode_len, 
			uchar priv_level, uint32 init_out_seqnum, 
			uint32 *session_seqnum, uint32 *session_id);
static int ipmilan_close_session(int sfd, struct sockaddr *destaddr, 
			int destaddr_len, uint32 session_id);

uchar
cksum(const uchar *buf, register int len)
{
        register uchar csum;
        register int i;
 
        /* 8-bit 2s compliment checksum */
        csum = 0;
        for (i = 0; i < len; i++)
           csum = (csum + buf[i]) % 256;
        csum = -csum;
        return(csum);
}

void show_LastError(char *tag, int err)
{
#ifdef WIN32
    fprintf(fperr,"%s LastError = %d  %s\n",tag,err,strlasterr(err));
#else
    char *s;
    s = strerror(err);
    if (s == NULL) s = "error";
    fprintf(fperr,"%s errno = %d, %s\n",tag,err,s);
#endif
}

int get_LastError( void )
{
#ifdef WIN32
   lasterr = WSAGetLastError();
#else
   lasterr = errno;
#endif
   return(lasterr);
}

char *decode_rv(int rv)
{
   char *msg;
   static char msgbuf[80];
   if (rv > 0) msg = decode_cc((ushort)0,(uchar)rv);
   else switch(rv) {
       case 0:                 msg = "completed successfully";  break;
       case -1:                msg = "error -1";    break;
       case LAN_ERR_SEND_FAIL: msg = "lan send failed";    break;
       case LAN_ERR_RECV_FAIL: msg = "lan receive failed"; break;
       case LAN_ERR_CONNECT:   msg = "cannot connect to BMC"; break;
       case LAN_ERR_ABORT:     msg = "abort signal caught"; break;
       case LAN_ERR_TIMEOUT:   msg = "timeout occurred"; break;
       case LAN_ERR_BADLENGTH: msg = "length greater than max"; break;
       case LAN_ERR_INVPARAM:   msg = "invalid parameter"; break;
       case LAN_ERR_NOTSUPPORT: msg = "request not supported"; break;
       case LAN_ERR_TOO_SHORT:  msg = "receive too short"; break;
       case LAN_ERR_HOSTNAME: msg = "error resolving hostname"; break;
       case LAN_ERR_PING:     msg = "error during ping"; break;
       case LAN_ERR_V1:       msg = "BMC only supports lan v1"; break;
       case LAN_ERR_V2:       msg = "BMC only supports lan v2"; break;
       case LAN_ERR_OTHER:    msg = "other error"; break; 
       case ERR_NO_DRV:       msg = "cannot open IPMI driver"; break; 
       default:           
           sprintf(msgbuf,"error %d",rv);
           msg = msgbuf;
           break;
   }
   return(msg);
}

static void cc_challenge(int cc)
{
   switch(cc) {
      case 0x81:  
          printf("GetSessChallenge: Invalid user name\n");
          break;
      case 0x82: 
          printf("GetSessChallenge: Null user name not enabled\n");
          break;
      default:
          printf("%s\n",decode_cc((ushort)0,(uchar)cc));
          break;
   }
}

static void cc_session(int cc)
{
   switch(cc) {
      case 0x81:  
          printf("ActivateSession: No session slots available from BMC\n");
          break;
      case 0x82: 
          printf("ActivateSession: No sessions available for this user\n");
          break;
      case 0x83: 
          printf("ActivateSession: No sessions for this user/privilege\n");
          break;
      case 0x84: 
          printf("ActivateSession: Session sequence number out of range\n");
          break;
      case 0x85: 
          printf("ActivateSession: Invalid session ID in request\n");
          break;
      case 0x86: 
          printf("ActivateSession: Privilege level exceeds user/channel limit\n");
          break;
      default:
          printf("%s\n",decode_cc((ushort)0,(uchar)cc));
          break;
   }
   return;
}

static void close_sockfd(int sfd);
static void close_sockfd(int sfd)
{
    if (sfd == 0) return;
#ifdef WIN32
    closesocket(sfd);   /*close lan socket */
    WSACleanup(); 
#else
    alarm(0);
    signal(SIGALRM,SIG_DFL);
    signal(SIGINT,SIG_DFL);
    close(sfd);   /*close lan socket */
#endif
    sockfd = 0;   /*set global to zero */
}

static int open_sockfd(char *node, int *sfd, struct sockaddr_in *daddr, 
                       int *daddr_len, int foutput);
static int open_sockfd(char *node, int *sfd, struct sockaddr_in *daddr, 
                       int *daddr_len, int foutput)
{
    int rv;
    int sockfd;
    struct hostent *hptr;
#ifdef WIN32
    DWORD rvl;

    if (sfd == NULL || daddr == NULL || daddr_len == NULL)
        return(-3);  /* invalid pointer */
    rvl = WSAStartup(0x0101,&lan_ws);
    if (rvl != 0) {
	   fprintf(fperr,"lan, WSAStartup(1.1) error %ld\n", rvl);
           return((int)rvl);
    }
#else
    if (sfd == NULL || daddr == NULL || daddr_len == NULL)
        return(-3);  /* invalid pointer */
#endif
	/* Open lan interface */
	rv = socket(AF_INET, SOCK_DGRAM, 0); 
	if (rv < 0) return (rv);
	else sockfd = rv;

	connect_state = 1;  
	memset(&_srcaddr, 0, sizeof(_srcaddr));
	_srcaddr.sin_family = AF_INET;
	_srcaddr.sin_port = htons(0);
	_srcaddr.sin_addr.s_addr = htonl(INADDR_ANY);
	rv = bind(sockfd, (struct sockaddr *)&_srcaddr, sizeof(_srcaddr));
	if (rv < 0) {
            close_sockfd(sockfd);
            return (rv);
        }

	memset(daddr, 0, sizeof(struct sockaddr_in));
	daddr->sin_family = AF_INET;
	daddr->sin_port = htons(RMCP_PRI_RMCP_PORT);  /*0x26f = 623.*/
	if ((hptr = gethostbyname(node)) == NULL) {
            if (foutput) {
#ifdef WIN32
                fprintf(fperr,"lan, gethostbyname(%s): errno=%d\n", node,errno);
#else
		fprintf(fperr,"lan, gethostbyname(%s): %s\n", node,hstrerror(errno));
#endif
            }
            close_sockfd(sockfd);
	    return(LAN_ERR_HOSTNAME);
	}
     daddr->sin_addr = *((struct in_addr *)hptr->h_addr);
     strncpy(gnodename,hptr->h_name,SZGNODE);

     *daddr_len = sizeof(struct sockaddr_in);
     *sfd = sockfd;
     return(rv);
}

static void
sig_timeout(int sig)
{
#ifndef WIN32
   alarm(0);
   signal(SIGALRM,SIG_DFL);
#endif
   fprintf(fpdbg,"ipmilan_cmd timeout, after %s\n",conn_state_str[connect_state]);
   _exit(LAN_ERR_TIMEOUT);
}

static void
sig_abort(int sig)
{
   static int sig_aborting = 0;
   uchar buf_rs[4]; 
   uchar *cmd_rs;
   int rv;

   if (sig_aborting == 0) {
     sig_aborting = 1;
     if (sockfd != 0) {  /* socket is open */
	  if (session_id != 0) {  /* session is open */
  	    cmd_rs = buf_rs;
	    rv = ipmilan_close_session(sockfd, (struct sockaddr *)&_destaddr, 
			  _destaddr_len, ipmi_hdr.sess_id);
	  }
          close_sockfd(sockfd);
     }
     signal(SIGINT,SIG_DFL);
     fprintf(fpdbg,"ipmilan_cmd interrupt, after %s\n", conn_state_str[connect_state]);
     _exit(LAN_ERR_ABORT);
   }  /*endif*/
} /*end sig_abort*/

int fd_wait(int fd, int nsec, int usec)
{
    fd_set readfds;
    struct timeval tv;
    int rv;
              
    FD_ZERO(&readfds);
    FD_SET(fd, &readfds);
    tv.tv_sec  = nsec; 
    tv.tv_usec = usec;
    rv = select(fd+1, &readfds, NULL, NULL, &tv);
    if (rv <= 0 || !FD_ISSET(fd,&readfds)) return(-1);
    else return(0);
}

void os_usleep(int s, int u)
{
#ifdef WIN32
   if (s == 0) {
      if (u >= 1000) Sleep(u/1000);
   } else {
      Sleep(s * 1000);
   }
#else
   if (s == 0) {
      usleep(u);
   } else {
      sleep(s);
   }
#endif
}

#ifndef WIN32
                  /*LINUX*/
/* signal handlers + sleep(3) is a bad idea */
static int do_sleep(unsigned int sleep_len)
{
  struct timeval tv;
  
  if (sleep_len == 0)
    return 0;

  tv.tv_sec = sleep_len;
  tv.tv_usec = 0;
  if (select(1, NULL, NULL, NULL, &tv) < 0)
    {
      if (errno != EINTR) return(errno);
    }
  return 0;
}
#endif

static void h2net(uint h, char *net, int n)
{
   int i = 0;
   net[i++] = h & 0xff;
   net[i++] = (h >> 8)  & 0xff;
   if (n == 2) return;
   net[i++] = (h >> 16)  & 0xff;
   net[i++] = (h >> 24)  & 0xff;
   return;
}

static void net2h(uint *h, uchar *net, int n)
{
   uint v;
   v  = (net[1] << 8) | net[0];
   if (n == 2) { *h = v; return; }
   v |= (net[3] << 24) | (net[2] << 16);
   *h = v;
   return;
}

/* 
 * _ipmilan_cmd 
 * local routine to send & receive each command.
 * called by global ipmicmd_lan()
 */
int _ipmilan_cmd(int sockfd, struct sockaddr *hostaddr, int hostaddr_len,
     uchar cmd, uchar netfn, uchar lun, uchar sa, 
     uchar *sdata, int slen, uchar *rdata, int *rlen, int fdebugcmd)
{
  uchar cmd_rq[RQ_LEN_MAX+3];
  uchar cmd_rs[RS_LEN_MAX+4];
  int rv = 0;
  int rs_len;
  uchar cc = 0;

  fdebuglan = fdebugcmd;
  if (sockfd == 0 || hostaddr == NULL ||
      sdata == NULL || rdata == NULL)
  	return(LAN_ERR_INVPARAM);;

  cmd_rq[0] = cmd;
  cmd_rq[1] = (netfn << 2) + (lun & 0x03);
  cmd_rq[2] = sa;
  memcpy(&cmd_rq[3],sdata,slen);
  rs_len = sizeof(cmd_rs);
  memset(cmd_rs, 0, rs_len);
  rv = _send_lan_cmd(sockfd, cmd_rq, slen+3, cmd_rs, &rs_len, 
			hostaddr, hostaddr_len);
  if (rv == 0 && rs_len == 0) cc = 0;
  else cc = cmd_rs[0];
  if (fdebugcmd) fprintf(fpdbg,"_ipmilan_cmd[%02x]: rv = %d, cc=%x rs_len=%d\n",
			 cmd_rq[0],rv,cc,rs_len);
  if (rv == 0 && cc != 0) {     /* completion code error */
  	if (fdebugcmd) {
                dump_buf("cmd_rq",cmd_rq,slen+3,0);
                dump_buf("cmd_rs",cmd_rs,rs_len,0);
	}
  }
  if (rv == 0) {
      if (rs_len < 0) rs_len = 0;
      if (*rlen <= 0) *rlen = 1;   /*failsafe*/
      if (rs_len > 0) {
          if (rs_len > *rlen) rs_len = *rlen;
          memcpy(rdata,&cmd_rs[0],rs_len);
      } else { /*(rs_len == 0)*/
          rv = LAN_ERR_TOO_SHORT; /*no completion code returned*/
      }
  } else {
      rdata[0] = cc;
  }
  *rlen = rs_len;

  return (rv);
}  /*end _ipmilan_cmd()*/


static void hash(uchar *pwd, uchar *id, uchar *chaldata, int chlen, uint32 seq, 
		 uchar *mda, uchar md)
{
   uchar pbuf[80];  /* 16 + 4 + 16 + 4 + 16 = 56 */
   int blen, n, i;

   blen = 0;      n = 16;
   memcpy(&pbuf[blen], pwd,n);   /* password   */
   blen += n;     n = 4;
   memcpy(&pbuf[blen],id,n);     /* session id */
   blen += n; 
   memcpy(&pbuf[blen],chaldata,chlen);  /* ipmi msg data, incl challenge */
   blen += chlen; n = 4;
   h2net(seq,&pbuf[blen],n);     /* in_seq num */
   blen += n;     n = 16;
   memcpy(&pbuf[blen],pwd,n);    /* password   */
   blen += n;
   if (md == IPMI_SESSION_AUTHTYPE_MD2) i = 2;
   else i = 5;
#ifdef TEST_AUTH
   if (fdebuglan) {
      fprintf(fpdbg,"hash: calling md%d_sum with seq %d\n",i,seq);
      dump_buf("pbuf",pbuf,blen,0);
      }
#endif
#ifdef MD2OK
   if (md == IPMI_SESSION_AUTHTYPE_MD2)
      md2_sum(pbuf,blen,mda);
   else   /* assume md5 */
#endif
      md5_sum(pbuf,blen,mda);
#ifdef TEST_AUTH
   if (fdebuglan) {
      fprintf(fpdbg,"Hashed MD%d AuthCode: \n",i);
      dump_buf("AuthCode",mda,16,0);
      }
#endif
} /* end hash() */

static int ipmilan_sendto(int s, const void *msg, size_t len, int flags, 
				const struct sockaddr *to, socklen_t tolen)
{
    int fusepad = 0;
    int n;
#ifdef TEST_LAN
    if (fdebuglan) {
        dump_buf("ipmilan_sendto",(uchar *)msg,len,0);
    }
#endif
    /* Check whether we need a pad byte */
    /* Note from Table 12-8, RMCP Packet for IPMI via Ethernet footnote. */
    if (len == 56 || len == 84 || len == 112 || len == 128 || len == 156) {
        /* include pad byte at end, require input buffer to have one extra */
        fusepad = 1;
        len += 1;
    }
    n = sendto(s,msg,len,flags,to,tolen);
    if (fusepad && (n > 0)) n--;
    return(n);
}

static int ipmilan_recvfrom(int s, void *buf, size_t len, int flags, 
		struct sockaddr *from, socklen_t *fromlen)
{
    int rv;
    rv = recvfrom(s,buf,len,flags,from,fromlen);
    /* Sometimes the OS sends an ECONNREFUSED error, but
     * retrying will catch the BMC's reply packet. */
#ifdef WIN32
    if (rv < 0) {
        int err;
        err = WSAGetLastError();
        if (err == WSAECONNREFUSED) /*10061*/
            rv = recvfrom(s,buf,len,flags,from,fromlen);
    }
#else
    if ((rv < 0) && (errno == ECONNREFUSED))
        rv = recvfrom(s,buf,len,flags,from,fromlen);
#endif
    return(rv);
}

static int ipmilan_poke1(int sfd, struct sockaddr *destaddr, int destlen)
{
   int rv;
   uchar asfpkt[16] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x07, 0x20, 0x18, 0xc8, 0xc2, 0x01, 0x01, 0x3c };
   if (fdebuglan) fprintf(fpdbg,"sending ipmilan poke1\n");
   rv = ipmilan_sendto(sockfd, asfpkt, 16, 0, destaddr, destlen);
   os_usleep(0,100);
   return rv;
}

static int ipmilan_poke2(int sfd, struct sockaddr *destaddr, int destlen)
{
   int rv;
   uchar asfpkt[16] = "poke2";  /*any junk*/
   if (fdebuglan) fprintf(fpdbg,"sending ipmilan poke2\n");
   rv = ipmilan_sendto(sockfd, asfpkt, 10, 0, destaddr, destlen);
   os_usleep(0,100);
   return rv;
}


static void do_hash(char *password, uchar *sessid, uchar *pdata, int sdata, 
             uint32 seq_num, uchar auth_type, uchar *auth_out)
{
       /* finish header with auth_code */
       if (auth_type != IPMI_SESSION_AUTHTYPE_NONE)  {  /*fdoauth==1*/
         if (auth_type == IPMI_SESSION_AUTHTYPE_MD5)
            hash(password, sessid, pdata, sdata, seq_num, 
		 auth_out,IPMI_SESSION_AUTHTYPE_MD5);
#ifdef MD2OK
         else if (auth_type == IPMI_SESSION_AUTHTYPE_MD2)
            hash(password, sessid, pdata, sdata, seq_num, 
		 auth_out,IPMI_SESSION_AUTHTYPE_MD2);
#endif
         else  /* IPMI_SESSION_AUTHTYPE_PASSWORD */ 
            memcpy(auth_out,password,16);
      }
}

static uint32 inc_seq_num(uint32 seq)
{
    seq++;
    if (seq == 0) seq = 1;  
    return(seq);
}

static int inc_sol_seq(int seq)
{
  seq++;
  if (seq > 0x0f) seq = 1;  /*limit to 4 bits*/
  return(seq);
}

/* 
 * _send_lan_cmd 
 * Internal routine called by local _ipmilan_cmd() and by
 * ipmilan_open_session().
 * Writes the data to the lan socket in IPMI LAN format.
 * Authentication, sequence numbers, checksums are handled here.
 *
 * Input Parameters:
 * s    = socket descriptor for this session
 * pcmd = buffer for the command and data
 *        Arbitrary pcmd format:
 *           cmd[0] = IPMI command 
 *           cmd[1] = NETFN/LUN byte
 *           cmd[2] = Slave Address (usu 0x20)
 *           cmd[3-N] = command data
 * scmd = size of command buffer (3+sdata)
 * presp = pointer to existing response buffer
 * sresp = On input, size of response buffer,
 *         On output, length of response data.
 * to    = sockaddr structure for to/destination
 * tolen = length of to sockaddr 
 */
static int _send_lan_cmd(int s, uchar *pcmd, int scmd, uchar *presp, int *sresp,
			   struct sockaddr *to, int tolen)
{
    uchar cbuf[RQ_LEN_MAX+RQ_HDR_LEN+7];
    uchar rbuf[RS_LEN_MAX+RQ_HDR_LEN+7];
    uint32 out_seq = 0;	   /* outbound */
    int clen, rlen, hlen, msglen;
    int flags; 
    int sz, n, i;
    uchar *pdata;
    int sdata;
    IPMI_HDR *phdr;
    uchar *psessid;
    uchar iauth[16];
    int fdoauth = 1;
    uint32 sess_id_tmp;
    int rv = 0; 
    int try;
    
    /* set up LAN req hdr */
    phdr = &ipmi_hdr;
    hlen = RQ_HDR_LEN;
    if (phdr->auth_type == IPMI_SESSION_AUTHTYPE_NONE) fdoauth = 0;
    else if (finsession && (fMsgAuth == 0)) fdoauth = 0;

    if (phdr->seq_num != 0) finsession = 1;
    if ( (pcmd[0] == CMD_ACTIVATE_SESSION) ||
         (pcmd[0] == CMD_SET_SESSION_PRIV) &&
         (pcmd[1] == (NETFN_APP << 2)) ) {
       finsession = 1;  /*so do seq_num*/
       fdoauth = 1;     /*use msg auth*/
    }
    if (fdoauth == 0) hlen = RQ_HDR_LEN - 16;

    /* copy command */
    if (scmd < 3) return(LAN_ERR_INVPARAM);
    sdata = scmd - 3;          /* scmd = 3 + datalen */
    msglen = 7 + sdata;
#ifdef TEST_LAN
    // if (fdebuglan) 
    {
        fprintf(fpdbg,"cmd=%02x, hlen=%d, msglen=%02x, authtype=%02x\n", 
		pcmd[0], hlen, msglen, phdr->auth_type);
    }
#endif
    clen = hlen + msglen;
    if (clen > sizeof(cbuf)) {
        fprintf(fpdbg,"message size %d > buffer size %d\n",clen,sizeof(cbuf));
        return(LAN_ERR_TOO_SHORT);
    }

    if ((pcmd[0] == SOL_DATA) && 
        (pcmd[1] == (NETFN_SOL << 2)) ) {  /*SOL 1.5 data packet*/
        hlen = SOL_HDR_LEN;   /*RMCP header + 26 */
        if (fdoauth == 0) hlen = SOL_HDR_LEN - 16;
        msglen = sdata;
        memcpy(&cbuf[0], phdr, 4);   /* copy RMCP header to buffer */
        pdata = &cbuf[4];
        pdata[0] = phdr->auth_type;
        memcpy(&pdata[1],&phdr->seq_num,4);
        sess_id_tmp = phdr->sess_id | SOL_MSG;
        memcpy(&pdata[5],&sess_id_tmp,4);
#ifdef TEST_LAN
        if (fdebuglan)
          printf("auth_type=%x/%x fdoauth=%d hlen=%d seq_num=%x\n", /*SOL*/
		 phdr->auth_type,gauth_type,fdoauth,hlen,phdr->seq_num);
#endif
        if (fdoauth) {
           psessid = (uchar *)&sess_id_tmp;
           do_hash(phdr->password, psessid, &cbuf[hlen],msglen, 
		   phdr->seq_num, phdr->auth_type, iauth);
           /* copy hashed authcode to header */
           memcpy(&pdata[9],iauth,16);
        }
        // pdata[hlen-1] = msglen;
        pdata = &cbuf[hlen];
        memcpy(pdata,&pcmd[3],msglen);
        clen = hlen + msglen;
#ifdef TEST_LAN
        if (fdebuglan)
            dump_buf("sol data, before",pdata,sdata,1);  /*SOL  TEST*/
#endif
    } else {  
        /* normal IPMI LAN commmand packet */
        pdata = &cbuf[hlen];
        pdata[0] = pcmd[2];        /*sa*/
        pdata[1] = pcmd[1];        /*netfn/lun*/
        pdata[2] = cksum(&pdata[0],2); /*cksum1*/
        pdata[3] = phdr->swid;     /*swid*/
        pdata[4] = (phdr->swseq << 2) + phdr->swlun; /*swseq/lun*/
        pdata[5] = pcmd[0];        /*cmd*/
        memcpy(&pdata[6],&pcmd[3],sdata);
        pdata[6+sdata] = cksum(&pdata[3],3+sdata); /*cksum2*/
    
        if (fdoauth) {
           psessid = (uchar *)&phdr->sess_id;
           do_hash(phdr->password, psessid, &cbuf[hlen],msglen, 
		   phdr->seq_num, phdr->auth_type, iauth);
           /* copy hashed authcode to header */
           memcpy(phdr->auth_code,iauth,16);
        }
        memcpy(&cbuf[0], phdr, hlen);   /* copy header to buffer */
    }  /*end-else normal IPMI */

    if (fdoauth == 0 && phdr->auth_type != IPMI_SESSION_AUTHTYPE_NONE) {  
        /* force the packet auth type to NONE (0) */
        IPMI_HDR *pchdr;
        pchdr = (IPMI_HDR *)&cbuf[0];
        pchdr->auth_type = IPMI_SESSION_AUTHTYPE_NONE;
    }
    cbuf[hlen-1] = msglen;       /* IPMI Message Length = 7 + data */
    if ((pcmd[0] == SOL_DATA) && 
        (pcmd[1] == (NETFN_SOL << 2)) ) {
        // phdr->sess_id = sess_id_sav; /*restore sess_id*/
#ifdef TEST_LAN
        if (fdebuglan) 
            dump_buf("sending lan sol data",cbuf,clen,1);  /*SOL  TEST*/
#endif
    }
    flags = 0;
    rlen = 0;

#ifdef TEST_LAN  
    if ((pcmd[0] == CMD_GET_CHAN_AUTH_CAP) && 
        (pcmd[1] == (NETFN_APP << 2)) ) 
        dump_buf("get_chan_auth_cap command",cbuf,clen,1);
#endif
    for (try = 0; (try < ipmi_try) && (rlen == 0); try++)
    {
      sz = ipmilan_sendto(s,cbuf,clen,flags,to,tolen);
      if (sz < 1) {
        lasterr = get_LastError();
        if (fdebuglan) show_LastError("ipmilan_sendto",lasterr);
        rv = LAN_ERR_SEND_FAIL; 
        os_usleep(0,5000);
        continue;  /* retry  */
      }

      /* receive the response */
      rv = fd_wait(s, ipmi_timeout,0);
      if (rv != 0) {
        if (fdebuglan)
           fprintf(fpdbg,"ipmilan_cmd timeout, after request, try=%d\n", try);
        rv = LAN_ERR_RECV_FAIL;
	if (fdopoke2) ipmilan_poke2(s, to, tolen);
        os_usleep(0,5000);
        continue;  /* retry  */
      }
      flags = RECV_MSG_FLAGS;
      rlen = ipmilan_recvfrom(s,rbuf,sizeof(rbuf),flags,to,&tolen);
      if (rlen < 0) {
        lasterr = get_LastError();
        if (fdebuglan) show_LastError("ipmilan_recvfrom",lasterr);
        rv = rlen;   /* -3 = LAN_ERR_RECV_FAIL */
        break; /* goto EXIT; */
      } else {  /* successful receive */
        if (fdebuglan) {
#ifdef TEST_LAN
           dump_buf("ipmilan_recvfrom", rbuf,rlen,0);
#else
           fprintf(fpdbg,"ipmilan_recvfrom rlen=%d\n", rlen);
#endif
        }
        net2h(&phdr->iseq_num,&rbuf[5],4);  /*incoming seq_num from hdr*/

        /* incoming auth_code (may differ from request auth code), Dell 1855*/
        if (rbuf[4] == IPMI_SESSION_AUTHTYPE_NONE) {  /* if AUTH_NONE */
            phdr->auth_type = IPMI_SESSION_AUTHTYPE_NONE;
            hlen = SOL_HDR_LEN - 16;
	} else {
            hlen = SOL_HDR_LEN;   /*RMCP header + 26 */
            // memcpy(phdr->iauth_code,&rbuf[13],16); /*iauthcode*/
        }

        i = hlen + 6;
        if (rlen <= i) rv = LAN_ERR_TOO_SHORT;
        else {              /* successful */
          n = rlen - i - 1;
          if (n > *sresp) n = *sresp;
          memcpy(presp,&rbuf[i],n);
          *sresp = n;
          rv = 0;
        }
      }  /*end else success*/
    } /*end for*/
// EXIT:
    /* do not increment sequence numbers for SEND_MESSAGE command */
    if ((pcmd[0] == IPMB_SEND_MESSAGE) && (pcmd[1] == (NETFN_APP << 2))) 
        finsession = 0;
    if (finsession) {
        /* increment seqnum - even if error */
        phdr->seq_num = inc_seq_num( phdr->seq_num );
        if (rlen > 0) in_seq = phdr->iseq_num;
        else in_seq++;
        phdr->swseq++;
    }
    return(rv);
}   /*end _send_lan_cmd*/


/* 
 * ipmilan_open_session
 * Performs the various command/response sequence needed to 
 * initiate an IPMI LAN session.
 */
static int ipmilan_open_session(int sfd, struct sockaddr *destaddr, 
			int destaddr_len, uchar auth_type, char *username, 
			char *authcode, int authcode_len, 
			uchar priv_level, uint32 init_out_seqnum, 
			uint32 *session_seqnum, uint32 *session_id)
{
    int rv = 0;
    uchar ibuf[RQ_LEN_MAX+3];
    uchar rbuf[RS_LEN_MAX+4];
    uint32 iseqn;
    uchar ipasswd[16];
    uchar iauthtype;
    uchar iauthcap;
    int rlen, ilen;
    IPMI_HDR *phdr;
    uchar cc;

    if (sfd == 0 || destaddr == NULL) return LAN_ERR_INVPARAM;
    phdr = &ipmi_hdr;
    /* Initialize ipmi_hdr fields */
    memset(phdr,0,sizeof(ipmi_hdr));
    phdr->rmcp_ver  = 0x06; 
    phdr->rmcp_res  = 0x00; 
    phdr->rmcp_seq  = 0xFF; 
    phdr->rmcp_type = 0x07; 
    phdr->swid  = 0x81; 
    phdr->swseq = 1; 
    phdr->priv_level = priv_level; 

    /* Get Channel Authentication */
    phdr->auth_type = IPMI_SESSION_AUTHTYPE_NONE; /*use none(0) at first*/
    ibuf[0] = 0x0e;  /*this channel*/
    ibuf[1] = phdr->priv_level; 
    rlen = sizeof(rbuf);
    if (fdebuglan) 
        fprintf(fpdbg,"GetChanAuth(sock %x, level %x) called\n",sfd,ibuf[1]);
    rv = _ipmilan_cmd(sfd,  destaddr, destaddr_len, CMD_GET_CHAN_AUTH_CAP, 
                  NETFN_APP,BMC_LUN,BMC_SA, ibuf,2, rbuf,&rlen, fdebuglan);
    if (rv != 0)
        rv = _ipmilan_cmd(sfd,  destaddr, destaddr_len, CMD_GET_CHAN_AUTH_CAP, 
                     NETFN_APP,BMC_LUN,BMC_SA, ibuf,2, rbuf,&rlen, fdebuglan);
    cc = rbuf[0];
    if (fdebuglan) 
	 fprintf(fpdbg,"GetChanAuth rv = %d, rbuf: %02x %02x %02x %02x\n",
                           rv, rbuf[0],rbuf[1],rbuf[2],rbuf[3]);
    if (rv != 0 || cc != 0) goto ERREXIT; 

    /* Check Channel Auth params */
    if ((rbuf[2] & 0x80) != 0) {
        if ( ((rbuf[4]&0x02) != 0) && ((rbuf[4]&0x01) == 0) ) {
            if (fdebuglan)
                fprintf(fpdbg,"GetChanAuth reports only v2 capability\n");
            rv = LAN_ERR_V2;  /*try v2 instead*/
            goto ERREXIT;
        }
    }
    /* check priv_level here?  */
    if ((rbuf[3] & 0x10) == 0) fMsgAuth = 1;
    else fMsgAuth = 0;
    iauthcap = rbuf[2] & 0x3f;
    if ((auth_type == AUTHTYPE_INIT || 
	      auth_type == IPMI_SESSION_AUTHTYPE_MD5)
             && (iauthcap & IPMI_MASK_AUTHTYPE_MD5))
        iauthtype = IPMI_SESSION_AUTHTYPE_MD5;
#ifdef MD2OK
    else if ((auth_type == AUTHTYPE_INIT || 
	      auth_type == IPMI_SESSION_AUTHTYPE_MD2)
             && (iauthcap & IPMI_MASK_AUTHTYPE_MD2))
        iauthtype = IPMI_SESSION_AUTHTYPE_MD2;
#endif
    else if ((auth_type == AUTHTYPE_INIT || 
	      auth_type == IPMI_SESSION_AUTHTYPE_PASSWORD)
             && (iauthcap & IPMI_MASK_AUTHTYPE_PASSWORD))
        iauthtype = IPMI_SESSION_AUTHTYPE_PASSWORD;
    else if ((auth_type == AUTHTYPE_INIT || 
	      auth_type == IPMI_SESSION_AUTHTYPE_OEM)
             && (iauthcap & IPMI_MASK_AUTHTYPE_OEM))
        iauthtype = IPMI_SESSION_AUTHTYPE_OEM;
    else 
        iauthtype = IPMI_SESSION_AUTHTYPE_NONE;
    if (fdebuglan) {
        fprintf(fpdbg,"auth_type %02x allow %02x, iauthtype %02x msgAuth=%d\n",
                           auth_type, iauthcap,iauthtype,fMsgAuth);
        if (auth_type != AUTHTYPE_INIT && auth_type != iauthtype) 
           fprintf(fpdbg,"auth_type %02x not allowed\n",auth_type);
    }

    /* get session challenge */
    phdr->auth_type = IPMI_SESSION_AUTHTYPE_NONE;
    memset(ibuf,0,17);
    ibuf[0] = iauthtype;
    if (username != NULL) 
        strncpy(&ibuf[1],username,16);
    rlen = sizeof(rbuf);
    rv = _ipmilan_cmd(sfd,  destaddr, destaddr_len, CMD_GET_SESSION_CHALLENGE,
                  NETFN_APP,BMC_LUN,BMC_SA, ibuf,17, rbuf,&rlen, fdebuglan);
    cc = rbuf[0];
    if (fdebuglan) fprintf(fpdbg,"GetSessionChallenge rv = %d, rbuf: %02x %02x\n",
			  rv, rbuf[0],rbuf[1]);
    if (rv != 0) goto ERREXIT;
    else if (cc != 0) { cc_challenge(cc); goto ERREXIT; }

    /* save challenge response data */
    memcpy(&phdr->sess_id,  &rbuf[1], 4);
    memcpy(phdr->challenge, &rbuf[5], 16);

    /* Save authtype/authcode in ipmi_hdr for later use in _send_lan_cmd. */
    phdr->auth_type = iauthtype;
    if (authcode_len > 16 || authcode_len < 0) authcode_len = 16;
    memset(&ipasswd,0,16);
    if (authcode != NULL && authcode_len > 0) 
        memcpy(&ipasswd,(uchar *)authcode,authcode_len); /* AuthCode=passwd */
    memcpy(phdr->password,&ipasswd,16);              /* save password */

    /* ActivateSession request */
    ibuf[0] = phdr->auth_type;  
    ibuf[1] = phdr->priv_level;
    memcpy(&ibuf[2],phdr->challenge,16);  /* copy challenge string to data */
    phdr->seq_num = 0;
    iseqn = init_out_seqnum; 
    h2net(iseqn,&ibuf[18],4);       /* write iseqn to buffer */
    ilen = 22;

    rlen = sizeof(rbuf);
    rv = _ipmilan_cmd(sfd,destaddr,destaddr_len,CMD_ACTIVATE_SESSION,NETFN_APP,
		  BMC_LUN, BMC_SA, ibuf, ilen, rbuf, &rlen, fdebuglan);
    cc = rbuf[0];
    if (fdebuglan) {
        if (rv > 0) fprintf(fpdbg,"ActivateSession rv = 0x%02x\n",rv); /*cc*/
        else fprintf(fpdbg,"ActivateSession rv = %d\n",rv);
    }
    if (rv != 0) goto ERREXIT;
    else if (cc != 0) { cc_session(cc); goto ERREXIT; }

    memcpy(&phdr->sess_id,&rbuf[2],4);  /* save new session id */
    net2h(&iseqn,  &rbuf[6],4);     /* save returned out_seq_num */
    if (iseqn == 0) ++iseqn;
    phdr->seq_num  = iseqn;   /* new session seqn */

    /* set session privileges */
    ibuf[0] = phdr->priv_level;
    rlen = sizeof(rbuf);
    rv = _ipmilan_cmd(sfd,  destaddr, destaddr_len, CMD_SET_SESSION_PRIV, 
                   NETFN_APP,BMC_LUN,BMC_SA, ibuf,1, rbuf,&rlen, fdebuglan);
    cc = rbuf[0];
    if (fdebuglan) fprintf(fpdbg,"SetSessionPriv rv = %d\n",rv);

    *session_id     = phdr->sess_id;
    *session_seqnum = phdr->seq_num;
ERREXIT:
    if (rv == 0 && cc != 0) rv = cc;
    return(rv);
}      /*end ipmilan_open_session*/

/* 
 * ipmilan_close_session
 */
static int ipmilan_close_session(int sfd, struct sockaddr *destaddr, 
			int destaddr_len, uint32 session_id)
{
    uchar ibuf[RQ_LEN_MAX+3];
    uchar rbuf[RS_LEN_MAX+4];
    int rlen;
    int rv = 0;

    if (session_id == 0) return(0);
    /* send close session command */
    memcpy(ibuf,&session_id,4);
    rlen = sizeof(rbuf);
    rv = _ipmilan_cmd(sfd,  destaddr, destaddr_len, CMD_CLOSE_SESSION, 
                  NETFN_APP,BMC_LUN,BMC_SA, ibuf,4, rbuf,&rlen, fdebuglan);
    if (fdebuglan) fprintf(fpdbg,"CloseSession rv = %d, cc = %02x\n",
			  rv, rbuf[0]);
    if (rbuf[0] != 0) rv = rbuf[0];
    ipmi_hdr.seq_num  = 0;
    ipmi_hdr.swseq    = 1;
    ipmi_hdr.iseq_num = 0;
    ipmi_hdr.sess_id  = 0;
    finsession = 0;
    return(rv);
}


int rmcp_ping(int sfd, struct sockaddr *saddr, int saddr_len, int foutput)
{
   /* The ASF spec says to use network byte order */
   uchar asf_pkt[40] = {06,0,0xFF,06,0x00,0x00,0x11,0xBE,0x80,0,0,0 };
   struct sockaddr from_addr;
   int from_len;
   int rv;

   /* Send RMCP ASF ping to verify IPMI LAN connection. */
	asf_pkt[9] = 1; /*tag*/	
	rv = ipmilan_sendto(sfd, asf_pkt, 12, 0, saddr, saddr_len);
	if (foutput) 
		fprintf(fpdbg,"ipmilan ping, sendto len=%d\n",rv);
	if (rv < 0) return(LAN_ERR_PING);

	from_len = sizeof(struct sockaddr);
        rv = fd_wait(sfd,ipmi_timeout,0);
        if (rv != 0) {
            fprintf(fpdbg,"pong timeout, after %s\n",
			conn_state_str[connect_state]);
	    rv = LAN_ERR_CONNECT;
        } else {
	   rv = ipmilan_recvfrom(sfd, asf_pkt, sizeof(asf_pkt), 0,
			       &from_addr,&from_len);
	   if (foutput) 
		fprintf(fpdbg,"ipmilan pong, recvfrom len=%d\n", rv);
	   if (rv < 0) return(LAN_ERR_CONNECT);
        }
   return(0);
}

int ping_bmc(char *node, int fdebugcmd)
{
    struct sockaddr_in toaddr;
    int toaddr_len;
    int sfd;
    int rv;

    rv = open_sockfd(node,&sfd, &toaddr, &toaddr_len, fdebugcmd);
    if (rv != 0) return(rv);

    rv = rmcp_ping(sfd, (struct sockaddr *)&toaddr,toaddr_len, fdebugcmd);

    close_sockfd(sfd);
    return(rv);
}

/* 
 * ipmi_open_lan
 */
int ipmi_open_lan(char *node, char *user, char *pswd, int fdebugcmd)
{
   char *username;
   uchar priv_level;
   char *temp;
   int rv = -1;

   fdebuglan = fdebugcmd;
   if (nodeislocal(node)) {
        fprintf(fpdbg,"ipmi_open_lan: node %s is local!\n",node);
        rv = LAN_ERR_INVPARAM;
        goto EXIT;
   } else {

        if ((gshutdown==0) || fdebugcmd) 
	   fprintf(fpdbg,"Opening connection to node %s ...\n",node);
	/* save nodename for sig_abort later */
	if (strlen(node) > 32) {
	       strncpy(nodename, node, 32); nodename[32] = 0;
	} else strcpy(nodename, node);

        rv = open_sockfd(node, &sockfd, &_destaddr, &_destaddr_len, 1);
        if (rv != 0) goto EXIT;

#ifdef WIN32
        /* check for ws2_32.lib(getnameinfo) resolution */
        gnodename[0] = 0;
/*
	int getnameinfo( const struct sockaddr  * sa, socklen_t    salen,
	     char  *      host, DWORD        hostlen,
	     char  *      serv, DWORD        servlen,
	     int          flags);
        rv = getnameinfo((SOCKADDR *)&_destaddr, _destaddr_len,
			gnodename,SZGNODE, NULL,0,0);
*/
#else
        rv = getnameinfo((struct sockaddr *)&_destaddr, _destaddr_len,
			gnodename,SZGNODE, NULL,0,0);
#endif
        if (rv != 0) {
            if (fdebugcmd) 
	        fprintf(fpdbg,"ipmi_open_lan: getnameinfo rv = %d\n",rv);
	    gnodename[0] = 0;
        }
	temp = inet_ntoa(_destaddr.sin_addr);
	fprintf(fpdbg,"Connected to node %s %s\n",gnodename,temp);
	strncpy(_dest_ip, temp, INET_ADDRSTRLEN);
	_dest_ip[INET_ADDRSTRLEN] = 0;

	/* Set up signals to handle errors & timeouts. */
#ifndef WIN32
	signal(SIGINT,sig_abort);
	signal(SIGALRM,sig_timeout);
#endif

	connect_state = 2;

        rv = rmcp_ping(sockfd,(struct sockaddr *)&_destaddr,_destaddr_len,
			fdebugcmd);
        if (fdopoke1 && rv != 0) { 
	    /* May sometimes need a poke to free up the BMC (cant hurt) */
	    ipmilan_poke1(sockfd,(struct sockaddr *)&_destaddr,_destaddr_len);
        }
        if (rv != 0) {
	    if (rv == LAN_ERR_CONNECT && frequireping == 0) {
               /* keep going even if ping/pong failure */
               rv = 0;
            } else {
               if (rv != LAN_ERR_PING) connect_state = 3; /*ping was sent ok*/
               close_sockfd(sockfd);
               rv = LAN_ERR_CONNECT;
               goto EXIT;
            }
        }
	connect_state = 4;

	{
            auth_type  = gauth_type;
            priv_level = gpriv_level;
	    username = user;
	    authcode = pswd;
	    authcode_len = (pswd) ? strlen(authcode) : 0;
	}
	rv = ipmilan_open_session(sockfd, (struct sockaddr *)&_destaddr, 
			_destaddr_len, auth_type, username, 
			authcode, authcode_len, priv_level, start_out_seq,
			&in_seq, &session_id);
	if (rv == 0) { /* successful (session active) */
	   connect_state = 5; /* bump connection state to active */
	} else {  /* open_session rv != 0 */
           if ((gshutdown==0) || fdebugcmd) {
              if (rv < 0) 
                   fprintf(fpdbg,"ipmilan_open_session error, rv = %d\n",rv);
              else fprintf(fpdbg,"ipmilan_open_session error, rv = 0x%x\n",rv);
           }
           close_sockfd(sockfd);
        }
   }
EXIT:
   if (rv != 0) {
      // if ((gshutdown==0) || fdebugcmd) 
          printf("ipmilan %s\n",decode_rv(rv));
          if (rv == -1 && lasterr != 0) show_LastError("ipmilan",lasterr);
   }
   return(rv);
}

int ipmi_close_lan(char *node)
{
   int rv;
   uchar buf_rs[4];  /* set_session_priv_level resp is 3 bytes */
   uchar *cmd_rs;

   /* use node for matching */
   if (!nodeislocal(node)) {  /* ipmilan, need to close & cleanup */
	if (sockfd != 0) {  /* socket is open */
          if (gshutdown) session_id = 0;
	  if (session_id != 0) {  /* session is open */
  	    cmd_rs = buf_rs;
	    rv = ipmilan_close_session(sockfd, (struct sockaddr *)&_destaddr, 
			  _destaddr_len, ipmi_hdr.sess_id);
	  }
          close_sockfd(sockfd);
	}
   } else {  /* kcs cleanup */
#ifndef WIN32
	alarm(0);
	signal(SIGALRM,SIG_DFL);
#endif
   }  /* endif */
   return (rv);
}
 
/* 
 * ipmicmd_lan
 * This is called by ipmi_cmd_lan, all commands come through here.
 */
int ipmicmd_lan(char *node, uchar cmd, uchar netfn, uchar lun, uchar sa,
		uchar *pdata, uchar sdata, uchar *presp, int *sresp, 
		uchar *pcc, char fdebugcmd)
{
   uchar rq_data[RQ_LEN_MAX+3];
   uchar cmd_rs[RS_LEN_MAX+4];
   uchar cc = 0;
   int rlen; 
   int rv = -1;

   fdebuglan = fdebugcmd;
   /* check sdata/sresp against MAX_ */
   if (sdata > RQ_LEN_MAX)  {
        if (fdebuglan) printf("sdata(%d) > RQ_LEN_MAX(%d)\n",sdata,RQ_LEN_MAX);
        return(LAN_ERR_BADLENGTH);
   }
   if (*sresp > RS_LEN_MAX) {
        if (fdebuglan) printf("sresp(%d) > RS_LEN_MAX(%d)\n",*sresp,RS_LEN_MAX);
        return(LAN_ERR_BADLENGTH);
   }
   if (pdata == NULL) { pdata = rq_data; sdata = 0; }
   rlen = *sresp;
 
   if (nodeislocal(node)) {  /*local, use kcs*/
      fprintf(fpdbg,"ipmicmd_lan: node %s is local", node);
      goto EXIT;
   } else { /* ipmilan */
        if (sockfd == 0) {  /* do re-open */
            if (fdebugcmd)
		fprintf(fpdbg,"sockfd==0, node %s needs re-open\n",node);
            rv = ipmi_open_lan(gnode, guser, gpswd, fdebugcmd);
            if (rv != 0) goto EXIT;
        }
        if (fdebugcmd) {
            fprintf(fpdbg,"lan_cmd(seq=%d) %02x %02x %02x %02x, (dlen=%d): ",
		    ipmi_hdr.seq_num, cmd,netfn,lun,sa,sdata);
            dump_buf("cmd data",pdata,sdata,0);
        }
#ifdef TEST_LAN
        printf("calling _ipmilan_cmd(%02x)\n",cmd); 
#endif
        rlen = sizeof(cmd_rs);
        rv = _ipmilan_cmd(sockfd, (struct sockaddr *)&_destaddr, _destaddr_len, 
                cmd, netfn, lun, sa, pdata, sdata,
                cmd_rs, &rlen, fdebugcmd);
   }

   cc = cmd_rs[0];
   if (rv == 0 && cc == 0) {  /* success */
	if (fdebugcmd) {
	   fprintf(fpdbg,"lan_rsp rv=0 cc=0 (rlen=%d): ",rlen);
           dump_buf("cmd rsp",cmd_rs,rlen,0);
	}
        rlen--;
	memcpy(presp,&cmd_rs[1],rlen);
	*sresp = rlen;
   } else {  /* error */
        if (fdebugcmd) 
	   fprintf(fpdbg,"ipmicmd_lan: cmd=%02x rv=%d, cc=%02x, rlen=%d\n",
		   cmd,rv,cc,rlen);
	presp[0] = 0; /*memset(presp,0,*sresp);*/
	*sresp = 0;
   }
 
EXIT:
   *pcc = cc;
   return(rv);
}  /*end ipmicmd_lan()*/

/* 
 * ipmi_cmd_lan
 * This is the entry point, called from ipmicmd.c
 */
int ipmi_cmd_lan(char *node, ushort cmd, uchar *pdata, uchar sdata,
                uchar *presp, int *sresp, uchar *pcc, char fdebugcmd)
{
    int rc, i;
    uchar mycmd;

    for (i = 0; i < NCMDS; i++) {
       if (ipmi_cmds[i].cmdtyp == cmd) break;
    }
    if (i >= NCMDS) {
        fprintf(fperr, "ipmi_cmd_lan: Unknown command %x\n",cmd);
        return(-1);
        }
   if (cmd >= CMDMASK) mycmd = (uchar)(cmd & CMDMASK);  /* unmask it */
   else mycmd = (uchar)cmd;
#ifdef TEST_LAN
   printf("ipmi_cmd_lan: cmd=%04x, mycmd=%02x\n",cmd,mycmd);
#endif
 
   rc = ipmicmd_lan(node,mycmd,ipmi_cmds[i].netfn,ipmi_cmds[i].lun,
                ipmi_cmds[i].sa, pdata,sdata,presp,sresp,pcc,fdebugcmd);
   return (rc);
}

int ipmi_cmdraw_lan(char *node, uchar cmd, uchar netfn, uchar lun, uchar sa,
		uchar bus, uchar *pdata, uchar sdata, uchar *presp, int *sresp, 
		uchar *pcc, char fdebugcmd)
{
   int rc;
#ifdef TEST_LAN
   printf("ipmi_cmdraw_lan: cmd=%02x, netfn=%02x\n",cmd,netfn);
#endif
   /* bus is not used for lan */
   rc = ipmicmd_lan(node, cmd, netfn, lun, sa, 
                    pdata,sdata,presp,sresp,pcc,fdebugcmd);
   return (rc);
}

/* 
 * ipmi_cmd_ipmb()
 * This is an entry point for IPMB indirect commands.
 *
 * The iseq value needs to be the same as ipmi_hdr.swseq if
 * the session is ipmilan, so this routine was moved here.
 * However, ipmi_cmd_ipmb will also work ok via ipmi_cmdraw 
 * for local commands.
 */
int ipmi_cmd_ipmb(uchar cmd, uchar netfn, uchar sa, uchar bus, uchar lun,
		uchar *pdata, uchar sdata, uchar *presp,
                int *sresp, uchar *pcc, char fdebugcmd)
{
    int rc, i, j;
    uchar idata[MAX_BUFFER_SIZE];
    uchar rdata[MAX_BUFFER_SIZE];
    uchar ilen;
    int rlen;
    uchar iseq;
    IPMI_HDR *phdr;
    char *pstr;
    uchar fneedclear = 0;
    uchar cc;

    phdr = &ipmi_hdr;
    iseq = phdr->swseq;
    i = 0;
    idata[i++] = bus;
    j = i;
    idata[i++] = sa;
    idata[i++] = (netfn << 2) | (lun & 0x03);
    idata[i++] = cksum(&idata[j],2);
    j = i;
    idata[i++] = BMC_SA;
    idata[i++] = (iseq << 2) | SMS_LUN;  /*Seq num, SMS Message LUN = 0x02*/
    idata[i++] = cmd;
    if (sdata > 0) {
       memcpy(&idata[i],pdata,sdata);
       i += sdata;
    }
    idata[i] = cksum(&idata[j],(i - j));
    ilen = ++i;
    rlen = sizeof(rdata);
    rc = ipmi_cmdraw(IPMB_SEND_MESSAGE,NETFN_APP, BMC_SA,PUBLIC_BUS,BMC_LUN,
                  idata, ilen, rdata, &rlen, pcc, fdebugcmd);
    if (rc == 0x83 || *pcc == 0x83) { /*retry*/
       rlen = sizeof(rdata);
       rc = ipmi_cmdraw(IPMB_SEND_MESSAGE,NETFN_APP, BMC_SA,PUBLIC_BUS,BMC_LUN,
                     idata, ilen, rdata, &rlen, pcc, fdebugcmd);
    }
    if (fdebugcmd) {
       if (rc == 0 && *pcc == 0) 
            dump_buf("ipmb sendmsg ok",rdata,rlen,0);
       else {
          if (*pcc == 0x80) {    pstr = "Invalid session handle"; }
          else if (*pcc == 0x81) pstr = "Lost Arbitration";
          else if (*pcc == 0x82) pstr = "Bus Error";
          else if (*pcc == 0x83) pstr = "NAK on Write";
          else pstr = "";
          fprintf(fpdbg,"ipmb sendmsg error %d, cc %x %s\n",rc,*pcc,pstr);
       }
    }
    if (rc != 0 || *pcc != 0) { *sresp = 0; return(rc); }
    /* sent ok, now issue a GET_MESSAGE command to get the response. */
    for (i = 0; i < 5; i++)
    {
       rlen = sizeof(rdata);
       rc = ipmi_cmdraw(IPMB_GET_MESSAGE,NETFN_APP, BMC_SA,PUBLIC_BUS,BMC_LUN,
                        idata, 0, rdata, &rlen, pcc, fdebugcmd);
       if (fdebugcmd) printf("ipmb get_message rc=%d cc=%x\n",rc,*pcc);
       if (rc == 0x80 || *pcc == 0x80) /*retry*/;
       else if (rc == 0x83 || *pcc == 0x83) /*retry*/;
       else break;       /* done w success or error */
       fd_wait(0,0,10);  /* wait 1 msec before retry */
    }
    if (rc == 0 && *pcc == 0) {
        if (fdebugcmd) 
           dump_buf("ipmb getmsg ok",rdata,rlen,0);
        /* copy the data */
        if (rlen > *sresp) rlen = *sresp;
        memcpy(presp,rdata,rlen);
    } else {
        if (*pcc == 0x80) pstr = "buffer empty";
        else { fneedclear = 1; pstr = ""; }
        if (fdebugcmd) 
           fprintf(fpdbg,"ipmb getmsg[%d] error %d, cc %x %s\n",i,rc,*pcc,pstr);
        if (fneedclear) {  /* Clear pending message flags */
              idata[0] = 0x03;  /* clear EvtMsgBuf & RecvMsgQueue */
              rlen = 16;
              rc = ipmi_cmdraw(IPMB_CLEAR_MSGF, NETFN_APP, 
			BMC_SA, PUBLIC_BUS,BMC_LUN, 
                        idata, 1, rdata, &rlen, &cc, fdebugcmd);
        }
        rlen = 0;
    }
    *sresp = rlen;
    return(rc);
}

int  lan_get_fd(void)
{
   return(sockfd);
}

/* static SOL v1.5 encryption routines */

static void sol15_cipherinit( uchar SeedCount, char *password, uint32 out_seq)
{
	uchar temp[ 40 ]; /* 16 + 4 + 8 + 4 = 32 bytes */
        int i;

        i = SeedCount & 0x0f;

        srand(time(NULL));
        g_Seed[i] = (int32_t)rand();

        if (password == NULL) 
             memset(&temp[0], 0, 16);
        else memcpy(&temp[0], password,16); /*16-byte password*/
        h2net(g_Seed[i],&temp[16],4);       /*4-byte seed */
	memset(&temp[20], 0, 8 );           /*8-byte pad*/
        h2net(out_seq,&temp[28],4);         /*4-byte seq num */
        md5_sum(temp,32, g_Cipher[i]);
}

static void sol15_encrypt( uchar *dst, uchar *src, int len, uchar SeedCount )
{
     uchar cipher;
     unsigned int val;
     int i;

	SeedCount &= 0x0f;
	if ( sol_Encryption ) {
		for (i = 0; i < len; i++, dst++, src++) {
			cipher = g_Cipher[ SeedCount ][ (i & 0x0f) ];
			val = (*src) << (cipher & 0x07);
			*dst = (uchar)((val | (val >> 8)) ^ cipher);
		}
	} else {
		memcpy( dst, src, len );
	}
}

static void sol15_decrypt( uchar *dst, uchar *src, int len, uchar SeedCount )
{
     uchar cipher;
     unsigned int val;
     int i;
	SeedCount &= 0x0f;
	if ( sol_Encryption ) {
		for (i = 0; i < len; i++, dst++, src++) {
			cipher = g_Cipher[ SeedCount ][ (i & 0x0f) ];
			val = ((*src) ^ cipher) << 8;
			val >>= (cipher & 0x07);
			*dst = (uchar)((val >> 8) | val);
		}
	} else {
		memcpy( dst, src, len );
	}
}

/* 
 * lan_get_sol_data
 * Called before ACTIVATE_SOL1 command 
 */
void lan_get_sol_data(uchar fEnc, uchar seed_cnt, uint32 *seed)
{
   if (seed_cnt != sol_seed_cnt && (seed_cnt < 16)) 
	sol_seed_cnt = seed_cnt;
   start_out_seq = ipmi_hdr.seq_num;
   sol_snd_seq   = start_out_seq;
   sol15_cipherinit(sol_seed_cnt, gpswd, start_out_seq);
   *seed = g_Seed[sol_seed_cnt];
}

/* 
 * lan_set_sol_data
 * Called after ACTIVATE_SOL1 response received 
 */
void lan_set_sol_data(uchar fenc, uchar auth, uchar seed_cnt, 
		      int insize, int outsize)
{
#ifdef TEST_LAN
   if (fdebuglan)
      printf("lan_set_sol_data: %02x %02x %02x %02x\n",   /*SOL*/
		auth,seed_cnt,insize,outsize);
#endif
   if (fenc || (auth & 0x07) == 1) {
      sol_op = 0x80;  /*OEM encryption*/
      sol_Encryption = 1;
   } else { 
      sol_op = 0x00;  /*no encryption*/
      sol_Encryption = 0;
   }
   if (seed_cnt != sol_seed_cnt && (seed_cnt < 16))  {
      /* if seed count changed, re-init the cipher. */
      sol_seed_cnt = seed_cnt;
      sol15_cipherinit(sol_seed_cnt, gpswd, start_out_seq);
   }
}

/*
 * lan_send_sol
 * buffer contains characters entered at console.
 * build an SOL data frame, which ends up
 * calling _send_lan_cmd().
 */
int  lan_send_sol( uchar *buffer, int len, SOL_RSP_PKT *rsp)
{
   int rv = -1;
   int ilen;
   uchar idata[MAX_BUFFER_SIZE]; /*=80 bytes*/
   uchar *pdata;
   int hlen, msglen;
   int sz;
   IPMI_HDR *phdr;
   int fdoauth = 1;
   uchar iauth[16];
   uint32 sess_id_tmp;
   uchar *psessid;
   int flags; 
              
   phdr = &ipmi_hdr;
   hlen = SOL_HDR_LEN;   /*RMCP header + 26 */
   if (phdr->auth_type == IPMI_SESSION_AUTHTYPE_NONE) fdoauth = 0;
   else if (fMsgAuth == 0) fdoauth = 0; /*always finsession*/
   if (fdoauth == 0) hlen = SOL_HDR_LEN - 16;

   pdata = &idata[hlen];
   if (len == 0) {
      pdata[0] = 0x00;  /*ack for keepalive*/
   } else {
      sol_snd_seq = inc_sol_seq(sol_snd_seq);
      pdata[0] = sol_snd_seq;
      sol15_encrypt(&pdata[5],buffer,len, sol_seed_cnt);
   }
   pdata[1] = sol_rcv_seq; /* seq to ack*/
   pdata[2] = sol_offset;  /* num bytes nack'd, always 0 */
   pdata[3] = sol_seed_cnt;
   pdata[4] = sol_op;      /* &= 0x01; * also flush rsp data */
   msglen = len + 5;
   {
        pdata = &idata[0];
        memcpy(&pdata[0], phdr, 4);   /* copy RMCP header to buffer */
        if (fdoauth == 0 && phdr->auth_type != IPMI_SESSION_AUTHTYPE_NONE) {
            /* force the packet auth type to NONE (0) */
            pdata[4] = IPMI_SESSION_AUTHTYPE_NONE;
        } else
            pdata[4] = phdr->auth_type;
        memcpy(&pdata[5],&phdr->seq_num,4);
        sess_id_tmp = phdr->sess_id | SOL_MSG;
        memcpy(&pdata[9],&sess_id_tmp,4);
#ifdef TEST_LAN
	if (fdebuglan)
           printf("auth_type=%x/%x fdoauth=%d hlen=%d seq_num=%x\n", /*SOL*/
                 phdr->auth_type,gauth_type,fdoauth,hlen,phdr->seq_num);
#endif
        if (fdoauth) {
           psessid = (uchar *)&sess_id_tmp;
           do_hash(phdr->password, psessid, &idata[hlen],msglen,
                   phdr->seq_num, phdr->auth_type, iauth);
           /* copy hashed authcode to header */
           memcpy(&pdata[13],iauth,16);
        }
   }
   idata[hlen-1] = msglen;
   ilen = hlen + msglen;
   // rlen = sizeof(rdata);;

// #ifdef TEST_LAN
   if (fdebuglan) 
      dump_buf("lan_send_sol",idata,ilen,1);
// #endif
   flags = 0;
   sz = ipmilan_sendto(sockfd,idata,ilen,flags,
			(struct sockaddr *)&_destaddr,_destaddr_len); 
   if (fdebuglan) printf("lan_send_sol ret = %d\n",sz);
   if (sz < 1) {
      lasterr = get_LastError();
      if (fdebuglan) show_LastError("lan_send_sol",lasterr);
      rv = LAN_ERR_SEND_FAIL;
      os_usleep(0,5000);
   }
   else {
      phdr->seq_num = inc_seq_num( phdr->seq_num );
      rv = 0;
   }

   if (rsp != NULL) rsp->len  = 0;
   return(rv);
}

int  lan_keepalive( uchar bType )
{
   int rv = 0;
   /* don't try if no data sent yet */
   if (sol_snd_seq == 0) return(rv);
   /* use lan_send_sol, but with 0 length data */
   rv = lan_send_sol(NULL,0,NULL);
   return(rv);
}

int  lan_recv_sol( SOL_RSP_PKT *rsp )
{
   static uchar rsdata[MAX_BUFFER_SIZE];  /*LANplus allows 1024*/
   uchar rdata[MAX_BUFFER_SIZE];
   int  rlen, hlen;
   IPMI_HDR *phdr;
   int fdoauth = 1;
   uchar *pdata;
   int try;
   int flags; 
   int rv = -1;

   phdr = &ipmi_hdr;
   rsp->data = rsdata;
   hlen = SOL_HDR_LEN;   /*RMCP header + 26 */
   if (phdr->auth_type == IPMI_SESSION_AUTHTYPE_NONE) fdoauth = 0;
   else if (fMsgAuth == 0) fdoauth = 0; /*always finsession*/
   if (fdoauth == 0) hlen = SOL_HDR_LEN - 16;
   rlen = 0;
#ifdef TEST_LAN
#endif
   if (fdebuglan)
      printf("lan_recv_sol, fdebug=%d, fpdbg=%p\n",fdebuglan,fpdbg);
   // for (try = 0; (try < ipmi_try) && (rlen == 0); try++)
   for (try = 0; (try < 1) && (rlen == 0); try++)
   {
      /* receive the response */
      rv = fd_wait(sockfd, ipmi_timeout,0);
      if (rv != 0) {
         if (fdebuglan) fprintf(fpdbg,"lan_recv_sol timeout\n");
         rv = LAN_ERR_RECV_FAIL;
         os_usleep(0,5000);
         continue; /* ok to retry  */
      }
      flags = RECV_MSG_FLAGS;
      rlen = ipmilan_recvfrom(sockfd,rdata,sizeof(rdata),flags,
                        (struct sockaddr *)&_destaddr,&_destaddr_len);
      if (rlen < 0) {
         lasterr = get_LastError();
         if (fdebuglan) show_LastError("ipmilan_recvfrom",lasterr);
         rv = rlen;   /* -3 = LAN_ERR_RECV_FAIL */
         break; /* goto EXIT; */
      } else {  /* successful receive */
         rv = 0;
         if (fdebuglan) {
            dump_buf("lan_recv_sol rdata",rdata,rlen,1);  /*SOL*/
         }
         if (rdata[4] == IPMI_SESSION_AUTHTYPE_NONE) {  /* if AUTH_NONE */
            /* may have tried with auth, but denied, Dell 1855 */
            phdr->auth_type = IPMI_SESSION_AUTHTYPE_NONE;
            hlen = SOL_HDR_LEN - 16;
         }
         net2h(&phdr->iseq_num,&rdata[5],4);  /*incoming seq_num from hdr*/ 
         if (rlen >= hlen) {
            pdata = &rdata[hlen];
            rlen -= hlen;
            if (rlen >= 5) sol_rcv_seq = pdata[1]; 
            rsp->type = PAYLOAD_TYPE_SOL;
            rsp->len  = rlen;
            sol15_decrypt(rsp->data,pdata,rlen, sol_seed_cnt);
            if (fdebuglan) 
               dump_buf("lan_recv_sol rsp",rsp->data,rlen,1);  /*SOL*/
         } else {
            if (fdebuglan) 
               printf("lan_recv_sol rlen %d < %d\n",rlen,hlen); /*fpdbg*/
            rsp->type = PAYLOAD_TYPE_SOL;
            rsp->len  = 0;
         }
         break;
      }
   } /*end for*/
   return(rv);
}

/* end ipmilan.c */
