/*M*
//  PVCS:
//      $Workfile:   ipmilan2.c  $
//      $Revision:   1.0  $
//      $Modtime:   21 Nov 2006 08:42:14  $
//      $Author:   arcress@users.sourceforge.net  $  
//
//  This implements support for the IPMI LAN 2.0 (RMCP+) interface natively.
// 
//  11/21/06 ARC - created from ipmilan.c.
 *M*/
/*----------------------------------------------------------------------*
The BSD License 

Copyright (c) 2005-2006, Intel Corporation
All rights reserved.

Redistribution and use in source and binary forms, with or without 
modification, are permitted provided that the following conditions are met:

  a.. Redistributions of source code must retain the above copyright notice, 
      this list of conditions and the following disclaimer. 
  b.. Redistributions in binary form must reproduce the above copyright notice,
      this list of conditions and the following disclaimer in the documentation 
      and/or other materials provided with the distribution. 
  c.. Neither the name of Intel Corporation nor the names of its contributors 
      may be used to endorse or promote products derived from this software 
      without specific prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR 
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *----------------------------------------------------------------------*/

#ifdef WIN32
#include <windows.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <winsock.h>
#include <io.h>

#define INET_ADDRSTRLEN 16
#define MSG_NONE     0x000 
#define MSG_WAITALL  0x100   /* Wait for a full request */ 
#define u_int32_t   unsigned int
#define uchar       unsigned char
typedef unsigned int socklen_t;
#define RECV_MSG_FLAGS  MSG_NONE

#else   /* Linux */
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/select.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <net/if.h>
#include <netdb.h>
#include <sys/time.h>
#define RECV_MSG_FLAGS  MSG_WAITALL
#endif

#include <signal.h>
#include "ipmicmd.h"
#include "ipmilan2.h"

#define V2_HDR_LEN    16  /*RMCP(4) + IPMI_Sess(10) + Msg_Len(2)*/
#define V2_PAY_MAX   243  /*send payload max*/
#define V2_RPAY_MAX  243  /*recv payload max*/
#define V2_TRAIL_MAX  66  /*Integ_Pad(32) + Pad(1) + Next(1) + Auth_Code(32)*/
#ifdef NON_GPL
#undef  MD2OK    /* use NON_GPL flag if non-GPL code is needed. */
#else
#define MD2OK     1
#endif

extern FILE *fperr;  /*defined in ipmicmd.c, usu stderr */
extern FILE *fpdbg;  /*defined in ipmicmd.c, usu stdout */
extern ipmi_cmd_t ipmi_cmds[NCMDS];

/* 
 * These variables pertain to ipmilan, for the node given at open time.
 * The assumption here is that these are utilities, so no more than one
 * node will be open at a given time.  
 * See also gnode, guser, gpswd in ipmicmd.c
 */
static int fdebuglan = 0;
static int fdopoke1  = 1;
static int fdopoke2  = 0;
static int frequireping = 0; /*=1 if ping is required, =0 ignore ping error */
static int       finsession = 0;
static uchar     fMsgAuth   = 1;
static u_int32_t session_id = 0;
static u_int32_t in_seq = 0x01;    /* inbound */
static int    sockfd = 0;
static char   nodename[33] = "";
static struct sockaddr_in _destaddr;
static struct sockaddr_in _srcaddr;
static char   _dest[MAXHOSTNAMELEN+1];
static char   _dest_ip[INET_ADDRSTRLEN+1];
static int    _destaddr_len = 0;
static uchar  auth_type = 0;
static char  *authcode = NULL;
static int    authcode_len = 0;
static int ipmi_timeout = 2;       /* timeout: 10 sec -> 2 sec */
static int ipmi_try = 4;           /* retries: 4 */
static int    connect_state = 0;   
	/*  0 = init, 1 = socket() complete, 2 = bind/gethost complete,
	 *  3 = ping sent, 4 = pong received, 5 = session activated.  */
static char  *conn_state_str[6] = {
	"init state", "socket complete", "bind complete", 
	"ping sent", "pong received", "session activated" };

#pragma pack(1)
typedef struct {
  /* RSP header */
  uchar  rmcp_sessid[4];
  uchar  rmcp_sessseq[4];
  /* RMCP header */
  uchar  rmcp_ver;
  uchar  rmcp_res;
  uchar  rmcp_seq;
  uchar  rmcp_type;
  /*IPMI header starts here, LO-HI ordering*/
  uchar  v2_auth;       /*auth type/format*/
  uchar  v2_tpayload;   /*payload type*/
  uint32 v2_sessid;     /*IPMI 2.0 session id*/
  uint32 v2_seq;        /*session sequence number*/
  ushort msg_len;       /*msg/payload length, 2 bytes*/
  /* confidentiality hdr (optional) */
  /* IPMI payload (variable length) */
  /* confidentiality trailer (optional) */
  /* integrity pad (optional) */
  /* pad length,  1 byte */
  /* next header, 1 byte */
  /* auth code (optional) */

  /* IPMI Payload fields */
  uchar  ipmi_tag;      /*message tag*/
  uchar  ipmi_lvl;      /*max priv level requested*/
  ushort ipmi_res;      /*reserved*/
  uint32 sess_id;       /*console session id*/
  uchar  auth_pay[8];
  uchar  integ_pay[8];
  uchar  conf_pay[8];
  uchar  trailer[4]; 

  /* Other session variables are kept here */
  uchar  auth_type; /*see v2_auth above*/
  uchar  swid;
  uchar  swseq;
  uchar  swlun;
  uint32 seq_num;   /*outgoing*/
  uint32 iseq_num;  /*incoming*/
  uint32 timeout;   
  uint32 retry;
  ushort port;
  uchar  password[16];
  uchar  challenge[16];  /*used for v15, v2?*/
  uchar  auth_code[16]; /*Msg Auth Code Code*/
  uint32 v1_sessid;     /*IPMI 1.5 session id*/
  uchar  cipher_suite_id;

  /* IPMI v2 session data */
  uchar  req_auth_alg;
  uchar  req_integ_alg;
  uchar  req_crypt_alg;
  uchar  auth_alg;
  uchar  integ_alg;
  uchar  crypt_alg;
  uchar  priv_level;
  uint32 console_id; 
  uint32 bmc_id; 
  uchar  rand_console[16];
  uchar  rand_bmc[16];
  uchar  bmc_guid[16];
  uchar  integ_key[20];  /* session integrity key */
  uchar  bmc_key[21];    /* BMC key (KG) */
  uchar  k1[20];
  uchar  k2[20];    /* first 16 bytes for AES */

  /* Serial Over LAN session data */
  ushort sol_max_inbound_payload_size;
  ushort sol_max_outbound_payload_size;
  ushort sol_port;
  char   sol_escape_char;
  uchar  sol_sequence_number;
  uchar  sol_last_received_sequence_number;
  uchar  sol_last_received_byte_count;
} IPMI2_HDR;
#pragma pack()
static IPMI2_HDR ipmi2_hdr; 
static IPMI2_HDR *phdr;

#ifdef WIN32
  WSADATA ws; 
#endif

#ifdef MD2OK
extern void md2_sum(uchar *string, int len, uchar *mda); /*from md2.c*/
#endif
extern void md5_sum(uchar *string, int len, uchar *mda); /*from md5.c*/
extern uchar cksum(const uchar *buf, register int len);  /*from ipmilan.c*/
extern int fd_wait(int fd, int nsec, int usec);          /*from ipmilan.c*/
extern int _ipmilan_cmd(int s, struct sockaddr *to, int tolen, /*ipmilan.c*/
     		uchar cmd, uchar netfn, uchar lun, uchar sa, 
		uchar *sdata, int slen, uchar *rdata, int *rlen, int fdebugcmd);

int _ipmilan2_cmd(int s, struct sockaddr *to, int tolen,
     		uchar cmd, uchar netfn, uchar lun, uchar sa, 
		uchar *sdata, int slen, uchar *rdata, int *rlen, int fdebugcmd);
static int _send_lan_cmd(int s, uchar *pcmd, int scmd, uchar *presp, int *sresp,
			 struct sockaddr *to, int tolen);
static int _recv_lan_cmd(int s, uchar *pcmd, int scmd, uchar *presp, int *sresp,
			 struct sockaddr *to, int tolen);
static int ipmilan_open_session(int sfd, struct sockaddr *destaddr, 
			int destaddr_len, uchar auth_type, char *username, 
			char *authcode, int authcode_len, 
			uchar priv_level, uint32 init_out_seqnum, 
			uint32 *session_seqnum, uint32 *session_id);
static int ipmilan_close_session(int sfd, struct sockaddr *destaddr, 
			int destaddr_len, uint32 session_id);

static void show_LastError(char *tag)
{
#ifdef WIN32
    int rv = 0;
    rv = WSAGetLastError();
    fprintf(fperr,"%s LastError = %d\n",tag,rv);
#else
    fprintf(fperr,"%s errno =  %d\n",tag,errno);
#endif
}

static void cc_challenge(int cc)
{
   switch(cc) {
      case 0x81:  
          printf("GetSessChallenge: Invalid user name\n");
          break;
      case 0x82: 
          printf("GetSessChallenge: Null user name not enabled\n");
          break;
      default:
          printf("%s\n",decode_cc((ushort)0,(uchar)cc));
          break;
   }
}

static void cc_session(int cc)
{
   switch(cc) {
      case 0x81:  
          printf("ActivateSession: No session slots available from BMC\n");
          break;
      case 0x82: 
          printf("ActivateSession: No sessions available for this user\n");
          break;
      case 0x83: 
          printf("ActivateSession: No sessions for this user/privilege\n");
          break;
      case 0x84: 
          printf("ActivateSession: Session sequence number out of range\n");
          break;
      case 0x85: 
          printf("ActivateSession: Invalid session ID in request\n");
          break;
      case 0x86: 
          printf("ActivateSession: Privilege level exceeds user/channel limit\n");
          break;
      default:
          printf("%s\n",decode_cc((ushort)0,(uchar)cc));
          break;
   }
   return;
}

static void close_sockfd(int sfd);
static void close_sockfd(int sfd)
{
    if (sfd == 0) return;
#ifdef WIN32
    closesocket(sfd);   /*close lan socket */
    WSACleanup(); 
#else
    alarm(0);
    signal(SIGALRM,SIG_DFL);
    signal(SIGINT,SIG_DFL);
    close(sfd);   /*close lan socket */
#endif
    sockfd = 0;   /*set global to zero */
}

static int open_sockfd(char *node, int *sfd, struct sockaddr_in *daddr, 
                       int *daddr_len, int foutput);
static int open_sockfd(char *node, int *sfd, struct sockaddr_in *daddr, 
                       int *daddr_len, int foutput)
{
    int rv;
    int sockfd;
    struct hostent *hptr;
#ifdef WIN32
    DWORD rvl;

    if (sfd == NULL || daddr == NULL || daddr_len == NULL)
        return(-3);  /* invalid pointer */
    rvl = WSAStartup(0x0101,&ws);
    if (rvl != 0) {
	   fprintf(fperr,"lan, WSAStartup(1.1) error %ld\n", rvl);
           return((int)rvl);
    }
#else
    if (sfd == NULL || daddr == NULL || daddr_len == NULL)
        return(-3);  /* invalid pointer */
#endif
	/* Open lan interface */
	rv = socket(AF_INET, SOCK_DGRAM, 0); 
	if (rv < 0) return (rv);
	else sockfd = rv;

	connect_state = 1;  
	memset(&_srcaddr, 0, sizeof(_srcaddr));
	_srcaddr.sin_family = AF_INET;
	_srcaddr.sin_port = htons(0);
	_srcaddr.sin_addr.s_addr = htonl(INADDR_ANY);
	rv = bind(sockfd, (struct sockaddr *)&_srcaddr, sizeof(_srcaddr));
	if (rv < 0) {
            close_sockfd(sockfd);
            return (rv);
        }

	memset(daddr, 0, sizeof(struct sockaddr_in));
	daddr->sin_family = AF_INET;
	daddr->sin_port = htons(RMCP_PRI_RMCP_PORT);  /*0x26f = 623.*/
	if ((hptr = gethostbyname(node)) == NULL) {
            if (foutput) {
#ifdef WIN32
                fprintf(fperr,"lan, gethostbyname(%s): errno=%d\n", node,errno);
#else
		fprintf(fperr,"lan, gethostbyname(%s): %s\n", node,hstrerror(errno));
#endif
            }
            close_sockfd(sockfd);
	    return(LAN_ERR_HOSTNAME);
	}
     daddr->sin_addr = *((struct in_addr *)hptr->h_addr);

     *daddr_len = sizeof(struct sockaddr_in);
     *sfd = sockfd;
     return(rv);
}

static void
sig_timeout(int sig)
{
#ifndef WIN32
   alarm(0);
   signal(SIGALRM,SIG_DFL);
#endif
   fprintf(fpdbg,"ipmilan_cmd timeout, after %s\n",conn_state_str[connect_state]);
   _exit(LAN_ERR_TIMEOUT);
}

static void
sig_abort(int sig)
{
   static int sig_aborting = 0;
   uchar buf_rs[4]; 
   uchar *cmd_rs;
   int rv;

   if (sig_aborting == 0) {
     sig_aborting = 1;
     if (sockfd != 0) {  /* socket is open */
	  if (session_id != 0) {  /* session is open */
  	    cmd_rs = buf_rs;
	    rv = ipmilan_close_session(sockfd, (struct sockaddr *)&_destaddr, 
			  _destaddr_len, ipmi2_hdr.sess_id);
	  }
          close_sockfd(sockfd);
     }
     signal(SIGINT,SIG_DFL);
     fprintf(fpdbg,"ipmilan_cmd interrupt, after %s\n", conn_state_str[connect_state]);
     _exit(LAN_ERR_ABORT);
   }  /*endif*/
} /*end sig_abort*/


/* extern void os_usleep(int s, int u);  * defined in ipmilan.c */

#ifndef WIN32
/* select only works on sockets if Windows  */
/* signal handlers + sleep(1) is a bad idea */
static int do_sleep(unsigned int sleep_len)
{
  struct timeval tv;
  
  if (sleep_len == 0)
    return 0;

  tv.tv_sec = sleep_len;
  tv.tv_usec = 0;
  if (select(1, NULL, NULL, NULL, &tv) < 0)
    {
      if (errno != EINTR) return(errno);
    }
  return 0;
}
#endif

static void h2net(uint h, char *net, int n)
{
   int i = 0;
   net[i++] = h & 0xff;
   net[i++] = (h >> 8)  & 0xff;
   if (n == 2) return;
   net[i++] = (h >> 16)  & 0xff;
   net[i++] = (h >> 24)  & 0xff;
   return;
}

static void net2h(uint *h, uchar *net, int n)
{
   uint v;
   v  = (net[1] << 8) | net[0];
   if (n == 2) { *h = v; return; }
   v |= (net[3] << 24) | (net[2] << 16);
   *h = v;
   return;
}

/* 
 * _ipmilan2_cmd 
 * local routine to send & receive each command.
 * called by global ipmicmd_lan2()
 */
int _ipmilan2_cmd(int sockfd, struct sockaddr *hostaddr, int hostaddr_len,
     uchar cmd, uchar netfn, uchar lun, uchar sa, 
     uchar *sdata, int slen, uchar *rdata, int *rlen, int fdebugcmd)
{
  uchar cmd_rq[V2_PAY_MAX+3];
  uchar cmd_rs[V2_RPAY_MAX+4];
  int rv = 0;
  int rs_len;
  uchar cc;

  fdebuglan = fdebugcmd;
  if (sockfd == 0 || hostaddr == NULL ||
      sdata == NULL || rdata == NULL)
  	return(LAN_ERR_INVPARAM);;

  cmd_rq[0] = cmd;
  cmd_rq[1] = (netfn << 2) + (lun & 0x03);
  cmd_rq[2] = sa;
  memcpy(&cmd_rq[3],sdata,slen);
  rs_len = sizeof(cmd_rs);
  memset(cmd_rs, 0, rs_len);
  rv = _send_lan_cmd(sockfd, cmd_rq, slen+3, cmd_rs, &rs_len, 
			hostaddr, hostaddr_len);
  if (rv == 0) 
      rv = _recv_lan_cmd(sockfd, cmd_rq, slen+3, cmd_rs, &rs_len, 
			hostaddr, hostaddr_len);
  if (rv == 0 && rs_len == 0) cc = 0;
  else cc = cmd_rs[0];
  if (fdebugcmd) fprintf(fpdbg,"_ipmilan2_cmd[%02x]: rv = %d, cc=%x rs_len=%d\n",
			 cmd_rq[0],rv,cc,rs_len);
  if (rv == 0 && cc != 0) {     /* completion code error */
  	if (fdebugcmd) {
		fprintf(fpdbg,"cmd_rq: %02x %02x %02x %02x \n",
			cmd_rq[0],cmd_rq[1], cmd_rq[2],cmd_rq[3]);
		fprintf(fpdbg,"cmd_rs: %02x %02x %02x %02x \n",
			cmd_rs[0],cmd_rs[1],cmd_rs[2],cmd_rs[3]);
	}
	rv = cc;
  }
  if (rv == 0) {
      if (rs_len < 0) rs_len = 0;
      if (rs_len > 0) {
          if (rs_len > *rlen) rs_len = *rlen;
          memcpy(rdata,&cmd_rs[0],rs_len);
      } else { /*(rs_len == 0)*/
          rv = LAN_ERR_TOO_SHORT; /*no completion code returned*/
      }
      *rlen = rs_len;
  } else {        /*errors*/
      rdata[0] = cc;
      *rlen = rs_len;  /* usu 1 or 0 */
  }

  return (rv);
}  /*end _ipmilan2_cmd()*/


static void hash(uchar *pwd, uchar *id, uchar *chaldata, int chlen, uint32 seq, 
		 uchar *mda, uchar md)
{
   uchar pbuf[80];  /* 16 + 4 + 16 + 4 + 16 = 56 */
   int blen, n, i;

   blen = 0;      n = 16;
   memcpy(&pbuf[blen], pwd,n);   /* password   */
   blen += n;     n = 4;
   memcpy(&pbuf[blen],id,n);     /* session id */
   blen += n; 
   memcpy(&pbuf[blen],chaldata,chlen);  /* ipmi msg data, incl challenge */
   blen += chlen; n = 4;
   h2net(seq,&pbuf[blen],n);     /* in_seq num */
   blen += n;     n = 16;
   memcpy(&pbuf[blen],pwd,n);    /* password   */
   blen += n;
   if (md == IPMI_SESSION_AUTHTYPE_MD2) i = 2;
   else i = 5;
   if (fdebuglan) {
      fprintf(fpdbg,"hash: calling md%d_sum with seq %d\n",i,seq);
      // dump_buf("pbuf",pbuf,blen,0);
      }
#ifdef MD2OK
   if (md == IPMI_SESSION_AUTHTYPE_MD2) 
      md2_sum(pbuf,blen,mda);
   else   /* assume md5 */
#endif
      md5_sum(pbuf,blen,mda);
#ifdef TEST_AUTH
   if (fdebuglan) {
      fprintf(fpdbg,"Hashed MD%d AuthCode: \n",i);
      dump_buf("AuthCode",mda,16,0);
      }
#endif
} /* end hash() */

static int ipmilan_sendto(int s, const void *msg, size_t len, int flags, 
				const struct sockaddr *to, socklen_t tolen)
{
    int fusepad = 0;
    int n;
#ifdef TEST
    if (fdebuglan) {
        dump_buf("ipmilan_sendto",(uchar *)msg,len,0);
    }
#endif
    /* Check whether we need a pad byte */
    /* Note from Table 12-8, RMCP Packet for IPMI via Ethernet footnote. */
    if (len == 56 || len == 84 || len == 112 || len == 128 || len == 156) {
        /* include pad byte at end, require input buffer to have one extra */
        fusepad = 1;
        len += 1;
    }
    n = sendto(s,msg,len,flags,to,tolen);
    if (fusepad && (n > 0)) n--;
    return(n);
}

static int ipmilan_recvfrom(int s, void *buf, size_t len, int flags, 
		struct sockaddr *from, socklen_t *fromlen)
{
    int rv;
    rv = recvfrom(s,buf,len,flags,from,fromlen);
#ifndef WIN32
    if ((rv < 0) && (errno == ECONNREFUSED))
        rv = recvfrom(s,buf,len,flags,from,fromlen);
#endif
    return(rv);
}

static int ipmilan_poke1(int sfd, struct sockaddr *destaddr, int destlen)
{
   int rv;
   uchar asfpkt[16] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x07, 0x20, 0x18, 0xc8, 0xc2, 0x01, 0x01, 0x3c };
   if (fdebuglan) fprintf(fpdbg,"sending ipmilan poke1\n");
   rv = ipmilan_sendto(sockfd, asfpkt, 16, 0, destaddr, destlen);
   os_usleep(0,100);
   return rv;
}

static int ipmilan_poke2(int sfd, struct sockaddr *destaddr, int destlen)
{
   int rv;
   uchar asfpkt[16] = "poke2";  /*any junk*/
   if (fdebuglan) fprintf(fpdbg,"sending ipmilan poke2\n");
   rv = ipmilan_sendto(sockfd, asfpkt, 10, 0, destaddr, destlen);
   os_usleep(0,100);
   return rv;
}

static int v2_build_payload(uchar *buf)
{
   return(0);
}

/* 
 * _send_lan_cmd 
 * called by local _ipmilan2_cmd() and ipmilan_open_session()
 * Write the data to the lan socket in IPMI LAN format.
 * Authentication, sequence numbers, checksums are handled here.
 */
static int _send_lan_cmd(int s, uchar *pcmd, int scmd, uchar *presp, int *sresp,
			   struct sockaddr *to, int tolen)
{
    uchar cbuf[V2_HDR_LEN+V2_PAY_MAX+V2_TRAIL_MAX];
    int clen, rlen, hlen, tlen, msglen;
    int flags; 
    int sz, n; // i;
    uchar *pdata;
    int sdata;
    IPMI2_HDR *phdr;
    uchar *psessid;
    uchar iauth[16];
    u_int32_t out_seq = 0;
    int fdoauth = 1;
    int rv = 0; 
    int try;
    
    /* set up LAN req hdr */
    phdr = &ipmi2_hdr;
    hlen = V2_HDR_LEN;
    tlen = V2_TRAIL_MAX;
    if (phdr->auth_type == IPMI_SESSION_AUTHTYPE_NONE) fdoauth = 0;
    else if (finsession && (fMsgAuth == 0)) fdoauth = 0;

    if (phdr->seq_num != 0) finsession = 1;
    if ( (pcmd[0] == CMD_ACTIVATE_SESSION) ||
         (pcmd[0] == CMD_SET_SESSION_PRIV) &&
         (pcmd[1] == (NETFN_APP << 2)) ) {
       finsession = 1;  /*so do seq_num*/
       fdoauth = 1;     /*use msg auth*/
    }

    /* copy command */
    if (scmd < 3) return(LAN_ERR_INVPARAM);
    if (fdoauth == 0) hlen = V2_HDR_LEN - 16;
    sdata = scmd - 3;          /* scmd also has 3 cmd bytes */
    msglen = sdata + 7;
    memset(cbuf,0,sizeof(cbuf));
#ifdef TEST
    if (fdebuglan) {
        fprintf(fpdbg,"cmd=%02x, hlen=%d, msglen=%02x, authtype=%02x\n", 
		pcmd[0], hlen, msglen, phdr->auth_type);
    }
#endif
    clen = hlen + msglen;
    if (clen > sizeof(cbuf)) {
        fprintf(fpdbg,"message size %d > buffer size %d\n",clen,sizeof(cbuf));
        return(LAN_ERR_TOO_SHORT);
    }
    pdata = &cbuf[hlen];
    pdata[0] = pcmd[2];        /*sa*/
    pdata[1] = pcmd[1];        /*netfn/lun*/
    pdata[2] = cksum(&pdata[0],2); /*cksum1*/
    pdata[3] = phdr->swid;  /*swid*/
    pdata[4] = (phdr->swseq << 2) + phdr->swlun; /*swseq/lun*/
    pdata[5] = pcmd[0];        /*cmd*/
    memcpy(&pdata[6],&pcmd[3],sdata);
    pdata[6+sdata] = cksum(&pdata[3],3+sdata); /*cksum2*/

    /* fill in header with auth_code */
    if ((phdr->auth_type != IPMI_SESSION_AUTHTYPE_NONE) && (fdoauth)) {
      n = msglen;
      psessid = (uchar *)&phdr->sess_id;
      if (phdr->auth_type == IPMI_SESSION_AUTHTYPE_MD5)
         hash(phdr->password, psessid, &cbuf[hlen],n, phdr->seq_num,
		iauth,IPMI_SESSION_AUTHTYPE_MD5);
#ifdef MD2OK
      else if (phdr->auth_type == IPMI_SESSION_AUTHTYPE_MD2)
         hash(phdr->password, psessid, &cbuf[hlen],n, phdr->seq_num,
		iauth,IPMI_SESSION_AUTHTYPE_MD2);
#endif
      else  /* IPMI_SESSION_AUTHTYPE_PASSWORD */ 
         memcpy(iauth,phdr->password,16);
      memcpy(phdr->auth_code,iauth,16);  /* copy hashed authcode to header */
    }
    /* do payload building */

    v2_build_payload(&cbuf[hlen]);  /* build payload in buffer */
                                    /* and make any length mods */
    memcpy(&cbuf[0], phdr, hlen);   /* copy header to buffer */
    /* build trailer in buffer */

    if (fdoauth == 0 && phdr->auth_type != IPMI_SESSION_AUTHTYPE_NONE) {  
        /* force the packet auth type to NONE (0) */
        IPMI2_HDR *pchdr;
        pchdr = (IPMI2_HDR *)&cbuf[0];
        pchdr->auth_type = IPMI_SESSION_AUTHTYPE_NONE;
    }
    cbuf[hlen-1] = msglen;       /* IPMI Message Length = 7 + data */
    flags = 0;
    rlen = 0;

    for (try = 0; (try < ipmi_try) && (rlen == 0); try++)
    {
      sz = ipmilan_sendto(s,cbuf,clen,flags,to,tolen);
      if (sz < 1) {
        if (fdebuglan) show_LastError("ipmilan_sendto");
        rv = LAN_ERR_SEND_FAIL; 
        os_usleep(0,100);
        continue;  /* retry sendto */
      }

    } /*end for*/
    return(rv);
}   /*end _send_lan_cmd*/

static int _recv_lan_cmd(int s, uchar *pcmd, int scmd, uchar *presp, int *sresp,
			   struct sockaddr *to, int tolen)
{
    IPMI2_HDR *phdr;
    uchar rbuf[V2_RPAY_MAX+V2_HDR_LEN+V2_TRAIL_MAX];
    int rlen, try;
    int flags; 
    int n, i;
    u_int32_t out_seq = 0;
    int rv;

    phdr = &ipmi2_hdr;
    rlen = 0;
    memset(rbuf,0,sizeof(rbuf));

    for (try = 0; (try < ipmi_try) && (rlen == 0); try++)
    {
      /* receive the response */
      rv = fd_wait(s, ipmi_timeout,0);
      if (rv != 0) {
        if (fdebuglan)
           fprintf(fpdbg,"ipmilan_cmd timeout, after request, try=%d\n", try);
        rv = LAN_ERR_RECV_FAIL;
	if (fdopoke2) ipmilan_poke2(s, to, tolen);
        os_usleep(0,5000);
        continue;  /* retry recv */
      }
      flags = RECV_MSG_FLAGS;
      rlen = ipmilan_recvfrom(s,rbuf,sizeof(rbuf),flags,to,&tolen);
      if (rlen < 0) {
        if (fdebuglan) show_LastError("ipmilan_recvfrom");
        rv = rlen;   /* -3 = LAN_ERR_RECV_FAIL */
        break; /* goto EXIT; */
      } else {  /* successful receive */
        if (fdebuglan) {
#ifdef TEST
           dump_buf("ipmilan_recvfrom", rbuf,rlen,0);
#else
           fprintf(fpdbg,"ipmilan_recvfrom rlen=%d\n", rlen);
#endif
        }
        net2h(&phdr->iseq_num,&rbuf[5],4);  /*incoming seq_num from hdr*/
        /* incoming auth_code */
        // if (rbuf[4] != 0)  /* if not AUTH_NONE */
        //    memcpy(phdr->iauth_code,&rbuf[13],16); /*iauthcode*/

        i = V2_HDR_LEN + 2;    /*min send length*/
        if (rlen <= i) rv = LAN_ERR_TOO_SHORT;
        else {              /* successful */
          n = rlen - i - 1;
          if (n > *sresp) n = *sresp;
          memcpy(presp,&rbuf[i],n);
          *sresp = n;
          rv = 0;
        }
      }  /*end else success*/
    }  /*end for*/

    /* global finsession was set in _send_lan_cmd already */
    /* do not increment sequence numbers for SEND_MESSAGE command */
    if ((pcmd[0] == IPMB_SEND_MESSAGE) && (pcmd[1] == (NETFN_APP << 2)))
        finsession = 0;
    if (finsession) {
        /* increment seqnum - even if error */
	out_seq = phdr->seq_num;
        out_seq++;
        if (out_seq == 0) out_seq = 1;  
	phdr->seq_num = out_seq;
        if (rlen > 0) in_seq = phdr->iseq_num;
        else in_seq++;
        phdr->swseq++;
    }
    return(rv);
}  /*end _recv_lan_cmd*/


/* 
 * ipmilan_open_session
 */
static int ipmilan_open_session(int sfd, struct sockaddr *destaddr, 
			int destaddr_len, uchar auth_type, char *username, 
			char *authcode, int authcode_len, 
			uchar priv_level, uint32 init_out_seqnum, 
			uint32 *session_seqnum, uint32 *session_id)
{
    int rv = 0;
    uchar ibuf[V2_PAY_MAX+3];
    uchar rbuf[V2_RPAY_MAX+4];
    uint32 iseqn;
    uchar ipasswd[16];
    uchar iauthtype;
    uchar iauthcap;
    int rlen, ilen;
    IPMI2_HDR *phdr;
    uchar cc;

    if (sfd == 0 || destaddr == NULL) return LAN_ERR_INVPARAM;
    phdr = &ipmi2_hdr;
    /* Initialize ipmi2_hdr fields */
    memset(phdr,0,sizeof(ipmi2_hdr));
    phdr->rmcp_ver  = 0x06;
    phdr->rmcp_res  = 0x00;
    phdr->rmcp_seq  = 0xFF;
    phdr->rmcp_type = 0x07;
    phdr->swid  = 0x81;
    phdr->swseq = 1;
    phdr->priv_level = priv_level;
    if (phdr->port == 0) phdr->port       = RMCP_PRI_RMCP_PORT; /*0x26F*/
    if (phdr->timeout == 0) phdr->timeout = ipmi_timeout;  /*2*/
    if (phdr->retry == 0) phdr->retry     = ipmi_try;  /*4*/
    if (phdr->priv_level == 0) phdr->priv_level = IPMI_PRIV_LEVEL_ADMIN; /*4*/

    /* Initialize lan2 session data */
    // phdr->session_state    = LANPLUS_STATE_PRESESSION;
    phdr->auth_alg         = IPMI_AUTH_RAKP_NONE;
    phdr->crypt_alg        = IPMI_CRYPT_NONE;
    phdr->console_id       = 0x00;
    phdr->bmc_id           = 0x00;
    phdr->sol_sequence_number = 1;

    /* Get Channel Authentication */
    phdr->auth_type = 0;
    ibuf[0] = 0x0e;  /*this channel*/
    ibuf[1] = phdr->priv_level; 
    rlen = sizeof(rbuf);
    if (fdebuglan) 
        fprintf(fpdbg,"GetChanAuth(sock %x, level %x) called\n",sfd,ibuf[1]);
    rv = _ipmilan_cmd(sfd,  destaddr, destaddr_len, CMD_GET_CHAN_AUTH_CAP, 
                  NETFN_APP, BMC_LUN, BMC_SA, ibuf, 2, rbuf, &rlen, fdebuglan);
    if (fdebuglan) 
	 fprintf(fpdbg,"GetChanAuth rv = %d rbuf: %02x %02x %02x %02x %02x\n",
                           rv, rbuf[0],rbuf[1],rbuf[2],rbuf[3],rbuf[4]);
    if (rv != 0 || rbuf[0] != 0) {  /* try a second time */
        rv = _ipmilan_cmd(sfd,  destaddr, destaddr_len, CMD_GET_CHAN_AUTH_CAP, 
                  NETFN_APP, BMC_LUN, BMC_SA, ibuf, 2, rbuf, &rlen, fdebuglan);
        if (fdebuglan) 
	   fprintf(fpdbg,"GetChanAuth rv = %d rbuf: %02x %02x %02x %02x %02x\n",
                           rv, rbuf[0],rbuf[1],rbuf[2],rbuf[3],rbuf[4]);
    }
    cc = rbuf[0];
    if (rv != 0 || cc != 0) goto ERREXIT; 

    /* Check Channel Auth params */
    /* Does it support IPMI 2.0 RMCP+ ? */
    if ((rbuf[2] & 0x80) == 0) {
        if (fdebuglan) 
	    fprintf(fpdbg,"GetChanAuth reports only v1 capability\n");
        rv = LAN_ERR_V1;  /*try v1 instead*/
        goto ERREXIT;
    }
    if (fdebuglan) 
        fprintf(fpdbg,"GetChanAuth channel %x v2=%x v1.5=%x\n",
		rbuf[1], (rbuf[4]&0x02),(rbuf[4]&0x01));
    if ((rbuf[3] & 0x10) == 0) fMsgAuth = 1;
    else fMsgAuth = 0;
    iauthcap = rbuf[2] & 0x3f;
    if ((auth_type == 0 || auth_type == IPMI_SESSION_AUTHTYPE_MD5)
             && (iauthcap & IPMI_MASK_AUTHTYPE_MD5))
        iauthtype = IPMI_SESSION_AUTHTYPE_MD5;
#ifdef MD2OK
    else if ((auth_type == 0 || auth_type == IPMI_SESSION_AUTHTYPE_MD2)
             && (iauthcap & IPMI_MASK_AUTHTYPE_MD2))
        iauthtype = IPMI_SESSION_AUTHTYPE_MD2;
#endif
    else if ((auth_type == 0 || auth_type == IPMI_SESSION_AUTHTYPE_PASSWORD)
             && (iauthcap & IPMI_MASK_AUTHTYPE_PASSWORD))
        iauthtype = IPMI_SESSION_AUTHTYPE_PASSWORD;
    else if ((auth_type == 0 || auth_type == IPMI_SESSION_AUTHTYPE_OEM)
             && (iauthcap & IPMI_MASK_AUTHTYPE_OEM))
        iauthtype = IPMI_SESSION_AUTHTYPE_OEM;
    else 
        iauthtype = IPMI_SESSION_AUTHTYPE_NONE;
    if (fdebuglan) {
        fprintf(fpdbg,"auth_type %02x allow %02x, iauthtype %02x msgAuth=%d\n",
                           auth_type, iauthcap,iauthtype,fMsgAuth);
        if (auth_type != 0 && auth_type != iauthtype) 
           fprintf(fpdbg,"auth_type %02x not allowed\n",auth_type);
    }

    /* TODO: fill in v2data now */

    /* get session challenge */
    phdr->auth_type = 0;
    memset(ibuf,0,17);
    ibuf[0] = iauthtype;
    if (username != NULL) 
        strncpy(&ibuf[1],username,16);
    rlen = sizeof(rbuf);
    rv = _ipmilan2_cmd(sfd,  destaddr, destaddr_len, CMD_GET_SESSION_CHALLENGE,
                  NETFN_APP, BMC_LUN, BMC_SA, ibuf, 17, rbuf, &rlen, fdebuglan);
    if (fdebuglan) fprintf(fpdbg,"GetSessionChallenge rv = %d, rbuf: %02x %02x\n",
			  rv, rbuf[0],rbuf[1]);
    if (rv > 0) cc_challenge(rv);
    if (rv != 0) goto ERREXIT;

    /* save challenge response data */
    memcpy(&phdr->sess_id,  &rbuf[1], 4);
    memcpy(phdr->challenge, &rbuf[5], 16);

    /* ActivateSession request */
    phdr->auth_type = iauthtype;
    if (authcode_len > 16 || authcode_len < 0) authcode_len = 16;
    memset(&ipasswd,0,16);
    if (authcode != NULL && authcode_len > 0) 
        memcpy(&ipasswd,(uchar *)authcode,authcode_len); /* AuthCode=passwd */
    memcpy(phdr->password,&ipasswd,16);              /* save password */

    ibuf[0] = phdr->auth_type;  
    ibuf[1] = phdr->priv_level;
    memcpy(&ibuf[2],phdr->challenge,16);  /* copy challenge string to data */
    phdr->seq_num = 0;
    iseqn = init_out_seqnum; 
    h2net(iseqn,&ibuf[18],4);       /* write iseqn to buffer */
    ilen = 22;

    rlen = sizeof(rbuf);
    rv = _ipmilan2_cmd(sfd,destaddr,destaddr_len,CMD_ACTIVATE_SESSION,NETFN_APP,
		  BMC_LUN, BMC_SA, ibuf, ilen, rbuf, &rlen, fdebuglan);
    if (fdebuglan) {
        if (rv > 0) fprintf(fpdbg,"ActivateSession rv = 0x%02x\n",rv); /*cc*/
        else fprintf(fpdbg,"ActivateSession rv = %d\n",rv);
    }
    if (rv != 0) {
       if (rv > 0) cc_session(rv);
       goto ERREXIT;
    }

    memcpy(&phdr->sess_id,&rbuf[2],4);  /* save new session id */
    net2h(&iseqn,  &rbuf[6],4);     /* save returned out_seq_num */
    if (iseqn == 0) ++iseqn;
    phdr->seq_num  = iseqn;   /* new session seqn */

    /* set session privileges */
    ibuf[0] = phdr->priv_level;
    rlen = sizeof(rbuf);
    rv = _ipmilan2_cmd(sfd,  destaddr, destaddr_len, CMD_SET_SESSION_PRIV, 
                   NETFN_APP, BMC_LUN, BMC_SA, ibuf, 1, rbuf, &rlen, fdebuglan);
    if (fdebuglan) fprintf(fpdbg,"SetSessionPriv rv = %d\n",rv);

    *session_id     = phdr->sess_id;
    *session_seqnum = phdr->seq_num;
ERREXIT:
    return(rv);
}      /*end ipmilan_open_session*/

/* 
 * ipmilan_close_session
 */
static int ipmilan_close_session(int sfd, struct sockaddr *destaddr, 
			int destaddr_len, uint32 session_id)
{
    uchar ibuf[V2_PAY_MAX+3];
    uchar rbuf[V2_RPAY_MAX+4];
    int rlen;
    int rv = 0;

    if (session_id == 0) return(0);
    /* send close session command */
    memcpy(ibuf,&session_id,4);
    rlen = sizeof(rbuf);
    rv = _ipmilan2_cmd(sfd,  destaddr, destaddr_len, CMD_CLOSE_SESSION, 
                  NETFN_APP, BMC_LUN, BMC_SA, ibuf, 4, rbuf, &rlen, fdebuglan);
    if (fdebuglan) fprintf(fpdbg,"CloseSession rv = %d, cc = %02x\n",
			  rv, rbuf[0]);
    if (rbuf[0] != 0) rv = rbuf[0];
    ipmi2_hdr.seq_num  = 0;
    ipmi2_hdr.swseq    = 1;
    ipmi2_hdr.iseq_num = 0;
    ipmi2_hdr.sess_id  = 0;
    finsession = 0;
    return(rv);
}


int rmcp_ping2(int sfd, struct sockaddr *saddr, int saddr_len, int foutput)
{
   uchar asf_pkt[40] = {06,0,0xFF,06,0x00,0x00,0x11,0xBE,0x80,0,0,0 };
   struct sockaddr from_addr;
   int from_len;
   int rv;

   /* Send RMCP ASF ping to verify IPMI LAN connection. */
	asf_pkt[9] = 1; /*tag*/	
	rv = ipmilan_sendto(sfd, asf_pkt, 12, 0, saddr, saddr_len);
	if (foutput) 
		fprintf(fpdbg,"ipmilan ping, sendto len=%d\n",rv);
	if (rv < 0) return(LAN_ERR_PING);

	from_len = sizeof(struct sockaddr);
        rv = fd_wait(sfd,ipmi_timeout,0);
        if (rv != 0) {
            fprintf(fpdbg,"pong timeout, after %s\n",
			conn_state_str[connect_state]);
	    rv = LAN_ERR_CONNECT;
        } else {
	   rv = ipmilan_recvfrom(sfd, asf_pkt, sizeof(asf_pkt), 0,
			       &from_addr,&from_len);
	   if (foutput) 
		fprintf(fpdbg,"ipmilan pong, recvfrom len=%d\n", rv);
	   if (rv < 0) return(LAN_ERR_CONNECT);
        }
   return(0);
}

/* 
 * ipmi_open_lan2
 */
int ipmi_open_lan2(char *node, char *user, char *pswd, int fdebugcmd)
{
   char *username;
   uchar priv_level;
   char *temp;
   int rv = -1;

   fdebuglan = fdebugcmd;
   if (nodeislocal(node)) {
        fprintf(fpdbg,"ipmi_open_lan: node %s is local!\n",node);
        rv = LAN_ERR_INVPARAM;
        goto EXIT;
   } else {

        if ((gshutdown==0) || fdebugcmd) 
	   fprintf(fpdbg,"Opening connection to node %s ...\n",node);
	/* save nodename for sig_abort later */
	if (strlen(node) > 32) {
	       strncpy(nodename, node, 32); nodename[32] = 0;
	} else strcpy(nodename, node);

        rv = open_sockfd(node, &sockfd, &_destaddr, &_destaddr_len, 1);
        if (rv != 0) goto EXIT;

	temp = inet_ntoa(_destaddr.sin_addr);
	strncpy(_dest_ip, temp, INET_ADDRSTRLEN);
	_dest_ip[INET_ADDRSTRLEN] = 0;

	/* Set up signals to handle errors & timeouts. */
#ifndef WIN32
	signal(SIGINT,sig_abort);
	signal(SIGALRM,sig_timeout);
#endif

	connect_state = 2;

        rv = rmcp_ping2(sockfd,(struct sockaddr *)&_destaddr,_destaddr_len,
			fdebugcmd);
        if (fdopoke1 && rv != 0) { 
	    /* May sometimes need a poke to free up the BMC (cant hurt) */
	    ipmilan_poke1(sockfd,(struct sockaddr *)&_destaddr,_destaddr_len);
        }
        if (rv != 0) {
	    if (rv == LAN_ERR_CONNECT && frequireping == 0) {
               /* keep going even if ping/pong failure */
               rv = 0;
            } else {
               if (rv != LAN_ERR_PING) connect_state = 3; /*ping was sent ok*/
               close_sockfd(sockfd);
               rv = LAN_ERR_CONNECT;
               goto EXIT;
            }
        }
	connect_state = 4;

	{
            auth_type  = gauth_type;
            priv_level = gpriv_level;
	    username = user;
	    authcode = pswd;
	    authcode_len = (pswd) ? strlen(authcode) : 0;
	}
	rv = ipmilan_open_session(sockfd, (struct sockaddr *)&_destaddr, 
			_destaddr_len, auth_type, username, 
			authcode, authcode_len, priv_level, 0x01,
			&in_seq, &session_id);
	if (rv == 0) { /* successful (session active) */
	   connect_state = 5; /* bump connection state to active */
	} else {  /* open_session rv != 0 */
           if ((gshutdown==0) || fdebugcmd) {
              if (rv < 0) 
                   fprintf(fpdbg,"ipmilan_open_session error, rv = %d\n",rv);
              else fprintf(fpdbg,"ipmilan_open_session error, rv = 0x%x\n",rv);
           }
           close_sockfd(sockfd);
        }
   }
EXIT:
   if (rv != 0) {
      if ((gshutdown==0) || fdebugcmd) printf("ipmilan %s\n",decode_rv(rv));
   }
   return(rv);
}

int ipmi_close_lan2(char *node)
{
   int rv;
   uchar buf_rs[4];  /* set_session_priv_level resp is 3 bytes */
   uchar *cmd_rs;

   /* use node for matching */
   if (!nodeislocal(node)) {  /* ipmilan, need to close & cleanup */
	if (sockfd != 0) {  /* socket is open */
          if (gshutdown) session_id = 0;
	  if (session_id != 0) {  /* session is open */
  	    cmd_rs = buf_rs;
	    rv = ipmilan_close_session(sockfd, (struct sockaddr *)&_destaddr, 
			  _destaddr_len, ipmi2_hdr.sess_id);
	  }
          close_sockfd(sockfd);
	}
   } else {  /* kcs cleanup */
#ifndef WIN32
	alarm(0);
	signal(SIGALRM,SIG_DFL);
#endif
   }  /* endif */
   return (rv);
}
 
/* 
 * ipmicmd_lan2
 * This is called by ipmi_cmd_lan2, all commands come through here.
 */
int ipmicmd_lan2(char *node, uchar cmd, uchar netfn, uchar lun, uchar sa,
		uchar *pdata, uchar sdata, uchar *presp, int *sresp, 
		uchar *pcc, char fdebugcmd)
{
   uchar rq_data[V2_PAY_MAX+3];
   uchar cmd_rs[V2_RPAY_MAX+4];
   uchar cc;
   int rlen; 
   int rv = -1;

   fdebuglan = fdebugcmd;
   /* check sdata/sresp against MAX_ */
   if (sdata > V2_PAY_MAX)  {
        if (fdebuglan) printf("sdata(%d) > V2_PAY_MAX(%d)\n",sdata,V2_PAY_MAX);
        return(LAN_ERR_BADLENGTH);
   }
   if (*sresp > V2_RPAY_MAX) {
        if (fdebuglan) printf("sresp(%d) > V2_RPAY_MAX(%d)\n",*sresp,V2_RPAY_MAX);
        return(LAN_ERR_BADLENGTH);
   }
   if (pdata == NULL) { pdata = rq_data; sdata = 0; }
   rlen = *sresp;
 
   if (nodeislocal(node)) {  /*local, use kcs*/
      fprintf(fpdbg,"ipmicmd_lan2: node %s is local", node);
      goto EXIT;
   } else { /* ipmilan */
        if (sockfd == 0) {  /* do re-open */
            if (fdebugcmd)
		fprintf(fpdbg,"sockfd==0, node %s needs re-open\n",node);
            rv = ipmi_open_lan2(gnode, guser, gpswd, fdebugcmd);
            if (rv != 0) goto EXIT;
        }
        if (fdebugcmd) {
            fprintf(fpdbg,"lan2_cmd(seq=%d) %02x %02x %02x %02x, (dlen=%d): ",
		   ipmi2_hdr.seq_num, cmd,netfn,lun,sa,sdata);
            dump_buf("cmd data",pdata,sdata,0);
        }
        rlen = sizeof(cmd_rs);
        rv = _ipmilan2_cmd(sockfd, (struct sockaddr *)&_destaddr, _destaddr_len, 
                cmd, netfn, lun, sa, pdata, sdata,
                cmd_rs, &rlen, fdebugcmd);
   }

   cc = cmd_rs[0];
   if (rv == 0 && cc == 0) {  /* success */
	if (fdebugcmd) {
	   fprintf(fpdbg,"lan2_rsp rv=0 cc=0 (rlen=%d): ",rlen);
           dump_buf("cmd rsp",cmd_rs,rlen,0);
	}
        rlen--;
	memcpy(presp,&cmd_rs[1],rlen);
	*sresp = rlen;
   } else {  /* error */
        if (fdebugcmd) 
	   fprintf(fpdbg,"ipmicmd_lan2: cmd=%02x rv=%d, cc=%02x, rlen=%d\n",
		   cmd,rv,cc,rlen);
	presp[0] = 0; /*memset(presp,0,*sresp);*/
	*sresp = 0;
   }
   *pcc = cc;
 
EXIT:
   return(rv);
}  /*end ipmicmd_lan2()*/

int ipmi_cmdraw_lan2(char *node, uchar cmd, uchar netfn, uchar lun, uchar sa,
		uchar bus, uchar *pdata, uchar sdata, 
		uchar *presp, int *sresp, uchar *pcc, char fdebugcmd)
{
   int rc;
   rc = ipmicmd_lan2(node, cmd, netfn, lun, sa, 
		pdata,sdata,presp,sresp,pcc,fdebugcmd);
   return (rc);
}

/* 
 * ipmi_cmd_lan2
 * This is the entry point, called from ipmicmd.c
 */
int ipmi_cmd_lan2(char *node, ushort cmd, uchar *pdata, uchar sdata,
                uchar *presp, int *sresp, uchar *pcc, char fdebugcmd)
{
   int rc, i;
   uchar mycmd;
   for (i = 0; i < NCMDS; i++) {
       if (ipmi_cmds[i].cmdtyp == cmd) break;
   }
   if (i >= NCMDS) {
        fprintf(fperr, "ipmi_cmd_lan2: Unknown command %x\n",cmd);
        return(-1);
        }
   if (cmd >= CMDMASK) mycmd = (uchar)(cmd & CMDMASK);  /* unmask it */
   else mycmd = (uchar)cmd;
 
   rc = ipmicmd_lan2(node,mycmd,ipmi_cmds[i].netfn,ipmi_cmds[i].lun,
                ipmi_cmds[i].sa, pdata,sdata,presp,sresp,pcc,fdebugcmd);
   return (rc);
}

/* end ipmilan2.c */
