/*
 * ipmilanplus.c
 * 
 * Interface to call libintf_lanplus from ipmitool to do RMCP+ protocol.
 *
 * 01/09/07 Andy Cress - created
 * 02/22/07 Andy Cress - initialize cipher_suite to 3 (was 0)
 */
#include <stdio.h>
#include <stdarg.h>
#include <string.h>

// #define DEBUG  1
#ifndef HAVE_LANPLUS
/* No lanplus, so stub these functions returning errors. */
#define uchar  unsigned char
#define ushort  unsigned short
#define LAN_ERR_INVPARAM  -8

int ipmi_open_lan2(char *node, char *user, char *pswd, int fdebugcmd)
{
    return(LAN_ERR_INVPARAM);
}

int ipmi_close_lan2(char *node)
{
    return(LAN_ERR_INVPARAM);
}

int ipmi_cmdraw_lan2(char *node, uchar cmd, uchar netfn, uchar lun, 
		uchar sa, uchar bus, uchar *pdata, uchar sdata,
                uchar *presp, int *sresp, uchar *pcc, char fdebugcmd)
{
    return(LAN_ERR_INVPARAM);
}

int ipmi_cmd_lan2(char *node, ushort cmd, uchar *pdata, uchar sdata,
                uchar *presp, int *sresp, uchar *pcc, char fdebugcmd)
{
    return(LAN_ERR_INVPARAM);
}

#else

#include "ipmilanplus.h"
#include "ipmicmd.h"
#define LOG_WARN  4
#define LOG_INFO  6

extern int gshutdown;    /* from ipmicmd.c */
extern int gauth_type;   /* from ipmicmd.c */
extern int gpriv_level;  /* from ipmicmd.c */
extern char gnodename[]; /* from ipmicmd.c */
extern FILE *fpdbg;      /* from ipmicmd.c */
extern FILE *fperr;      /* from ipmicmd.c */
extern FILE *fplog;      /* from ipmicmd.c */
extern ipmi_cmd_t ipmi_cmds[]; /* from ipmicmd.c */
extern char lan2_nodename[]; /*from lib/lanplus/lanplus.c */
extern struct ipmi_intf ipmi_lanplus_intf;  /*from libintf_lanplus.a*/
extern int gcipher_suite; /*from ipmicmd.c, see table 22-19 IPMI 2.0 spec*/

int   verbose = 0;  
char  fdbglog = 0;
static int loglevel = 4; /*3=LOG_ERR, 4=LOG_WARNING, 6=LOG_INFO, 7=LOG_DEBUG*/
static int lan2_fd = 0;
static struct ipmi_intf *intf = NULL;

// #define LAN_ERR_INVPARAM  -8
#define PSWD_MAX       16

#ifdef WIN32
#ifdef OLD
/* Now moved to lib/lanplus/lanplus.c */
void ipmilanplus_init(struct ipmi_intf *intf)
{
         strcpy(intf->name,"lanplus");
         intf->setup  =      ipmi_lanplus_setup;
         intf->open   =      ipmi_lanplus_open;
         intf->close  =      ipmi_lanplus_close;
         intf->sendrecv  =   ipmi_lanplus_send_ipmi_cmd;
         intf->recv_sol  =   ipmi_lanplus_recv_sol;
         intf->send_sol  =   ipmi_lanplus_send_sol;
         intf->keepalive =   ipmi_lanplus_keepalive;
         intf->target_addr = IPMI_BMC_SLAVE_ADDR;  /*0x20*/
}
#else
extern void ipmilanplus_init(struct ipmi_intf *intf);
#endif
#endif

struct ipmi_intf * ipmi_intf_load(char * name)
{
    if (strcmp(name,"lanplus") == 0) {
#ifdef WIN32
         /* initialize the intf */
         ipmilanplus_init(&ipmi_lanplus_intf);
#endif
         return (&ipmi_lanplus_intf);
    }
    else return (NULL);
}

/* 
 * ipmi_open_lan2
 */
int ipmi_open_lan2(char *node, char *user, char *pswd, int fdebugcmd)
{
   // char *username;
   // uchar priv_level;
   // char *temp;
   int rv = -1;
   int n;

   if (fdebugcmd)   
      printf("ipmi_open_lan2(%s,%s,%p,%d)\n",node,user,pswd,fdebugcmd);
   if (fdebugcmd) {
      loglevel = 7;  /* LOG_DEBUG; max=8 (LOG_DEBUG+1) */
      verbose = 1;   /* =fdebugcmd; * usually 0 or 1, but could be up to 8. */
   } else if (fdbglog) {
      loglevel = 6; /* 4=LOG_WARNING, 6=LOG_INFO */
      verbose = 5;   /* show packets ++++ */
   }
#if DEBUG
   if (fdebugcmd) { verbose = 8; loglevel = 8; }
#endif
   if (nodeislocal(node)) {
        fprintf(fpdbg,"ipmi_open_lan2: node %s is local!\n",node);
        rv = LAN_ERR_INVPARAM;
        goto EXIT;
   } else {
        if ((gshutdown==0) || fdebugcmd) 
	   fprintf(fpdbg,"Opening connection to node %s ...\n",node);

        rv = 0;
        if (intf == NULL) {
            /* fill in the intf structure */
            intf = ipmi_intf_load("lanplus");
	    if (intf == NULL) return(-1);
	    if (intf->setup == NULL) return(-1);

            rv = intf->setup(intf);  /*allocates session struct*/
            if (fdebugcmd)
                printf("lan2 intf setup returned %d\n",rv);
        } 

        if (rv == 0) {
	    if (intf->open == NULL) return(-1);
            if (intf->session == NULL) return(-1);
            intf->session->authtype_set = gauth_type; 
	    intf->session->privlvl      = gpriv_level;
            intf->session->cipher_suite_id = gcipher_suite; 
            if (node != NULL) {
                strcpy(intf->session->hostname,node);
            }
            if (user != NULL) {
                strcpy(intf->session->username,user);
            }
            if (pswd == NULL || pswd[0] == 0) 
	      intf->session->password = 0;
	    else {
	      intf->session->password = 1;
              n = strlen(pswd);
              if (n > PSWD_MAX) n = PSWD_MAX;
              memset(intf->session->authcode,0,PSWD_MAX);
              strncpy(intf->session->authcode, pswd, n);
            }
            lan2_fd = intf->open(intf);
            if (fdebugcmd)
                printf("lan2 open.intf(auth=%d,priv=%d,cipher=%d) returned fd %d\n",
			 gauth_type,gpriv_level,gcipher_suite, lan2_fd);
            if (lan2_fd > 0) {
		// strcpy(gnodename,lan2_nodename);
		rv = 0;
            } else rv = -1;
        }
   }
EXIT:
   if (rv != 0) {
      if ((gshutdown == 0) || fdebugcmd) 
             fprintf(fperr, "ipmi_open_lan2 error %d\n",rv);
   }
   return(rv);
}

int ipmi_close_lan2(char *node)
{
   int rv = 0;

   if (!nodeislocal(node)) {  /* ipmilan, need to close & cleanup */
      if (intf != NULL) { 
        if (intf->opened > 0 && intf->close != NULL)
            intf->close(intf);   /* do the close */
        /* ipmi_cleanup(intf);     * does ipmi_sdr_list_empty(intf) */
      }
   } 
   return (rv);
}
 
int ipmi_cmdraw_lan2(char *node, uchar cmd, uchar netfn, uchar lun, 
		uchar sa, uchar bus, uchar *pdata, uchar sdata,
                uchar *presp, int *sresp, uchar *pcc, char fdebugcmd)
{
   int rc, n; 
   struct ipmi_rq req;
   struct ipmi_rs *rsp;

   verbose = fdebugcmd;
#if DEBUG
   if (fdebugcmd) verbose = 8; 
#endif
   if (intf == NULL || (intf->opened == 0)) {
        rc = ipmi_open_lan2(node,guser,gpswd,fdebugcmd);
        if (rc != 0) {
           if (fdebugcmd)
              fprintf(fperr, "ipmi_cmd_lan2: interface open error %d\n",rc);
           return(rc);
        }
   }
 
   /* do the command */
   memset(&req, 0, sizeof(req));
   req.msg.cmd      = cmd;
   req.msg.netfn    = netfn;
   req.msg.lun      = lun;
   req.msg.target_cmd = sa;
   req.msg.data     = pdata;
   req.msg.data_len = sdata;

   rsp = intf->sendrecv(intf, &req);
   if (rsp == NULL) rc = -1;
   else {
      *pcc = rsp->ccode;
      rc = rsp->ccode; 
   }

   if (rc == 0) {
      /* copy data */
      if (rsp->data_len > *sresp) n = *sresp;
      else n = rsp->data_len;
      memcpy(presp,rsp->data,n);
      *sresp = n;
   } else {
      *sresp = 0;
      if (fdebugcmd)
          fprintf(fperr, "ipmi_cmd_lan2 error %d\n",rc);
   }
   return (rc);
}

/* 
 * ipmi_cmd_lan2
 * This is the entry point, called from ipmicmd.c
 */
int ipmi_cmd_lan2(char *node, ushort cmd, uchar *pdata, uchar sdata,
                uchar *presp, int *sresp, uchar *pcc, char fdebugcmd)
{
   int rc, i;
   uchar mycmd;

   for (i = 0; i < NCMDS; i++) {
       if (ipmi_cmds[i].cmdtyp == cmd) break;
   }
   if (i >= NCMDS) {
        fprintf(fperr, "ipmi_cmd_lan2: Unknown command %x\n",cmd);
        return(-1);
        }
   if (cmd >= CMDMASK) mycmd = (uchar)(cmd & CMDMASK);  /* unmask it */
   else mycmd = (uchar)cmd;
   
   rc = ipmi_cmdraw_lan2(node,mycmd, ipmi_cmds[i].netfn, ipmi_cmds[i].lun,
			ipmi_cmds[i].sa, ipmi_cmds[i].bus,
                        pdata, sdata, presp, sresp, pcc, fdebugcmd);
   return(rc);
}


/* misc ipmitool helper routines for lanplus plugin */
void lprintf(int level, const char * format, ...)
{
        va_list vptr;
        static char logtmp[LOG_MSG_LENGTH];
	FILE *fp = stderr; 

        if (level > loglevel) return;
	if (fdbglog) {
	   if (fplog != NULL) fp = fplog;
	}
#ifdef WIN32
        va_start(vptr, format);
        vfprintf(fp, format, vptr);
        va_end(vptr);
        fprintf(fp,"\r\n");
#else
        va_start(vptr, format);
        vsnprintf(logtmp, LOG_MSG_LENGTH, format, vptr);
        va_end(vptr);
        fprintf(fp, "%s\r\n", logtmp);
#endif
        return;
}

void lperror(int level, const char * format, ...)
{
        va_list vptr;
        if (level > loglevel) return;
        va_start(vptr, format);
        vfprintf(stderr, format, vptr); 
        va_end(vptr);
        fprintf(stderr,"\r\n");
        return;
}

#define NOEM 3
static struct { int id; char *name; } oem_list[NOEM] = {
   {0x000157, "intelplus"},
   {0x002A7C, "supermicro"},
   /* {       0, "icts"},  *ICTS testware, needs user option*/
   {0x000002, "ibm"}
};
#define NIBMC  3
static int intel_ibmc[NIBMC] = {  /*Intel product codes for IBMC*/
   0x003a, 0x003d, 0x003e
};

int ipmi_oem_active(struct ipmi_intf * intf, const char * oemtype)
{
        int i, vend, prod;
        vend  = my_devid[6] + (my_devid[7] << 8) + (my_devid[8] << 16); 
        prod  = my_devid[9] + (my_devid[10] << 8);
        if (verbose) 
	    lprintf(LOG_WARN,"oem_active: type=%s vend=%x prod=%x", oemtype,vend,prod);
        if (strncmp("intelplus", oemtype, strlen(oemtype)) == 0) {
            /* special case to detect intelplus, not all Intel platforms */
            if (vend == VENDOR_INTEL) {
                /* If IBMC, does not use intelplus */
                for (i = 0; i < NIBMC; i++) {
                  if (prod == intel_ibmc[i]) {  
                    if (verbose) lprintf(LOG_WARN,"intel IBMC detected");
                    return 0;
                  }
                }
                if (verbose) lprintf(LOG_WARN,"intelplus detected, vend=%x",vend);
		return 1;
            }
        } else {
           // if (intf->oem == NULL) return 0;
           for (i = 0; i < NOEM; i++) {
             if (strncmp(oem_list[i].name,oemtype,strlen(oemtype)) == 0)
                if (oem_list[i].id == vend) {
                   if (verbose) lprintf(LOG_WARN,"%s detected, vend=%x",oemtype,vend);
                   return 1;
                }
           }
        }
        return 0;
}

/*
 * lan2_set_sol_data
 * called from isolconsole when SOL 2.0 session is activated.
 */
void lan2_set_sol_data(int insize, int outsize, int port, void *handler)
{
    if (intf == NULL) return;
    intf->session->sol_data.max_inbound_payload_size = insize;
    intf->session->sol_data.max_outbound_payload_size = outsize;
    intf->session->sol_data.port = port;
    intf->session->sol_data.sol_input_handler = handler; 
    intf->session->timeout = 2;  /*see lib/.../lanplus.h: IPMI_LAN_TIMEOUT */
    intf->session->sol_escape_char = '~';
}

int lan2_keepalive(int type)
{
    int rv = 0;
    if (type == 2) {  /*send empty SOL data*/
        struct ipmi_v2_payload v2_payload;
        struct ipmi_rs * rsp = NULL;
        memset(&v2_payload, 0, sizeof(v2_payload));
        v2_payload.payload.sol_packet.character_count = 0;
        rsp = intf->send_sol(intf, &v2_payload);
    } else 
        rv = intf->keepalive(intf);  /*get_device_id*/
    return(rv);
}

typedef struct {
  int type;
  int len;
  char *data;
  } SOL_RSP_PKT;

int lan2_send_sol( uchar *payload, int len, SOL_RSP_PKT *rsp)
{
    struct ipmi_rs *rs;
    struct ipmi_v2_payload v2_payload;
    int rv = 0;

    if (rsp) rsp->len  = 0;  /*just in case*/
    if (intf == NULL) return -1;
    memset(&v2_payload, 0, sizeof(v2_payload));
    memcpy(v2_payload.payload.sol_packet.data,payload,len);
    v2_payload.payload.sol_packet.character_count = len;
    rs = intf->send_sol(intf, &v2_payload);
    if (rs == NULL) {
       rv = -1;
       lprintf(LOG_INFO,"send_sol error (%d bytes)",len);
    } else {
       rsp->type = rs->session.payloadtype;
       rsp->len  = rs->data_len;
       rsp->data = rs->data;
       rv = 0;
       lprintf(LOG_INFO,"send_sol: rq_seq=%d rs_seq=%d (0x%02x)",
	   v2_payload.payload.ipmi_request.rq_seq,
		rs->session.seq, rs->session.seq);
    }
    return(rv);
}

int lan2_recv_sol( SOL_RSP_PKT *rsp )
{
    struct ipmi_rs * rs;

    if (rsp == NULL) return -1;
    rsp->len  = 0;
    if (intf == NULL) return -1;
    rs = intf->recv_sol(intf);
    if (rs == NULL) return -1;
    rsp->type = rs->session.payloadtype;
    rsp->len  = rs->data_len;
    rsp->data = rs->data;
    lprintf(LOG_INFO,"recv_sol: rs_seq=%d (0x%02x)",
		rs->session.seq, rs->session.seq);
    return(rsp->len);
}

int lan2_get_fd(void)
{
   if (intf == NULL) return 1;
   return(intf->fd);
}
#endif

/* end ipmilanplus.c */
