/*M*
//  PVCS:
//      $Workfile:   ipmimv.c  $
//      $Revision:   1.1  $
//      $Modtime:   08 Apr 2003 15:31:14  $
//      $Author:   arcress@users.sourceforge.net  $  
//
//  This implements support for the /dev/ipmi0 native interface from
//  the MontaVista OpenIPMI driver.
// 
//  01/29/03 ARC - created, derived from ipmi_test.c and ipmitool_mv.c
//  04/08/03 ARC - don't watch stdin on select, since stdin may not be
//                 valid if invoked from cron, etc.
//  06/11/03 ARC - ignore EMSGSIZE errno for get_wdt command
//  05/05/04 ARC - only open/close device once per application,
//		   rely on each app calling ipmi_close, helps performance.
//  08/10/04 ARC - handle alternate device filenames for some 2.6 kernels
//  03/01/05 ARC - fix /dev/ipmi0 IPMB requests (to other than BMC_SA)
//  04/12/07 ARC - check for IPMI_ASYNC_EVENT_RECV_TYPE in ipmicmd_mv
 *M*/
/*----------------------------------------------------------------------*
The BSD License 

Copyright (c) 2002-2005, Intel Corporation
All rights reserved.

Redistribution and use in source and binary forms, with or without 
modification, are permitted provided that the following conditions are met:

  a.. Redistributions of source code must retain the above copyright notice, 
      this list of conditions and the following disclaimer. 
  b.. Redistributions in binary form must reproduce the above copyright notice,
      this list of conditions and the following disclaimer in the documentation 
      and/or other materials provided with the distribution. 
  c.. Neither the name of Intel Corporation nor the names of its contributors 
      may be used to endorse or promote products derived from this software 
      without specific prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR 
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *----------------------------------------------------------------------*/

#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>
#ifdef SCO_UW
#include <sys/ioccom.h>
#endif
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "ipmicmd.h"
// #define uchar    unsigned char
#define IPMI_MAX_ADDR_SIZE 32
#define IPMI_RESPONSE_RECV_TYPE         1
#define IPMI_ASYNC_EVENT_RECV_TYPE      2
#define IPMI_CMD_RECV_TYPE              3

#ifdef TV_PORT
/* use this to define timeval if it is a portability issue */
struct timeval {
        long int     tv_sec;         /* (time_t) seconds */
        long int     tv_usec;        /* (suseconds_t) microseconds */
};
#endif

extern FILE *fperr;  /*defined in ipmicmd.c*/
extern FILE *fpdbg;  /*defined in ipmicmd.c*/
extern ipmi_cmd_t ipmi_cmds[NCMDS];
int ipmi_timeout_mv = 10;   /* 10 seconds, was 5 sec */

struct ipmi_addr
{
        int   adrtype;
        short channel;
        char  data[IPMI_MAX_ADDR_SIZE];
};

struct ipmi_msg
{
        uchar  netfn;
        uchar  cmd;
        ushort data_len;
        uchar  *data;
};

struct ipmi_req
{
        unsigned char *addr; /* Address to send the message to. */
        unsigned int  addr_len;
        long    msgid; /* The sequence number for the message.  */
        struct ipmi_msg msg;
};
 
struct ipmi_recv
{
        int     recv_type;  	/* Is this a command, response, etc. */
        unsigned char *addr;    /* Address the message was from */
	int  addr_len;  	/* The size of the address buffer. */
        long    msgid;  	/* The sequence number from the request */
        struct ipmi_msg msg; 	/* The data field must point to a buffer. */
};

#define IPMI_IOC_MAGIC 'i'
#define IPMICTL_RECEIVE_MSG         _IOWR(IPMI_IOC_MAGIC, 12, struct ipmi_recv)
#define IPMICTL_RECEIVE_MSG_TRUNC   _IOWR(IPMI_IOC_MAGIC, 11, struct ipmi_recv)
#define IPMICTL_SEND_COMMAND        _IOR(IPMI_IOC_MAGIC,  13, struct ipmi_req)
#define IPMICTL_SET_GETS_EVENTS_CMD _IOR(IPMI_IOC_MAGIC,  16, int)

#define BMC_SA   	  0x20
#define IPMI_BMC_CHANNEL  0xf
#define IPMI_SYSTEM_INTERFACE_ADDR_TYPE 0x0c
#define IPMI_IPMB_ADDR_TYPE             0x01
#define IPMI_IPMB_BROADCAST_ADDR_TYPE   0x41
/* Broadcast get device id is used as described in IPMI 1.5 section 17.9. */

struct ipmi_system_interface_addr
{
        int           adrtype;
        short         channel;
        unsigned char lun;
};

struct ipmi_ipmb_addr
{
        int    adrtype;
        short  channel;
        uchar  slave_addr;
        uchar  lun;
};

static int ipmi_fd = -1;
static int curr_seq = 0;

int ipmi_open_mv(char fdebugcmd)
{
    char *pdev;

    if (ipmi_fd != -1) return(0);
    pdev = "/dev/ipmi/0";
    ipmi_fd = open("/dev/ipmi/0", O_RDWR);
    if (ipmi_fd == -1) {
	if (fdebugcmd) printf("ipmi_open_mv: cannot open %s\n",pdev);
	pdev = "/dev/ipmi0";
        ipmi_fd = open(pdev, O_RDWR);
    }
    if (ipmi_fd == -1) {
	if (fdebugcmd) printf("ipmi_open_mv: cannot open %s\n",pdev);
	pdev = "/dev/ipmidev0";
        ipmi_fd = open(pdev, O_RDWR);
    }
    if (ipmi_fd == -1) {
	if (fdebugcmd) printf("ipmi_open_mv: cannot open %s\n",pdev);
	pdev = "/dev/ipmidev/0";
        ipmi_fd = open(pdev, O_RDWR);
    }
    if (ipmi_fd == -1) { 
	if (fdebugcmd) printf("ipmi_open_mv: cannot open %s\n",pdev);
	return(-1);
    }
    if (fdebugcmd) printf("ipmi_open_mv: succesfully opened %s\n",pdev);
    return(0);
}

int ipmi_close_mv(void)
{
    int rc = 0;
    if (ipmi_fd != -1) { 
	rc = close(ipmi_fd);
	ipmi_fd = -1; 
    }
    return(rc);
}

int ipmicmd_mv(uchar cmd, uchar netfn, uchar lun, uchar sa, uchar bus,
		uchar *pdata, uchar sdata, uchar *presp, int sresp, int *rlen)
{
    fd_set readfds;
    struct timeval tv;
    struct ipmi_req       req;
    struct ipmi_recv      rsp;
    struct ipmi_addr      addr;
    struct ipmi_ipmb_addr             ipmb_addr;
    struct ipmi_system_interface_addr bmc_addr;
    int    i, done;
    int    rv;

    rv = ipmi_open_mv(0);
    if (rv != 0) return(rv);

    i = 1;
    rv = ioctl(ipmi_fd, IPMICTL_SET_GETS_EVENTS_CMD, &i);
    if (rv) { return(errno); }

    FD_ZERO(&readfds);
    // FD_SET(0, &readfds);  /* dont watch stdin */
    FD_SET(ipmi_fd, &readfds);  /* only watch ipmi_fd for input */

    /* Special handling for ReadEventMsgBuffer, etc. */
#ifdef TEST
    recv.msg.data = data;
    recv.msg.data_len = sizeof(data);
    recv.addr = (unsigned char *) &addr;
    recv.addr_len = sizeof(addr);
    rv = ioctl(fd, IPMICTL_RECEIVE_MSG_TRUNC, &recv);
    if (rv == -1) {
        if (errno == EMSGSIZE) {
            /* The message was truncated, handle it as such. */
            data[0] = IPMI_REQUESTED_DATA_LENGTH_EXCEEDED_CC;
            rv = 0;
        } else
            goto out;
    }
#endif

    /* Send the IPMI command */ 
    if (sa == BMC_SA) {
	bmc_addr.adrtype = IPMI_SYSTEM_INTERFACE_ADDR_TYPE;
	bmc_addr.channel = IPMI_BMC_CHANNEL;
	bmc_addr.lun = lun;       /* usu BMC_LUN = 0 */
	req.addr = (char *) &bmc_addr;
	req.addr_len = sizeof(bmc_addr);
    } else {
	ipmb_addr.adrtype = IPMI_IPMB_ADDR_TYPE;
	ipmb_addr.channel = bus;   /* usu PUBLIC_BUS = 0 */
	ipmb_addr.slave_addr = sa;
	ipmb_addr.lun = lun;
	req.addr = (char *) &ipmb_addr;
	req.addr_len = sizeof(ipmb_addr);
    }
    req.msg.cmd = cmd;
    req.msg.netfn = netfn;   
    req.msgid = curr_seq;
    req.msg.data = pdata;
    req.msg.data_len = sdata;
    rv = ioctl(ipmi_fd, IPMICTL_SEND_COMMAND, &req);
    curr_seq++;
    if (rv == -1) { rv = errno; }

    done = 0;
    if (rv == 0) while (!done) {
        done = 1;
	tv.tv_sec=ipmi_timeout_mv;
	tv.tv_usec=0;
	rv = select(ipmi_fd+1, &readfds, NULL, NULL, &tv);
	/* expect select rv = 1 here */
	if (rv <= 0) { /* no data within 5 seconds */
           fprintf(fperr,"mv select timeout, fd = %d, isset = %d, rv = %d, errno = %d\n",
		  ipmi_fd,FD_ISSET(ipmi_fd, &readfds),rv,errno);
	   if (rv == 0) rv = -3;
	   else rv = errno;
	} else {
	   /* receive the IPMI response */
	   rsp.addr = (char *) &addr;
	   rsp.addr_len = sizeof(addr);
	   rsp.msg.data = presp;
	   rsp.msg.data_len = sresp;
	   rv = ioctl(ipmi_fd, IPMICTL_RECEIVE_MSG_TRUNC, &rsp);
	   if (rv == -1) { 
	      if ((errno == EMSGSIZE) && (rsp.msg.data_len == sresp))
		 rv = 0;   /* errno 90 is ok */
	      else { 
		 rv = errno; 
                 fprintf(fperr,"mv rcv_trunc errno = %d, len = %d\n",
			errno, rsp.msg.data_len);
	      }
	   } else rv = 0;
	   /* Driver should ensure matching req.msgid and rsp.msgid */
           /* Skip & retry if async events, only listen for those in 
            * getevent_mv() below. */
           if (rsp.recv_type == IPMI_ASYNC_EVENT_RECV_TYPE) 
	       done = 0;
	   *rlen = rsp.msg.data_len;
	}
    } /*endif send ok, while select/recv*/

    /* ipmi_close_mv();  * rely on the app calling ipmi_close */
    return(rv);
}

int ipmi_cmdraw_mv(uchar cmd, uchar netfn, uchar lun, uchar sa, uchar bus,
		uchar *pdata, uchar sdata, uchar *presp, int *sresp, 
		uchar *pcc, char fdebugcmd)
{
    uchar  buf[MAX_BUFFER_SIZE];
    int rc;
    int rlen = 0;
    uchar cc;

    if (fdebugcmd) { 
         fprintf(fpdbg,"mv cmd=%02x netfn=%02x lun=%02x sdata=%d sresp=%d\n",
		 cmd,netfn,lun,sdata,*sresp); 
         dump_buf("mv cmd data",pdata,sdata,0);
    }
    rc = ipmicmd_mv(cmd,netfn,lun,sa, bus, pdata,sdata,
			buf,(*sresp + 1),&rlen);
    if (fdebugcmd) {
        fprintf(fpdbg,"ipmi_cmdraw_mv: status=%x, ccode=%x\n",(uint)rc,cc);
        if (rc == 0) dump_buf("mv rsp data",buf,rlen,0);
    }
    cc = buf[0];
    if (rlen > 0) { /* copy data, except first byte */
       rlen -= 1;
       if (rlen > *sresp) rlen = *sresp;
       memcpy(presp,&buf[1],rlen);
    }
    *pcc = cc;
    *sresp = rlen;
    return(rc);
}

int ipmi_cmd_mv(ushort cmd, uchar *pdata, uchar sdata, uchar *presp, 
		int *sresp, uchar *pcc, char fdebugcmd)
{
    uchar  buf[MAX_BUFFER_SIZE];
    int rc, i, buflen;
    uchar   cc;
    int rlen = 0;
    int xlen, j;

    for (i = 0; i < NCMDS; i++) {
       if (ipmi_cmds[i].cmdtyp == cmd) break;
    }
    if (i >= NCMDS) {
	fprintf(fperr, "ipmi_cmd_mv: Unknown command %x\n",cmd);
	return(-1); 
	}
    if (cmd >= CMDMASK) cmd = cmd & CMDMASK;  /* unmask it */

    if (fdebugcmd) { 
         fprintf(fpdbg,
		"mv cmd=%02x netfn=%02x lun=%02x sdata=%d sresp=%d\n",
		cmd,ipmi_cmds[i].netfn,ipmi_cmds[i].lun,sdata, *sresp); 
         dump_buf("mv cmd data",pdata,sdata,0);
    }
    rc = ipmicmd_mv(cmd,ipmi_cmds[i].netfn,ipmi_cmds[i].lun,
			ipmi_cmds[i].sa, ipmi_cmds[i].bus,
			pdata,sdata,buf,(*sresp + 1),&rlen);
    // if (rc == -1) printf("ipmi_cmd_mv: cannot open /dev/ipmi0\n");
    cc = buf[0];
    if (fdebugcmd) {
	    fprintf(fpdbg,"ipmi_cmd_mv: ipmicmd_mv status=%x, ccode=%x\n", 
                      (uint)rc, cc);
            if (rc == ACCESS_OK) {
               uchar * pc; int sz;
               sz = rlen;
               pc = (uchar *)buf;
               fprintf(fpdbg,"ipmi_cmd_mv: response (len=%d): ",sz);
               for (j = 0; j < sz; j++) fprintf(fpdbg,"%02x ",pc[j]);
               fprintf(fpdbg,"\n");
            }
        }
    xlen = ipmi_cmds[i].rslen + 1;
    if ((ipmi_cmds[i].cmdtyp == GET_SEL_ENTRY) && 
        (rlen < xlen) && (rc == 0) && (cc != 0) &&
        (i > 0) && (rlen > 1))                  /*not temp slot, have data*/ 
    {
          /* Detect/Handle MV driver SEL bug returning missing bytes */
          if (fdebugcmd) {
              fprintf(fpdbg,"ipmi_cmd_mv[%d] BUG: returned %d, expected %d\n",
				i,rlen,xlen);
          }
          cc = 0x80;  /*flag as busy, retry*/
          j = xlen - rlen;
          j--;  /* omit cc */
          for (i = 0; i < j; i++) presp[i] = 0xff;
          if ((rlen+j) > *sresp) rlen = *sresp - j;
          memcpy(&presp[j],&buf[0],rlen);
          rlen += j;
    }
    if (rlen > 0) {
       /* copy data, except first byte */
       rlen -= 1;
       if (rlen > *sresp) rlen = *sresp;
       memcpy(presp,&buf[1],rlen);
    }
    *pcc = cc;
    *sresp = rlen;

    return(rc);
}  /*end ipmi_cmd_mv*/

int getevent_mv(uchar *evt_data, int *evt_len, uchar *cc)
{
    struct ipmi_recv recv;
    uchar data[36];  /* #define MAX_IPMI_DATA_SIZE 36 */
    struct ipmi_addr  addr;
    int rv = 0;
    int n;
 
    recv.msg.data = data;
    recv.msg.data_len = sizeof(data);
    recv.addr = (unsigned char *) &addr;
    recv.addr_len = sizeof(addr);
    rv = ioctl(ipmi_fd, IPMICTL_RECEIVE_MSG_TRUNC, &recv);
    if (rv == -1) {
        if (errno == EMSGSIZE) { /* The message was truncated */
            *cc = 0xC8;   /*IPMI_REQUESTED_DATA_LENGTH_EXCEEDED_CC;*/
            rv = 0;
        }
    } else *cc = 0;
    if (rv == 0) {
        n = recv.msg.data_len;
        if (n > 0) 
           memcpy(evt_data,&data[0],n);
        *evt_len = n;
    } else if (rv == -1 || rv == -11) {
        rv = 0x80;  /* -EAGAIN, no data, try again */
    }
    return(rv);
}

#ifdef TEST
int
main(int argc, char *argv[])
{
    fd_set readfds;
    struct timeval tv;
    char   data[40];
    int    i, j;
    int    err;
    int    rlen;

    err = ipmicmd_mv(0x01, 0x06, 0, NULL, 0, data, sizeof(data), &rlen);
    printf("ipmicmd_mv ret=%d, cc=%02x\n",err,(uchar)data[0]);
    printf(" ** Return Code: %2.2X\n", data[0]);
    printf(" ** Data[%d]: ",rlen);
    for (i=1; i < rlen; i++)
	    printf("%2.2X ", (uchar)data[i]);
    printf("\n");

    printf("\n");
    ipmi_close_mv();
    return 0;
}
#endif
