/*
 * isolconsole.c
 *      IPMI Serial-Over-LAN console application
 *
 * Author:  Andy Cress  arcress@users.sourceforge.net
 * Copyright (c) 2006 Intel Corporation. 
 *
 * 10/12/06 Andy Cress - created, just a framework for now
 * 02/08/07 Andy Cress - now working for IPMI LAN 2.0 SOL in Linux,
 *                       still stdio problems in Windows
 * 02/23/07 Andy Cress - fixed Windows stdin, still Windows output issues,
 *                       need more in xlate_vt100_win().
 * 04/06/07 Andy Cress - more vt100-to-windows translation, 
 *                       added -r for raw termio
 * 04/19/07 Andy Cress - more vt100-to-windows translation, 
 * 05/08/07 Andy Cress - more logic for 1.5 SOL.  Session opens but 
 *                       1.5 SOL data packets are not yet working.
 * 05/24/07 Andy Cress - Fix Enter key confusion in BIOS Setup (use CR+LF)
 * 06/15/07 Andy Cress - More fixes to Windows translation (fUseWinCon)
 * 08/20/07 Andy Cress - moved Windows translation to isolwin.c
 *
 * NOTE: There are a few other options for Serial-Over-LAN console 
 * applications:
 *   ipmitool.sf.net (has v2 sol console capability for Linux)
 *   Intel dpccli (CLI console for Linux and Windows)
 *   Intel ISM console (for Windows)
 *   Intel System Management Software (LANDesk) console 
 * The Intel applications support both IPMI 1.5 and 2.0 SOL 
 * protocols.
 */
/*M*
Copyright (c) 2006-2007, Intel Corporation
All rights reserved.

Redistribution and use in source and binary forms, with or without 
modification, are permitted provided that the following conditions are met:

  a.. Redistributions of source code must retain the above copyright notice, 
      this list of conditions and the following disclaimer. 
  b.. Redistributions in binary form must reproduce the above copyright notice,
      this list of conditions and the following disclaimer in the documentation 
      and/or other materials provided with the distribution. 
  c.. Neither the name of Intel Corporation nor the names of its contributors 
      may be used to endorse or promote products derived from this software 
      without specific prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR 
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *M*/
#ifdef WIN32
#include <windows.h>
#include <stdio.h>
#include <winsock.h>
#include <io.h>
#include "getopt.h"
#else
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdarg.h> 
#include <getopt.h>
#endif
#include <string.h>
#include <time.h>
#include "ipmicmd.h"
#include "ipmilanplus.h"
 
#define ACTIVATE_PAYLOAD    0x48
#define DEACTIVATE_PAYLOAD  0x49
#define PAYLOAD_TYPE_SOL    0x01
#define SOL_SERIAL_ALERT_MASK_DEFERRED 0x04
#define SOL_BMC_ASSERTS_CTS_MASK_TRUE  0x00  
#define SOL_BMC_ASSERTS_CTS_MASK_FALSE 0x02
#define IPMI_PAYLOAD_TYPE_SOL  0x01
#define AUTHTYPE_RMCP_PLUS  0x06  
#define RV_END   -2
#define CH_CR   '\r'    /*0x0D =='\r'*/
#define CH_LF   '\n'    /*0x0A =='\n' LineFeed(Down) */
#define CH_ESC  '\033'  /*0x1B == ASCII Escape */
#define CH_CTL  '\316'  /*0xCE == VT100 Control */
#define CH_DEL  '\177'  /*0x7F == Delete key */
#define CH_BS   '\010'  /*0x08 == Backspace */
#define CH_TAB  '\011'  /*0x09 == Tab */
#define CH_UP   '\013'  /*0x0B == Up Arrow */

/* Input data buffer size: screen size (80*24) = 1920 */
/* IPMI_BUF_SIZE = 1024, less RMCP+IPMI hdr */
/* IDATA_SZ=2048 was too big, used =300 thru v2.1.0 */
#define IDATA_SZ  512
#define IPKT_SZ   512

typedef struct {
  int type;
  int len;
  char *data;
  } SOL_RSP_PKT;

extern void tty_setraw(int mode);      /*from ipmicmd.c*/
extern void tty_setnormal(int mode);   /*from ipmicmd.c*/
extern int  lan_get_fd(void);           /*from ipmilan.c*/
extern int  lan_send_sol( uchar *payload, int len, SOL_RSP_PKT *rsp);
extern int  lan_recv_sol( SOL_RSP_PKT *rsp );
extern int  lan_keepalive(int type);
extern void lan_get_sol_data(uchar fEnc, uchar iseed, uint32 *seed);
extern void lan_set_sol_data(uchar fEnc, uchar auth, uchar iseed, 
				int insize, int outsize);
#ifdef HAVE_LANPLUS
extern int  lan2_get_fd(void);          /*from ipmilanplus.c*/
extern int  lan2_send_sol( uchar *payload, int len, SOL_RSP_PKT *rsp);
extern int  lan2_recv_sol( SOL_RSP_PKT *rsp );
extern void lan2_set_sol_data(int insize, int outsize, int port, void*hnd);
extern int  lan2_keepalive(int type);
#else
int  lan2_get_fd(void) { return(-1); }
int  lan2_send_sol( uchar *payload, int len, SOL_RSP_PKT *rsp) { return(-1); }
int  lan2_recv_sol( SOL_RSP_PKT *rsp ) { return(-1); }
void lan2_set_sol_data(int insize, int outsize, int port, void*hnd)
{ return; }
int  lan2_keepalive(int type) { return(-1); }
#endif
void dbglog( char *pattn, ... ); /*local prototype*/
extern char  fdbglog;    /*see ipmilanplus.c*/
extern char log_name[];  /*see ipmicmd.c*/

/*
 * Global variables 
 */
static char * progver   = "2.12";
static char * progname  = "isolconsole";
static char  fdebug        = 0;
static char  fpicmg        = 0;
static char  factivate     = 0;
static char  fdeactivate   = 0;
static char  bDriver       = 0;
static uchar bSolVer       = 0;
static uchar bKeepAlive    = 2;  /*1=GetDeviceID, 2=empty SOL data*/
static char  sol_esc_seq[3] = {'~',0,0};  /* SOL escape char ('~') */
static char  sol_esc_fn[3] = {'q','r','h'}; /* SOL escape functions */
static uchar ipmi_maj = 0;
static uchar ipmi_min = 0;
static int   mfg      = 0;  /* Manufacturer IANA vendor id */
static uchar fSolEncryption     = 1;
static uchar fSolAuthentication = 1;
static int   sol_timeout = 30;  /*send keepalive every 30 seconds*/
static time_t keepalive_start; 
char  fUseWinCon    = 1;
uchar fRaw      = 0;
uchar fCRLF     = 0;    /* =1 to use legacy CR/LF translation for BIOS */
uchar bTextMode = 0;  
int sol_done = 0;

#ifdef WIN32
extern DWORD WINAPI input_thread( LPVOID lpParam);
extern void console_open(uchar fdebugcmd);
extern void console_close(void);
extern void console_out(uchar *pdata, int len);
extern int  console_in(uchar c, uchar c1, uchar *pdata, int len);
extern int  os_select(int infd, int sfd, fd_set *read_fds);
extern int  os_read(int fd, uchar *buf, int sz);
#else
/*======== start Linux ===============================*/
int os_read(int fd, uchar *buf, int sz)
{
    int len = 0;
    len = read(fd, buf, sz);
    return(len);
}

int os_select(int infd, int sfd, fd_set *read_fds)
{
   int rv = 0;
   struct timeval tv;

   /* Linux handles both stdin & socket via select() */
   FD_ZERO(read_fds);
   FD_SET(infd, read_fds);
   FD_SET(sfd, read_fds);
   tv.tv_sec =  0;
   tv.tv_usec = 500000;
   rv = select(sfd + 1, read_fds, NULL, NULL, &tv);
   return rv;
}

void CheckTextMode(uchar *buffer, int sbuf)
{
    int c, i, j, imatch;
    uchar pattern[4] = {CH_ESC,'[','0','m'};
    int spattern = 4;
 
    /* if CH_ESC,"[0m", TextMode change */
    j = 0;
    imatch = 0;
    for (j = 0; j < sbuf; j++) {
        if ((sbuf - j) < spattern && imatch == 0) break;
        c = buffer[j];
        if (c == pattern[imatch]) {
            imatch++;
        } else if (pattern[imatch] == '?') {  /*wildcard char*/
            imatch++;
        } else if (pattern[imatch] == '0' && (c>='0' && c <='9')) { /*numeric*/
            imatch++;
        } else {
            if (imatch > 0) {
               if (j > 0) j--; /* try again with the first match char */
               imatch = 0;
            } 
        }
        if (imatch == spattern) { /*match found*/
            i = (j+1) - imatch;  /*buffer[i] is the match */
            bTextMode = buffer[i+2] & 0x0f;  /* usu 0 or 1 */
            dbglog("Text Mode changed to %d\n",bTextMode);
        }
        else if (imatch > spattern) break;
    }
    return;
}

/*======== end Linux =================================*/
#endif

void dbglog( char *pattn, ... )
{
    va_list arglist;
#ifdef WIN32
    if (fdebug == 0 && fdbglog == 0) return;
        va_start(arglist, pattn);
        vfprintf(stderr, pattn, arglist);
        va_end(arglist);
        fprintf(stderr,"\r\n");
#else
    char logtmp[LOG_MSG_LENGTH];
    if (fdebug == 0 && fdbglog == 0) return;
    va_start( arglist, pattn );
    vsnprintf(logtmp, LOG_MSG_LENGTH, pattn, arglist);
    va_end( arglist );
    if (fdbglog) print_log(logtmp);
    if (fdebug) fprintf( stderr, logtmp);
#endif
}
             
void dbg_dump(char *tag, uchar *pdata, int len, int fascii)
{
    if (fdebug == 0 && fdbglog == 0) return;
    dump_log(NULL,tag,pdata,len,fascii); /*uses fplog or fpdbg */
}

static int sol_send(uchar *buf, int len, SOL_RSP_PKT *rsp)
{
    int rv = 0;
    if (bSolVer == 1) {
       rv = lan_send_sol(buf, len, rsp);
    } else 
       rv = lan2_send_sol(buf, len, rsp);
    return(rv);
}

static int sol_recv( SOL_RSP_PKT *rsp)
{
    int rv = 0;
    if (bSolVer == 1) {
       rv = lan_recv_sol( rsp );
    } else 
       rv = lan2_recv_sol( rsp );
    return(rv);
}

static int  sol_get_fd(void)
{
    int fd = 0;
    if (bSolVer == 1) {
       fd = lan_get_fd();
    } else 
       fd = lan2_get_fd();
    return(fd);
}

static int sol_keepalive(int type)
{
   int rv = 0;
   time_t end; 

   time(&end); 
   if ((end - keepalive_start) > sol_timeout) {
       if (bSolVer == 1) {
           rv = lan_keepalive(bKeepAlive);   
       } else 
           rv = lan2_keepalive(bKeepAlive);   
       time(&keepalive_start);  /*time in seconds*/
       dbglog("sol_keepalive sent, rv = %d\n",rv);
       flush_log();
   }
   return(rv);
}

static void sol_keepalive_reset(void)
{
    /* not idle, so reset the keepalive timer */
    time(&keepalive_start);  /*time in seconds*/
}

void sol_output_handler(SOL_RSP_PKT *rsp)
{
    if (rsp &&
        (rsp->type == PAYLOAD_TYPE_SOL))
    {
#ifdef WIN32
	 console_out(rsp->data,rsp->len);
#else
         uchar *pdata;
         int len;
         int i;
         int mode;
         uchar c;

         pdata = rsp->data;
         len   = rsp->len;
         if (fdbglog) {
             // dbglog("sol_output: len = %d\n",len);
             dbg_dump("sol_output",pdata,len,1); /*like printlog*/
         }
         CheckTextMode(pdata,len);
         for (i = 0; i < len; i++) {
	      c = pdata[i];
              /* do special CH_CR handling here? */
              putc(c, stdout);
         }
         fflush(stdout);
#endif
    }
}

static int    sol_retries = 3;
static uchar  sol15_iseed = 0x01;  /*initial seed count, usu 0 or 1 */
static uint32 sol15_wseed = 0;     /*32-bit seed value*/

int sol_input_handler( uchar *input, int ilen)
{
   int rv = 0;
   uchar payload[IPKT_SZ]; 
   SOL_RSP_PKT  rs;
   int i, length, t;
   static uchar lastc = 0;
   uchar c, c1;

   memset(&payload, 0, sizeof(payload)); 
   length = 0;
   rs.len = 0;
   for (i = 0; i < ilen; ++i)
   {
        c = input[i];
        c1 = input[i+1];

        /* look for escape sequences */
        // if (lastc == '\n' && c == sol_esc_ch) 
        if (c == sol_esc_seq[0]) {  /*'~'*/
           dbglog("sol_input RV_END\n");
           /* TODO: handle sol_esc_fn[] */
           return(RV_END);
        }
#ifdef WIN32
        if (c < 0x20 || c > 0x7A) { /*not displayable char*/
	    dbglog("sol_input[%d]: lastc=%02x c=%02x, %02x %02x\n", 
			i,lastc,c, input[i+1],input[i+2]);
        }
        switch(c) {
          case 0xe0:  /*Arrow keys*/
              payload[length++] = CH_ESC;
              payload[length++] = '[';
	      switch(c1) {
              case 0x48: /*Up*/
		 c = 'A'; /* ESC [ A */
                 break;
              case 0x50: /*Down*/
		 c = 'B'; /* ESC [ B */
                 break;
              case 0x4d: /*Right*/
                 c = 'C'; /* ESC [ C */
                 break;
              case 0x4b: /*Left*/
                 c = 'D'; /* ESC [ D */
                 break;
	      default:    
		 c = 'B'; /* ESC [ B */
                 break;
              }
              i++;
              break;
          case 0x00:  /*Function keys 0x00,[0x3b-0x44]*/
              payload[length++] = CH_ESC;
	      switch(c1) {
	      case 0x3b:  /*F1*/
                 payload[length++] = 'O';
                 c = 'P'; /* PF1 = ESC O P */
                 break;
	      case 0x3c:  /*F2*/
                 payload[length++] = 'O';
                 c = 'Q'; /* PF2 =ESC O Q */
                 break;
	      case 0x44:  
                 c = '0'; /* F10 = ESC 0 */
                 break;
	      default:   /* F3-F9 (use ESC,'3' thru ESC,'9') */
                 t = c1 - 0x3a;  /* t = Fkey number */
                 if (t > 10) c = (--t & 0x0f) + '0';
                 else c = (t & 0x0f) + '0';
                 dbglog("Fkey: %x,%x -> ESC %c\n",input[i+1],t,c);
                 break;
              }
              i++;
              break;
          default:
	      /* check input to see if it should be sent */
              t = console_in(c,c1,&payload[length],2);
              if (t > 0) length += t;
              break;
        }
        /* copy current data byte to payload */
        payload[length++] = c;
        lastc = c;
#else
        /*
         * Linux LF -> CR 
         * Note that this fCRLF logic works for BIOS, but causes
         * extra returns after Linux commands. 
         */
        if (c == CH_LF) { //  && lastc != CH_CR) 
            if (fCRLF) payload[length++] = CH_CR; /*insert CR here, LF below*/
            else  c = CH_CR;
        }
        /* copy other data to payload */
        payload[length++] = c;
        lastc = c;
#endif
   }
   if (length > 0) {
      for (t = 0; t < sol_retries; t++) {
         rv = sol_send(payload, length, &rs);
	 dbglog("sol_send(%d,%d) rv=%d rs=%d\n", t,length,rv,rs.len);
         if (rv == 0) break;
         os_usleep(0,5000);  /* wait between retries 5000 us */
      }
      if (rv >= 0 && rs.len != 0) { /*handle the rsp data*/
         if (rs.type == IPMI_PAYLOAD_TYPE_SOL) {
	    dbglog("output: handler(%d)\n",rs.len);
            sol_output_handler(&rs);
	 } else
	    dbglog("WARNING: after sol_send, rs.type=%d, not SOL\n",rs.type);
      }
   }
   return(rv);
}

static int send_activate_15sol(void)
{
   int rv = 0;
   uchar actcmd;
   uchar netfn;
   uchar ibuf[64];
   uchar rbuf[64];
   int ilen, rlen;
   uchar cc;

   lan_get_sol_data(fSolEncryption,sol15_iseed,&sol15_wseed);
   actcmd  = (uchar)(ACTIVATE_SOL1 & 0x00ff); 
   netfn   = NETFN_SOL; 
   if (fSolEncryption) ibuf[0] = 0x11;   /* use encryption, activate SOL */
   else ibuf[0] = 0x01;   /*no encryptionn, no serial alerts, activate SOL */
   memcpy(&ibuf[1],&sol15_wseed,4);  /*32-bit seed*/
   ilen = 5;
   rlen = sizeof(rbuf);
   rv = ipmi_cmdraw(actcmd,netfn,BMC_SA, PUBLIC_BUS, BMC_LUN,
                   ibuf,(uchar)ilen,rbuf,&rlen,&cc,fdebug);
   if (rv == 0 && cc != 0) rv = cc;
   /* switch(cc) { case 0x80: SOL already active on another session
    *              case 0x81: SOL disabled
    *              case 0x82: SOL activation limit reached  }
    */
   if (rv == 0) {  /*success*/
      dbg_dump("sol 15 act_resp",rbuf,rlen,0);
      /* sol 15 act_resp (len=4): 0000: 00 01 64 64 */
      /* 00=auth, 01=seed_cnt, 64=in_payload_sz, 64=out_payload_sz */
      lan_set_sol_data(fSolEncryption,rbuf[0],rbuf[1], rbuf[2], rbuf[3]);
   }
   return(rv);
}

static int send_deactivate_15sol(void)
{
   int rv = -1;

   uchar actcmd;
   uchar netfn;
   uchar ibuf[64];
   uchar rbuf[64];
   int ilen, rlen;
   uchar cc;

   /* SOL for IPMI 1.5 does not have an explicit deactivate, so
    * use an activate with data0 = 00. 
    */
   actcmd  = (uchar)(ACTIVATE_SOL1 & 0x00ff); 
   netfn   = NETFN_SOL; 
   if (fSolEncryption) ibuf[0] = 0x10;  /* deactivate, use encryption */
   else ibuf[0] = 0x00;    /*deactivate*/
   memcpy(&ibuf[1],&sol15_wseed,4);  /*32-bit seed*/
   ilen = 5;
   rlen = sizeof(rbuf);
   rv = ipmi_cmdraw(actcmd,netfn,BMC_SA, PUBLIC_BUS, BMC_LUN,
                   ibuf,(uchar)ilen,rbuf,&rlen,&cc,fdebug);
   if (rv == 0 && cc != 0) rv = cc;
   dbg_dump("sol 15 deact_resp",rbuf,rlen,0); /* 80 00 32 ff */
   return(rv);
}

int send_activate_sol(uchar verSOL)
{
   int rv = 0;
   int in_payload_size, out_payload_size, udp_port;
   uchar actcmd;
   uchar netfn;
   uchar ibuf[64];
   uchar rbuf[64];
   int ilen, rlen;
   uchar cc;
   /* activate SOL = 0x01, get SOL status = 0x05 */
   if (verSOL == 1) {
      rv = send_activate_15sol();
      return(rv);
   } else {   /* use IPMI 2.0 method */
      actcmd  = ACTIVATE_PAYLOAD;
      netfn   = NETFN_APP;  
      ibuf[0] = PAYLOAD_TYPE_SOL;
      ibuf[1] = 1;        /* payload instance */
      ibuf[2] = SOL_SERIAL_ALERT_MASK_DEFERRED;
      if (fSolEncryption)     ibuf[2] |= 0x80;
      if (fSolAuthentication) ibuf[2] |= 0x40;
      if (mfg == VENDOR_INTEL) 
           ibuf[2] |= SOL_BMC_ASSERTS_CTS_MASK_TRUE;
      else ibuf[2] |= SOL_BMC_ASSERTS_CTS_MASK_FALSE;
      ibuf[3] = 0;
      ibuf[4] = 0;
      ibuf[5] = 0;
      ilen = 6;
   }
   rlen = sizeof(rbuf);
   rv = ipmi_cmdraw(actcmd,netfn,BMC_SA, PUBLIC_BUS, BMC_LUN,
                   ibuf,(uchar)ilen,rbuf,&rlen,&cc,fdebug);
   dbg_dump("sol act_cmd",ibuf,ilen,0);
   dbglog("send_activate v2(%x,%x) rv = %d cc = %x\n",actcmd,netfn,rv,cc);
   if (rv >= 0) {
      rv = cc;
      switch(cc) {
         case 0x00: 
	       if (rlen != 12) {
	         printf("Unexpected SOL response data received, len=%d\n",rlen);
	         rv = -1;
               }
               break;
         case 0x80: 
	       printf("SOL payload already active on another session\n");
               break;
         case 0x81: 
	       printf("SOL payload disabled\n");
               break;
         case 0x82: 
	       printf("SOL payload activation limit reached\n");
               break;
         case 0x83: 
	       printf("Cannot activate SOL payload with encryption\n");
               break;
         case 0x84: 
	       printf("Cannot activate SOL payload without encryption\n");
               break;
         default:
	       printf("Cannot activate SOL, ccode = 0x%02x\n",cc);
               break;
      }
   }
   if (rv == 0) {   /* success, use the response data */
      /* only here if response data is from IPMI 2.0 method */
      dbg_dump("sol act_resp",rbuf,rlen,0);
      in_payload_size  = rbuf[4] + (rbuf[5] << 8);
      out_payload_size = rbuf[6] + (rbuf[7] << 8);
      udp_port         = rbuf[8] + (rbuf[9] << 8);
      dbglog("activate ok, in=%d out=%d port=%d\n",
			in_payload_size, out_payload_size, udp_port);
      if (bSolVer == 2) 
         lan2_set_sol_data(in_payload_size, out_payload_size, udp_port,
			   (void *)sol_output_handler);
   }
   return(rv);
}

int send_deactivate_sol(uchar verSOL)
{
   int rv = 0;
   uchar deactcmd;
   uchar netfn;
   uchar ibuf[64];
   uchar rbuf[64];
   int ilen, rlen;
   uchar cc;

   if (verSOL == 1) {
      return(send_deactivate_15sol());
   } else {   /* use IPMI 2.0 method */
      deactcmd  = DEACTIVATE_PAYLOAD;
      netfn   = NETFN_APP;
      ibuf[0] = PAYLOAD_TYPE_SOL;   /* payload type */
      ibuf[1] = 1;                  /* payload instance */
      ibuf[2] = 0;
      ibuf[3] = 0;
      ibuf[4] = 0;
      ibuf[5] = 0;
      ilen = 6;
   }
   rlen = sizeof(rbuf);
   rv = ipmi_cmdraw(deactcmd,netfn,BMC_SA, PUBLIC_BUS, BMC_LUN,
                   ibuf,(uchar)ilen,rbuf,&rlen,&cc,fdebug);
   dbglog("sol deactivate rv = %d, cc = %x\n",rv,cc);
   if (rv == 0 && cc != 0) rv = cc;
   return(rv);
}

int sol_data_loop( void )
{
   int istdin;
   int rv = 0;
   uchar ibuf[IDATA_SZ];
   int szibuf;
   int fd;
   int len;
   fd_set read_fds;
   long ltime1 = 0;
   long ltime2 = 0;
#ifdef WIN32
   HANDLE thnd;
   istdin = _fileno(stdin);
#else
   istdin = fileno(stdin);
#endif

   /* 
    * OK, now loop and listen to 
    *   stdin for user input 
    *   interface fd for incoming SOL data 
    */
   szibuf = sizeof(ibuf);  /* or via intf sol data */
   fd = sol_get_fd();
   dbglog("stdin = %d, intf->fd = %d\n",istdin,fd);
   printf("[SOL session is running, use '%s' to end session.]\n",sol_esc_seq);

   tty_setraw(2);
   sol_keepalive_reset();
#ifdef WIN32
   thnd = CreateThread(NULL, 0, &input_thread, NULL, 0, NULL);
   if (thnd == NULL) { 
       printf("CreateThread error, aborting\n");
       rv = -1; 
       sol_done = 1; 
   }
   console_open(fdebug);
#endif
   if (bSolVer == 1) {   /* jump start the session v1.5 data */
      ibuf[0] = 0x1b; /*escape*/
      ibuf[1] = '{';
      rv = sol_input_handler(ibuf,2);
   }
   if (fdbglog) time(&ltime1);
		   
   while (sol_done == 0) 
   {
       if (bKeepAlive > 0)   /* send KeepAlive if needed */
           rv = sol_keepalive(bKeepAlive);   

       rv = os_select(istdin,fd, &read_fds);
       if (rv < 0) { 
	   dbglog("os_select(%d) error %d\n",fd,rv);
	   perror("select"); 
           sol_done = 1; 
       } else if (rv > 0) {
	   if (fdbglog) { time(&ltime2); } 
           dbglog("select rv = %d time=%ld\n",rv,(ltime2 - ltime1));

           if (FD_ISSET(fd, &read_fds)) {
               SOL_RSP_PKT rs;
               rs.len = 0;
               /* receive output from BMC SOL */
               rv = sol_recv(&rs);
#ifdef WIN32
		/* Windows sometimes gets rv = -1 here */
               if (rv == -1) rv = -(WSAGetLastError());
#endif
               dbglog("output: recv rv = %d\n",rv);
               if (rv < 0) {
                   dbglog("Error %d reading SOL socket\n",rv);
                   printf("Error %d reading SOL socket\n",rv);
                   sol_done = 1;
               } else {
		   sol_output_handler(&rs);
                   sol_keepalive_reset();
	       }
           }  /*endif fd*/

           if (FD_ISSET(istdin, &read_fds)) {
               /* input from stdin (user) */
               memset(ibuf,0,szibuf);
               len = os_read(istdin, ibuf, szibuf);
                dbglog("stdin: read len=%d, %02x %02x %02x\n",
					len,ibuf[0],ibuf[1],ibuf[2]);
                if (len <= 0) {
                  dbglog("Error %d reading stdin\n",len);
                  printf("Error %d reading stdin\n",len);
                  sol_done = 1;
               } else {
                  rv = sol_input_handler(ibuf,len);
                  dbglog("input: handler rv = %d\n",rv);
                  if (rv < 0) {
                      sol_done = 1;
                      if (rv == RV_END) {
		          dbglog("sol_data RV_END\n");
		          printf("\n%s exit via user input \n",progname);
                      } else printf("Error %d processing input\n",rv);
                  }
		  else sol_keepalive_reset();
               }
           }  /*endif stdin*/

       }  /*endif select rv > 0 */
   }  /*end while*/
   if (rv == RV_END) rv = 0;
#ifdef WIN32
   os_usleep(0,5000);   /*wait 5 ms for thread to close itself*/ 
   // CloseHandle(thnd);
   console_close();
#endif
   tty_setnormal(2);
   return(rv);
}

static void show_usage(void)
{
                printf("Usage: %s [-adcex -NUPREFTVY]\n", progname);
                printf(" where -a     activate SOL console session\n");
                printf("       -d     deactivate SOL console session\n");
                printf("       -c'^'  set escape Char to '^', default='~'\n");
                printf("       -e     Encryption off for SOL session\n");
                printf("       -l     Legacy mode for BIOS/DOS CR+LF\n");
#ifdef WIN32
                printf("       -r     Raw termio, no VT-ANSI translation\n");
                printf("       -w     use Windows Console buffer, not stdio\n");
#endif
                printf("       -x     show eXtra debug messages\n");
		print_lan_opt_usage();
}

#ifdef METACOMMAND
int i_sol(int argc, char **argv)
#else
#ifdef WIN32
int __cdecl
#else
int
#endif
main(int argc, char **argv)
#endif
{
   int ret = 0;
   int i;
   uchar devrec[16];
   char c;

   printf("%s ver %s\n", progname,progver);

   while ( (c = getopt( argc, argv,"ac:delrwxzEF:T:V:J:N:P:R:U:Y?")) != EOF ) 
      switch(c) {
          case 'a': factivate = 1;   break;    /*activate*/
          case 'd': fdeactivate = 1; break;    /*deactivate*/
          case 'c':     		       /* escape char */
                if (strncmp(optarg,"0x",2) == 0) 
			sol_esc_seq[0] = htoi(&optarg[2]);  /*hex char value*/
                else if (optarg[0] >= '0' && optarg[0] <= '9') 
			sol_esc_seq[0] = atoi(optarg);  /*decimal char value*/
		else    sol_esc_seq[0] = optarg[0];  /*single ascii char*/
		break;
          case 'e': fSolEncryption = 0; break; /*encryption off*/
          case 'l': fCRLF = 1; break;          /*do legacy CR/LF translation*/
          case 'r': fRaw = 1; fCRLF = 0; fUseWinCon = 0;
			break; /*raw termio, xlate off*/
          case 'w': fUseWinCon = 0;  break;    /*do not use Console routines*/
          case 'x': fdbglog = 1; fdebug = 1;  break;    /*debug messages */
          case 'z': fdbglog = 1;   break;    /*only dbglog */
          case 'N':    /* nodename */
          case 'U':    /* remote username */
          case 'P':    /* remote password */
          case 'R':    /* remote password */
          case 'E':    /* get password from IPMI_PASSWORD environment var */
          case 'F':    /* force driver type */
          case 'T':    /* auth type */
          case 'J':    /* cipher suite */ 
          case 'V':    /* priv level */
          case 'Y':    /* prompt for remote password */
                parse_lan_options(c,optarg,fdebug);
                break;
	  default:
                show_usage();
                exit(1);
      }

   if (fipmi_lan == 0) {  /* no node specified */
      printf("Serial-Over-Lan Console requires a -N nodename\n");
      exit(1);
   }

   if (factivate == 0 && fdeactivate == 0) {
      show_usage();
      printf("Error: Need either -a or -d for sol\n");
      exit(1);
   }
#ifdef WIN32
   else if (factivate) {
      system("cls");
      printf("%s ver %s\n", progname,progver);
   }
#endif
   if (fdbglog) {
      strcpy(log_name,"isoldbg.log"); 
      open_log("ipmiutil_sol");
   }

   i = get_driver_type(); /*see if user explictly set type*/
   if (i == 0) bDriver = 0;  /*no driver type specified*/
   else bDriver = i;

   ret = ipmi_getdeviceid(devrec,16,fdebug);
   if (ret != 0) {
        show_outcome(progname,ret);  
	ipmi_close_();
	exit(1);
   } else {
      ipmi_maj = devrec[4] & 0x0f;
      ipmi_min = devrec[4] >> 4;
      mfg  = devrec[6] + (devrec[7] << 8) + (devrec[8] << 16); 
      printf("-- BMC version %x.%x, IPMI version %d.%d \n",
             devrec[2],  devrec[3], ipmi_maj, ipmi_min);
      if (mfg == 0x002A7C || mfg == VENDOR_LMC || mfg == 0x28C5) /*SuperMicro*/
           sol_timeout = 10;  /* 10 sec SOL keepalive timeout */
   }

   ret = ipmi_getpicmg(devrec,16,fdebug);
   if (ret == 0) fpicmg = 1;

   dbglog("driver=%d mfg=%x\n",bDriver,mfg); 
   /* check for SOL support */
   if (ipmi_maj >= 2) { 
       if ((bDriver == DRV_LAN) && (mfg == VENDOR_INTEL)) {
          /* user specified to use IPMI LAN 1.5 SOL on Intel */
          bSolVer = 1;      /*IPMI 1.5 SOL*/
       } else {
          bSolVer = 2;      /*IPMI 2.0 SOL*/
          if (get_driver_type() == DRV_LAN) { /*now using IPMI LAN 1.5*/
            ipmi_close_();           /*close current IPMI LAN 1.5*/
            i = set_driver_type("lan2"); /*switch to IPMI LAN 2.0*/
          }
       } 
   } else if (ipmi_maj == 1) {
       if (ipmi_min >= 5) bSolVer = 1;  /* IPMI 1.5 */
       else bSolVer = 0;  /* IPMI 1.0 */
   } else bSolVer = 0;
   if (bSolVer == 0) {
       printf("Serial Over Lan not supported for this IPMI version\n");
       exit(1);
   }
#ifndef HAVE_LANPLUS
   if (bSolVer == 2) {
       printf("2.0 LanPlus module not available, trying 1.5 SOL instead\n");
       bSolVer = 1;
   }
#endif
   /* May also want to verify that SOL is implemented here */

   /*
    * Spawn a console raw terminal thread now, which will wait for the 
    * "Activating cmd (0x02)" on success 
    * Using globals: gnode,guser,gpswd, gauth_type, gpriv_level
    */
   if (fdeactivate) {
      ret = send_deactivate_sol(bSolVer); 
      dbglog("send_deactivate_sol rv = %d\n",ret);
   } else if (factivate) {
      ret = send_activate_sol(bSolVer);
      dbglog("send_activate_sol rv = %d\n",ret);
      if (ret == 0) {
         ret = sol_data_loop();
      } /*endif activate ok*/
   }  /*endif do activate */

   ipmi_close_();
   show_outcome(progname,ret); 
   if (fdbglog) close_log();
   exit (ret);
}  /* end main()*/

/* end isolconsole.c */
