/*
 * isolwin.c
 *      IPMI Serial-Over-LAN console Windows terminal emulation
 *
 * Author:  Andy Cress  arcress@users.sourceforge.net
 * Copyright (c) 2006 Intel Corporation. 
 *
 * 08/20/07 Andy Cress - extracted from isolconsole.c/WIN32,
 *                       added os_usleep to input_thread
 * 09/24/07 Andy Cress - various termio fixes, e.g. scroll(), BUF_SZ
 */
/*M*
Copyright (c) 2006-2007, Intel Corporation
All rights reserved.

Redistribution and use in source and binary forms, with or without 
modification, are permitted provided that the following conditions are met:

  a.. Redistributions of source code must retain the above copyright notice, 
      this list of conditions and the following disclaimer. 
  b.. Redistributions in binary form must reproduce the above copyright notice,
      this list of conditions and the following disclaimer in the documentation 
      and/or other materials provided with the distribution. 
  c.. Neither the name of Intel Corporation nor the names of its contributors 
      may be used to endorse or promote products derived from this software 
      without specific prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR 
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *M*/
#include <windows.h>
#include <stdio.h>
#include <winsock.h>
#include <io.h>
#include <conio.h>
#include <wincon.h>
#include <string.h>
#include <time.h>

#define uchar   unsigned char
#define RV_END   -2
#define CH_CR   '\r'    /*0x0D =='\r' CarriageReturn*/
#define CH_LF   '\n'    /*0x0A =='\n' LineFeed(Down)*/
#define CH_BELL '\007'  /*0x07 == bell */
#define CH_BS   '\010'  /*0x08 == Backspace */
#define CH_TAB  '\011'  /*0x09 == Tab */
#define CH_UP   '\013'  /*0x0B == Up Arrow (VerticalTab) */
#define CH_ESC  '\033'  /*0x1B == ASCII Escape */
#define CH_RS   '\036'  /*0x1E == RecordSeparator */
#define CH_DLE  '\020'  /*0x10 == DLE */
#define CH_CTL  '\316'  /*0xCE == VT100 Control */
#define CH_DEL  '\177'  /*0x7F == Delete key */
#define BG_WHITE (BACKGROUND_RED | BACKGROUND_GREEN | BACKGROUND_BLUE)
#define FG_WHITE (FOREGROUND_RED | FOREGROUND_GREEN | FOREGROUND_BLUE | FOREGROUND_INTENSITY)

typedef struct {
  int type;
  int len;
  char *data;
  } SOL_RSP_PKT;

extern void dbglog( char *pattn, ... ); /*from isolconsole.c*/
extern void os_usleep(int s, int u);    /*from ipmilan.c*/

/*
 * Global variables 
 */
extern int   sol_done;
extern char  fUseWinCon;
extern uchar fRaw;
extern uchar fCRLF;    /* =1 to use legacy CR/LF translation for BIOS */
extern uchar bTextMode;
static uchar fdebugsol = 0;

/*======== start Windows =============================*/
    /* Windows os_read, os_select and supporting code */
#define TERMBUF_SZ  300
// #define TERMBUF_SZ  128
#define INBUF_SZ    64
static uchar rg_stdin[INBUF_SZ];
static int   n_stdin = 0;
static HANDLE g_hConOut = INVALID_HANDLE_VALUE;
static SMALL_RECT g_ScrollRegion;
static COORD      g_coord;
static WORD       g_wAttr; /*orig Attributes*/
static WORD       g_cAttr; /*current Attributes*/
static DWORD      conmodeout, conmode1;
static int g_Rows = 25;
static int g_Columns = 80;
static int iTab  = 8;  /* tab stops every 8 chars by default */
/* extern int _cdecl _getche(void);   * see conio.h */
/* extern int _cdecl _getch(void);    * see conio.h */

#define  NESCS   53
int esc_type = 0;  
static struct { 
   int  vt_sz;
   char vt[16];
   char win[16];
   int  win_sz;
} xlate_esc[NESCS] = {
/* 0*/  3, {CH_ESC,"[K"},   { "" }, 0,  /*erase line */
/* 1*/  3, {CH_ESC,"[J"},   { "" }, 0,  /*erase screen*/
/* 2*/ 11, {CH_ESC,"[00;00m00m"}, {CH_ESC,"|00;00m" },8, /*v;h cursor position*/
/* 3*/  8, {CH_ESC,"[00;00m"},    {CH_ESC,"|00;00m" },8, /*v;h cursor position*/
/* 4*/  5, {CH_ESC,"[00m"},    { CH_ESC,"|00m" }, 0,    /*text color/mode*/
/* 5*/  6, {CH_ESC,"[0;0H"},   {CH_ESC,"|0;0H" }, 6,    /*v;h home position*/ 
/* 6*/  7, {CH_ESC,"[00;0H"},  {CH_ESC,"|00;0H" }, 7,   /*v;h home position*/
/* 7*/  7, {CH_ESC,"[00;0H"},  {CH_ESC,"|00;0H" }, 7,   /*v;h home position*/
/* 8*/  8, {CH_ESC,"[0;000H"}, {CH_ESC,"|0;000H" }, 8,  /*v;h home position*/
/* 9*/  8, {CH_ESC,"[00;00H"}, {CH_ESC,"|00;00H" }, 8,  /*v;h home position*/
/*10*/  3, {CH_ESC,"[H"},      { "" }, 0,    /* home */
/*11*/  8, {CH_ESC,"[00:00r"}, { "" }, 0,    /* set scroll region*/
/*12*/  7, {CH_ESC,"[0:00r"},  { "" }, 0,    /* set scroll region*/
/*13*/  5, {CH_ESC,"[00A"}, {CH_CTL,"H"}, 2,      /*cursor up  */
/*14*/  5, {CH_ESC,"[00B"}, {CH_CTL,"P"}, 2,      /*cursor down  */
/*15*/  5, {CH_ESC,"[00C"}, {CH_CTL,"M" }, 2,     /*cursor right  */ 
/*16*/  5, {CH_ESC,"[00D"}, {CH_CTL,"K" }, 2,     /*cursor left  */
/*17*/  3, {CH_ESC,"[A"},   {CH_CTL,"H"}, 2,      /*cursor up  */
/*18*/  3, {CH_ESC,"[B"},   {CH_CTL,"P"}, 2,      /*cursor down  */
/*19*/  3, {CH_ESC,"[C"},   {CH_CTL,"M"}, 2,      /*cursor right  */
/*20*/  3, {CH_ESC,"[D"},   {CH_CTL,"K"}, 2,      /*cursor left  */
/*21*/  3, {CH_ESC,"OA"},   {CH_CTL,"H"}, 2,      /*cursor up  */
/*22*/  3, {CH_ESC,"OB"},   {CH_CTL,"P"}, 2,      /*cursor down  */
/*23*/  3, {CH_ESC,"OC"},   {CH_CTL,"M"}, 2,      /*cursor right  */
/*24*/  3, {CH_ESC,"OD"},   {CH_CTL,"K"}, 2,      /*cursor left  */
/*25*/  6, {CH_ESC,"[0;0f"}, {CH_ESC,"|0;0f" }, 6,  /*cursor position*/
/*26*/  7, {CH_ESC,"[00;0f"}, {CH_ESC,"|00;0f" }, 7,  /*cursor position*/
/*27*/  7, {CH_ESC,"[0;00f"}, {CH_ESC,"|0;00f" }, 7,  /*cursor position*/
/*28*/  8, {CH_ESC,"[00;00f"}, {CH_ESC,"|00;00f" }, 8,  /*cursor position*/
/*29*/  4, {CH_ESC,"[0A"}, {CH_CTL,"H"}, 2,      /*cursor up  */
/*30*/  4, {CH_ESC,"[0B"}, {CH_CTL,"P"}, 2,      /*cursor down  */
/*31*/  4, {CH_ESC,"[0C"}, {CH_CTL,"M" }, 2,     /*cursor right  */ 
/*32*/  4, {CH_ESC,"[0D"}, {CH_CTL,"K" }, 2,     /*cursor left  */
/*33*/  5, {CH_ESC,"[Pm0"}, { "" }, 0,  /*char attributes*/
/*34*/  2, {CH_ESC,"E"},    { "" }, 0,  /*next line */
/*35*/  2, {CH_ESC,"="},    { "" }, 0,  /*enter alt keypad mode */
/*36*/  5, {CH_ESC,"[?0h"}, { "" }, 0,  /*  */
/*37*/  5, {CH_ESC,"[?0l"}, { "" }, 0,  /*  */
/*38*/  6, {CH_ESC,"[?00h"}, { "" }, 0,  
/*39*/  6, {CH_ESC,"[?00l"}, { "" }, 0,  
/*40*/  3, {CH_ESC,"0h"}, { "" }, 0,    /*  */
/*41*/  3, {CH_ESC,"0l"}, { "" }, 0,    /*  */
/*42*/  4, {CH_ESC,"00h"}, { "" }, 0,  
/*43*/  4, {CH_ESC,"00l"}, { "" }, 0,  
/*44*/  3, {CH_ESC,"[m"},  { "" }, 0,   /*normal text mode*/
/*45*/  4, {CH_ESC,"[0m"}, { "" }, 0,   /*text mode, 0=normal 1=bold*/
/*46*/  4, {CH_ESC,"[0J"}, { "" } , 0,  /*clear screen, etc */
/*47*/  4, {CH_ESC,"[0K"}, { "" } , 0,  /*clear line, etc */
/*48*/  4, {CH_ESC,"[0L"}, { "" } , 0,  /*delete lines */
/*49*/  4, {CH_ESC,"[0M"}, { "" } , 0,  /*insert lines */
/*50*/  3, {CH_ESC,"(0"},   { "" }, 0,  /*character set*/
/*51*/  2, {CH_ESC,"<"},    { "" }, 0,  /*exit alt keypad*/  
/*52*/  2, {CH_ESC,">"},    { "" }, 0   /*enter ansi keypad mode*/
};

static int goto_xy( int x, int y);

static char *fmtstr(uchar *p, int sz)
{
   static char str_tmp[100];
   char *pret;
   if (sz >= (sizeof(str_tmp) - 1)) sz = sizeof(str_tmp) - 1;
   memcpy(str_tmp,p,sz);
   str_tmp[sz] = 0;  /*stringify*/
   pret = str_tmp;
   return(pret);
}

static int EscMatch(uchar *p1, uchar *p2, int len)
{
   int i;
   int rv = 0;
   for (i = 0; i < len; i++) {
      if (p1[i] == p2[i]) continue;
      else if ((p1[i] >= '0') && (p1[i] <= '9') && 
               (p2[i] >= '0') && (p2[i] <= '9')) continue;
      else break;
   }
   if (i >= len) rv = 1;
   return(rv);
}

#define NCTLS  7
static int IsCtl(uchar c)
{
   char ctls[NCTLS] = {CH_LF, CH_CR, CH_DEL, CH_BS, CH_TAB, CH_CTL, CH_DEL};
   int i;
   for (i = 0; i < NCTLS; i++)
      if (c == ctls[i]) break;
   if (i < NCTLS) return(1);
   if (c == CH_ESC) {  /* error if here, should be handled via IsVtEsc */
      dbglog("IsCtl(%02x) ESC not handled\n",c);
      return(1);
   }
   if (c < 0x20) return(1);
   else return(0);
}


static int IsVtEsc(uchar c, uchar *buf, int idx)
{
   int ret = 0;
   uchar tmpbuf[20];
   int i;

   /* add this char to the esc sequence for matching */
   memcpy(tmpbuf,buf,idx);
   tmpbuf[idx] = c;
   idx++;
   if (idx >= sizeof(tmpbuf)) { dbglog("esc idx>MAX(%d)\n",idx); }
   tmpbuf[idx] = 0;  /*stringify*/
   /* see if it matches a known VT esc sequence */
   for (i = 0; i < NESCS; i++) {
      if (c == CH_ESC) {
	  ret = 1; 
	  if (idx != 1) {  /* need to restart esc sequence */
	     dbglog("new esc[%d]: %s\n",idx,&tmpbuf[1]);
	     ret = 3; 
          }
          break; 
      } else if (EscMatch(xlate_esc[i].vt,tmpbuf,idx)) { /*matches*/
          ret = 1;
          if (idx >= xlate_esc[i].vt_sz) {
             esc_type = i;
             ret = 2; /*complete*/
	     dbglog("esc match, type[%d]: %s\n",esc_type,&tmpbuf[1]);
             if (fdebugsol) 
	       printf("esc match, type[%d]: %s\n",esc_type,&tmpbuf[1]);
             break;
          }
      } 
   }  /*end NESCS loop*/
   if ((idx > 1) && (ret == 0)) {
      if (fdebugsol) 
	  printf("no match, esc seq(%d): %s, c=%c\n",idx,&tmpbuf[1],c);
      dbglog("no match, esc seq(%d): %s, c=%c (%02x)\n",idx,&tmpbuf[1],c,c);
   }

   return(ret);
}

static void console_write_chars(uchar *buf, int sbuf)
{
   DWORD written = 0;
   uchar c;
   int i;

   if (g_hConOut != INVALID_HANDLE_VALUE) {
      if (sbuf > 0) {  /* skip this if zero bytes */
#ifdef OLD
        int rv;
        /* Normally Unicode, but WriteConsoleOutputCharacterA is ANSI */
        /* rv = WriteConsoleOutputAttribute(g_hConOut,&g_cAttr,sbuf,g_coord,&written); */
        rv = WriteConsoleOutputCharacterA(g_hConOut,buf,sbuf,g_coord,&written);
        if (rv == 0 || written == 0) written = 1;
        // buf[sbuf] = 0;  /*stringify*/
        // printf(buf);    /*stdio uses attributes also*/
#endif
        written = fwrite(buf,sbuf,1,stdout);
        dbglog("write_chars (%d+%d=%d) x,y=(%d,%d), buf: 0x%02x, %s\n",
	       g_coord.X,sbuf,(g_coord.X+sbuf), g_coord.X,g_coord.Y,
		buf[0], fmtstr(buf,sbuf));
        g_coord.X += sbuf;
        /* do not let cursor position go beyond scroll region bounds */
        while (g_coord.X > g_ScrollRegion.Right) {
            g_coord.X -= g_Columns;
            if (g_coord.Y < g_ScrollRegion.Bottom) g_coord.Y++;
        }
        dbglog("write_chars new position = (%d,%d)\n",g_coord.X,g_coord.Y);
        SetConsoleCursorPosition(g_hConOut, g_coord);
      }
   } else {  /*do stdout method*/
      for (i = 0; i < sbuf; i++) {
          c = buf[i];
          /* do special CH_CR & CH_BS handling here? */
          putc(c, stdout);
      }
      fflush(stdout);
   }
}

static void console_get_coord(void)
{
   CONSOLE_SCREEN_BUFFER_INFO csbi;
   /* leave unchanged if any errors */
   if (g_hConOut != INVALID_HANDLE_VALUE) {
      if (GetConsoleScreenBufferInfo(g_hConOut, &csbi)) {
          g_ScrollRegion = csbi.srWindow;
          g_coord.X = csbi.dwCursorPosition.X;
          g_coord.Y = csbi.dwCursorPosition.Y;
          g_wAttr   = csbi.wAttributes;
          g_cAttr   = g_wAttr;
      }
      dbglog("console_get_coord: (%d,%d) max=(%d,%d)\n", 
			g_coord.X, g_coord.Y , g_Columns, g_Rows );
   }
}

static int delete_lines(unsigned int n)
{
    SMALL_RECT src;
    COORD      dst;
    CHAR_INFO  filler;
    DWORD dwDummy;
    int cnt, i;

    if (g_hConOut == INVALID_HANDLE_VALUE) return(-1);
    dst.X = 0;
    dst.Y = g_coord.Y;
    src.Left   = 0;
    src.Right  = g_ScrollRegion.Right;
    src.Top    = g_coord.Y + n;
    src.Bottom = g_ScrollRegion.Bottom;
    filler.Char.AsciiChar = ' ';
    filler.Attributes = g_cAttr;  /*current attribute*/
    ScrollConsoleScreenBuffer(g_hConOut, &src, NULL, dst, &filler);
    cnt = dst.Y + (src.Bottom - src.Top) + 1;
    if (cnt < src.Top) {
        COORD coord1;
        coord1.X = 0;
        coord1.Y = cnt;
        i = g_Columns * (src.Top - cnt);
        FillConsoleOutputCharacterA(g_hConOut,' ',i,coord1, &dwDummy);
    }
    return(0);
}

static int insert_lines(unsigned int n)
{
    SMALL_RECT src;
    COORD      dst;
    CHAR_INFO  filler;
    DWORD dwDummy;
    int cnt, i;

    if (g_hConOut == INVALID_HANDLE_VALUE) return(-1);
    dst.X = 0;
    dst.Y = g_coord.Y + n;
    src.Left   = 0;
    src.Right  = g_ScrollRegion.Right;
    src.Top    = g_coord.Y;
    src.Bottom = g_ScrollRegion.Bottom - n;
    filler.Char.AsciiChar = ' ';
    filler.Attributes = g_cAttr;  /*current attribute*/
    ScrollConsoleScreenBuffer(g_hConOut, &src, NULL, dst, &filler);
    cnt = dst.Y + (src.Bottom - src.Top) + 1;
    if (src.Bottom < dst.Y) {
        COORD coord1;
        coord1.X = 0;
        coord1.Y = src.Bottom;
        i = g_Columns * (dst.Y - src.Bottom);
        FillConsoleOutputCharacterA(g_hConOut,' ',i,coord1, &dwDummy);
    }
    return(0);
}

static void scroll(unsigned int nLines)
{
    COORD coord0;
    coord0 = g_coord;
    goto_xy(g_ScrollRegion.Left, g_ScrollRegion.Top);
    delete_lines(nLines);
    g_coord = coord0;
}

static int goto_xy( int x, int y)
{
    if (g_hConOut == INVALID_HANDLE_VALUE) return(-1);
    dbglog("goto_xy(%d,%d)\n",x,y);
    if (y < 0) y = 0;
    if (x < 0) x = 0;
    if (x >= g_Columns) x = g_Columns; /*or x >= Right */
    // if (y >= Bottom) { scroll(1); }
    /* zero-based coordinates */
    g_coord.X = x;
    g_coord.Y = y;
    SetConsoleCursorPosition(g_hConOut, g_coord);
    return(0);
}

static int goto_x( int x )
{
    if (g_hConOut == INVALID_HANDLE_VALUE) return(-1);
    dbglog("goto_x(%d,%d)\n",x,g_coord.Y);
    if (x < 0) x = 0;
    if (x >= g_Columns) x = g_Columns;
    g_coord.X = x;
    // same g_coord.Y 
    SetConsoleCursorPosition(g_hConOut, g_coord);
    return(0);
}

static int goto_nextline( void )
{
    if (g_hConOut == INVALID_HANDLE_VALUE) return(-1);
    /* zero-based coordinates */
    g_coord.X = 0;
    dbglog("goto_nextline(%d,%d)\n", g_coord.X, g_coord.Y);
    if (g_coord.Y >= g_ScrollRegion.Bottom) {
	scroll(1);
    } else {
        g_coord.Y += 1;
    }
    SetConsoleCursorPosition(g_hConOut, g_coord);
    return(0);
}

static int console_backspace( void )
{
    DWORD dwDummy;
    if (g_hConOut == INVALID_HANDLE_VALUE) return(-1);
    if (g_coord.X > 0) {
       g_coord.X--;
       SetConsoleCursorPosition(g_hConOut, g_coord);
       FillConsoleOutputCharacterA(g_hConOut,' ',1,g_coord, &dwDummy);
       dbglog("backspace x=%d, y=%d\n", g_coord.X,g_coord.Y);
    }
    return(0);
}


static int clear_line( void)
{
    DWORD dwDummy;
    unsigned int n;
    if (g_hConOut == INVALID_HANDLE_VALUE) return(-1);
    else {
       dbglog("clear_line(x=%d, y=%d)\n", g_coord.X,g_coord.Y);
       n = g_Columns - g_coord.X;
       FillConsoleOutputCharacterA(g_hConOut,' ',n,g_coord, &dwDummy);
    } 
    return(0);
}

static int clear_screen( int opt)
{
    DWORD dwDummy;
    unsigned int n, x, y;
		
    if (g_hConOut == INVALID_HANDLE_VALUE) return(-1);
    else {
       x = g_coord.X;
       y = g_coord.Y;
       dbglog("clear_screen(%d), was x=%d, y=%d\n", opt,x,y);
       if (opt == 0) {        /* [0J =erase here to bottom */
          n = (g_Columns * g_Rows) - ((x+1) * (y+1));
       } else if (opt == 1) { /* [1J =erase top to here  */
          g_coord.X = 0;
          g_coord.Y = 0;
          SetConsoleCursorPosition(g_hConOut, g_coord);
          n = ((x+1) * (y+1));
       } else {               /* [2J =erase entire screen */
          g_coord.X = 0;
          g_coord.Y = 0;
          SetConsoleCursorPosition(g_hConOut, g_coord);
          n = g_Columns * g_Rows;
	  x = 0;
          y = 0;
       }
       FillConsoleOutputCharacterA(g_hConOut,' ',n,g_coord, &dwDummy);
       g_coord.X = x;
       g_coord.Y = y;
       SetConsoleCursorPosition(g_hConOut, g_coord);
    }
    return(0);
}

static void parse_xy(uchar *buf, int sbuf, int *x, int *y)
{
    int r,c;    /*row, col*/
    char *p;
    /* Input buf is a VT100 esc sequence, like {ESC,"[00:00H"} 
     * So, r comes from buf[2,3], and c is usu buf[5,6].  */
    c = 0;
    r = 0;
    p = strchr(buf,';');
    buf[sbuf-1] = 0;  /*end at H char*/
    if (p != NULL) { 
	p[0] = 0; 
        r = atoi(&buf[2]);
	p++; 
	c = atoi(p);
    }
    else c = atoi(&buf[2]);  /*assign x to the only value*/
    *x = c;  /*x is the column*/
    *y = r;  /*y is the row*/
}

static void console_write_esc(int itype, uchar *ebuf, int elen, 
				uchar *obuf, int *olen)
{
     int n, len; 
     int x,y;
     WORD bAttr;

     /* For this vt esc, output valid ANSI */
     n = *olen;
     len = xlate_esc[itype].win_sz;
     switch (itype) {
           case 0:   /* "[K" clear line */
           case 47:  /* "[2K" clear line */
                parse_xy(ebuf,elen,&x,&y);
		if (clear_line() != 0) 
                    obuf[n++] = '\b';  /*0x08,backspace*/
                break;
           case 1:   /* "[J" clear screen*/
           case 46:  /* "[2J" clear screen */
                parse_xy(ebuf,elen,&x,&y);
		if (clear_screen(x) != 0) {
		   obuf[n++] = '\f';  /* 0x0c, FormFeed*/
                }
                break;
           case 2:
           case 3:  /*"[00:00m" text mode */
                parse_xy(ebuf,elen,&x,&y);
                dbglog("%d: esc,[00:00m text mode (%d,%d)\n", itype,x,y);
                break;
           case 4:  /* "[00m" text/background color change */
                parse_xy(ebuf,elen,&x,&y);
		switch(x) {
		   case 30: /*black text*/
		      bAttr = 0;  
		      break;
		   case 31: /*red text*/
		      bAttr = FOREGROUND_RED;
		      break;
		   case 32: /*green text*/
		      bAttr = FOREGROUND_GREEN;
		      break;
		   case 33: /*yellow text*/
		      bAttr = FOREGROUND_RED | FOREGROUND_GREEN | FOREGROUND_INTENSITY;
		      break;
		   case 34: /*blue text*/
		      bAttr = FOREGROUND_BLUE;
		      break;
		   case 35: /*magenta text*/
		      bAttr = FOREGROUND_RED | FOREGROUND_BLUE;
		      break;
		   case 36: /*cyan text*/
		      bAttr = FOREGROUND_GREEN | FOREGROUND_BLUE;
		      break;
		   case 37: /*white text*/
		      bAttr = FG_WHITE;
		      break;
		   case 40: /*black background*/
                      bAttr = 0;
		      break;
		   case 41: /*red background*/
		      bAttr = BACKGROUND_RED;
		      break;
		   case 42: /*green background*/
		      bAttr = BACKGROUND_GREEN;
		      break;
		   case 43: /*yellow background*/
		      bAttr = BACKGROUND_RED | BACKGROUND_GREEN | BACKGROUND_INTENSITY;
		      break;
		   case 44: /*blue background*/
		      bAttr = BACKGROUND_BLUE;
		      break;
		   case 45: /*magenta background*/
		      bAttr = BACKGROUND_RED | BACKGROUND_BLUE;
		      break;
		   case 46: /*cyan background*/
		      bAttr = BACKGROUND_GREEN | BACKGROUND_BLUE;
		      break;
		   case 47: /*white background*/
		      bAttr = BG_WHITE;
		      break;
		   case 0:
		      bAttr = g_wAttr;  /*default colors*/
		      break;
		   default:
		      bAttr = FG_WHITE;
		      break;
		}
                if (x < 40) {  /*foreground*/
                   g_cAttr &= 0xf0;    /*preserve bg*/
                } else {       /*background*/
                   g_cAttr &= 0x0f;    /*preserve fg*/
                }
                g_cAttr |= bAttr;
                dbglog("%d: [00m color (%d) cAttr=%02x\n", itype,x,g_cAttr);
                SetConsoleTextAttribute(g_hConOut, g_cAttr); 
                break;
           case 45:  /* "[0m" text mode change */
                parse_xy(ebuf,elen,&x,&y);
		bTextMode = x;
		switch(bTextMode) {
		   case 0: /*Normal*/
		      g_cAttr &= ~(FOREGROUND_INTENSITY | BACKGROUND_INTENSITY);
                      break;
		   case 1: /*Bold*/
		      g_cAttr |= (FOREGROUND_INTENSITY | BACKGROUND_INTENSITY);
                      break;
		   case 4: /*Underlined*/
		      g_cAttr |= 0x8000; /* COMMMON_LVB_UNDERSCORE */
		      break;
		   case 5: /*Blinking*/
		      break;
		   case 7: /*Reverse Video*/
		      g_cAttr |= 0x4000; /* COMMMON_LVB_REVERSE_VIDEO */
		      break;
		   default:  /*other, do nothing*/
		      break;
                }
                dbglog("%d: text mode changed to %d, cAttr=%02x\n", 
			itype,bTextMode,g_cAttr);
                SetConsoleTextAttribute(g_hConOut, g_cAttr);
                break;
           case 5:  /*cursor positioning "[x;yH"*/
           case 6:  /*cursor positioning "[x;yH"*/
           case 7:  /*cursor positioning "[x;yH"*/
           case 8:  /*cursor positioning "[x;yH"*/
           case 9:  /*cursor positioning "[x;yH"*/
                parse_xy(ebuf,elen,&x,&y);
		if (x > 0) x--;  /*esc is 1-based, goto is 0-based*/
		if (y > 0) y--;
		if (goto_xy(x,y) != 0) {
                   obuf[n++] = '\n';  /* 0x0a, LineFeed*/
                }
                break;
           case 10:  /* "[H" Home */
		dbglog("Home from (%d,%d)\n",g_coord.X,g_coord.Y);
		goto_xy(0,0);  /*default home is 0,0*/
                break;
           case 13:  /* "[00A" up*/
                parse_xy(ebuf,elen,&x,&y);
		y = g_coord.Y - x;
	        if (y < 0) y = 0;
		goto_xy(g_coord.X,y); 
                break;
           case 14:  /* "[00B" down*/
                parse_xy(ebuf,elen,&x,&y);
	        y = g_coord.Y + x;
		goto_xy(g_coord.X,y); 
                break;
           case 15:  /* "[00C" right*/
                parse_xy(ebuf,elen,&x,&y);
	        x = (g_coord.X + x);
		goto_xy(x,g_coord.Y); 
                break;
           case 16:  /* "[00D" left*/
                parse_xy(ebuf,elen,&x,&y);
		if (g_coord.X > 0) x = (g_coord.X - x);
		goto_xy(x,g_coord.Y); 
                break;
           case 17:  /* "[A" up*/
           case 21:  /* "OA" up*/
           case 29:  /* "[0A" up*/
		goto_xy(g_coord.X,(g_coord.Y-1)); 
                break;
           case 18:  /* "[B" down*/
           case 22:  /* "OB" down*/
           case 30:  /* "[0B" down*/
		goto_xy(g_coord.X,(g_coord.Y+1)); 
                break;
           case 19:  /* "[C" right*/
           case 23:  /* "OC" right*/
           case 31:  /* "[0C" right*/
		goto_xy((g_coord.X + 1),g_coord.Y); 
                break;
           case 20:  /* "[D" left*/
           case 24:  /* "OD" left*/
           case 32:  /* "[0D" left*/
		x = g_coord.X;
		if (x > 0) x--;
		goto_xy(x,g_coord.Y); 
                break;
           case 25:  /* "[x;yf"*/
           case 26:  /* "[x;yf"*/
           case 27:  /* "[x;yf"*/
           case 28:  /* "[x;yf"*/
                parse_xy(ebuf,elen,&x,&y);
                dbglog("%d: esc,[00:00f (%d,%d)\n", itype,x,y);
		if (x > 0) x--;  /*esc is 1-based, goto is 0-based*/
		if (y > 0) y--;
		if (goto_xy(x,y) != 0) {
                   obuf[n++] = '\b';  /* 0x08, backspace */
                }
                break;
           case 44:  /* "[m" normal text mode */
                dbglog("%d: esc,[m normal text mode\n", itype);
                g_cAttr = g_wAttr;  /*default/normal text mode*/
                SetConsoleTextAttribute(g_hConOut, g_cAttr);
                break;
           case 35:   /* "=" enter alt keypad mode (Application)*/
           case 51:   /* "<" exit alt keypad mode */
           case 52:   /* ">" enter ansi keypad mode (Numeric) */
                dbglog("%d: change keypad mode\n",itype);
                break;
	   /*TODO: add more here */
           default:
                dbglog("%d: default esc, no action at (%d,%d)\n", itype,
			g_coord.X,g_coord.Y);
                ; /*do nothing by default*/
     }
     *olen = n;
}

static void xlate_term(uchar *pin, int ilen, uchar *pout, int *olen)
{
    int i, j, k, l;
    static int iesc = 0;
    static uchar escbuf[20];
    static uchar outtmp[100];
    uchar *pdata;
    int n;

    /* Translate vt100 escape sequences to ansi here. */
    j = 0;   /*index into pout*/
    l = 0;   /*length of pdata*/
    pdata = pout;
    dbglog("xlate_term(%d,%d), iesc=%d inbuf[%d]: %s\n",g_coord.X,g_coord.Y,
		iesc,ilen,fmtstr(pin,ilen));
    for (i = 0; i < ilen; i++) {
       if (j >= TERMBUF_SZ) break;
       k = IsVtEsc(pin[i],escbuf,iesc);
       if (k > 0) {  /*vt100 esc sequence*/
          if (k == 3) iesc = 0;    /*flush any existing seq and start fresh */
          escbuf[iesc++] = pin[i]; /*add pending char to escbuf */
          if (k == 2) {    /*sequence complete, flush and start over*/
             escbuf[iesc]   = 0;   /*stringify*/
             console_write_chars(pdata,l);  /*write what we have so far*/
             l = 0;
             pdata = &pout[j];
             console_write_esc(esc_type,escbuf,iesc,outtmp,&l); /*escape seq*/
             iesc = 0;
             j += l;
          }
       } else {  /*standard chars pass through*/
          iesc = 0;
          if ((fUseWinCon) && IsCtl(pin[i])) {
             console_write_chars(pdata,l);  /*write what we have so far*/
             pdata = &pout[j];
             l = 0;
             switch(pin[i]) {
               case CH_CR: 
                 dbglog("ctl CR\n");
		 // goto_x(0);
		 break;
               case CH_LF: 
                 dbglog("ctl LF\n"); 
		 goto_nextline();  break;
               case CH_BS: 
                 dbglog("ctl BS\n"); 
		 console_backspace();  break;
               case CH_DEL:  /* same as BS? */
		 dbglog("ctl DEL, skipped\n");
		 break;
               case CH_TAB:  
		 dbglog("ctl TAB\n");
		 n = (g_coord.X % iTab);
		 goto_x(g_coord.X + (iTab-n));
	       default:
		 dbglog("ctl %02x, skipped\n",pin[i]);
                 // pout[j++] = pin[i];
		 break;
             }
          } else {
             pout[j++] = pin[i];
             l++;
          }
       }
    }
   
    dbglog("remainder write_chars[%d,%d] iesc=%d (%d,%d): %s\n",
		g_coord.X,g_coord.Y, iesc, l,j,fmtstr(pdata,l));
    console_write_chars(pdata,l);  /*write the rest*/
    *olen = j;
    return;
}

/*
 * console_in: * called from sol_input_handler for Windows console
 */
int console_in(uchar c, uchar c1, uchar *pdata, int len)
{
   int i = 0;
   if (pdata == NULL || len < 1) return(i);
   if (!fRaw) {
       if (fUseWinCon) {
          if (c == CH_CR) c = CH_LF;
       } else {
          if (c == CH_CR) { /*0x0D = '\r' from Windows Enter*/
              if (fCRLF) pdata[i++] = c;  /*insert CH_CR*/
              c = CH_LF;    /*0x0A = '\n' newline */
          }
       }
   }
   return(i);
}

/*
 * console_out: * called from sol_output for Windows console
 */
void console_out(uchar *pdata, int len)
{
         uchar buf[TERMBUF_SZ];
         int blen;
 
         if (fRaw) {
            console_write_chars(pdata,len);
         } else {
            xlate_term(pdata,len,buf,&blen);
         }
         dbglog("console_out: inlen=%d outlen = %d\n",len,blen);
}

void console_open(uchar fdebugcmd)
{
   fdebugsol = fdebugcmd;
   if (fUseWinCon && !fRaw) 
      g_hConOut = GetStdHandle(STD_OUTPUT_HANDLE); 
   GetConsoleMode(g_hConOut,&conmodeout);
   conmode1 = conmodeout;
   conmodeout |= ENABLE_WRAP_AT_EOL_OUTPUT;
             /* (ENABLE_WRAP_AT_EOL_OUTPUT | ENABLE_PROCESSED_OUTPUT); */
   SetConsoleMode(g_hConOut,conmodeout);
   g_coord.X = 0;
   g_coord.Y = 0;
   console_get_coord();
}

void console_close(void)
{
   if (g_hConOut != INVALID_HANDLE_VALUE) {
      SetConsoleMode(g_hConOut,conmode1);
      SetConsoleTextAttribute(g_hConOut, g_wAttr); /* or 0x00 */
      CloseHandle(g_hConOut);
      g_hConOut = INVALID_HANDLE_VALUE;
   }
}

DWORD WINAPI input_thread( LPVOID lpParam) 
{
   // HANDLE hstdin;
   DWORD rv = 0;
   int c;

   // hstdin = GetStdHandle(STD_INPUT_HANDLE); 
   while (!sol_done)
   {
      c = _getch();   /*was getc(stdin) or _getche() */
      if (c != EOF) {
         if (n_stdin < sizeof(rg_stdin)) {
	    if (c < 0x20 || c > 0x7A) 
	        dbglog("rg_stdin[%d] = %02x\n",n_stdin,c);
            rg_stdin[n_stdin] = c;
            n_stdin++;
         } 
      } 
      else {   /*EOF*/
         os_usleep(0,5000);   /*wait 5 ms*/
      }
   }
   return(rv);
}

int os_read(int fd, uchar *buf, int sz)
{
    int len = 0;
    int i;
    if ((fd == 0) && (n_stdin > 0)) {
       for (i = 0; i < n_stdin; i++)
          buf[i] = rg_stdin[i];
       len = n_stdin;
       n_stdin = 0;
    } else {
       // ReadFile(fd, buf, sz, &len, NULL);
       // len = _read(fd, buf, sz);
       len = recv(fd, buf, sz, 0);
    }
    return(len);
}

int os_select(int infd, int sfd, fd_set *read_fds)
{
   int rv = 0;
   struct timeval tv;
   fd_set err_fds;
   int rv0 = -1;

   /* check the socket for new data via select */
   /* Windows select only works on WSA sockets */
   {
      FD_ZERO(read_fds);
      FD_SET(sfd, read_fds);
      FD_ZERO(&err_fds);
      tv.tv_sec =  0;
      tv.tv_usec = 50000; /* 50 ms */
      rv = select(sfd + 1, read_fds, NULL, NULL, &tv);
      if (rv < 0) {
         rv = -(WSAGetLastError());
         if (fdebugsol) printf("select(%d) error %d\n",sfd,rv);
      }
      else if (FD_ISSET(sfd, &err_fds)) {
         if (fdebugsol) printf("select(%d) err_fds rv=%d err=%d\n",sfd, rv, 
			 WSAGetLastError());
      }
      if (rv > 0 && fdebugsol) printf("select(%d) got socket data %d\n",sfd,rv);
   }  
   if (infd == 0) {  /* check stdin */
      rv0 = -1;
      if (n_stdin > 0) rv0 = 0;
   }
   if (rv0 == 0) {
      FD_SET(infd, read_fds);
      if (rv < 0) rv = 1;
      else rv++;
   }
   return rv;
}
/*======== end Windows ===============================*/


/*end solwin.c*/
