/*---------------------------------------------------------------------------
 * Filename:   xmlconfig.c
 *
 * Author:     arcress@users.sourceforge.net
 * Copyright (c) 2005 Intel Corporation. 
 *
 * Abstract:
 * This utility reads IPMI LAN and Serial parameters and outputs XML
 * suitable for SysCon use.
 *
 * ----------- Change History -----------------------------------------------
 * 02/14/05 Andy Cress - created from pefconfig.c, prototype
 * Note that this would need to be re-ported from pefconfig.c if xmlconfig
 * were to be released.
 */
/*M*
 *---------------------------------------------------------------------------
Copyright (c) 2005, Intel Corporation
All rights reserved.

Redistribution and use in source and binary forms, with or without 
modification, are permitted provided that the following conditions are met:

  a.. Redistributions of source code must retain the above copyright notice, 
      this list of conditions and the following disclaimer. 
  b.. Redistributions in binary form must reproduce the above copyright notice,
      this list of conditions and the following disclaimer in the documentation 
      and/or other materials provided with the distribution. 
  c.. Neither the name of Intel Corporation nor the names of its contributors 
      may be used to endorse or promote products derived from this software 
      without specific prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR 
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *---------------------------------------------------------------------------
 *M*/
#ifdef WIN32
#include <windows.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "getopt.h"
#else
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <getopt.h>
#include <string.h>
#include <sys/socket.h>    /* for Get_IPMac_Addr() */
#include <sys/ioctl.h>     /* for Get_IPMac_Addr() */
#include <net/if.h>        /* for Get_IPMac_Addr() */
#include <errno.h>
#endif
#include "ipmicmd.h" 
 
extern int errno;
#define SELprintf          printf
#define RTF_UP          0x0001   /* route usable */

#define SOL_ENABLE_PARAM		0x01
#define SOL_AUTHENTICATION_PARAM 	0x02
#define SOL_ENABLE_FLAG			0x01
#define SOL_PRIVILEGE_LEVEL_USER	0x02
#define SOL_BAUD_RATE_PARAM		0x05
#define SOL_PREFERRED_BAUD_RATE		0x07

/* Channel Access values */
#define CHAN_ACC_DISABLE   0x20   /* PEF off, disabled*/
#define CHAN_ACC_PEFON     0x02   /* PEF on, always avail */
#define CHAN_ACC_PEFOFF    0x22   /* PEF off, always avail*/
/* special channel access values for ia64 */
#define CHAN_ACC_PEFON64   0x0A   /* PEF on, always avail, UserLevelAuth=off */
#define CHAN_ACC_PEFOFF64  0x2A   /* PEF off, always avail, UserLevelAuth=off */

   /* Channels: 0=IPMB, 1=Serial/EMP, 6=LAN2, 7=LAN1 */
   /* for Torrey Pines, LAN channel = 1 instead, see lan_ch below */
#define LAN_CH   7
#define SER_CH   1
#define MAXCHAN  12  /*was 16, reduced for gnu ipmi_lan*/
#define NUM_DEVICES_TO_CHECK		32   /*for GetBmcEthDevice()*/

typedef struct
{            /* See IPMI Table 15-2 */
        uchar              rec_id;
        uchar              fconfig;
        uchar              action;
        uchar              policy;
        uchar              severity;
        uchar              genid1;
        uchar              genid2;
        uchar              sensor_type;
        uchar              sensor_no;
        uchar              event_trigger;
        uchar              data1;
        uchar              mask1;
        uchar              res[9];
}       PEF_RECORD;

typedef struct
{            /* See IPMI Table 19-3 */
        uchar              data[30];
}       LAN_RECORD;

/*
 * Global variables 
 */
static char * progver   = "0.20";
static char * progname  = "xmlconfig";
static char   fdebug    = 0;
static char   fIPMI10   = 0;      /* =1 if IPMI v1.0 or less */
static char   fIPMI20   = 0;      /* =1 if IPMI v2.0 or greater */
static char   fpicmg    = 0;
static char   freadonly = 1;
static char   fdisable  = 0;
static char   fenable   = 0;      /* =1 to config BMC LAN and PEF */
static char   fgetser   = 1;
static char   fsetifn   = 0;	  /* =1 if user specified ifname */
static char   fpassword = 0;	  /* =1 user-specified a password, so set it. */
static char   pefnum    = 0x0c;   /* 11 pre-defined entries, adding 12th */
static char   pefmax    = 20;     /* 20 for Sahalee, 30 for miniBMC */
static char  *myuser    = NULL;   /* username to set, specified by -u */
static uchar  rgmyip[4]   = {0,0,0,0};
static uchar  rggwyip[4]  = {0,0,0,0};
static uchar  rgdestip[4] = {0,0,0,0};
static uchar  rgsubnet[4] = {0,0,0,0};  /* default subnet, i.e. 255.255.255.0 */
static uchar  rgmymac[6]   = {0xFF,0,0,0,0,0};
static uchar  rggwymac[6]  = {0xFF,0,0,0,0,0};
static uchar  rgdestmac[6] = {0xFF,0,0,0,0,0};
static char   rgcommunity[19] = "public";  /* default community */
static char   fsetcommunity = 0;    /* =1 if user-specified community */
static char   passwordData[17] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0}; 
static uchar  authmask      = 0;  /* usu 0x17, mBMC = 0x15 */
static uchar  lan_access    = 0;  /* usu 0x04 */
static uchar  fmBMC         = 0;  
static char   ifname[16]    = "eth0";
static uchar  lan_ch        = LAN_CH;
static int    lan_dhcp      = 0;
static uchar  chan_pefon    = CHAN_ACC_PEFON;
static uchar  chan_pefoff   = CHAN_ACC_PEFOFF;
#define MAXPEF 31	 	/* max pefnum offset = 30 (31 entries) */
#define MAX_PEFPARAMS  14	/* max pef params = 14 */
static uchar peflen[MAX_PEFPARAMS] = {0,1,1,1,1,1,21,2,1,4,17,1,3,18}; 
static uchar pef_array[MAXPEF][21];    /* array of all PEF entries read, */
                                /* sizeof(PEF_RECORD) = 21  */
static uchar pef_defaults[11][21] = {  /*array of first 11 default PEF entries*/
0x01,0x80,1,1,0,0xff,0xff,0x01,0xff,0x01,0x95,0x0a,0,0,0,0,0,0,0,0,0, 
0x02,0x80,1,1,0,0xff,0xff,0x02,0xff,0x01,0x95,0x0a,0,0,0,0,0,0,0,0,0, 
0x03,0x80,1,1,0,0xff,0xff,0x04,0xff,0x01,0x95,0x0a,0,0,0,0,0,0,0,0,0, 
0x04,0x80,1,1,0,0xff,0xff,0x05,0x05,0x03,0x01,0x00,0,0,0,0,0,0,0,0,0, 
0x05,0x80,1,1,0,0xff,0xff,0x08,0xff,0x6f,0x06,0x00,0,0,0,0,0,0,0,0,0, 
0x06,0x80,1,1,0,0xff,0xff,0x0c,0x08,0x6f,0x02,0x00,0,0,0,0,0,0,0,0,0, 
0x07,0x80,1,1,0,0xff,0xff,0x0f,0x06,0x6f,0x01,0x00,0,0,0,0,0,0,0,0,0, 
0x08,0x80,1,1,0,0xff,0xff,0x07,0xff,0x6f,0x1c,0x00,0,0,0,0,0,0,0,0,0, 
0x09,0x80,1,1,0,0xff,0xff,0x13,0xff,0x6f,0x3e,0x03,0,0,0,0,0,0,0,0,0, 
0x0a,0x80,1,1,0,0xff,0xff,0x23,0x03,0x6f,0x0e,0x00,0,0,0,0,0,0,0,0,0, 
0x0b,0x80,1,1,0,0xff,0xff,0x12,0xff,0x6f,0x02,0x00,0,0,0,0,0,0,0,0,0 };

static char **pefdesc;
static char *pefdesc1[MAXPEF] = {    /* for Sahalee BMC */
/* 0 0x00 */ "",
/* 1 0x01 */ "Temperature Sensor",
/* 2 0x02 */ "Voltage Sensor",
/* 3 0x04 */ "Fan Failure",
/* 4 0x05 */ "Chassis Intrusion",
/* 5 0x08 */ "Power Supply Fault",
/* 6 0x0c */ "Memory ECC Error",
/* 7 0x0f */ "FRB Failure",
/* 8 0x07 */ "BIOS POST Error",
/* 9 0x13 */ "Fatal NMI",
/*10 0x23 */ "Watchdog Timer Reset",
/*11 0x12 */ "System Restart",
/*12 0x20 */ "OS Critical Stop",
/*13 0x00 */ "reserved",
/*14 0x00 */ "reserved",
/*15 0x00 */ "reserved",
/*16 0x00 */ "reserved",
/*17 */ "reserved",
/*18 */ "reserved",
/*19 */ "reserved",
/*20 */ "reserved",
/*21 */ "reserved",
/*22 */ "reserved",
/*23 */ "reserved",
/*24 */ "reserved",
/*25 */ "reserved",
/*26 */ "reserved",
/*27 */ "reserved",
/*28 */ "reserved",
/*29 */ "unused",
/*30 */ "unused" };

static char *pefdesc2[MAXPEF] = {    /* for NSC miniBMC */
/* 0 */ "",
/* 1 0x02*/ "Voltage Sensor Assert",
/* 2 0x23*/ "Watchdog FRB Timeout",  /* was "Proc FRB Thermal", */
/* 3 0x02*/ "Voltage Sensor Deassert",
/* 4 0x07*/ "Proc1 IERR",
/* 5 0xff*/ "Digital Sensor OK",
/* 6 0x14*/ "Chassis Identify",
/* 7 0x13*/ "NMI Button",
/* 8 0x14*/ "Clear CMOS via Panel",
/* 9 0x0f*/ "OS Load POST Code",
/*10 0x20*/ "OS Critical Stop",
/*11 0x00*/ "reserved",
/*12 0x00*/ "reserved",
/*13 */ "reserved",
/*14 */ "reserved",
/*15 */ "reserved",
/*16 */ "reserved",
/*17 */ "reserved",
/*18 */ "reserved",
/*19 */ "reserved",
/*20 */ "reserved",
/*21 */ "reserved",
/*22 */ "reserved",
/*23 */ "reserved",
/*24 */ "reserved",
/*25 */ "reserved",
/*26 0x05*/ "Chassis Intrusion",
/*27 0x0f*/ "POST Code Error",
/*28 0x02*/ "Voltage Failure",
/*29 0x04*/ "Fan Failure",
/*30 0x01*/ "Temperature Failure"};

static char suffix[41];
#define NLAN  30
static struct {
  int cmd;
  int sz;
  char desc[40];
} lanparams[NLAN] = {   /* see IPMI Table 19-4 */
 /*  0 */  0, 1, "Set in progress",
 /*  1 */  1, 1, "Auth type support",
 /*  2 */  2, 5, "<ipmi15:AuthenticationTypeEnablesV001 >",
 /*  3 */  3, 4, "<ipmi15:IPAddress>",
 /*  4 */  4, 1, "<ipmi15:IPAddressSource>",  /* (DHCP/Static) */
 /*  5 */  5, 6, "<ipmi15:MACAddress>",
 /*  6 */  6, 4, "<ipmi15:SubnetMask>",
 /*  7 */  7, 3, "<ipmi15:IPv4Header>",
 /*  8 */  8, 2, "Prim RMCP port ",
 /*  9 */  9, 2, "Sec RMCP port ",
 /* 10 */ 10, 1, "<ipmi15:enableGratuitousArp>",
 /* 11 */ 11, 1, "<ipmi15:ArpInterval>",
 /* 12 */ 12, 4, "<ipmi15:DefaultGateway>",
 /* 13 */ 13, 6, "<ipmi15:DefaultGatewayMacAddress>",
 /* 14 */ 14, 4, "<ipmi15:BackupGateway>",
 /* 15 */ 15, 6, "<ipmi15:BackupGatewayMacAddress>",
 /* 16 */ 16,18, "<ipmi15:CommunityString>",
 /* 17 */ 17, 1, "Num dest",
 /* 18 */ 18, 5, "Dest type",
 /* 19 */ 19, 13, "<ipmi15:AlertIPAddress>",
 /* 20 */ 20, 2,  "VLAN ID",
 /* 21 */ 21, 1,  "VLAN Priority",
 /* 22 */ 25, 4,  "VLAN Dest Tag",
 /* 23 */ 96, 28, "OEM Alert String",
 /* 24 */ 97,  1, "Alert Retry Algorithm",
 /* 25 */ 98,  3, "UTC Offset",
 /* 26 */ 192, 4, "<ipmi15:DHCPServerIP>",
 /* 27 */ 193, 6, "<ipmi15:DHCPMacAddress>",
 /* 28 */ 194, 1, "DHCP Enable",
 /* 29 */ 201, 2, "<ipmi15:LanChannelAccessMode>" 
};

#define NSER  22   /* max=32 */
static struct {
  int cmd;
  int sz;
  char desc[40];
} serparams[NSER] = {   /* see IPMI Table 20-4 */
 /*  0 */  0, 1, "Set in progress",
 /*  1 */  1, 1, "Auth type support",
 /*  2 */  2, 5, "<ipmi15:AuthenticationTypeEnablesV001 >",
 /*  3 */  3, 1, "<ipmi15:ConnectionMode>",
 /*  4 */  4, 1, "<ipmi15:SessionInactivityTimeout>",
 /*  5 */  5, 5, "<ipmi15:ChannelCallback>",
 /*  6 */  6, 1, "<ipmi15:SessionTermination>",
 /*  7 */  7, 2, "<ipmi15:IPMIMessageComm>",
 /*  8 */  8, 2, "<ipmi15:MuxSwitch>",
 /*  9 */  9, 2, "Modem Ring Time",
 /* 10 */ 10,17, "<ipmi15:ModemInitString>",
 /* 11 */ 11, 5, "<ipmi15:ModemEscapeSeq>",
 /* 12 */ 12, 8, "<ipmi15:ModemHangupSeq>",
 /* 13 */ 13, 8, "<ipmi15:ModemDialCommand>",
 /* 14 */ 14, 1, "Page Blackout Interval",
 /* 15 */ 15,18, "Community String",
 /* 16 */ 16, 1, "Num of Alert Dest",
 /* 17 */ 17, 5, "<ipmi15:Destination Info>",
 /* 18 */ 18, 1, "<ipmi15:CallRetryInterval>",
 /* 19 */ 19, 3, "<ipmi15:DestinationCommSettings>",
 /* 20 */ 29, 2, "<ipmi15:TerminalModeConfig>",
 /* 21 */ 201, 2,"<ipmi15:SerialChannelAccessMode>" 
};


int GetDeviceID(LAN_RECORD *pLanRecord)
{
	uchar responseData[MAX_BUFFER_SIZE];
	int responseLength = MAX_BUFFER_SIZE;
	int status;
	uchar inputData[24];
	uchar completionCode;

	if (pLanRecord == NULL) return(-1);

        status = ipmi_cmd(GET_DEVICE_ID, inputData, 0, responseData,
                        &responseLength, &completionCode, fdebug); 

	if (status == ACCESS_OK) {
		if( completionCode ) {
			SELprintf("GetDeviceID: completion code=%x\n", 
				completionCode); 
		} else {
			memcpy(pLanRecord,&responseData[0],responseLength);
			memcpy(my_devid,&responseData[0],responseLength);
			return(0);  // successful, done
		}
	}  /* endif */
	/* if get here, error */
 	return(-1);
}  /*end GetDeviceID() */

int GetChanAcc(uchar chan, uchar parm, LAN_RECORD *pLanRecord)
{
	uchar responseData[MAX_BUFFER_SIZE];
	int responseLength = MAX_BUFFER_SIZE;
	int status;
	uchar inputData[24];
	uchar completionCode;

	if (pLanRecord == NULL) return(-1);
	responseLength = 3;
	inputData[0] = chan;
	inputData[1] = parm;  /* 0x80 = active, 0x40 = non-volatile */
	responseLength = sizeof(responseData);
        status = ipmi_cmd(GET_CHANNEL_ACC, inputData, 2, responseData,
                        &responseLength, &completionCode, fdebug); 

	if (status == ACCESS_OK) {
		if( completionCode ) {
			SELprintf("GetChanAcc: completion code=%x\n", 
				completionCode); 
		} else {
			// dont copy first byte (Parameter revision, usu 0x11)
			memcpy(pLanRecord,&responseData[0],responseLength);
			return(0);  // successful, done
		}
	}  /* endif */
	/* if get here, error */
 	return(-1);
}  /*GetChanAcc()*/

int SetChanAcc(uchar chan, uchar parm, uchar val)
{
	uchar responseData[MAX_BUFFER_SIZE];
	int responseLength = MAX_BUFFER_SIZE;
	int status;
	uchar inputData[24];
	uchar completionCode;

	if (fmBMC) return(0);  /* mBMC doesn't support this */
        /* parm: 0x80 = active, 0x40 = set non-vol*/
	responseLength = 1;
	inputData[0] = chan;  /* channel */
	inputData[1] = (parm & 0xc0) | (val & 0x3F); 
#ifdef OLD
	if (chan == LAN_CH) inputData[2] = lan_access;  /* LAN access, no chg */
	else                   /* serial defaults to 0x02 = User priv level */
#endif
	inputData[2] = (parm & 0xc0) | lan_access; /* set priv level to Admin */

	responseLength = sizeof(responseData);
        status = ipmi_cmd(SET_CHANNEL_ACC, inputData, 3, responseData,
                        &responseLength, &completionCode, fdebug); 

	if (status == ACCESS_OK) {
		if( completionCode ) {
			SELprintf("SetChanAcc: completion code=%x\n", 
				completionCode); 
		} else {
			return(0);  // successful, done
		}
	}  /* endif */
	/* if get here, error */
 	return(-1);
}  /*SetChanAcc()*/

int
SetPasswd(int usernum, char *uname, char *upswd)
{
   uchar responseData[MAX_BUFFER_SIZE];
   int responseLength = MAX_BUFFER_SIZE;
   int status, i;
   uchar completionCode;
   char  inputData[24];
   int ret = 0;

	inputData[0] = usernum;  /*user 1 = null user */
	responseLength = sizeof(responseData);
        status = ipmi_cmd(GET_USER_NAME, inputData, 1, responseData, 
		        &responseLength, &completionCode, fdebug);
        SELprintf("GET_USERNAME: %x %x %x, status = %x, ccode=%x\n",
                responseData[0],responseData[1],responseData[2],
		status,completionCode);
	if (fdebug) {
		char aname[17];
                printf("User %d: ",usernum);
		for (i = 0; i < responseLength; i++) {
                   printf("%02x ",responseData[i]);
		   if (responseData[i] < 0x20) aname[i] = '.';
		   else aname[i] = responseData[i];
		}
		aname[16] = 0;
                printf(" %s\n",aname);
	}

	if (usernum != 1) {  /* user 2 = specified lan username */
	   inputData[0] = usernum;
           memset(&inputData[1],0,16);
           memcpy(&inputData[1],uname,strlen(uname));
           status = ipmi_cmd(SET_USER_NAME, inputData, 17, responseData, 
		        &responseLength, &completionCode, fdebug);
           SELprintf("SETUSERNAME - %x %x %x  status = %x, ccode=%x\n",
                inputData[0],inputData[1],inputData[2],
		status,completionCode);
	   if (status != 0) ret = completionCode;
	}

        inputData[0] = usernum;  /*user 1 = null user */
        inputData[1] = 0x01;  /*enable user*/
	responseLength = sizeof(responseData);
        status = ipmi_cmd(SET_USER_PASSWORD, inputData, 2, responseData, 
		        &responseLength, &completionCode, fdebug);
        SELprintf("SETUSERPSW - inputData = %x %x %x, status = %x, ccode=%x\n",
                inputData[0],inputData[1],inputData[2],
		status,completionCode);
	if (status != 0) ret = completionCode;

         inputData[0] = usernum;  /*user 1 = null user */
         inputData[1] = 0x02;  /*set password*/
         memset(&inputData[2],0,16);
         strcpy(&inputData[2],upswd);
	 responseLength = sizeof(responseData);
	 if (fdebug) {
		char apsw[17];
		char c;
                printf("Pswd %d: ",usernum);
		for (i = 0; i < 16; i++) {
		   c = inputData[i+2];
                   printf("%02x ",(unsigned char)c);
		   if (c < 0x20) apsw[i] = '.';
		   else apsw[i] = c;
		}
		apsw[16] = 0;
                printf(" %s\n",apsw);
	 }
         status = ipmi_cmd(SET_USER_PASSWORD, inputData, 18, responseData, 
		        &responseLength, &completionCode, fdebug);
         SELprintf("SETUSERPSW - inputData = %x %x %x, status = %x, ccode=%x\n",
                inputData[0],inputData[1],inputData[2],
		status,completionCode);
	 if (status != 0) ret = completionCode;

         inputData[0] = usernum;  /*user 1 = null user */
         inputData[1] = 0x03;  /*test password*/
         memset(&inputData[2],0,16);
         strcpy(&inputData[2],upswd);
	 responseLength = sizeof(responseData);
         status = ipmi_cmd(SET_USER_PASSWORD, inputData, 18, responseData, 
		        &responseLength, &completionCode, fdebug);
         SELprintf("TESTUSERPSW - inputData = %x %x %x, status = %x, ccode=%x\n",
                inputData[0],inputData[1],inputData[2],
		status,completionCode);

        inputData[0] = 0x90 | lan_ch;   /* = 0x97 */
        inputData[1] = usernum;   /* user 1 */
        inputData[2] = lan_access;   /* admin */
        inputData[3] = 0x00;   /* User Session Limit, 0=not limited*/
	responseLength = sizeof(responseData);
        status = ipmi_cmd(SET_USER_ACCESS, inputData, 4, responseData,
                        &responseLength, &completionCode, fdebug);
        SELprintf("SETUSER_ACCESS - inputData = %x %x %x, status = %x, ccode=%x\n",
		   (uchar)inputData[0],inputData[1],inputData[2],
		   status,completionCode);
	if (status != 0) ret = completionCode;

   return(ret);
}  /*end SetPswd()*/

int SetUser(char *uname, char *passwd)
{
      int ret = 0; 
      if (fpassword) {  /* if the user specified a password, set it. */
	   if (uname == NULL) 
		ret = SetPasswd(1, "",   passwd);
	   else    /* set user 2 username also */
		ret = SetPasswd(2, uname,passwd);
      }  /*endif fpassword*/
      return(ret);
}  /*end SetUser()*/

int GetUser(uchar chan, uchar usernum)
{
      uchar responseData[MAX_BUFFER_SIZE];
      int responseLength = MAX_BUFFER_SIZE;
      int status;
      uchar completionCode;
      char  inputData[24];

         inputData[0] = chan;
         inputData[1] = usernum;  /* usually = 1 for BMC LAN */
	 responseLength = sizeof(responseData);
         status = ipmi_cmd(GET_USER_ACCESS, inputData, 2, responseData,
                        &responseLength, &completionCode, fdebug);
	 if (status == 0 && completionCode == 0) {
	    uchar c;
            SELprintf("Get User Access(%d:%d): %02x %02x %02x %02x ",
		        chan,usernum, responseData[0],responseData[1],
		 	responseData[2], responseData[3]);
	    c = responseData[3];
            if (c & 0x10) printf("IPMI, ");
            c = (c & 0x0f);
            switch(c) {
		case 1:    printf("Callback\n"); break;
		case 2:    printf("User  \n"); break;
		case 3:    printf("Operator\n"); break;
		case 4:    printf("Admin \n"); break;
		case 5:    printf("OEM   \n"); break;
		case 0x0f: printf("No access\n"); break;
		default:   printf("Reserved\n");
            }
	 } else 
            SELprintf("Get User Access(%d): status=%x, ccode=%x\n",usernum,
	       		status, completionCode);
	return(status);
}  /*end GetUser()*/

int GetSerEntry(uchar subfunc, LAN_RECORD *pLanRecord)
{
	uchar responseData[MAX_BUFFER_SIZE];
	int responseLength = MAX_BUFFER_SIZE;
	uchar inputData[24];
	int status;
	uchar completionCode;
	uchar chan; uchar bset;

	if (pLanRecord == NULL)
	{
	   if (fdebug)
	      printf("GetSerEntry(%d): error, output buffer is NULL\n",subfunc);
 	   return (-1);
	}

        chan = SER_CH;  /* 1=EMP, 0=IPMB, 6=LAN2, 7=LAN1 */
	bset = 0;

	inputData[0]            = chan;  // flags, channel 3:0 (1=EMP)
	inputData[1]            = subfunc;  // Param selector 
	inputData[2]            = bset;  // Set selector 
	inputData[3]            = 0;  // Block selector
	if (subfunc == 10)  {
	   inputData[2] = 0;
	   inputData[3] = 1;
 	}

        status = ipmi_cmd(GET_SER_CONFIG, inputData, 4, responseData,
                        &responseLength, &completionCode, fdebug); 

	if (status == ACCESS_OK) {
		if( completionCode ) {
			SELprintf("GetSerEntry(%d): completion code=%x\n", 
				subfunc,completionCode); // responseData[0]);
		} else {
			// dont copy first byte (Parameter revision, usu 0x11)
			memcpy(pLanRecord,&responseData[1],responseLength-1);
			//successful, done
			return(0);
		}
	}

	// we are here because completionCode is not COMPLETION_CODE_OK
	if (fdebug) 
		SELprintf("GetSerEntry(%d): ipmi_cmd status=%x ccode=%x\n",
                          subfunc,status,completionCode);
 	return -1;
}

int GetLanEntry(uchar subfunc, LAN_RECORD *pLanRecord)
{
	uchar responseData[MAX_BUFFER_SIZE];
	int responseLength = MAX_BUFFER_SIZE;
	uchar inputData[24];
	int status;
	uchar completionCode;
	uchar chan; uchar bset;

	if (pLanRecord == NULL)
	{
	   if (fdebug)
	       printf("GetLanEntry: error, output buffer is NULL\n");
 	   return (-1);
	}

        chan = lan_ch;  /* LAN 1 = 7 */
	if (subfunc == 18 || subfunc == 19) bset = 1;  /* dest id = 1 */
	else bset = 0;

	inputData[0]            = chan;  // flags, channel 3:0 (LAN 1)
	inputData[1]            = subfunc;  // Param selector (3 = ip addr)
	inputData[2]            = bset;  // Set selector 
	inputData[3]            = 0;  // Block selector

        status = ipmi_cmd(GET_LAN_CONFIG, inputData, 4, responseData,
                        &responseLength, &completionCode, fdebug); 

	if (status == ACCESS_OK) {
		if( completionCode ) {
			SELprintf("GetLanEntry: completion code=%x\n", 
				completionCode); // responseData[0]);
		} else {
			// dont copy first byte (Parameter revision, usu 0x11)
			memcpy(pLanRecord,&responseData[1],responseLength-1);
			//successful, done
			return(0);
		}
	}

	// we are here because completionCode is not COMPLETION_CODE_OK
	if (fdebug) 
		SELprintf("GetLanEntry: ipmi_cmd status=%d completionCode=%x\n",
                          status,completionCode);
 	return -1;
}  /* end GetLanEntry() */

int SetLanEntry(uchar subfunc, LAN_RECORD *pLanRecord, int reqlen)
{
	uchar responseData[MAX_BUFFER_SIZE];
	int responseLength = MAX_BUFFER_SIZE;
	uchar inputData[24];
	int status;
	uchar completionCode;

	if (pLanRecord == NULL)
	{
	   if (fdebug)
	       printf("SetLanEntry: error, input buffer is NULL\n");
 	   return (-1);
	}

	inputData[0]            = lan_ch;  // flags, channel 3:0 (LAN 1)
	inputData[1]            = subfunc;  // Param selector (3 = ip addr)
	memcpy(&inputData[2],pLanRecord,reqlen);

        status = ipmi_cmd(SET_LAN_CONFIG, inputData, (uchar)(reqlen+2), 
			responseData, &responseLength,&completionCode,fdebug);

	if (status == ACCESS_OK) {
		if( completionCode ) {
			SELprintf("SetLanEntry: completion code=%x\n", 
				completionCode); // responseData[0]);
		} else {
			//successful, done
			return(0);
		}
	}

	// we are here because completionCode is not COMPLETION_CODE_OK
	if (fdebug) 
		SELprintf("SetLanEntry: sendImbRequest completion code=%x\n",
                          completionCode);
 	return -1;
}  /* end SetLanEntry() */

int GetPefEntry(uchar subfunc, ushort rec_id, PEF_RECORD *pPefRecord)
{
	uchar responseData[MAX_BUFFER_SIZE];
	int responseLength = MAX_BUFFER_SIZE;
	uchar inputData[24];
	int status;
	uchar completionCode;

	if (pPefRecord == NULL)
	{
	   if (fdebug)
	      printf("GetPefEntry(%d): error, output buffer is NULL\n",subfunc);
 	   return (-1);
	}

	inputData[0]            = subfunc; // Parameter = Evt Filter Table
	inputData[1]            = (uchar)rec_id; 
	inputData[2]            = 0; 

        status = ipmi_cmd(GET_PEF_CONFIG, inputData, 3, responseData,
                        &responseLength, &completionCode, fdebug); 

        if (status == ACCESS_OK) {
                if( completionCode ) {
                        SELprintf("GetPefEntry(%d/%d): completion code=%x\n",
                                subfunc,rec_id,completionCode); 
                } else {
                        // dont copy first byte (Parameter revision, usu 0x11)
                        memcpy(pPefRecord,&responseData[1],responseLength-1);
                        //successful, done
                        return(0);
                }
	}

	// we are here because completionCode is not COMPLETION_CODE_OK
	if (fdebug) 
	      SELprintf("GetPefEntry: ipmi_cmd status=%x completionCode=%x\n",
                         status, completionCode);
 	return -1;
}  /* end GetPefEntry() */

int SetPefEntry(PEF_RECORD *pPefRecord)
{
	uchar responseData[MAX_BUFFER_SIZE];
	int responseLength = MAX_BUFFER_SIZE;
	uchar inputData[24]; /* sizeof(PEF_RECORD) = 21 +1=22 */
	int status;
	uchar completionCode;
	uchar subfunc;

	subfunc = 0x06; // Parameter = Evt Filter Table

	if (pPefRecord == NULL) {
	   if (fdebug)
	       printf("SetPefEntry: error, output buffer is NULL\n");
 	   return (-1);
	}

        //  0  1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 17 18 19 20 21
        // 06 0c 80 01 01 00 ff ff 20 ff 6f ff 00 00 00 00 00 00 00 00 00 00
	// memset(&inputData[0],0,requestData.dataLength);
	inputData[0]            = subfunc; 
	memcpy(&inputData[1],pPefRecord,sizeof(PEF_RECORD));

        status = ipmi_cmd(SET_PEF_CONFIG, inputData, sizeof(PEF_RECORD)+1, 
		       responseData, &responseLength, &completionCode, fdebug); 

	if (status == ACCESS_OK) {
		if( completionCode ) {
			SELprintf("SetPefEntry: completion code=%x\n", 
				completionCode); // responseData[0]);
		} else {
			//successful, done
			return(0);
		}

	}

	// we are here because completionCode is not COMPLETION_CODE_OK
	if (fdebug) 
		SELprintf("SetPefEntry: sendImbRequest completion code=%x\n",
                          completionCode);
 	return(-1);
 
}  /* end SetPefEntry() */

int DisablePef(void)
{
	uchar iData[24]; /* sizeof(PEF_RECORD) = 21 +1=22 */
	uchar rData[MAX_BUFFER_SIZE];
	int rLength = MAX_BUFFER_SIZE;
	uchar cc;
	int status;

        status = SetChanAcc(lan_ch, 0x80, CHAN_ACC_DISABLE);
        if (fdebug) SELprintf("SetChanAcc(lan/active), ret = %d\n",status);
        status = SetChanAcc(lan_ch, 0x40, CHAN_ACC_DISABLE); 
        SELprintf("SetChanAcc(lan), ret = %d\n",status);
	if (status != 0) return(status);

	iData[0] = 0x01;        /* PEF Control Param */
	iData[1] = 0x00;	/* PEF disable */
	rLength = MAX_BUFFER_SIZE;
        status = ipmi_cmd(SET_PEF_CONFIG, iData, 2, rData, &rLength, 
			  &cc, fdebug); 
	if (status != 0) return(status);
	if( cc ) {
		SELprintf("DisablePef[%d]: completion code=%x\n",iData[0],cc);
		return(-1);
	}
	return(status);
}

int ShowPef(void)
{
	uchar iData[24]; /* sizeof(PEF_RECORD) = 21 +1=22 */
	uchar rData[MAX_BUFFER_SIZE];
	int rLength = MAX_BUFFER_SIZE;
	uchar cc;
	int status, i,j;
	 
	for (j = 1; j < MAX_PEFPARAMS; j++) {
	   if (j == 4 && fmBMC) {
		/* fmBMC gets cc=0x80 for param 4, so skip it. */
		continue;
	   }
	   iData[0] = j;         /* PEF Control Param */
	   if (j == 6 || j == 7 || j == 9) iData[1] = 1; 
	   else iData[1] = 0x00;	 /* PEF Set Selector */
	   if (j == 13) iData[2] = 1; 
	   else iData[2] = 0x00;	 /* PEF Block Selector */
	   rLength = MAX_BUFFER_SIZE;
	   status = ipmi_cmd(GET_PEF_CONFIG, iData, 3, rData, &rLength, 
				&cc, fdebug); 
	   if (status == 0 && cc == 0) {
		SELprintf("PefParam[%d]: ",iData[0]);
		if (rLength > 0)
		  for (i=0;i<peflen[j];i++) SELprintf("%02x ", rData[1+i]);
		SELprintf("\n");
	   } else
		SELprintf("PefParam[%d]: GET_PEF status=%d cc=%x\n",
			  	iData[0],status,cc);
	}
	return(status);
}

int EnablePef(void)
{
	uchar iData[24]; /* sizeof(PEF_RECORD) = 21 +1=22 */
	uchar rData[MAX_BUFFER_SIZE];
	int rLength = MAX_BUFFER_SIZE;
	uchar cc;
	int status;
	uchar sdelay;

        status = SetChanAcc(lan_ch, 0x80, chan_pefon);  
        if (fdebug) SELprintf("SetChanAcc(lan/active), ret = %d\n",status);
        status = SetChanAcc(lan_ch, 0x40, chan_pefon); 
        SELprintf("SetChanAcc(lan), ret = %d\n",status);
	if (status != 0) return(status);

	{
	  iData[0] = 0x01;         /* PEF Control Param */
	  iData[1] = 0x00;	   /* PEF Set Selector */
	  iData[2] = 0x00;	   /* PEF Block Selector */
	  rLength = MAX_BUFFER_SIZE;
          status = ipmi_cmd(GET_PEF_CONFIG, iData, 3, rData, &rLength, 
			  &cc, fdebug); 
	  if (status != 0 || cc != 0) sdelay = 0;
	  else sdelay = rData[1];
	  if (fdebug) SELprintf("EnablePef[%d]: get cc=%x, control=%02x\n",
			  	iData[0],cc,sdelay);
	  iData[0] = 0x01;         /* PEF Control Param (0x01 or 0x05) */
	  iData[1] = 0x01;	   /* PEF enable, & no startup delay */
	  rLength = MAX_BUFFER_SIZE;
          status = ipmi_cmd(SET_PEF_CONFIG, iData, 2, rData, &rLength, 
			  &cc, fdebug); 
	  if (status != 0) return(status);
	  if( cc ) {
		SELprintf("EnablePef[%d]: completion code=%x\n",iData[0],cc);
		return(-1);
	  }

#ifdef TEST
	  iData[0] = 0x01;         /* Serial Channel */
	  iData[1] = 0x13;         /* Dest Com settings = 19. */
	  iData[2] = 0x01;         /* POL Default Dest */
	  iData[3] = 0x60;         
	  iData[4] = 0x07;        
          status = ipmi_cmd(SET_SER_CONFIG, iData, 5, rData, &rLength, 
			  &cc, fdebug); 
#endif

	  iData[0] = 0x02;         /* PEF Action Param */
	  iData[1] = 0x00;	   /* PEF Set Selector */
	  iData[2] = 0x00;	   /* PEF Block Selector */
	  rLength = MAX_BUFFER_SIZE;
          status = ipmi_cmd(GET_PEF_CONFIG, iData, 3, rData, &rLength, 
			  &cc, fdebug); 
	  if (fdebug) SELprintf("EnablePef[%d]: get cc=%x, val=%02x\n",
			  	iData[0],cc,rData[1]);
	  iData[0] = 0x02;         /* PEF Action Param */
	  iData[1] = 0x2f;         /* enable alerts, reset, power cycle/down */
	  rLength = MAX_BUFFER_SIZE;
          status = ipmi_cmd(SET_PEF_CONFIG, iData, 2, rData, &rLength, 
			  &cc, fdebug); 
	  if (status != 0) return(status);
	  if( cc ) {
		SELprintf("EnablePef[%d]: completion code=%x\n",iData[0],cc);
		return(-1);
	  }

	  if ((sdelay & 0x04) != 0) {  /* startup delay is supported */
	    iData[0] = 0x03;         /* PEF Startup Delay Param */
	    iData[1] = 0x00;         /* 0 seconds, default is 0x3c (60 sec) */
	    rLength = MAX_BUFFER_SIZE;
            status = ipmi_cmd(SET_PEF_CONFIG, iData, 2, rData, &rLength, 
			  &cc, fdebug); 
	    if (status != 0) return(status);
	    if( cc ) {
		SELprintf("EnablePef[%d]: completion code=%x\n",iData[0],cc);
		// return(-1);
	    }
	    iData[0] = 0x04;         /* PEF Alert Startup Delay Param */
	    iData[1] = 0x00;         /* 0 seconds, default is 0x3c (60 sec) */
	    rLength = MAX_BUFFER_SIZE;
            status = ipmi_cmd(SET_PEF_CONFIG, iData, 2, rData, &rLength, 
			  &cc, fdebug); 
	    if (status != 0) return(status);
	    if( cc ) {
		SELprintf("EnablePef[%d]: completion code=%x\n",iData[0],cc);
		// return(-1);
	    }
	  } /*endif sdelay*/

	  iData[0] = 0x09;        /* PEF Alert Policy Table */
	  iData[1] = 0x01;        /* Policy number */
	  iData[2] = 0x18;        /* PEF LAN, always alert, policy enable */
	  iData[3] = (lan_ch << 4) + 0x01;  /* LAN_CH=70, default dest=01 */
	  iData[4] = 0x00;        /* No alert string */
	  rLength = MAX_BUFFER_SIZE;
          status = ipmi_cmd(SET_PEF_CONFIG, iData, 5, rData, &rLength, 
			  &cc, fdebug); 
	  if (status != 0) return(status);
	  if( cc ) {
		SELprintf("EnablePef[%d]: completion code=%x\n",iData[0],cc);
		return(-1);
	  }
	}  /*endif IPMI 1.5 */

	return(status);
}  /* end EnablePef */

static int ReadSELinfo()
{
	uchar responseData[MAX_BUFFER_SIZE];
	int responseLength = MAX_BUFFER_SIZE;
	uchar completionCode;
	uchar inputData[6];
	int status;

        status = ipmi_cmd(GET_SEL_INFO, inputData, 0, responseData, 
			&responseLength, &completionCode, fdebug); 
        if (fdebug) 
		printf("ipmi_cmd(GET_SEL_INFO) status = %x, rlen=%d\n",
			status, responseLength);

	if (status == ACCESS_OK) {
              /*  SELprintf("Code %d SEL Ver %d Support %d\n",
                          completionCode,
                          responseData[0], responseData[13]); */
		//successful, done
		return(0);
	} else  return(1);

}  /*end ReadSELinfo()*/


/*
 * atomac - converts ASCII string to binary MAC address (array).
 * Accepts input formatted as 11:22:33:44:55:66 or 11-22-33-44-55-66.
 */
void atomac(uchar *array, char *instr)
{
   int i,j,n;
   char *pi;
   j = 0;
   pi = instr;
   n = strlen(instr);
   for (i = 0; i <= n; i++) {
      if (instr[i] == ':') {
	array[j++] = htoi(pi);
	pi = &instr[i+1];
      } else if (instr[i] == '-') {
	array[j++] = htoi(pi);
	pi = &instr[i+1];
      } else if (instr[i] == 0) {
	array[j++] = htoi(pi);
	}
   }
   if (fdebug) 
      printf("atomac: %02x %02x %02x %02x %02x %02x\n", 
         array[0],array[1],array[2],array[3], array[4],array[5]);
}  /*end atomac()*/

void atoip(uchar *array,char *instr)
{
   int i,j,n;
   char *pi;
   /* converts ASCII input string into binary IP Address (array) */
   j = 0;
   pi = instr;
   n = strlen(instr);
   for (i = 0; i <= n; i++) {
      if (instr[i] == '.') {
	instr[i] = 0;
	array[j++] = atoi(pi);
	pi = &instr[i+1];
	}
      else if (instr[i] == 0) {
	array[j++] = atoi(pi);
	}
   }
   if (fdebug) 
      printf("atoip: %d %d %d %d\n", array[0],array[1],array[2],array[3]);
}  /*end atoip()*/

int GetBmcEthDevice(void)
{
    LAN_RECORD LanRecord;
    int devnum = -1;
    int ret;
    uchar bmcMacAddress[ 6 ];  /*MAC_LENGTH*/
#ifndef WIN32
    struct ifreq ifr;  /*only used for Linux*/
    int skfd;
    int nCurDevice;
    char szDeviceName[ 7 ];    /* MAX_DEVICE_NAME_LENGTH + 1 */
#endif
    int rlen;
    uchar iData[2];
    uchar rData[10];
    uchar cc;
    int i = 0;
    int j;

    /* find the LAN channel via Channel Info */
    for (j = 1; j < MAXCHAN; j++) {
        rlen = sizeof(rData);
        iData[0] = j; /*channel #*/
	memset(rData,0,9); /*initialize recv data*/
        ret = ipmi_cmd(GET_CHANNEL_INFO, iData, 1, rData, &rlen, &cc, fdebug); 
	if (ret == 0xcc || cc == 0xcc) /* special case for gnu ipmi_lan */
		continue;
	if (ret != 0) {
    		if (fdebug) printf("get_chan_info rc = %x\n",ret);
		break;
	}
	if (rData[1] == 4) { /* LAN type*/
		if (fdebug) printf("chan[%d] = lan\n",j);
		lan_ch = j;
		i++;
	} else if (rData[1] == 5) { /* serial type*/
		if (fdebug) printf("chan[%d] = serial\n",j);
	} else  /* 7 = SMBus, 12 = System Interface */
		if (fdebug) printf("chan[%d] = %d\n",j,rData[1]);
    }
    if (i == 0) return(-2);  /* no lan channels found */
    if (fdebug) printf("lan_ch = %d\n",lan_ch);

	ret = GetLanEntry( 5 /*MAC_ADDRESS_LAN_PARAM*/, &LanRecord );
	if ( ret < 0 ) {
		printf( "GetBmcEthDevice: GetLanEntry failed\n" );
		return devnum;
	}

	if (fdebug) printf("BMC MAC %x:%x:%x:%x:%x:%x\n",
		LanRecord.data[0], LanRecord.data[1], LanRecord.data[2],
		LanRecord.data[3], LanRecord.data[4], LanRecord.data[5] );
	memcpy( bmcMacAddress, LanRecord.data, sizeof(bmcMacAddress) );
	/* Note: BMC MAC may not be valid yet. */

#ifdef WIN32
	devnum = 0;   /* BMC LAN is always eth0 for Windows */
#else
	if ( ( skfd = socket(AF_INET, SOCK_DGRAM, 0 ) ) < 0) {
		if ( fdebug ) {
			perror("socket");
			return devnum;
		}
	}

	for ( 	nCurDevice = 0 ;
		(nCurDevice < NUM_DEVICES_TO_CHECK) && (devnum == -1);
		nCurDevice++ )
	{
		sprintf( szDeviceName, "eth%d", nCurDevice );
		strcpy(ifr.ifr_name, szDeviceName );
		if (ioctl(skfd, SIOCGIFHWADDR, &ifr) > 0) {
			if ( fdebug ) 
				printf( "GetBmcEthDevice: Could not get MAC address for device %d\n", nCurDevice );
		} else {
			if (memcmp(ifr.ifr_hwaddr.sa_data, bmcMacAddress, sizeof(bmcMacAddress)) == 0) {
				devnum = nCurDevice;
				break;
			}
		}
	}

	close(skfd);
#endif
	if ( fdebug ) 
		printf( "GetBmcEthDevice: Returning Device %d\n", devnum );
	return devnum;
}

#ifdef WIN32
/* 
 * findmatch 
 * returns offset of the match if found, or -1 if not found.  
 */
static int
findmatch(char *buffer, int sbuf, char *pattern, int spattern, char figncase)
{
    int c, i, j, imatch;

    j = 0;
    imatch = 0;
    for (j = 0; j < sbuf; j++) {
        if ((sbuf - j) < spattern && imatch == 0) return(-1);
        c = buffer[j];
        if (c == pattern[imatch]) {
            imatch++;
        } else if ((figncase == 1) &&
                   ((c & 0x5f) == (pattern[imatch] & 0x5f))) {
            imatch++;
        } else if (pattern[imatch] == '?') {  /*wildcard char*/
            imatch++;
        } else {
            if (imatch > 0) {
               imatch = 0;
               if (j > 0) j--; /* try again with the first match char */
	    }
        }
        if (imatch == spattern) break;
    }
    if (imatch == spattern) {
        i = (j+1) - imatch;  /*buffer[i] is the match */
        return(i);
    } else return (-1);  /*not found*/
}				/*end findmatch */

int file_grep(char *fname, char *pattn, char *line, int sline, char bfirst)
{
    FILE *fp;
    char buff[1024];
    int ret = -1;
    int i, plen, blen;

    fp = fopen(fname,"r");
    if (fp == NULL) {
          fprintf(stderr,"file_grep: Cannot open %s, errno = %d\n",
			fname,errno);
	  ret = -1;
    } else {
	 plen = strlen(pattn);
         while (fgets(buff, 1023, fp)) {
	   blen = strlen(buff);
	   /* check for pattern in this line */
	   i = findmatch(buff,blen,pattn,plen,0);
  	   if (i >= 0) {
		ret = 0;  /* found it, success */
		if (blen >= sline) blen = sline - 1;
		strncpy(line,buff,blen);
		line[blen] = 0;  /*stringify*/
		if (bfirst) break;  
	 	/* else keep looking, use last one if multiples */
	   }
         } /*end while*/
         fclose(fp);
    }  /*end else file opened*/
    return(ret); 
}  /*end file_grep*/
#endif

int Get_Mac(uchar *ipadr,uchar *macadr)
{
#ifdef WIN32
   char cmd[128];
   char arpfile[128];  /* %TEMP%\arp.tmp */
   char ipstr[16];
   char arpline[128];
   int ret = -1;
   char *pb;
   char *tempdir; 
   int i;

   tempdir = getenv("TEMP");
   sprintf(arpfile,"%s\\arp.tmp",tempdir);
   sprintf(ipstr,"%d.%d.%d.%d",ipadr[0],ipadr[1],ipadr[2],ipadr[3]);
   sprintf(cmd,"ping -n 1 %s >NUL: \n", ipstr);
   if (fdebug) printf(cmd);
   system(cmd);  /*ping*/
   sprintf(cmd,"arp -a >%s \n", arpfile);
   if (fdebug) printf(cmd);
   system(cmd);  /*arp*/

   /* grep [ip_addr] arp.tmp */
   if (file_grep(arpfile,ipstr,arpline,sizeof(arpline),1) == 0) {
	if (fdebug) printf("arp line/mac: %s",arpline); /*incl '\n'*/
	/* find mac in this line */
	i = strspn(arpline," \t");  /*skip leading spaces*/
	pb = &arpline[i];
	i = strcspn(pb," \t\r\n");  /*skip first word (ip addr)*/
	pb += i;
	i = strspn(pb," \t");  /*skip spaces*/
	pb += i;
	i = strcspn(pb," \t\r\n");  /*find next space, end of word*/
	pb[i] = 0;  /* end string here */
	if (macadr[0] == 0xFF) atomac(macadr,pb);
	ret = 0;
   }
   return(ret);
#else
   FILE *fparp;
   char buff[1024]; 
   /* char arpfile[] = "/proc/net/arp";  */
   char alertfile[] = "/tmp/dest.arping";
   char arping_cmd[128];
   char *pb, *pm, *px;
   int num, i;
   int foundit = 0;
   int ret = 0;

   /* Get a MAC address for a given IP address */
   if (ipadr[0] != 0) {   /* if valid IP address */

      /* make sure the destination is in the arp cache */
      sprintf(arping_cmd,
	      "arping -I %s -c 2 %d.%d.%d.%d |grep reply |tail -n1 >%s\n",
              ifname,ipadr[0],ipadr[1],ipadr[2],ipadr[3],alertfile);
      if (fdebug) printf(arping_cmd);
      system(arping_cmd);

      fparp = fopen(alertfile,"r");
      if (fparp == NULL) {
          fprintf(stderr,"Get_Mac: Cannot open %s, errno = %d\n",
			alertfile,errno);
	  ret = -1;
       } else {
         while (fgets(buff, 1023, fparp)) {
	   /* should only run through loop once */
	   num = strcspn(buff," \t");    /* skip 1st word ("Unicast") */
	   i = strspn(&buff[num]," \t");
	   pb = &buff[num+i];
  	   if (strncmp(pb,"reply",5) == 0) {  /* valid output */
	      /* Find the ip address */
	      pb += 6 + 5;         /* skip "reply from " */
	      num = strcspn(pb," \t");
	      pb[num] = 0;
	      if (fdebug) printf("Alert ip=%s\n",pb);
	      // IP address should match input param 
 	      // if (rgdestip[0] == 0) atoip(rgdestip,pb);
	      /* Now find the mac address */
	      pm = strchr(&pb[num+1],'[');
	      if (pm == NULL) pm = &pb[num+2];  /* just in case */
	      pm++;
	      px = strchr(pm,']');
	      if (px == NULL) px = pm + 17;    /* just in case */
	      px[0] = 0;
	      if (fdebug) printf("Alert mac=%s\n",pm);
	      foundit = 1;
 	      if (macadr[0] == 0xFF) 
                  atomac(macadr,pm);
	      break;
	   }
         } /*end while*/
         fclose(fparp);
       }  /*end else file opened*/
   }  /*endif valid IP */
   else ret = -1;

   if (ret == -1 || foundit == 0) {  /* couldn't get it */
      if (rgdestmac[0] == 0xFF)  /* not specified by user */
         memcpy(macadr,rggwymac,6);  /* get to it from the default gateway */
      }
   return(ret);
#endif
}  /* end Get_Mac()*/

/* 
   Get_IPMac_Addr
   This routine finds the IP and MAC for the default gateway and 
   SNMP destination from the OS information.

   snmpd.conf locations:
   RedHat, MontaVista:  /etc/snmp/snmpd.conf
   SuSE SLES 8:   /etc/ucdsnmpd.conf
   SuSE SLES 9:   /etc/snmpd.conf
 */
int Get_IPMac_Addr()
{
#ifdef WIN32
   char cmd[128];
   char ipfile[128];  /* %TEMP%\ip.tmp */
   char ipstr[]  = "IP Address";
   char macstr[] = "Physical Address";
   char gwystr[] = "Default Gateway";
   char substr[] = "Subnet Mask";
   char line[128];
   char *pb;
   char *tempdir;
   int i;

   tempdir = getenv("TEMP");
   sprintf(ipfile,"%s\\ip.tmp",tempdir);
   /* Use "ipconfig /all" to get my "IP Address", "Physical Address",
      and "Default Gateway" from Windows OS. */
   sprintf(cmd,"ipconfig /all >%s \n", ipfile);
   if (fdebug) printf(cmd);
   system(cmd);  /*ipconfig*/

   /* Get OS Default for My IP Address */
   /* This returns the last one in ipconfig, which is eth0 for Windows. */
   if (file_grep(ipfile,ipstr,line,sizeof(line),0) == 0) {
	pb = strchr(line,':');
	if (pb != NULL) { 
		pb += 2;  /*skip ": "*/
		i = strcspn(pb," \t");
		pb[i] = 0;
		if (fdebug) printf("my ip=%s\n",pb);
		if (rgmyip[0] == 0) {  /* if ip not user-specified */
		    atoip(rgmyip,pb);
		    /* Now get my MAC address */
   		    if (file_grep(ipfile,macstr,line,sizeof(line),0) == 0) {
			pb = strchr(line,':');
			if (pb != NULL) { 
			   pb += 2;  /*skip ": "*/
			   i = strcspn(pb," \t");
			   pb[i] = 0;
			   if (fdebug) printf("my mac=%s\n",pb);
			   atomac(rgmymac,pb);
			}
		    }
		}
	}
   }
   /* Get OS Default for Subnet Mask */
   if (file_grep(ipfile,substr,line,sizeof(line),0) == 0) {
	pb = strchr(line,':');
	if (pb != NULL) { 
		pb += 2;  /*skip ": "*/
		i = strcspn(pb," \t");
		pb[i] = 0;
		if (fdebug) printf("subnet=%s\n",pb);
		if (rgsubnet[0] == 0) {  /* if not user-specified */
		    atoip(rgsubnet,pb);
		}
	}
   }
   /* Get OS Default for Default Gateway */
   if (file_grep(ipfile,gwystr,line,sizeof(line),0) == 0) {
	pb = strchr(line,':');
	if (pb != NULL) { 
		pb += 2;  /*skip ": "*/
		i = strcspn(pb," \t");
		pb[i] = 0;
		if (fdebug) printf("gwy ip=%s\n",pb);
		if (rggwyip[0] == 0) {  /* if not user-specified */
		    atoip(rggwyip,pb);
		    if (rggwyip[0] != 0)  /* if valid */
		       Get_Mac(rggwyip,rggwymac);
		}
	}
   }
   return(0);

#else
   char rtfile[] = "/proc/net/route";
   char arpfile[] = "/proc/net/arp";
   char snmpfile[] = "/etc/snmp/snmpd.conf";
   char alertfile[] = "/tmp/alert.arping";
   FILE *fprt;
   FILE *fparp;
   int fd, skfd; 
   uchar *pc;
   int rc = 0;
   int i,j;
   uchar bnetadr[4];
   uchar bgateadr[4];
   char gate_addr[128]; 
   char defcommunity[19] = "public"; 
   char buff[1024]; 
   char alertname[60];
   int num;
   struct ifreq ifr;

   /* Get the IP address and associated MAC address specified. */
   /* Local for eth0; Default Gateway; Alert Destination */

   /* Create a channel to the NET kernel. */
   if ((skfd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
        perror("socket");
        return(-3);
   }

   strcpy(ifr.ifr_name, ifname);
   if (ioctl(skfd, SIOCGIFHWADDR, &ifr) > 0)
	printf("ioctl(SIOCGIFHWADDR) error, errno = %d\n",errno);
   else {
	if (rgmymac[0] == 0xFF)   /* if not user-specified */
           memcpy(rgmymac, ifr.ifr_hwaddr.sa_data, 6);   
   }

   fd = skfd;   // get_socket_for_af(AF_INET);
   if (fd >= 0) {
        strcpy(ifr.ifr_name, ifname);
        ifr.ifr_addr.sa_family = AF_INET;
        if (ioctl(fd, SIOCGIFADDR, &ifr) < 0) {
	    printf("ioctl(SIOCGIFADDR) error, errno = %d\n",errno);
	    /* errno 99 here means that eth0 is not up/defined. */
	} else {
	    pc = &ifr.ifr_addr.sa_data[2];
	    if (fdebug)
	      printf("addr = %d.%d.%d.%d \n", pc[0],pc[1],pc[2],pc[3]);
 	    if (rgmyip[0] == 0)   /* if not user-specified */
               memcpy(rgmyip, pc, 4);   

            strcpy(ifr.ifr_name, ifname);
            if (ioctl(fd, SIOCGIFNETMASK, &ifr) < 0)
	        printf("ioctl(SIOCGIFNETMASK) error, errno = %d\n",errno);
            else {              // sizeof(struct sockaddr)
	       pc = &ifr.ifr_netmask.sa_data[2];
	       if (fdebug)
	          printf("subnet = %d.%d.%d.%d \n", pc[0],pc[1],pc[2],pc[3]);
 	       if (rgsubnet[0] == 0)   /* if not user-specified */
                  memcpy(rgsubnet, pc, 4); 
	    }
	}
	close(skfd);  /* done, close the socket */
   }

   /* Get the default gateway IP */
   /* cat /proc/net/route and save Gwy if Dest == 0 and Gateway != 0 */
   fprt = fopen(rtfile,"r");
   if (fprt == NULL) {
      fprintf(stderr,"route: Cannot open %s, errno = %d\n",rtfile,errno);
   } else {
      char rtfmt[]  = "%16s %128s %128s %X %d %d %d %128s %d %d %d\n";
      int  iflags, refcnt, use, metric, mss, window, irtt;
      char iface[16]; 
      char mask_addr[128], net_addr[128];
      ulong *pnet;
      ulong *pgate;

      pnet = (long *)&bnetadr[0];
      pgate = (long *)&bgateadr[0];
      while (fgets(buff, 1023, fprt)) {
        num = sscanf(buff, rtfmt,
                     iface, net_addr, gate_addr,
                     &iflags, &refcnt, &use, &metric, mask_addr,
                     &mss, &window, &irtt); 
        if (num < 10 || !(iflags & RTF_UP))
            continue;

	j = 6;
	for (i = 0; i < 4; i ++) {
            bnetadr[i] = htoi(&net_addr[j]);
            bgateadr[i] = htoi(&gate_addr[j]);
	    j -= 2;
	    }
        if ((*pnet == 0) && (*pgate != 0)) {  /* found default gateway */
	    if (fdebug)
	       printf("default gateway: %s, %d.%d.%d.%d\n",gate_addr,
                   bgateadr[0], bgateadr[1], bgateadr[2], bgateadr[3]);
 	    if (rggwyip[0] == 0)   /* if not user-specified */
	       memcpy(rggwyip,bgateadr,4);
	}
      } /*end while*/
      fclose(fprt);
   }  /*end-else good open */

   /* Get the default gateway MAC */
   /* scan the arp cache for a match with gateway */
   if (rggwyip[0] != 0) {
    char arping_cmd[128];
    sprintf(gate_addr,"%d.%d.%d.%d",
              rggwyip[0], rggwyip[1], rggwyip[2], rggwyip[3]);
    /* make sure the gateway is in the arp cache */
    sprintf(arping_cmd,"arping -I %s -c 2 %s >/dev/null \n",ifname,gate_addr);
    if (fdebug) printf(arping_cmd);
    system(arping_cmd);

    fparp = fopen(arpfile,"r");
    if (fparp == NULL) {
      fprintf(stderr,"arp: Cannot open %s, errno = %d\n",arpfile,errno);
    } else {
      char arpfmt[] = "%128s %16s %16s %128s %16s %16s\n";
      char iface[16], hw_type[16], rgflags[16], rgmask[16];
      char hw_addr[128], net_addr[128];

      while (fgets(buff, 1023, fparp)) {
        num = sscanf(buff, arpfmt,
                     net_addr, hw_type, rgflags, hw_addr,
		     rgmask,iface);
        if (num < 6) continue;
	if (fdebug)
	   printf("arp: %s %s\n",net_addr,hw_addr);
	if (strcmp(gate_addr,net_addr) == 0) {  /* found it */
	   if (fdebug)
	      printf("gateway mac: %s\n",hw_addr);
 	   if (rggwymac[0] == 0xFF)   /* if not user-specified */
	      atomac(rggwymac,hw_addr);
	}
      }
      fclose(fparp);
    }
   }  /*endif have a gateway ip */

   /* Get the Alert Destination IP */
   /* By default, attempt to obtain this from /etc/snmp/snmpd.conf. */
   /* cat /etc/snmp/snmpd.conf | grep trapsink |tail -n1 | cut -f2 -d' ' */
   alertname[0] = 0;  /* default to null string */
   fprt = fopen(snmpfile,"r");
   if (fprt == NULL) {
      fprintf(stderr,"snmp: Cannot open %s, errno = %d\n",snmpfile,errno);
   } else {
      // char snmpfmt[] = "%20s %60s\n";
      // char *keywd, *value;
      while (fgets(buff, 1023, fprt)) {
	/* parse each line */
	if (buff[0] == '#') continue;  /*skip comment lines*/
	/* skip leading whitespace here */
	j = strspn(&buff[0]," \t");
	if (strncmp(&buff[j],"com2sec",7) == 0) {  /* found community line */
	   /* usu like this: "com2sec demouser default  public" */
	   i = j + 7;
	   for (j = 0; j < 3; j++) {
	     num = strspn(&buff[i]," \t");
	     i += num;
	     num = strcspn(&buff[i]," \t\r\n");
	     if (j < 2) i += num;
	   }
	   buff[i+num] = 0;
 	   if (fsetcommunity == 0) {  /* if not user-specified */
	      strcpy(rgcommunity,&buff[i]);
	      strcpy(defcommunity,&buff[i]);
	   }
	}
	if (strncmp(&buff[j],"trapsink",8) == 0) {  /* found trapsink line */
	   num = strspn(&buff[j+8]," \t");
	   i = j + 8 + num;
	   if (buff[i] == '`') continue;
	   num = strcspn(&buff[i]," \t\r\n");
	   strncpy(alertname,&buff[i],num);   /* save alert destination */
	   alertname[num] = 0;
	   i += num;
	   num = strspn(&buff[i]," \t"); /*skip whitespace*/
	   i += num;
	   num = strcspn(&buff[i]," \t\r\n");  /*span next word*/
	   if (num != 0) {  /* there is another word, that is community */
 	      if (fsetcommunity == 0) {  /* if not user-specified */
	         strncpy(rgcommunity,&buff[i],num); /* save community */
		 rgcommunity[num] = 0;
		 }
	   } else {  /*trapsink node with no community*/
	      /* use previously discovered default community from above */
	      strcpy(rgcommunity,defcommunity);
	   }
	   /* dont break, keep looking, use the last one */
	}
      } /*end while*/
      fclose(fprt);
      if (fdebug) 
	printf("snmp alertname=%s community=%s\n",alertname,rgcommunity);
   }  /*end else snmpfile*/
   
   /* Get the Alert Destination MAC from the alertname. */
   if (alertname[0] != 0) {
      char arping_cmd[128];
      char *pb, *pm, *px;
      int num, i;

      if (strncmp(alertname,"localhost",9) == 0)
      {   /* localhost (self) is the SNMP alert destination */
	 if (rgdestip[0] == 0)    /* if not user-specified */
	    memcpy(rgdestip,rgmyip,4);
	 if (rgdestmac[0] == 0xFF)   /* if not user-specified */
	    memcpy(rgdestmac,rgmymac,6);
      } else {
       /* make sure the destination is in the arp cache */
       sprintf(arping_cmd,"arping -I %s -c 2 %s |grep reply |tail -n1 >%s\n",
               ifname,alertname,alertfile);
       if (fdebug) printf(arping_cmd);
       system(arping_cmd);

       fparp = fopen(alertfile,"r");
       if (fparp == NULL) {
          fprintf(stderr,"alert: Cannot open %s, errno = %d\n",alertfile,errno);
       } else {
         while (fgets(buff, 1023, fparp)) {
	   /* should only run through loop once */
	   num = strcspn(buff," \t");    /* skip 1st word ("Unicast") */
	   i = strspn(&buff[num]," \t");
	   pb = &buff[num+i];
  	   if (strncmp(pb,"reply",5) == 0) {  /* valid output */
	      /* Find the ip address */
	      pb += 6 + 5;         /* skip "reply from " */
	      num = strcspn(pb," \t");
	      pb[num] = 0;
	      if (fdebug) printf("Alert ip=%s\n",pb);
 	      if (rgdestip[0] == 0) {  /* if ip not user-specified */
	         atoip(rgdestip,pb);
		 /* Now find the mac address */
		 pm = strchr(&pb[num+1],'[');
	         if (pm == NULL) pm = &pb[num+2];  /* just in case */
	         pm++;
	         px = strchr(pm,']');
	         if (px == NULL) px = pm + 17;    /* just in case */
	         px[0] = 0;
	         if (fdebug) printf("Alert mac=%s\n",pm);
 	         if (rgdestmac[0] == 0xFF)   /* if not user-specified */
	            atomac(rgdestmac,pm);
	      }
	      break;
	   } 
         } /*end while*/
         fclose(fparp);
	 if (fdebug) {
 	    if (rgdestip[0] == 0) 
		printf("No reply from Alert dest: %s\n",alertname);
	 }
       }  /*end else file opened*/
      } /*endif not local */
   }  /*endif have alertname*/

   return(rc);
#endif
} /* end Get_IPMac_Addr */

int ShowChanAcc(uchar bchan)
{
    LAN_RECORD LanRecord;
    int ret;
    uchar access;
    char *pstr;
    uchar *pb;
    int bret;

    bret = 0;
    if (bchan == lan_ch) pstr = "lan";
    else if (bchan == SER_CH) pstr = "ser";
    else pstr = "";
    ret = GetChanAcc(bchan, 0x40, &LanRecord);
    if (ret == 0) {
	pb = (uchar *)&bret;
	pb[0] = LanRecord.data[0];
	pb[1] = LanRecord.data[1];
    }
    access = LanRecord.data[0];
    printf("GetChanAcc(%s), ret = %d, new value = %02x\n",
		pstr,ret,access);
    switch (access & 0x03) {
	     case 0: printf("  Access = Disabled, ");     break;
	     case 1: printf("  Access = Pre-Boot, ");     break;
	     case 2: printf("  Access = Always Avail, "); break;
	     case 3: printf("  Access = Shared, ");       break;
	     }
    if (access & 0x20) printf("PEF Alerts Disabled\n"); /*0*/
    else printf("PEF Alerts Enabled\n");   /*1*/
    return(bret);
}

int SetupSerialOverLan( void )
{
	uchar requestData[24];
	uchar responseData[MAX_BUFFER_SIZE];
	int responseLength = MAX_BUFFER_SIZE;
	int status;
	uchar completionCode;

	memset(requestData, 0, sizeof(requestData));  /* zero-fill */
	requestData[0] = 0x00;
	requestData[1] = SOL_ENABLE_PARAM;
	requestData[2] = SOL_ENABLE_FLAG;
        status = ipmi_cmd(SET_SOL_CONFIG, requestData,3,responseData,
                        &responseLength, &completionCode, fdebug); 
	if (status == ACCESS_OK) {
		switch( completionCode ) {
		   case 0x00: /* success */
			break;
		   case 0xC1: /* unsupported command */
			SELprintf("SetupSerialOverLan: SOL not available on this platform\n");
			return 0;
		   default: /* other error */
			SELprintf("SetupSerialOverLan: SOL_ENABLE_PARAM ccode=%x\n",
					completionCode);
			return -1;
			break;
		}
	} else {
		SELprintf( "SET_SOL_CONFIG, enable SOL failed\n" );
	 	return -1;
 	}

	requestData[0] = 0x00;
	requestData[1] = SOL_AUTHENTICATION_PARAM;
	requestData[2] = 0x00; // block
	requestData[3] = 0x00; // selector
        status = ipmi_cmd(GET_SOL_CONFIG, requestData,4,responseData,
                        &responseLength, &completionCode, fdebug); 
	if (status == ACCESS_OK) {
		if( completionCode ) {
			SELprintf("SetupSerialOverLan: GET_SOL_AUTHENTICATION_PARAM code=%x\n",
				completionCode);

			return -1;
		}
	} else {
		SELprintf( "SET_SOL_CONFIG, get SOL authentication failed\n" );
	 	return -1;
 	}

	requestData[0] = 0x00;
	requestData[1] = SOL_AUTHENTICATION_PARAM;
	requestData[2] = SOL_PRIVILEGE_LEVEL_USER | ( responseData[1] & 0x80 );
	status = ipmi_cmd(SET_SOL_CONFIG, requestData,3,responseData,
			&responseLength, &completionCode, fdebug);
	if (status == ACCESS_OK) {
		if( completionCode ) {
			SELprintf("SetupSerialOverLan: SET_SOL_AUTHENTICATION_PARAM code=%x\n",
				completionCode);

			return -1;
		}
	} else {
		SELprintf( "SET_SOL_CONFIG, set SOL authentication failed\n" );
	 	return -1;
 	}

	requestData[0] = 0x00;
	requestData[1] = SOL_BAUD_RATE_PARAM;
	requestData[2] = SOL_PREFERRED_BAUD_RATE;
	status = ipmi_cmd(SET_SOL_CONFIG, requestData,3,responseData,
			&responseLength, &completionCode, fdebug);
	if (status == ACCESS_OK) {
	   if( completionCode ) {
		SELprintf("SetupSerialOverLan: SET SOL_PREFFERED_BAUD code=%x\n",
			completionCode);

		return -1;
	   }
	} else {
		SELprintf( "SET_SOL_CONFIG, set SOL BAUD failed\n" );
	 	return -1;
 	}
	return 0;
}  /* SetupSerialOverLan */


char guidstr[40];  /* 36 bytes + 1 null */

#ifdef METACOMMAND
int i_xml(int argc, char **argv)
#else
#ifdef WIN32
int __cdecl
#else
int
#endif
main(int argc, char **argv)
#endif
{
   int ret;
   PEF_RECORD PefRecord;
   LAN_RECORD LanRecord;
   int i, idx, j;
   int prod_id;
   char c;

   // progname = argv[0];
   printf("<! %s ver %s >\n",progname,progver);

   while ((c = getopt(argc, argv,"dersxDn:I:M:S:G:H:A:B:C:p:u:i:L:T:V:J:EYF:P:N:R:U:?")) != EOF)
      switch(c) {
          case 'x': fdebug = 1;     break;
          case 'r': freadonly = 1; fenable = 0; break;
          case 'd': fenable = 0; fdisable = 1; freadonly = 0; break;
          case 'e': fenable = 1; fdisable = 0; freadonly = 0; break;
          case 'D': lan_dhcp = 1;   break;
          case 's': fgetser = 1;    break;
          case 'I':      /* My IP Address (eth0) */
		atoip(rgmyip,optarg);
		break;
          case 'M':      /* My MAC Address */
		atomac(rgmymac,optarg);
		break;
          case 'S':      /* Subnet IP Address */
		atoip(rgsubnet,optarg);
		break;
          case 'G':      /* Gateway IP Address */
		atoip(rggwyip,optarg);
		break;
          case 'H':      /* Gateway MAC Address */
		atomac(rggwymac,optarg);
		break;
          case 'A':      /* Alert Dest IP Address */
		atoip(rgdestip,optarg);
		break;
          case 'B':      /* Alert Dest MAC Address */
		atomac(rgdestmac,optarg);
		break;
          case 'C':      /* Community String */
		if (strlen(optarg) > 18) optarg[18] = 0;
		fsetcommunity = 1;
		strcpy(rgcommunity,optarg);
		break;
	  case 'u': 	 /* username to set */
		myuser = strdup_(optarg);  /*remote username */
		break;
	  case 'p':      /* Password to set*/
		fpassword = 1;
		if (strlen(optarg) > 16) optarg[16] = 0;
		strcpy(passwordData,optarg);
                if (fdebug) printf("Password = %s\n",passwordData);
		break;
          case 'L': 
		lan_ch = atoi(optarg);
		if (lan_ch >= MAXCHAN) lan_ch = LAN_CH; /*set back to default*/
		break;
          case 'n':      /* number/index in PEF table to insert new entry */
		pefnum = atoi(optarg);
		if (pefnum > pefmax) pefnum = pefmax;
		break;
	  case 'i':      /* eth interface (ifname) */
          	fsetifn = 1;
		if (strlen(optarg) > 16) optarg[16] = 0;
		strcpy(ifname,optarg);
                if (fdebug) printf("ifname = %s\n",ifname);
		break;
          case 'N':    /* nodename */
          case 'U':    /* remote username */
          case 'P':    /* remote password */
          case 'R':    /* remote password */
          case 'E':    /* get password from IPMI_PASSWORD environment var */
          case 'F':    /* force driver type */
          case 'T':    /* auth type */
          case 'J':    /* cipher suite */ 
          case 'V':    /* priv level */
          case 'Y':    /* prompt for remote password */
                parse_lan_options(c,optarg,fdebug);
                break;
	  default:
                printf("Usage: %s [-drsxD -n pefnum -i eth1 ]\n", progname);
                printf("         \t [-I ipadr -M macadr -S subnet]\n");
                printf("         \t [-G gwyip -H gwymac -L lan_channel_num]\n");
                printf("         \t [-A alertip -B alertmac -C community ]\n");
                printf("         \t [-u username_to_set -p password_to_set]\n");
                printf("         \t [-NUPREFTVY] \n");
                printf("where -d  Disables BMC LAN & PEF\n");
                printf("      -r  Read-only BMC LAN & PEF settings\n");
                printf("      -s  Show some Serial settings also \n");
                printf("      -x  Show eXtra debug messages\n");
                printf("      -D  Use DHCP instead of static IP\n");
		print_lan_opt_usage();
                exit(1);
      }

   ret = GetDeviceID( &LanRecord);
   if (ret != 0) {
        show_outcome(progname,ret);  
	ipmi_close_();
	exit(ret);
   } else {  /* success */
      uchar ipmi_maj, ipmi_min;
      ipmi_maj = LanRecord.data[4] & 0x0f;
      ipmi_min = LanRecord.data[4] >> 4;
      // printf("-- BMC version %x.%x, IPMI version %d.%d \n",
      //        LanRecord.data[2],  LanRecord.data[3],
      //        ipmi_maj, ipmi_min);
      if (ipmi_maj == 0)  fIPMI10 = 1;
      else if (ipmi_maj == 1 && ipmi_min < 5) fIPMI10 = 1; 
      else  fIPMI10 = 0;    /* >= IPMI 1.5 is ok */
      if (ipmi_maj >= 2) fIPMI20 = 1;
      if (fIPMI10) {
         printf("This IPMI v%d.%d system does not support PEF records.\n",
			 ipmi_maj,ipmi_min);
	 fIPMI10 = 1;
	 /* wont handle PEF, but let it continue and look for BMC LAN anyway */
         }
      prod_id = LanRecord.data[9] + (LanRecord.data[10] << 8);
      /* check Device ID response for Manufacturer ID = 0x0322 (NSC) */
      if ((LanRecord.data[6] == 0x22) && (LanRecord.data[7] == 0x03)) {
	 fmBMC = 1;  /*NSC mBMC*/
	 if (pefnum == 12) pefnum = 10;  /* change default pefnum to 0x0a */
	 pefdesc = &pefdesc2[0];
	 pefmax = 30;
      } else {  /* else assume Intel = 0x000157 */
	 fmBMC = 0;  /*Intel Sahalee BMC*/
	 pefdesc = &pefdesc1[0];
	 pefmax  = 20;
	 /* also check for ia64, and set chan_pefon, chan_pefoff accordingly. */
	 if (prod_id == 0x0100) { /* Intel Tiger 2 */
	   chan_pefon  = CHAN_ACC_PEFON64;
	   chan_pefoff = CHAN_ACC_PEFOFF64;
	 }
      }
   }

   ret = ipmi_getpicmg( &LanRecord.data[0], 16, fdebug);
   if (ret == 0) fpicmg = 1;

   {  /*get IPMI SystemGUID */
     uchar *presp;
     int sresp; 
     uchar cc;
     char *s;

     presp = &LanRecord.data[0];
     sresp = sizeof(LanRecord);
     ret = ipmi_cmd(GET_SYSTEM_GUID,NULL,0,presp,&sresp,&cc,fdebug);
     if (fdebug) printf("system_guid: ret = %d, cc = %x\n",ret,cc);
     if (ret == 0 && cc == 0) {
	j = 0;
        for (i=0; i < 16; i++) {
	   if ((i == 4) || (i == 6) || (i == 8) || (i == 10)) s = "-";
           else s = "";
           sprintf(&guidstr[j],"%s%02x",s,presp[i]);
	   j += 2 + strlen(s);
	}
        guidstr[j] = 0;
     }  else {
	strcpy(guidstr,"error");
     }
   }

   /* Get the BMC LAN channel & match it to an OS eth if. */
   i = GetBmcEthDevice();
   if (i == -2) {  /* no lan channels */
	printf("This system does not support BMC LAN channels.\n");
	ipmi_close_();
	exit(1);
   } else if (i < 0) {  /* mac not found, use platform defaults */
	if (prod_id == 0x001b) /*Intel TIGPR2U*/ i = 1;
	else i = 0;   /*default to first channel*/
   }
   if (fsetifn == 0) sprintf(ifname,"eth%d",i);

   ret = ReadSELinfo();
   if (ret == 0) {  /* valid so continue */

    if (!fIPMI10) {
      // printf("\n%s: GetPefEntry ...\n",progname);
      for (idx = 1; idx <= pefmax; idx++)
      {
         ret = GetPefEntry( 0x06, (ushort)idx, &PefRecord);
         if (ret == 0) {    // Show the PEF record
            uchar * pc; int sz;
            pc = (uchar *)&PefRecord;
            sz = 21;        // sizeof(PEF_RECORD) = 21
	    if (PefRecord.sensor_type == 0) {
	      if (idx <= pefnum) 
                 printf("PEFilter(%02d): empty\n",idx);
	    } else {
              memcpy(pef_array[idx], &PefRecord, sz);
	      if (PefRecord.fconfig & 0x80) pc = "enabled";
	      else pc = "disabled";
	      i = PefRecord.rec_id;
              printf("PEFilter(%02d): %02x %s event - %s\n",
		  idx, PefRecord.sensor_type, pefdesc[i],pc);
	    }
	    if (fdebug) {  /* show raw PEFilter record */
		pc = &PefRecord.rec_id;
		printf("raw PEF(%.2d):  ",pc[0]);
		for (i = 0; i < sz; i++) printf("%02x ",pc[i]);
		printf("\n");
	    }
         } else 
            printf("GetPefEntry(%d), ret = %d\n",idx,ret);
      }
      if (fdebug) ShowPef();
      ret = GetPefEntry(0x01, 0,(PEF_RECORD *)&LanRecord);  
      if (ret == 0) printf("PEF Control: %02x\n",LanRecord.data[0]);
      ret = GetPefEntry(0x02, 0,(PEF_RECORD *)&LanRecord);
      if (ret == 0) printf("PEF Action: %02x\n",LanRecord.data[0]);
      ret = GetPefEntry(0x03, 0,(PEF_RECORD *)&LanRecord);
      if (ret == 0) printf("PEF Startup Delay: %d sec\n",LanRecord.data[0]);
      if (!fmBMC) {
		ret = GetPefEntry(0x04, 0,(PEF_RECORD *)&LanRecord);
		if (ret == 0) printf("PEF Alert Startup Delay: %d sec\n",
					LanRecord.data[0]);
		/* fmBMC gets cc=0x80 here */
      }
      ret = GetPefEntry(0x09, 1,(PEF_RECORD *)&LanRecord);
      if (ret == 0) printf("PEF Alert Policy: %02x %02x %02x %02x\n",
				LanRecord.data[0], LanRecord.data[1], 
				LanRecord.data[2], LanRecord.data[3]);

      if (!freadonly) {  /* fenable */
       printf("\n%s: SetPefEntry(%d) ...\n",progname,pefnum);
       for (idx = 1; idx <= pefnum; idx++)
       {
	 // Set & Enable all PEF records
         memset(&PefRecord.rec_id,0,sizeof(PEF_RECORD));
         PefRecord.rec_id  = idx; /* next record, or user-specified */
	 if (idx < pefnum) {  /* pefnum defaults to 0x0c */
	    if (pef_array[idx][7] == 0) /*empty pef record, set to default*/
               memcpy(&PefRecord.rec_id,pef_defaults[idx],sizeof(PEF_RECORD));
	    else /* set config however it was previously */
               memcpy(&PefRecord.rec_id,pef_array[idx],sizeof(PEF_RECORD));
	 } else {  /* new OS Crit Stop entry */
            // Set PEF values for 0x20, OS Critical Stop event 
            PefRecord.genid1        = 0xff;
            PefRecord.genid2        = 0xff;
            PefRecord.sensor_type   = 0x20; /* OS Critical Stop */
            PefRecord.sensor_no     = 0xff;
            PefRecord.event_trigger = 0x6f;
            PefRecord.data1         = 0xff;
            PefRecord.mask1         = 0x00;
	 }
         if (fdisable) {
	    if (idx < pefnum) PefRecord.fconfig = 0x40; /* disabled, preset */
	    else PefRecord.fconfig  = 0x00;             /* disabled, software */
            PefRecord.action   = 0x00;
            PefRecord.policy   = 0x00;
	 } else {
            if (PefRecord.sensor_type != 0) { /* not an empty PEF entry */
              PefRecord.action   = 0x01;
              PefRecord.policy   = 0x01;
	      if (idx >= pefnum) 
	          PefRecord.fconfig  = 0x80;         /* enabled, software */
	      else {          /* special handling for presets, 1 thru 11 */
	          PefRecord.fconfig  = 0x80;    /* enabled, software */
                  ret = SetPefEntry(&PefRecord);
                  if (ret == 0 && fdebug) 
                     printf("SetPefEntry(%d/80) successful\n",PefRecord.rec_id);
		  PefRecord.fconfig  = 0xC0;     /* enabled, preset */
	      }
	    }  /*endif not empty*/
	 }
         {  // Show the new PEF record before setting it.
            uchar * pc; int sz;
            pc = (uchar *)&PefRecord;
            sz = 21;
            printf("PEFilter(%d): ",PefRecord.rec_id);
            for (i = 0; i < sz; i++) printf("%02x ",pc[i]);
                  printf("\n");
         } 
         ret = SetPefEntry(&PefRecord);
         if (ret == 0 && fdebug) 
             printf("SetPefEntry(%d) successful\n", PefRecord.rec_id); 
       }  /*end for*/
      }  /*if set PEF*/
     }  /*end if not fIPMI10*/

     // printf("\n%s: GetLanEntry ...\n",progname);
     printf("<syscfg:LanConfigurationV001 UUID=%s>\n",guidstr);
     for (idx = 0; idx < NLAN; idx++)
     {
	 int ival;
	 ival = lanparams[idx].cmd;
         if (ival == 8 || ival == 9) continue;	  /* not implemented */
         if (ival >= 96 && ival <= 98) continue;  /* not implemented */
         if (fmBMC && (ival >= 192 && ival <= 194)) continue;  /* no DHCP */
         if (!fIPMI20 && (ival >= 20 && ival <= 25)) continue; /*VLAN if 2.0*/
         if (ival == 201) {
             ret = GetChanAcc(lan_ch, 0x40, &LanRecord);
#if 0
	     uchar *pb;
             j = ShowChanAcc(lan_ch);  /*calls GetChanAcc() */
	     pb = (uchar *)&j;
	     LanRecord.data[0] = pb[0];
	     LanRecord.data[1] = pb[1];
	     if (j == 0) ret = -1;
	     else ret = 0;
#endif
	 } else {
             ret = GetLanEntry((uchar)ival, &LanRecord);
	 }
         if (ret == 0) {    // Show the LAN record
            uchar * pc; int sz;
 	    char *prefix;
            pc = (uchar *)&LanRecord;
            sz = lanparams[idx].sz; 
	    prefix = lanparams[idx].desc;
	    if (prefix[0] == '<') {  /* is an xml param */
                 printf("   %s",prefix);
	    } else continue;  /* don't show non-xml params at all */
	    // } else printf("Lan Param(%d) %s: ",ival,prefix);
	    if (ival == 1) {
		authmask = pc[0]; /* auth type support mask */
		/* if (fmBMC) authmask is usually 0x15, else 0x14 */
		}
	    if (ival == 16) { printf("%s",pc);      /* community string */
	    } else {  /* print results for others */
              for (i = 0; i < sz; i++) {
	  	if (ival == 3 || ival == 6 || ival == 12 || ival == 14 || 
		    ival == 192) {      /* show IP addresses in dec */
		   if (i == (sz - 1)) printf("%d",pc[i]);
		   else printf("%d.",pc[i]);  
		} else if (ival == 4) {    /* IP address source */
		   /* pc[0]: 01 = static, 02 = DHCP, 03 = BIOS DHCP  */
		   if (pc[0] == 1) printf("Static");
		   else printf("DHCP");
		} else if (ival == 19)  {  /* dest addr */
		   char *sepstr;
		   if (i <= 2) continue;  /* 01 00 00 */
		   if (i > 2 && i < 7) {  /* IP address in dec */
			if (i == 6) sepstr = "";
			else if (i >=3 && i < 6) sepstr = ".";
			else sepstr = " ";
		        printf("%d%s",pc[i],sepstr);
		   } else if (i == 7) {
	              suffix[0] = '<';
	              suffix[1] = '/';
	              strcpy(&suffix[2],&prefix[1]);
		      prefix = "<ipmi15:AlertMacAddress>";
		      printf("%s\n   %s%02x-",suffix,prefix,pc[i]);
		   } else if (i == 12) printf("%02x",pc[i]);
		   else printf("%02x-",pc[i]);  
		}   /*endif ival==19*/
		else if (ival == 5 || ival == 13 || ival == 15 || 
			 ival == 193) {   /* Mac addresses */
		   if (i == 5) printf("%02x",pc[i]);
		   else printf("%02x-",pc[i]);
		}
		else printf("%02x ",pc[i]);  
	      }  /*end for*/
	    }  /*end else*/
	    if (prefix[0] == '<') {
	         suffix[0] = '<';
	         suffix[1] = '/';
	         strcpy(&suffix[2],&prefix[1]);
	    }  else suffix[0] = 0;
            printf("%s\n",suffix);
         } else  /* ret != 0 */
            printf("GetLanEntry(%d), ret = %d\n",ival,ret);
     }  /*end for*/
     if (fmBMC) lan_access = 0x04; 
     else lan_access = 0x04;
#if 0
     GetUser(lan_ch,1);
     if (!fmBMC) {  /* mBMC doesn't support more than 1 user */
	GetUser(lan_ch,2);
	GetUser(lan_ch,3);
     }
#endif
     printf("</syscfg:LanConfigurationV001>\n");


     if (fgetser) {
       printf("<syscfg:SerialConfigurationV001 UUID=%s>\n",guidstr);
       if (fmBMC)   /* mBMC doesn't support serial */
	  printf("No serial channel support on this platform\n");	
       else
       for (idx = 0; idx < NSER; idx++) {
   	 int ival;
	 ival = serparams[idx].cmd;
         // if (ival == 9) continue; /* not implemented */
         if (ival == 201) {
             ret = GetChanAcc(SER_CH, 0x40, &LanRecord);
	 } else {
             ret = GetSerEntry((uchar)ival, &LanRecord);
	 }
         if (ret == 0) {    // Show the SER record
            uchar * pc; int sz;
	    char *prefix;
            pc = (uchar *)&LanRecord;
            sz = serparams[idx].sz; 
	    prefix = serparams[idx].desc;
	    if (prefix[0] == '<') {  /* is an xml param */
                 printf("   %s",prefix);
	    } else continue;   /* dont show non-xml params at all */
	    // } else printf("Serial Param(%d) %s: ",ival,prefix);
	    /* show the data */
            if (ival == 10) {  /* modem init string */
                pc[sz] = 0;
		if (prefix[0] == '<') printf("%s",&pc[1]);
                else printf("%02x %s",pc[0],&pc[1]);
                }
            else if ((ival >= 11 && ival <= 13) || ival == 15) { /*strings*/
                printf("%s",pc);
                }
            else {
              for (i = 0; i < sz; i++) {
		printf("%02x ",pc[i]);  /* show in hex */
		}
              //printf("\n");
	    } /*end else*/
	    if (prefix[0] == '<') {  /*do xml end tag*/
	         suffix[0] = '<';
	         suffix[1] = '/';
	         strcpy(&suffix[2],&prefix[1]);
	    }  else suffix[0] = 0;
            printf("%s\n",suffix);
	 } /*endif ok*/
       }  /*end for*/
       printf("</syscfg:SerialConfigurationV001>\n");
      }  /*endif fgetser*/

      if (!freadonly && !fipmi_lan)  /* Set not valid via ipmi_lan. */
      {
	 /* Set LAN parameters.  fenable or fdisable */
         printf("\n%s: SetLanEntry ...\n",progname);
         if (fdisable) {
	     if (!fIPMI10) {
               ret = DisablePef();
               printf("DisablePef, ret = %d\n",ret);
	     }
	 } else {  /*fenable*/
             ret = SetChanAcc(lan_ch, 0x80, chan_pefoff);
             if (fdebug) printf("SetChanAcc(lan/active), ret = %d\n",ret);
             ret = SetChanAcc(lan_ch, 0x40, chan_pefoff);
             if (fdebug) printf("SetChanAcc(lan/nonvol), ret = %d\n",ret);
	     j = SetUser(myuser,passwordData);
	     }
         printf("SetChanAcc(lan), ret = %d\n",ret);
	 j = ShowChanAcc(lan_ch);
	 if (authmask == 0) authmask = 0x17;
	 LanRecord.data[0] = (0x16 & authmask); /*Callback level*/
	 LanRecord.data[1] = (0x16 & authmask); /*User level    */
	 LanRecord.data[2] = (0x16 & authmask); /*Operator level*/
	 LanRecord.data[3] = (0x16 & authmask); /*Admin level   */
	 LanRecord.data[4] = 0;
	 if (fdebug) printf("SetLanEntry(2): %02x %02x %02x %02x %02x\n",
	 		LanRecord.data[0],LanRecord.data[1],LanRecord.data[2],
	 		LanRecord.data[3],LanRecord.data[4]);
         ret = SetLanEntry(2, &LanRecord, 5);
	 printf("SetLanEntry(2), ret = %d\n",ret);
	 // LanRecord.data[0] = 0x01;  

         /* Get the values to use from Linux eth0, etc. */
         ret = Get_IPMac_Addr();
	 if (ret == -3) { ipmi_close_(); exit(1); }
	 if (lan_dhcp) {  /* use DHCP */
	    LanRecord.data[0] = 0x03;  /* address source = BIOS using DHCP */
            ret = SetLanEntry(4, &LanRecord, 1);
            printf("SetLanEntry(4), ret = %d\n",ret);
	    /* DHCP also relates to lan params 192, 193, 194 */
	    /* SSU/BIOS set up DHCP Server IP in param 192. */
	 } else {    /* use static IP */
          printf("%s \t\tip=%d.%d.%d.%d mac=%02x:%02x:%02x:%02x:%02x:%02x\n",
		ifname, rgmyip[0], rgmyip[1], rgmyip[2], rgmyip[3], 
		rgmymac[0], rgmymac[1], rgmymac[2], rgmymac[3], 
		rgmymac[4], rgmymac[5]);
	  if (rgmyip[0] != 0) {
	    memcpy(&LanRecord,rgmyip,4);
            ret = SetLanEntry(3, &LanRecord, 4);
            printf("SetLanEntry(3), ret = %d\n",ret);
	    LanRecord.data[0] = 0x01;    /* static IP address source */
            ret = SetLanEntry(4, &LanRecord, 1);
            printf("SetLanEntry(4), ret = %d\n",ret);
	    memcpy(&LanRecord,rgmymac,6);
            ret = SetLanEntry(5, &LanRecord, 6);
            printf("SetLanEntry(5), ret = %d\n",ret);
	    if (rgsubnet[0] == 0) { /* set subnet to default (255.255.255.0) */
		rgsubnet[0] = 255; rgsubnet[1] = 255; rgsubnet[2] = 255; }
	    memcpy(&LanRecord,rgsubnet,4);
            ret = SetLanEntry(6, &LanRecord, 4);
            printf("SetLanEntry(6), ret = %d\n",ret);
	    LanRecord.data[0] = 0x01;    /*grat arp*/
            ret = SetLanEntry(10, &LanRecord, 1);
	    printf("SetLanEntry(10), ret = %d\n",ret);
	    LanRecord.data[0] = 0x04;    /*grat arp interval*/
            ret = SetLanEntry(11, &LanRecord, 1);
	    printf("SetLanEntry(11), ret = %d\n",ret);
	  } else {      /* error, don't continue */
            printf("Missing IP Address, can't continue\n");
	    ipmi_close_();
	    exit(1);
	  }
	  if (rggwyip[0] != 0) {
	    if (rggwymac[0] == 0xFF) /* if MAC not set yet */
               ret = Get_Mac(rggwyip,rggwymac);
            printf("gateway \tip=%d.%d.%d.%d mac=%02x:%02x:%02x:%02x:%02x:%02x\n",
		rggwyip[0], rggwyip[1], rggwyip[2], rggwyip[3],
		rggwymac[0], rggwymac[1], rggwymac[2], rggwymac[3], 
		rggwymac[4], rggwymac[5]);
	    /* Set the Default Gateway IP & MAC */
	    memcpy(&LanRecord,rggwyip,4);
            ret = SetLanEntry(12, &LanRecord, 4);
            printf("SetLanEntry(12), ret = %d\n",ret);
	    memcpy(&LanRecord,rggwymac,6);
            ret = SetLanEntry(13, &LanRecord, 6);
	    printf("SetLanEntry(13), ret = %d\n",ret);
	  }
	 } /* end-else static IP */
	 ret = SetupSerialOverLan();
	 SELprintf("SetupSerialOverLan: ret = %d\n",ret);
	 if (rgdestip[0] == 0) {
            printf("alert dest \taddress not specified\n");
	 } else {   /* valid alert dest ip */
	    if (rgdestmac[0] == 0xFF) /* if MAC not set from defaults/user */
               ret = Get_Mac(rgdestip,rgdestmac);
            printf("alert dest \tip=%d.%d.%d.%d mac=%02x:%02x:%02x:%02x:%02x:%02x\n",
		rgdestip[0], rgdestip[1], rgdestip[2], rgdestip[3],
		rgdestmac[0], rgdestmac[1], rgdestmac[2], rgdestmac[3], 
		rgdestmac[4], rgdestmac[5]);
            printf("snmp community \t%s\n",rgcommunity);
	    /* Only need the SNMP community if there is an Alert Destination */
	    memset(&LanRecord.data[0], 0, 18);  /* make sure zero-filled */
	    strcpy(&LanRecord.data[0],rgcommunity);
            ret = SetLanEntry(16, &LanRecord, 18);  
            printf("SetLanEntry(16), ret = %d\n",ret);
	    /* Set Alert Destination Type */
	    LanRecord.data[0] = 0x01;     /* dest id = 1 */
	    LanRecord.data[1] = 0x00;     /* dest type = PET, no ack */
	    LanRecord.data[2] = 0x01;     /* ack timeout / retry interval */
	    LanRecord.data[3] = 0x00;     /* no retries */
	    // LanRecord.data[4] = 0x69; 
            ret = SetLanEntry(18, &LanRecord, 4);
            printf("SetLanEntry(18), ret = %d\n",ret);
	    /* Set the Alert Destination IP & MAC */
	    LanRecord.data[0] = 0x01;     /* dest id = 1 */
	    LanRecord.data[1] = 0x00; 
	    LanRecord.data[2] = 0x00; 
	    memcpy(&LanRecord.data[3],rgdestip,4);
	    memcpy(&LanRecord.data[7],rgdestmac,6);
            ret = SetLanEntry(19, &LanRecord, 13);
            printf("SetLanEntry(19), ret = %d\n", ret);

	    /* Now enable/disable PEF since we have an Alert destination. */
	    if (!fdisable && !fIPMI10) { /*fenable*/
              ret = EnablePef();
              printf("EnablePef, ret = %d\n",ret);
	      j = ShowChanAcc(lan_ch);
	    }
	 } /*endif valid alert*/
      }  /*endif not readonly*/

   }  /* if ok */
   ipmi_close_();
   show_outcome(progname,ret); 
   exit(ret);
}  /* end main()*/

/* end xmlconfig.c */
