/*
 * ProFTPD: mod_vroot -- a module implementing a virtual chroot capability
 *                       (without requiring root privs) via the FSIO API
 *
 * Copyright (c) 2002-2006 TJ Saunders
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307, USA.
 *
 * As a special exemption, TJ Saunders and other respective copyright holders
 * give permission to link this program with OpenSSL, and distribute the
 * resulting executable, without including the source code for OpenSSL in the
 * source distribution.
 *
 * This is mod_vroot, contrib software for proftpd 1.2 and above.
 * For more information contact TJ Saunders <tj@castaglia.org>.
 *
 * $Id: mod_vroot.c,v 1.11 2006/12/06 15:53:22 tj Exp tj $
 */

#include "conf.h"
#include "privs.h"

#define MOD_VROOT_VERSION 	"mod_vroot/0.7.2"

/* Make sure the version of proftpd is as necessary. */
#if PROFTPD_VERSION_NUMBER < 0x0001021004
# error "ProFTPD 1.2.10 or later required"
#endif

static const char *vroot_log = NULL;
static int vroot_logfd = -1;

static char vroot_cwd[PR_TUNABLE_PATH_MAX];
static char vroot_base[PR_TUNABLE_PATH_MAX];
static size_t vroot_baselen = 0;
static unsigned char vroot_engine = FALSE;

static unsigned int vroot_opts = 0;
#define	VROOT_OPT_ALLOW_SYMLINKS	0x0001

/* Support routines note: some of these support functions are borrowed from
 * pure-ftpd.
 */

static void strmove(register char *dst, register const char *src) {
  if (!dst || !src)
    return;

  while (*src != 0) {
    *dst++ = *src++;
  }

  *dst = 0;
}

static void vroot_clean_path(char *path) {
  char *p;

  if (path == NULL || *path == 0)
      return;

  while ((p = strstr(path, "//")) != NULL)
    strmove(p, p + 1);

  while ((p = strstr(path, "/./")) != NULL)
    strmove(p, p + 2);

  while (strncmp(path, "../", 3) == 0)
    path += 3;

  p = strstr(path, "/../");
  if (p != NULL) {
    if (p == path) {
      while (strncmp(path, "/../", 4) == 0)
        strmove(path, path + 3);

      p = strstr(path, "/../");
    }

    while (p != NULL) {
      char *next_elem = p + 4;

      if (p != path && *p == '/') 
        p--;

      while (p != path && *p != '/')
        p--;

      if (*p == '/')
        p++;

      strmove(p, next_elem);
      p = strstr(path, "/../");
    }
  }

  p = path;

  if (*p == '.') {
    p++;

    if (*p == '\0')
      return;

    if (*p == '/') {
      while (*p == '/') 
        p++;

      strmove(path, p);
    }
  }
}

static int vroot_lookup_path(char *path, size_t pathlen, const char *dir) {
  char buf[PR_TUNABLE_PATH_MAX], *bufp = NULL;

  memset(buf, '\0', sizeof(buf));
  memset(path, '\0', pathlen);

  if (strcmp(dir, ".") != 0)
    sstrncpy(buf, dir, sizeof(buf));
  else
    sstrncpy(buf, pr_fs_getcwd(), sizeof(buf));

  vroot_clean_path(buf);

  bufp = buf;

  if (strncmp(bufp, vroot_base, vroot_baselen) == 0) {
    bufp += vroot_baselen;
  }

  loop:
  if (bufp[0] == '.' &&
      bufp[1] == '.' &&
      (bufp[2] == '\0' ||
       bufp[2] == '/')) {
    char *tmp = NULL;

    tmp = strrchr(path, '/');
    if (tmp != NULL)
      *tmp = '\0';
    else
      *path = '\0';

    if (strncmp(path, vroot_base, vroot_baselen) == 0 ||
         path[vroot_baselen] != '/') {
      snprintf(path, pathlen, "%s/", vroot_base);
    }

    if (bufp[0] == '.' &&
        bufp[1] == '.' &&
        bufp[2] == '/') {
      bufp += 3;
      goto loop;
    }

  } else if (*bufp == '/') {
    snprintf(path, pathlen, "%s/", vroot_base);

    bufp += 1;
    goto loop;

  } else if (*bufp != '\0') {
    size_t buflen, tmplen;

    if (strstr(bufp, "..") != NULL) {
      errno = EPERM;
      return -1;
    }

    buflen = strlen(bufp) + 1;
    tmplen = strlen(path);

    if (tmplen + buflen >= pathlen) {
      errno = ENAMETOOLONG;
      return -1;
    }

    path[tmplen] = '/';
    memcpy(path + tmplen + 1, bufp, buflen);
  }

  /* Clean any unnecessary characters added by the above processing. */
  vroot_clean_path(path);

  return 0;
}

/* FS callbacks
 */

static int vroot_stat(pr_fs_t *fs, const char *path, struct stat *sbuf) {
  char vpath[PR_TUNABLE_PATH_MAX];

  if (session.curr_phase == LOG_CMD ||
      session.curr_phase == LOG_CMD_ERR ||
      (session.sf_flags & SF_ABORT) ||
      *vroot_base == '\0')
    /* NOTE: once stackable FS modules are supported, have this fall through
     * to the next module in the stack.
     */
    return stat(path, sbuf);

  if (vroot_lookup_path(vpath, sizeof(vpath), path) < 0)
    return -1;

  return stat(vpath, sbuf);
}

static int vroot_lstat(pr_fs_t *fs, const char *path, struct stat *sbuf) {
  char vpath[PR_TUNABLE_PATH_MAX];

  if (session.curr_phase == LOG_CMD ||
      session.curr_phase == LOG_CMD_ERR ||
      (session.sf_flags & SF_ABORT) ||
      *vroot_base == '\0')
    /* NOTE: once stackable FS modules are supported, have this fall through
     * to the next module in the stack.
     */
    return lstat(path, sbuf);

  if (vroot_lookup_path(vpath, sizeof(vpath), path) < 0)
    return -1;

  if (vroot_opts & VROOT_OPT_ALLOW_SYMLINKS) {
    if (lstat(vpath, sbuf) < 0)
      return -1;

    return stat(vpath, sbuf);
  }

  return lstat(vpath, sbuf);
}

static int vroot_rename(pr_fs_t *fs, const char *rnfm, const char *rnto) {
  char vpath1[PR_TUNABLE_PATH_MAX], vpath2[PR_TUNABLE_PATH_MAX];

  if (session.curr_phase == LOG_CMD ||
      session.curr_phase == LOG_CMD_ERR ||
      (session.sf_flags & SF_ABORT) ||
      *vroot_base == '\0')
    /* NOTE: once stackable FS modules are supported, have this fall through
     * to the next module in the stack.
     */
    return rename(rnfm, rnto);

  if (vroot_lookup_path(vpath1, sizeof(vpath1), rnfm) < 0 ||
      vroot_lookup_path(vpath2, sizeof(vpath2), rnto) < 0)
    return -1;

  return rename(vpath1, vpath2);
}

static int vroot_unlink(pr_fs_t *fs, const char *path) {
  char vpath[PR_TUNABLE_PATH_MAX];

  if (session.curr_phase == LOG_CMD ||
      session.curr_phase == LOG_CMD_ERR ||
      (session.sf_flags & SF_ABORT) ||
      *vroot_base == '\0')
    /* NOTE: once stackable FS modules are supported, have this fall through
     * to the next module in the stack.
     */
    return unlink(path);

  if (vroot_lookup_path(vpath, sizeof(vpath), path) < 0)
    return -1;

  return unlink(vpath);
}

static int vroot_open(pr_fh_t *fh, const char *path, int flags) {
  char vpath[PR_TUNABLE_PATH_MAX];

  if (session.curr_phase == LOG_CMD ||
      session.curr_phase == LOG_CMD_ERR ||
      (session.sf_flags & SF_ABORT) ||
      *vroot_base == '\0')
    /* NOTE: once stackable FS modules are supported, have this fall through
     * to the next module in the stack.
     */
    return open(path, flags, PR_OPEN_MODE);

  if (vroot_lookup_path(vpath, sizeof(vpath), path) < 0)
    return -1;

  return open(vpath, flags, PR_OPEN_MODE);
}

static int vroot_creat(pr_fh_t *fh, const char *path, mode_t mode) {
  char vpath[PR_TUNABLE_PATH_MAX];

  if (session.curr_phase == LOG_CMD ||
      session.curr_phase == LOG_CMD_ERR ||
      (session.sf_flags & SF_ABORT) ||
      *vroot_base == '\0')
    /* NOTE: once stackable FS modules are supported, have this fall through
     * to the next module in the stack.
     */
    return creat(path, mode);

  if (vroot_lookup_path(vpath, sizeof(vpath), path) < 0)
    return -1;

  return creat(vpath, mode);
}

static int vroot_link(pr_fs_t *fs, const char *path1, const char *path2) {
  char vpath1[PR_TUNABLE_PATH_MAX], vpath2[PR_TUNABLE_PATH_MAX];

  if (session.curr_phase == LOG_CMD ||
      session.curr_phase == LOG_CMD_ERR ||
      (session.sf_flags & SF_ABORT) ||
      *vroot_base == '\0')
    /* NOTE: once stackable FS modules are supported, have this fall through
     * to the next module in the stack.
     */
    return link(path1, path2);

  if (vroot_lookup_path(vpath1, sizeof(vpath1), path1) < 0 ||
      vroot_lookup_path(vpath2, sizeof(vpath2), path2) < 0)
    return -1;

  return link(vpath1, vpath2);
}

static int vroot_symlink(pr_fs_t *fs, const char *path1, const char *path2) {
  char vpath1[PR_TUNABLE_PATH_MAX], vpath2[PR_TUNABLE_PATH_MAX];

  if (session.curr_phase == LOG_CMD ||
      session.curr_phase == LOG_CMD_ERR ||
      (session.sf_flags & SF_ABORT) ||
      *vroot_base == '\0')
    /* NOTE: once stackable FS modules are supported, have this fall through
     * to the next module in the stack.
     */
    return symlink(path1, path2);

  if (vroot_lookup_path(vpath1, sizeof(vpath1), path1) < 0 ||
      vroot_lookup_path(vpath2, sizeof(vpath2), path2) < 0)
    return -1;

  return symlink(vpath1, vpath2);
}

static int vroot_readlink(pr_fs_t *fs, const char *path, char *buf,
    size_t max) {
  char vpath[PR_TUNABLE_PATH_MAX];

  if (session.curr_phase == LOG_CMD ||
      session.curr_phase == LOG_CMD_ERR ||
      (session.sf_flags & SF_ABORT) ||
      *vroot_base == '\0')
    /* NOTE: once stackable FS modules are supported, have this fall through
     * to the next module in the stack.
     */
    return readlink(path, buf, max);

  if (vroot_lookup_path(vpath, sizeof(vpath), path) < 0)
    return -1;

  return readlink(vpath, buf, max);
}

static int vroot_truncate(pr_fs_t *fs, const char *path, off_t length) {
  char vpath[PR_TUNABLE_PATH_MAX];

  if (session.curr_phase == LOG_CMD ||
      session.curr_phase == LOG_CMD_ERR ||
      (session.sf_flags & SF_ABORT) ||
      *vroot_base == '\0')
    /* NOTE: once stackable FS modules are supported, have this fall through
     * to the next module in the stack.
     */
    return truncate(path, length);

  if (vroot_lookup_path(vpath, sizeof(vpath), path) < 0)
    return -1;

  return truncate(vpath, length);
}

static int vroot_chmod(pr_fs_t *fs, const char *path, mode_t mode) {
  char vpath[PR_TUNABLE_PATH_MAX];

  if (session.curr_phase == LOG_CMD ||
      session.curr_phase == LOG_CMD_ERR ||
      (session.sf_flags & SF_ABORT) ||
      *vroot_base == '\0')
    /* NOTE: once stackable FS modules are supported, have this fall through
     * to the next module in the stack.
     */
    return chmod(path, mode);

  if (vroot_lookup_path(vpath, sizeof(vpath), path) < 0)
    return -1;

  return chmod(vpath, mode);
}

static int vroot_chown(pr_fs_t *fs, const char *path, uid_t uid, gid_t gid) {
  char vpath[PR_TUNABLE_PATH_MAX];

  if (session.curr_phase == LOG_CMD ||
      session.curr_phase == LOG_CMD_ERR ||
      (session.sf_flags & SF_ABORT) ||
      *vroot_base == '\0')
    /* NOTE: once stackable FS modules are supported, have this fall through
     * to the next module in the stack.
     */
    return chown(path, uid, gid);

  if (vroot_lookup_path(vpath, sizeof(vpath), path) < 0)
    return -1;

  return chown(vpath, uid, gid);
}

static int vroot_chroot(pr_fs_t *fs, const char *path) {
  char *tmp = NULL;

  if (!path || *path == '\0') {
    errno = EINVAL;
    return -1;
  }

  memset(vroot_base, '\0', sizeof(vroot_base));

  if (path[0] == '/' && path[1] == '\0')
    /* chrooting to '/', nothing needs to be done. */
    return 0;

  pr_fs_clean_path(path, vroot_base, sizeof(vroot_base));

  tmp = vroot_base;

  /* Advance to the end of the path. */
  while (*tmp != '\0')
    tmp++;

  for (;;) {
    tmp--;

    if (tmp == vroot_base || *tmp != '/')
      break;

    *tmp = '\0';
  }

  vroot_baselen = strlen(vroot_base);
  if (vroot_baselen >= sizeof(vroot_cwd)) {
    errno = ENAMETOOLONG;
    return -1;
  }

  session.chroot_path = "/";
  return 0;
}

static int vroot_chdir(pr_fs_t *fs, const char *path) {
  char vpath[PR_TUNABLE_PATH_MAX], *vpathp = NULL;

  if (session.curr_phase == LOG_CMD ||
      session.curr_phase == LOG_CMD_ERR ||
      (session.sf_flags & SF_ABORT) ||
      *vroot_base == '\0')
    /* NOTE: once stackable FS modules are supported, have this fall through
     * to the next module in the stack.
     */
    return chdir(path);

  if (vroot_lookup_path(vpath, sizeof(vpath), path) < 0)
    return -1;

  if (chdir(vpath) < 0)
    return -1;

  vpathp = vpath;

  if (strncmp(vpathp, vroot_base, vroot_baselen) == 0)
    vpathp += vroot_baselen;

  pr_fs_setcwd(vpathp);
  return 0;
}

static void *vroot_opendir(pr_fs_t *fs, const char *path) {
  char vpath[PR_TUNABLE_PATH_MAX];

  if (session.curr_phase == LOG_CMD ||
      session.curr_phase == LOG_CMD_ERR ||
      (session.sf_flags & SF_ABORT) ||
      *vroot_base == '\0')
    /* NOTE: once stackable FS modules are supported, have this fall through
     * to the next module in the stack.
     */
    return opendir(path);

  if (vroot_lookup_path(vpath, sizeof(vpath), path) < 0)
    return NULL;

  return opendir(vpath);
}

static int vroot_mkdir(pr_fs_t *fs, const char *path, mode_t mode) {
  char vpath[PR_TUNABLE_PATH_MAX];

  if (session.curr_phase == LOG_CMD ||
      session.curr_phase == LOG_CMD_ERR ||
      (session.sf_flags & SF_ABORT) ||
      *vroot_base == '\0')
    /* NOTE: once stackable FS modules are supported, have this fall through
     * to the next module in the stack.
     */
    return mkdir(path, mode);

  if (vroot_lookup_path(vpath, sizeof(vpath), path) < 0)
    return -1;

  return mkdir(vpath, mode);
}

static int vroot_rmdir(pr_fs_t *fs, const char *path) {
  char vpath[PR_TUNABLE_PATH_MAX];

  if (session.curr_phase == LOG_CMD ||
      session.curr_phase == LOG_CMD_ERR ||
      (session.sf_flags & SF_ABORT) ||
      *vroot_base == '\0')
    /* NOTE: once stackable FS modules are supported, have this fall through
     * to the next module in the stack.
     */
    return rmdir(path);

  if (vroot_lookup_path(vpath, sizeof(vpath), path) < 0)
    return -1;

  return rmdir(vpath);
}

/* Configuration handlers
 */

/* usage: VRootEngine on|off */
MODRET set_vrootengine(cmd_rec *cmd) {
  int bool = -1;
  config_rec *c = NULL;

  CHECK_ARGS(cmd, 1);
  CHECK_CONF(cmd, CONF_ROOT|CONF_VIRTUAL|CONF_GLOBAL);

  bool = get_boolean(cmd, 1);
  if (bool == -1)
    CONF_ERROR(cmd, "expected Boolean parameter");

  c = add_config_param(cmd->argv[0], 1, NULL);
  c->argv[0] = pcalloc(c->pool, sizeof(unsigned char));
  *((unsigned char *) c->argv[0]) = bool;

  return HANDLED(cmd);
}

/* usage: VRootLog path|"none" */
MODRET set_vrootlog(cmd_rec *cmd) {
  CHECK_ARGS(cmd, 1);
  CHECK_CONF(cmd, CONF_ROOT);

  if (pr_fs_valid_path(cmd->argv[1]) < 0)
    CONF_ERROR(cmd, "must be an absolute path");

  add_config_param_str(cmd->argv[0], 1, cmd->argv[1]);
  return HANDLED(cmd);
}

/* usage: VRootOptions opt1 opt2 ... optN */
MODRET set_vrootoptions(cmd_rec *cmd) {
  config_rec *c = NULL;
  register unsigned int i;
  unsigned int opts = 0U;

  if (cmd->argc-1 == 0)
    CONF_ERROR(cmd, "wrong number of parameters");

  CHECK_CONF(cmd, CONF_ROOT|CONF_VIRTUAL|CONF_GLOBAL);

  c = add_config_param(cmd->argv[0], 1, NULL);
  for (i = 1; i < cmd->argc; i++) {
    if (strcmp(cmd->argv[i], "allowSymlinks") == 0) {
      opts |= VROOT_OPT_ALLOW_SYMLINKS;

    } else {
      CONF_ERROR(cmd, pstrcat(cmd->tmp_pool, ": unknown VRootOption: '",
        cmd->argv[i], "'", NULL));
    }
  }

  c->argv[0] = pcalloc(c->pool, sizeof(unsigned int));
  *((unsigned int *) c->argv[0]) = opts;

  return HANDLED(cmd);
}

/* Command handlers
 */

MODRET vroot_pre_pass(cmd_rec *cmd) {
  pr_fs_t *fs = NULL;
  unsigned char *use_vroot = NULL;

  use_vroot = get_param_ptr(main_server->conf, "VRootEngine", FALSE); 
  
  if (!use_vroot || *use_vroot == FALSE) {
    vroot_engine = FALSE;
    return DECLINED(cmd);
  }

  fs = pr_register_fs(main_server->pool, "vroot", "/");
  if (fs == NULL) {
    pr_log_debug(DEBUG3, MOD_VROOT_VERSION ": error registering fs: %s",
      strerror(errno));
    return DECLINED(cmd);
  }

  pr_log_debug(DEBUG5, MOD_VROOT_VERSION ": vroot registered");

  /* Add the module's custom FS callbacks here. This module does not
   * provide callbacks for the following (as they are unnecessary):
   * close(), read(), write(), lseek(), readdir(), and closedir().
   */
  fs->stat = vroot_stat;
  fs->lstat = vroot_lstat;
  fs->rename = vroot_rename;
  fs->unlink = vroot_unlink;
  fs->open = vroot_open;
  fs->creat = vroot_creat;
  fs->link = vroot_link;
  fs->readlink = vroot_readlink;
  fs->symlink = vroot_symlink;
  fs->truncate = vroot_truncate;
  fs->chmod = vroot_chmod;
  fs->chown = vroot_chown;
  fs->chdir = vroot_chdir;
  fs->chroot = vroot_chroot;
  fs->opendir = vroot_opendir;
  fs->mkdir = vroot_mkdir;
  fs->rmdir = vroot_rmdir;

  vroot_engine = TRUE;
  return DECLINED(cmd);
}

MODRET vroot_post_pass(cmd_rec *cmd) {
  if (vroot_engine) {

    /* If not chrooted, unregister vroot. */
    if (!session.chroot_path) {
      if (pr_unregister_fs("/") < 0)
        pr_log_debug(DEBUG2, MOD_VROOT_VERSION
          ": error unregistering vroot: %s", strerror(errno));

      else {
        pr_log_debug(DEBUG5, MOD_VROOT_VERSION
          ": vroot unregistered");
        pr_fs_setcwd(pr_fs_getvwd());
        pr_fs_clear_cache();
      }

    } else {

      /* Otherwise, lookup and process any VRootOptions. */
      config_rec *c = find_config(main_server->conf, CONF_PARAM,
        "VRootOptions", FALSE);

      if (c)
        vroot_opts = *((unsigned int *) c->argv[0]);
    }
  }

  return DECLINED(cmd);
}

MODRET vroot_post_pass_err(cmd_rec *cmd) {
  if (vroot_engine) {

    /* If not chrooted, unregister vroot. */
    if (!session.chroot_path) {
      if (pr_unregister_fs("/") < 0)
        pr_log_debug(DEBUG2, MOD_VROOT_VERSION
          ": error unregistering vroot: %s", strerror(errno));
      else
        pr_log_debug(DEBUG5, MOD_VROOT_VERSION
          ": vroot unregistered");
    }
  }

  return DECLINED(cmd);
}

/* Initialization routines
 */

static int vroot_sess_init(void) {
  config_rec *c;

  c = find_config(main_server->conf, CONF_PARAM, "VRootLog", FALSE);
  if (c)
    vroot_log = c->argv[0];

  if (vroot_log &&
      strcasecmp(vroot_log, "none") != 0) {
    int res;

    PRIVS_ROOT
    res = pr_log_openfile(vroot_log, &vroot_logfd, 0660);
    PRIVS_RELINQUISH

    switch (res) {
      case 0:
        break;

      case -1:
        pr_log_debug(DEBUG1, MOD_VROOT_VERSION
          ": unable to open VRootLog '%s': %s", vroot_log, strerror(errno));
        break;

      case PR_LOG_SYMLINK:
        pr_log_debug(DEBUG1, MOD_VROOT_VERSION
          ": unable to open VRootLog '%s': %s", vroot_log, "is a symlink");
        break;

      case PR_LOG_WRITABLE_DIR:
        pr_log_debug(DEBUG1, MOD_VROOT_VERSION
          ": unable to open VRootLog '%s': %s", vroot_log,
          "parent directory is world-writable");
        break;
    }
  }

  return 0;
}

/* Module API tables
 */

static conftable vroot_conftab[] = {
  { "VRootEngine",	set_vrootengine,	NULL },
  { "VRootLog",		set_vrootlog,		NULL },
  { "VRootOptions",	set_vrootoptions,	NULL },
  { NULL }
};

static cmdtable vroot_cmdtab[] = {
  { PRE_CMD,		C_PASS,	G_NONE,	vroot_pre_pass, FALSE, FALSE },
  { POST_CMD,		C_PASS,	G_NONE,	vroot_post_pass, FALSE, FALSE },
  { POST_CMD_ERR,	C_PASS,	G_NONE,	vroot_post_pass_err, FALSE, FALSE },
  { 0, NULL }
};

module vroot_module = {
  NULL, NULL,

  /* Module API version 2.0 */
  0x20,

  /* Module name */
  "vroot",

  /* Module configuration handler table */
  vroot_conftab,

  /* Module command handler table */
  vroot_cmdtab,

  /* Module authentication handler table */
  NULL,

  /* Module initialization function */
  NULL,

  /* Session initialization function */
  vroot_sess_init,

  /* Module version */
  MOD_VROOT_VERSION
};
