/*
 * mmap a file, truncate it.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <time.h>
#include <setjmp.h>
#include <sys/wait.h>
#include <sys/mman.h>

#define ALIGN(x,a) (((x)+(a)-1)&~((a)-1))

char *ourname;
char *filename;
unsigned long truncatesize;
unsigned long mmapsize;
int got_bus_error;
unsigned long bus_error_offset = ~0UL;
unsigned long no_bus_error_offset = ~0UL;

void usage(void)
{
	fprintf(stderr,
	"Usage: %s [-b offset] [-B offset] filename mmapsize truncatesize\n"
		"  -b offset             : expect a bus error at offset\n"
		"  -B offset             : expect no bus error at offset\n"
		"",
			ourname);
	exit(1);
}

jmp_buf env;

void sigbus(int sig)
{
	got_bus_error = 1;
	longjmp(env, 1);
}

volatile signed char c;

int bus_error(char *p)
{
	signal(SIGBUS, sigbus);
	got_bus_error = 0;
	if (!setjmp(env)) {
		volatile char *pp = p;

		c = *pp;
	}
	return got_bus_error;
}

unsigned long getnum(char *p)
{
	unsigned long ret;

	if (!strncmp(p, "0x", 2))
		ret = strtoul(p + 2, NULL, 16);
	else
		ret = strtoul(p, NULL, 10);
	return ret;
}

int main(int argc, char *argv[])
{
	char *map;
	int fd;

	ourname = argv[0];
	while ((c = getopt(argc, argv, "b:B:")) != -1) {
		switch (c) {
		case 'b':
			bus_error_offset = getnum(optarg);
			break;
		case 'B':
			no_bus_error_offset = getnum(optarg);
			break;
		default:
			usage();
		}
	}

	if (optind == argc)
		usage();
	filename = argv[optind++];

	if (optind == argc)
		usage();
	mmapsize = getnum(argv[optind++]);

	if (optind == argc)
		usage();
	truncatesize = getnum(argv[optind++]);

	if (optind != argc)
		usage();

	fd = open(filename, O_RDWR, 0666);
	if (fd < 0) {
		fprintf(stderr, "%s: failed to open `%s': %s\n",
			ourname, filename, strerror(errno));
		exit(1);
	}

	map = mmap(NULL, mmapsize, PROT_READ|PROT_WRITE, MAP_PRIVATE, fd, 0);
	if (map == MAP_FAILED) {
		perror("mmap");
		exit(1);
	}

	if (ftruncate(fd, truncatesize) < 0) {
		perror("truncate");
		exit(1);
	}

	printf("mapping size 0x%lx, truncated to 0x%lx\n",
			mmapsize, truncatesize);

	getppid();		/* kgdb hook */

	if (bus_error_offset != ~0UL) {
		if (bus_error(map + bus_error_offset)) {
			printf("OK, bus error at offset 0x%lx (0x%lx)\n",
				bus_error_offset,
				(long)(map + bus_error_offset));
		} else {
			printf("error: no bus error at offset 0x%lx (0x%lx)\n",
				bus_error_offset,
				(long)(map + bus_error_offset));
		}
	}
	
	if (no_bus_error_offset != ~0UL) {
		if (!bus_error(map + no_bus_error_offset)) {
			printf("OK, no bus error at offset 0x%lx (0x%lx)\n",
				no_bus_error_offset,
				(long)(map + no_bus_error_offset));
		} else {
			printf("error: bus error at offset 0x%lx 0x%lx)\n",
				bus_error_offset,
				(long)(map + bus_error_offset));
		}
	}

	fflush(stdout);
	sleep(1);	
	exit(0);
}
