/*
 * test_create.c - test pfm_create_context
 *
 * Copyright (c) 2008 Google, Inc
 * Contributed by Stephane Eranian <eranian@google.com>
 */

#include <sys/types.h>
#include <inttypes.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>
#include <sys/wait.h>
#include <sys/resource.h>

#include <perfmon/perfmon.h>
#include <perfmon/perfmon_dfl_smpl.h>

#include "pfm_tests.h"


/*
 * test simple per-thread context creation
 */
static int
test1(void)
{
	pfarg_ctx_t ctx;
	int ctx_fd;

	memset(&ctx, 0, sizeof(ctx));

	ctx_fd = pfm_create_context(&ctx, NULL, NULL, 0);
	if (ctx_fd == -1)  {
		PFM_LOG("cannot create PFM context: %s\n", strerror(errno));
		return -1;
	}
	close(ctx_fd);
	return 0;
}

/*
 * test PFM_FL_NOTIFY_BLOCK accepted in per-thread context
 */
static int
test2(void)
{
	pfarg_ctx_t ctx;
	int ctx_fd;

	memset(&ctx, 0, sizeof(ctx));

	ctx.ctx_flags = PFM_FL_NOTIFY_BLOCK;
	ctx_fd = pfm_create_context(&ctx, NULL, NULL, 0);
	if (ctx_fd == -1) {
		PFM_LOG("NOTIFY_BLOCK not accepted on per-thread session: %s", strerror(errno));
		return -1;
	}
	close(ctx_fd);
	return 0;
}

/*
 * test system-wide context creation
 */
static int
test3(void)
{
	pfarg_ctx_t ctx;
	int ctx_fd;

	memset(&ctx, 0, sizeof(ctx));

	ctx.ctx_flags = PFM_FL_SYSTEM_WIDE;
	ctx_fd = pfm_create_context(&ctx, NULL, NULL, 0);
	if (ctx_fd == -1) {
		PFM_LOG("syswide session failed: %s", strerror(errno));
		return -1;
	}

	close(ctx_fd);
	return 0;
}

/*
 * test PFM_FL_NOTIFY_BLOCK refused for system-wide context
 */
static int
test4(void)
{
	pfarg_ctx_t ctx;
	int ctx_fd;

	memset(&ctx, 0, sizeof(ctx));

	ctx.ctx_flags = PFM_FL_NOTIFY_BLOCK | PFM_FL_SYSTEM_WIDE;
	ctx_fd = pfm_create_context(&ctx, NULL, NULL, 0);

	if (ctx_fd == 0) {
		PFM_LOG("NOTIFY_BLOCK accepted on syswide session");
		return -1;
	}
	close(ctx_fd);
	return 0;
}

/*
 * check task_group restriction forusers group work
 */
static int
test5(void)
{
	pfarg_ctx_t ctx;
	pid_t pid;
	char *str;
	gid_t gid, ogid;
	int ret, fd, status;
	char buf[16];

	memset(&ctx, 0, sizeof(ctx));

	/*
 	 * test requires root privileges as we change group
 	 */
	if (geteuid() != 0) {
		PFM_LOG("euid not root");
		return -2;
	}

	str = read_sysfs("/sys/kernel/perfmon/task_group");
	if (!str)
		return -1;

	gid = ogid = atoi(str);

	/* gid == -1: any group is allowed */
	gid = 100 + random() % (65535-100);

	switch(pid = fork()) {
		case -1: return -1;
		case 0:
			sprintf(buf, "%d", gid);
			ret = write_sysfs("/sys/kernel/perfmon/task_group", buf);
			if (ret)
				exit(1);	

			ret = setegid(gid);
			if (ret == -1)
				PFM_LOG("seteuid(gid=task_group) error : %s", strerror(errno));
	
			fd = pfm_create_context(&ctx, NULL, NULL, 0);
			if (fd == -1) {
				PFM_LOG("cannot create session for gid=task_group=%d: %s", gid, strerror(errno));
				exit(1);
			}
			close(fd);

			gid++;
			if (gid == 0)
				gid = 100 + random() % (65535-100);

			ret = setegid(gid);
			if (ret == -1)
				PFM_LOG("seteuid(gid=task_group) error : %s", strerror(errno));

			fd = pfm_create_context(&ctx, NULL, NULL, 0);
			if (fd == 0) {
				PFM_LOG("cannot create session for gid=task_group=%d: %s", gid, strerror(errno));
				close(fd);
				exit(1);
			}

			if (errno != EPERM) {
				PFM_LOG("unexpected errno=%d expected=EPERM", errno);
				close(fd);
				exit(1);
			}

			exit(0);
		default:
			waitpid(pid, &status, 0);
	}
	/*
 	 * undo any changes
 	 * special case -1 (any) because sysfs pdoes not parse signed integers
 	 */
	if (ogid == -1)
		write_sysfs("/sys/kernel/perfmon/task_group", "4294967295");
	else	
		write_sysfs("/sys/kernel/perfmon/task_group", str);

	/* allocated by read_sysfs */
	free(str);

	return WEXITSTATUS(status) == 0 ? 0 : -1;
}

/*
 * test sys_group users group restriction works for system-wide context
 */
static int
test6(void)
{
	pfarg_ctx_t ctx;
	pid_t pid;
	char *str;
	gid_t gid, ogid;
	int ret, fd, status;
	char buf[16];

	memset(&ctx, 0, sizeof(ctx));

	ctx.ctx_flags = PFM_FL_SYSTEM_WIDE;

	/*
 	 * test requires root privileges as we change group
 	 */
	if (geteuid() != 0) {
		PFM_LOG("euid not root");
		return -2;
	}

	str = read_sysfs("/sys/kernel/perfmon/sys_group");
	if (!str)
		return -1;

	gid = ogid = atoi(str);



	/* gid == -1: any group is allowed */
	gid = 100 + random() % (65535-100);

	switch(pid = fork()) {
		case -1: return -1;
		case 0:
			sprintf(buf, "%d", gid);
			ret = write_sysfs("/sys/kernel/perfmon/sys_group", buf);
			if (ret)
				exit(1);	

			ret = setegid(gid);
			if (ret == -1)
				PFM_LOG("seteuid(gid=task_group) error : %s", strerror(errno));
	
			ctx.ctx_flags = PFM_FL_SYSTEM_WIDE;

			fd = pfm_create_context(&ctx, NULL, NULL, 0);
			if (fd == -1) {
				PFM_LOG("cannot create session for gid=task_group=%d: %s", gid, strerror(errno));
				exit(1);
			}
			close(fd);

			gid++;
			if (gid == 0)
				gid = 100 + random() % (65535-100);

			ret = setegid(gid);
			if (ret == -1)
				PFM_LOG("seteuid(gid=task_group) error : %s", strerror(errno));

			ctx.ctx_flags = PFM_FL_SYSTEM_WIDE;

			fd = pfm_create_context(&ctx, NULL, NULL, 0);
			if (fd == 0) {
				PFM_LOG("cannot create session for gid=task_group=%d: %s", gid, strerror(errno));
				close(fd);
				exit(1);
			}

			if (errno != EPERM) {
				PFM_LOG("unexpected errno=%d expected=EPERM", errno);
				close(fd);
				exit(1);
			}
			close(fd);
			exit(0);
		default:
			waitpid(pid, &status, 0);
	}
	/*
 	 * undo any changes
 	 * special case -1 (any) because sysfs pdoes not parse signed integers
 	 */
	if (ogid == -1)
		write_sysfs("/sys/kernel/perfmon/sys_group", "4294967295");
	else
		write_sysfs("/sys/kernel/perfmon/sys_group", str);

	/* allocated by read_sysfs */
	free(str);

	return WEXITSTATUS(status) == 0 ? 0 : -1;
}

/*
 * test invalid sampling format is rejected
 */
static int
test7(void)
{
	pfarg_ctx_t ctx;
	char buf[8];
	int ctx_fd;
	int ret = -1;

	memset(&ctx, 0, sizeof(ctx));

	sprintf(buf, "%ld", random() % 10000);

	ctx_fd = pfm_create_context(&ctx, buf, NULL, 0);
	if (ctx_fd == 0) {
		PFM_LOG("should not have accepted bogus sampling format name %s", buf);
		goto error;
	}

	ret = 0;
error:
	close(ctx_fd);
	return ret;
}

/*
 * check that when sampling format has argument, it must be passed on creation
 */
static int
test8(void)
{
	pfarg_ctx_t ctx;
	int ctx_fd;
	int ret = -1;

	memset(&ctx, 0, sizeof(ctx));

	ctx_fd = pfm_create_context(&ctx, "default", NULL, 0);
	if (ctx_fd == 0) {
		PFM_LOG("should not have accepted default format without parameters");
		goto error;
	}

	ret = 0;
error:
	close(ctx_fd);
	return ret;
}

/*
 * check that when sampling format has argument, the arg_size is checked
 */
static int
test9(void)
{
	pfarg_ctx_t ctx;
	pfm_dfl_smpl_arg_t smpl_arg;
	int ctx_fd;
	int ret = -1;

	memset(&ctx, 0, sizeof(ctx));
	memset(&smpl_arg, 0, sizeof(smpl_arg));

	ctx_fd = pfm_create_context(&ctx, "default", &smpl_arg, 0);
	if (ctx_fd == 0) {
		PFM_LOG("should not have accepted default format without size parameters");
		goto error;
	}

	ret = 0;
error:
	close(ctx_fd);
	return ret;
}

/*
 * check that when sampling format has argument, the arg_size is checked
 */
static int
test10(void)
{
	pfarg_ctx_t ctx;
	pfm_dfl_smpl_arg_t smpl_arg;
	int ctx_fd;
	int ret = -1;

	memset(&ctx, 0, sizeof(ctx));
	memset(&smpl_arg, 0, sizeof(smpl_arg));

	ctx_fd = pfm_create_context(&ctx, "default", &smpl_arg, 4);
	if (ctx_fd == 0) {
		PFM_LOG("should not have accepted default format arg with wrong size");
		goto error;
	}

	ret = 0;
error:
	close(ctx_fd);
	return ret;
}

/*
 * check minimal sampling buffer size imposed by default format
 */
static int
test11(void)
{
	pfarg_ctx_t ctx;
	pfm_dfl_smpl_arg_t smpl_arg;
	int ctx_fd;
	int ret = -1;

	memset(&ctx, 0, sizeof(ctx));
	memset(&smpl_arg, 0, sizeof(smpl_arg));

	/* size smaller than minimal size */
	smpl_arg.buf_size = 32;

	ctx_fd = pfm_create_context(&ctx, "default", &smpl_arg, sizeof(smpl_arg));
	if (ctx_fd == 0) {
		PFM_LOG("should not have accepted default format arg with wrong size");
		goto error;
	}

	ret = 0;
error:
	close(ctx_fd);
	return ret;
}

/*
 * check default sampling format sampling buffer creation in per-thread mdoe
 */
static int
test12(void)
{
	pfarg_ctx_t ctx;
	pfm_dfl_smpl_arg_t smpl_arg;
	int ctx_fd;
	int ret = -1;

	memset(&ctx, 0, sizeof(ctx));
	memset(&smpl_arg, 0, sizeof(smpl_arg));

	smpl_arg.buf_size = getpagesize();

	ctx_fd = pfm_create_context(&ctx, "default", &smpl_arg, sizeof(smpl_arg));
	if (ctx_fd == -1) {
		PFM_LOG("failed context creation with default format: %s", strerror(errno));
		goto error;
	}

	ret = 0;
	close(ctx_fd);
error:
	return ret;
}

/*
 * check default sampling format sampling buffer creation in system-wide mode
 */
static int
test13(void)
{
	pfarg_ctx_t ctx;
	pfm_dfl_smpl_arg_t smpl_arg;
	int ctx_fd;
	int ret = -1;

	memset(&ctx, 0, sizeof(ctx));
	memset(&smpl_arg, 0, sizeof(smpl_arg));

	smpl_arg.buf_size = getpagesize();

	/* test in system-wide mode */
	ctx.ctx_flags = PFM_FL_SYSTEM_WIDE;

	ctx_fd = pfm_create_context(&ctx, "default", &smpl_arg, sizeof(smpl_arg));
	if (ctx_fd == -1) {
		PFM_LOG("failed context creation with default format: %s", strerror(errno));
		goto error;
	}

	ret = 0;
	close(ctx_fd);
error:
	return ret;
}

/*
 * check per-process resource limit for RLIMIT_MEMLOCK when using sampling buffer
 */
static int
test14(void)
{
	pfarg_ctx_t ctx;
	pfm_dfl_smpl_arg_t smpl_arg;
	struct rlimit rlim, rlim_orig;
	int ctx_fd;
	int ret = -1;

	memset(&ctx, 0, sizeof(ctx));
	memset(&smpl_arg, 0, sizeof(smpl_arg));

	ret = getrlimit(RLIMIT_MEMLOCK, &rlim_orig);
	if (ret == -1) {
		PFM_LOG("cannot retrieve maximum memory useable by buffer");
		return -1;
	}

	rlim = rlim_orig;

	if (rlim.rlim_cur == RLIM_INFINITY) {
		rlim.rlim_cur = 8 * getpagesize();

		ret = setrlimit(RLIMIT_MEMLOCK, &rlim);
		if (ret == -1) {
			PFM_LOG("cannot set rlimit: %s", strerror(errno));
			return -1;
		}
	}

	if ((rlim.rlim_cur + 1) < rlim.rlim_cur)
		rlim.rlim_cur--;

	/* oversubscribe */
	smpl_arg.buf_size = rlim.rlim_cur + 1;
	
	ctx_fd = pfm_create_context(&ctx, "default", &smpl_arg, sizeof(smpl_arg));
	if (ctx_fd == 0) {
		PFM_LOG("should not have accepted buffer creation");
		goto error;
	}

	ret = 0;
	close(ctx_fd);
error:
	setrlimit(RLIMIT_MEMLOCK, &rlim_orig);
	return ret;
}


static int (*ctx_tests[])(void)={
	test1,
	test2,
	test3,
	test4,
	test5,
	test6,
	test7,
	test8,
	test9,
	test10,
	test11,
	test12,
	test13,
	test14,
	NULL
};

int
main_pfm_create_context(int argc, char **argv)
{
	int i, ret;
	int (**pf)(void);

	for (pf = ctx_tests, i = 0; *pf ; pf++, i++) {
		printf("pfm_create_context.test%-3d ", i);fflush(stdout);
		ret = (*pf)();
		printf("[%s]\n", ret == -1 ? "FAIL" : ret == -2 ? "SKIP" : "PASS");
		if (ret)
			return ret;
	}
	return 0;
}
