/*
 * test_load.c - test pfm_load_context/pfm_unload_context
 *
 * Copyright (c) 2008 Google, Inc
 * Contributed by Stephane Eranian <eranian@google.com>
 */
#ifndef _GNU_SOURCE
#define _GNU_SOURCE /* sched_setaffinity */
#endif
#include <sys/types.h>
#include <inttypes.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <sched.h>
#include <sys/wait.h>
#include <sys/resource.h>
#include <perfmon/perfmon.h>
#include <perfmon/pfmlib.h>

#include "pfm_tests.h"
#include "detect_pmcs.h"

static int
test1(void)
{
	pfarg_load_t load;
	int ret;

	memset(&load, 0, sizeof(load));

	/* invalid fd */
	ret = pfm_load_context(0, &load);
	if (ret == 0) {
		PFM_LOG("should not have accepted the call");
		return -1;
	}

	ret = pfm_unload_context(0);
	if (ret == 0) {
		PFM_LOG("should not have accepted the call");
		return -1;
	}
	return 0;
}

static int
test2(void)
{
	pfarg_ctx_t ctx;
	pfarg_load_t load;
	int fd, ret;

	memset(&ctx, 0, sizeof(ctx));
	memset(&load, 0, sizeof(load));

	fd = pfm_create_context(&ctx, NULL, NULL, 0);
	if (fd == -1) {
		PFM_LOG("cannot create context");
		return -1;
	}

	/* invalid load arg */
	ret = pfm_load_context(fd, NULL);
	close(fd);
	if (ret == 0) {
		PFM_LOG("should not have accepted the call");
		return -1;
	}
	return 0;
}

static int
test3(void)
{
	pfarg_ctx_t ctx;
	pfarg_load_t load;
	int fd, ret;

	memset(&ctx, 0, sizeof(ctx));
	memset(&load, 0, sizeof(load));

	fd = pfm_create_context(&ctx, NULL, NULL, 0);
	if (fd == -1) {
		PFM_LOG("cannot create context");
		return -1;
	}

	load.load_pid = getpid();

	ret = pfm_load_context(fd, &load);
	if (ret == -1) {
		PFM_LOG("load failed: %s", strerror(errno));
		close(fd);
		return -1;
	}

	ret = pfm_unload_context(fd);
	if (ret == -1) {
		PFM_LOG("unload failed: %s", strerror(errno));
		close(fd);
		return -1;
	}
	close(fd);
	return 0;
}

static int
test4(void)
{
	pfarg_ctx_t ctx;
	pfarg_load_t load;
	int fd, ret;

	memset(&ctx, 0, sizeof(ctx));
	memset(&load, 0, sizeof(load));

	fd = pfm_create_context(&ctx, NULL, NULL, 0);
	if (fd == -1) {
		PFM_LOG("cannot create context");
		return -1;
	}

	load.load_pid = 1;
	/* cannot attach to init */
	ret = pfm_load_context(fd, &load);
	if (ret == 0) {
		PFM_LOG("should not have accepted attach to init");
		close(fd);
		return -1;
	}
	close(fd);
	return 0;
}

static int
test5(void)
{
	pfarg_ctx_t ctx;
	pfarg_load_t load;
	FILE *fp;
	int fd, ret, pid;

	
	fp = popen("ps ax | fgrep [aio | cut -d ' ' -f 3| head -1", "r");
	if (!fp) {
		PFM_LOG("popen failed");
		return -1;
	}
	ret = fscanf(fp, "%d", &pid);
	if (ret != 1) {
		PFM_LOG("cannot extract pid for kernel thread");
		fclose(fp);
		return -1;
	}
	fclose(fp);

	memset(&ctx, 0, sizeof(ctx));
	memset(&load, 0, sizeof(load));

	fd = pfm_create_context(&ctx, NULL, NULL, 0);
	if (fd == -1) {
		PFM_LOG("cannot create context");
		return -1;
	}

	load.load_pid = pid;

	/* cannot attach to kernel thread */
	ret = pfm_load_context(fd, &load);
	if (ret == 0) {
		PFM_LOG("should not have accepted attach to init");
		close(fd);
		return -1;
	}
	close(fd);
	return 0;
}

static int
test6_child(pid_t pid)
{
	pfarg_ctx_t ctx;
	pfarg_load_t load;
	int fd, ret;


	memset(&ctx, 0, sizeof(ctx));

	fd = pfm_create_context(&ctx, NULL, NULL, 0);
	if (fd == -1) {
		PFM_LOG("cannot create context: %s", strerror(errno));
		return -1;
	}
	memset(&load, 0, sizeof(load));

	load.load_pid = pid;

	/* must fail because pid is not stopped unless we run on utrace kernel */
	ret = pfm_load_context(fd, &load);
	if (ret == 0) {
		PFM_LOG("should not have accepted the load (unless utrace)");
		close(fd);
		return -1;
	}
	/* looks like ptrace return ESRCH was rocess is not ptraced */
	if (errno != ESRCH) {
		PFM_LOG("unexpected errno=%d vs. ESRCH", errno);
		close(fd);
		return -1;
	}
	close(fd);	
	return 0;
}

static int
test6(void)
{
	pid_t pid1, pid2;
	int status;

	/*
 	 * test requires root privileges as we change userid
 	 */
	if (geteuid() != 0) {
		PFM_LOG("euid not root");
		return -2;
	}

	pid1 = fork();
	if (pid1 == -1) {
		PFM_LOG("cannot create process");
		return -1;
	}

	/* start process to be monitored */
	if (pid1 == 0) {
		for(;;);
		exit(0);
	}

	pid2 = fork();
	if (pid2 == -1) {
		PFM_LOG("cannot create process");
		kill(pid1, SIGTERM);
		waitpid(pid1, NULL, 0);
		return -1;
	}

	/* start monitoring process */
	if (pid2 == 0)
		exit(test6_child(pid1));

	waitpid(pid2, &status, 0);
	kill(pid1, SIGTERM);
	waitpid(pid1, NULL, 0);

	return WEXITSTATUS(status) == 0 ? 0 : -1;
}


static int
get_cpu(int mode)
{
	FILE *fp;
	int ret, cpu;
	
	if (mode == 0)
		fp = popen("cat /proc/cpuinfo  | fgrep processor | cut -d : -f 2 | head -1 | tr -d ' '", "r");
	else
		fp = popen("cat /proc/cpuinfo  | fgrep processor | cut -d : -f 2 | tail -1 | tr -d ' '", "r");
	if (!fp) {
		PFM_LOG("popen failed");
		return -1;
	}
	ret = fscanf(fp, "%d", &cpu);
	if (ret != 1) {
		PFM_LOG("cannot extract pid for kernel thread");
		fclose(fp);
		return -1;
	} 

	fclose(fp);

	return cpu;
}

static int
test7(void)
{
	pfarg_ctx_t ctx;
	pfarg_load_t load;
	pfmlib_regmask_t una_pmcs, una_pmds;
	pfarg_pmc_t pc[1];
	pfarg_pmd_t pd[1];
	cpu_set_t set0, set1;
	int i, fd, ret;
	int cpu0, cpu1;

	cpu0 = get_cpu(0);
	cpu1 = get_cpu(1);

	if (cpu0 == cpu1) {
		PFM_LOG("not enough CPU to run test (needs >= 2)");
		return -1;
	}

	if (cpu0 == -1 || cpu1 == -1) {
		PFM_LOG("cannot get 2 cpus");
		return -1;
	}

	memset(&ctx, 0, sizeof(ctx));
	memset(&load, 0, sizeof(load));
	memset(&pc, 0, sizeof(pc));
	memset(&pd, 0, sizeof(pd));

	ctx.ctx_flags = PFM_FL_SYSTEM_WIDE;
	fd = pfm_create_context(&ctx, NULL, NULL, 0);
	if (fd == -1) {
		PFM_LOG("cannot create context");
		return -1;
	}
	ret = detect_unavail_pmu_regs(fd, &una_pmcs, &una_pmds);
	if (ret == -1) {
		PFM_LOG("cannot get unavailable registers");
		close(fd);
		return -1;
	}
	for (i=0; i < PFMLIB_REG_MAX; i++) {
		if (!pfm_regmask_isset(&una_pmcs, i))
			break;
	}
	if (i == PFMLIB_REG_MAX) {
		PFM_LOG("no available pmcs");
		close(fd);
		return -1;
	}
	pc[0].reg_num = i;

	for (i=0; i < PFMLIB_REG_MAX; i++) {
		if (!pfm_regmask_isset(&una_pmds, i))
			break;
	}
	pd[0].reg_num = i;
	if (i == PFMLIB_REG_MAX) {
		PFM_LOG("no available pmds");
		close(fd);
		return -1;
	}

	ret = sched_getaffinity(getpid(), sizeof(cpu_set_t), &set0);
	if (ret == -1) {
		PFM_LOG("cannot get affinity: %s", strerror(errno));
		close(fd);
		return -1;
	}
	CPU_ZERO(&set1);
	CPU_SET(cpu0, &set1);

	ret = sched_setaffinity(getpid(), sizeof(cpu_set_t), &set1);
	if (ret == -1) {
		PFM_LOG("cannot set affinity: %s", strerror(errno));
		close(fd);
		return -1;
	}
	
	load.load_pid = cpu1;
	/* will fail because not on correct cpu */
	ret = pfm_load_context(fd, &load);
	if (ret == 0) {
		PFM_LOG("should not have accepted attach to init");
		ret = -1;
		goto error;
	}
	if (errno != EINVAL) {
		PFM_LOG("unexpected errno=%d vs. EINVAL", errno);
		goto error;
	}

	CPU_ZERO(&set1);
	CPU_SET(cpu1, &set1);

	/* put back on correct CPU */
	ret = sched_setaffinity(getpid(), sizeof(cpu_set_t), &set1);
	if (ret == -1) {
		PFM_LOG("cannot get affinity: %s", strerror(errno));
		goto error;
	}

	/* load must succeed */
	ret = pfm_load_context(fd, &load);
	if (ret == -1) {
		PFM_LOG("cannot load: %s", strerror(errno));
		goto error;
	}

	CPU_ZERO(&set1);
	CPU_SET(cpu0, &set1);

	/* go back on wrong cpu */
	ret = sched_setaffinity(getpid(), sizeof(cpu_set_t), &set1);
	if (ret == -1) {
		PFM_LOG("cannot get affinity: %s", strerror(errno));
		goto error;
	}

	ret = pfm_write_pmcs(fd, pc, 1);
	if (ret == 0) {
		PFM_LOG("should not have accepted write pmcs");
		ret = -1;
		goto error;
	}

	ret = pfm_write_pmds(fd, pd, 1);
	if (ret == 0) {
		PFM_LOG("should not have accepted write pmds");
		ret = -1;
		goto error;
	}

	ret = pfm_read_pmds(fd, pd, 1);
	if (ret == 0) {
		PFM_LOG("should not have accepted read pmds");
		ret = -1;
		goto error;
	}

	ret = pfm_start(fd, NULL);
	if (ret == 0) {
		PFM_LOG("should not have accepted pfm_start");
		ret = -1;
		goto error;
	}

	ret = pfm_stop(fd);
	if (ret == 0) {
		PFM_LOG("should not have accepted pfm_stop");
		ret = -1;
	} else {
		ret = 0;
	}

error:
	/* restore originial affinity mask */
	sched_setaffinity(getpid(), sizeof(cpu_set_t), &set0);

	close(fd);

	return ret;
}

static int
test8(void)
{
	pfarg_ctx_t ctx;
	int fd, ret;

	memset(&ctx, 0, sizeof(ctx));

	fd = pfm_create_context(&ctx, NULL, NULL, 0);
	if (fd == -1) {
		PFM_LOG("cannot create context");
		return -1;
	}

	/* unloading an unloaded context is harmless */
	ret = pfm_unload_context(fd);
	close(fd);
	if (ret == -1) {
		PFM_LOG("unexpected error: %s", strerror(errno));
	}
	return ret;
}


/*
 * tests of pfm_write_pmcs() after pfm_load_context() are deferred to
 * a series after the tests for pfm_load_context
 */

static int (*ctx_tests[])(void)={
	test1,
	test2,
	test3,
	test4,
	test5,
	test6,
	test7,
	test8,
	NULL
};

int
main_pfm_load_context(int argc, char **argv)
{
	int i, ret = -1;
	int (**pf)(void);

	for (pf = ctx_tests, i= 0; *pf ; pf++, i++) {
		printf("pfm_load_context.test%-3d", i); fflush(stdout);
		ret = (*pf)();
		printf("[%s]\n", ret == -1 ? "FAIL" : ret == -2 ? "SKIP" : "PASS");
		/* -2 means skipped */
		if (ret == -1)
			break;
	}
	return ret;
}
