/*
 * test_mmap.c - test samlping buffer mmap
 *
 * Copyright (c) 2008 Google, Inc
 * Contributed by Stephane Eranian <eranian@google.com>
 */

#include <sys/types.h>
#include <inttypes.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>
#include <setjmp.h>
#include <signal.h>
#include <sys/mman.h>

#include <perfmon/perfmon.h>
#include <perfmon/perfmon_dfl_smpl.h>

#include "pfm_tests.h"

static jmp_buf jbuf;

static void segv_handler(int n)
{
	longjmp(jbuf, 1);
}

/*
 * test sampling buffer remapping via mmap()
 */
static int
test1(void)
{
	pfarg_ctx_t ctx;
	pfm_dfl_smpl_arg_t smpl_arg;
	pfm_dfl_smpl_hdr_t *hdr;
	int fd;
	int ret, retval = -1;
	void *addr, *end, *p;
	uint64_t sum = 0;

	memset(&ctx, 0, sizeof(ctx));
	memset(&smpl_arg, 0, sizeof(smpl_arg));

	signal(SIGSEGV, segv_handler);

	smpl_arg.buf_size = 2 * getpagesize();

	fd = pfm_create_context(&ctx, "default", &smpl_arg, sizeof(smpl_arg));
	if (fd == -1) {
		PFM_LOG("cannot create context: %s", strerror(errno));
		return -1;
	}

	addr = mmap(NULL, 0, PROT_READ, MAP_PRIVATE, fd, 0);
	if (addr != MAP_FAILED) {
		PFM_LOG("should have rfused a zero mmap");
		goto error;
	}
	addr = mmap(NULL, (size_t)smpl_arg.buf_size + 1, PROT_READ, MAP_PRIVATE, fd, 0);
	if (addr != MAP_FAILED) {
		PFM_LOG("should have rfused a zero mmap");
		goto error;
	}

	addr = mmap(NULL, (size_t)smpl_arg.buf_size, PROT_READ, MAP_PRIVATE, fd, 0);
	if (addr == MAP_FAILED) {
		PFM_LOG("unexpected mmap failure: %s", strerror(errno));
		goto error;
	}
	end = addr + smpl_arg.buf_size;

	/* touch entire buffer */
	for (p = addr; p < end; p++) sum += *(char *)p;

	if (sum == 0) {
		PFM_LOG("invalid buffer sum=0");
		goto error_unmap;
	}

	hdr = addr;
	if (hdr->hdr_version == 0) {
		PFM_LOG("invalid version number 0");
		goto error_unmap;
	}

	if (hdr->hdr_buf_size != smpl_arg.buf_size) {
		PFM_LOG("unexpected buffer size=%"PRIu64" vs. %"PRIu64, hdr->hdr_buf_size, smpl_arg.buf_size);
		goto error_unmap;
	}

	/* flags should be zero because we did not set them */
	if (hdr->hdr_buf_flags) {
		PFM_LOG("unexpected buffer flags=0x%x vs. 0", hdr->hdr_buf_flags);
		goto error_unmap;
	}
	/* initial offset should be right after header */
	if (hdr->hdr_cur_offs != sizeof(*hdr)) {
		PFM_LOG("invalid buffer offset=%"PRIu64" vs. %zu", hdr->hdr_cur_offs, sizeof(*hdr));
		goto error_unmap;
	}
	sum = 0;
	for (p = (char *)(hdr+1); p < end; p++) sum += *(char *)p;
	
	if (sum) {
		PFM_LOG("body not zeroed sum=%"PRIu64, sum);
		goto error_unmap;
	}

	ret = munmap(addr+getpagesize(), getpagesize());
	if (ret == -1) {
		PFM_LOG("cannot unmap first page: %s", strerror(errno));
		goto error;
	}

	/* use sigsetjmp to ensure SEGV is not blocked when we return form handler */
	ret = sigsetjmp(jbuf, 1);
	if (ret)
		goto test2;

	p = end - 1;
	if (*(char *)p) {
		PFM_LOG("last byte non null");
		goto error;
	}

test2:
	ret = munmap(addr, getpagesize());
	if (ret == -1) {
		PFM_LOG("cannot unmap first page: %s", strerror(errno));
		goto error;
	}

	ret = sigsetjmp(jbuf, 1);
	if (ret)
		goto test3;

	if (hdr->hdr_version == 0) {
		PFM_LOG("invalid version number 0");
		goto error_unmap;
	}
test3:

	/* remap and verify again */

	addr = mmap(NULL, (size_t)smpl_arg.buf_size, PROT_READ, MAP_PRIVATE, fd, 0);
	if (addr == MAP_FAILED) {
		PFM_LOG("unexpected mmap failure: %s", strerror(errno));
		goto error;
	}
	end = addr + smpl_arg.buf_size;

	/* touch entire buffer */
	for (p = addr; p < end; p++) sum += *(char *)p;

	if (sum == 0) {
		PFM_LOG("invalid buffer sum=0");
		goto error_unmap;
	}

	hdr = addr;
	if (hdr->hdr_version == 0) {
		PFM_LOG("invalid version number 0");
		goto error_unmap;
	}

	close(fd);

	ret = sigsetjmp(jbuf, 1);
	if (ret) {
		PFM_LOG("buffer disappeared after close, error");
		goto error_unmap;
	}

	/* buffer remains accessible */
	if (hdr->hdr_version == 0) {
		PFM_LOG("invalid version number 0");
		goto error_unmap;
	}

	retval = 0;
error_unmap:
	munmap(addr, (size_t)smpl_arg.buf_size);
error:
	close(fd);
	return retval;
}

static int (*ctx_tests[])(void)={
	test1,
	NULL
};

int
main_mmap(int argc, char **argv)
{
	int ret, i;
	int (**pf)(void);

	for (pf = ctx_tests, i=0; *pf ; pf++, i++) {
		printf("test_mmap.test%-3d", i); fflush(stdout);
		ret = (*pf)();
		printf("[%s]\n", ret == -1 ? "FAIL" : ret == -2 ? "SKIP" : "PASS");
		if (ret)
			return ret;
	}
	return 0;
}
