/*
 * test_syst.c - test system-wide context usage
 *
 * Copyright (c) 2008 Google, Inc
 * Contributed by Stephane Eranian <eranian@google.com>
 */
#ifndef _GNU_SOURCE
#define _GNU_SOURCE /* sched_setaffinity */
#endif
#include <sys/types.h>
#include <inttypes.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>
#include <sched.h>
#include <signal.h>
#include <sys/wait.h>
#include <sys/resource.h>
#include <sys/mman.h>

#include <perfmon/perfmon.h>
#include <perfmon/perfmon_dfl_smpl.h>
#include <perfmon/pfmlib.h>

#include "pfm_tests.h"
#include "detect_pmcs.h"

static int
test1_child(int cpu)
{
	pfmlib_input_param_t inp;
	pfmlib_output_param_t outp;
	pfarg_ctx_t ctx;
	pfarg_load_t load;
	pfarg_pmc_t pc[8];
	pfarg_pmd_t pd[8];
	cpu_set_t set;
	int i, fd, ret;

	memset(&inp,0, sizeof(inp));
	memset(&outp,0, sizeof(outp));
	memset(&pc, 0, sizeof(pc));
	memset(&pd, 0, sizeof(pd));

	CPU_ZERO(&set);
	CPU_SET(cpu, &set);

	ret = pfm_get_cycle_event(&inp.pfp_events[0]);
	if (ret != PFMLIB_SUCCESS) {
		PFM_LOG("cannot find cycle event: %s", pfm_strerror(ret));
		return -1;
	}
	inp.pfp_dfl_plm = PFM_PLM3 | PFM_PLM0;
	inp.pfp_event_count = 1;

	memset(&ctx, 0, sizeof(ctx));
	ctx.ctx_flags = PFM_FL_SYSTEM_WIDE;
	fd = pfm_create_context(&ctx, NULL, NULL, 0);
	if (!fd)
		return -1;

	detect_unavail_pmcs(fd, &inp.pfp_unavail_pmcs);

	ret = pfm_dispatch_events(&inp, NULL, &outp, NULL);
	if (ret != PFMLIB_SUCCESS) {
		PFM_LOG("failed dispatch_events: %s", pfm_strerror(ret));
		goto error;
	}

	if (outp.pfp_pmc_count == 0) {
		PFM_LOG("errno no pmc to write");
		goto error;
	}

	if (outp.pfp_pmc_count*sizeof(pfarg_pmc_t) > sizeof(pc)) {
		PFM_LOG("pc vector is too small");
		goto error;
	}

	if (outp.pfp_pmd_count*sizeof(pfarg_pmd_t) > sizeof(pd)) {
		PFM_LOG("pd vector is too small");
		goto error;
	}


	for (i=0; i < outp.pfp_pmc_count; i++) {
		pc[i].reg_num   = outp.pfp_pmcs[i].reg_num;
		pc[i].reg_value = outp.pfp_pmcs[i].reg_value;
	}

	for (i=0; i < outp.pfp_pmd_count; i++)
		pd[i].reg_num   = outp.pfp_pmds[i].reg_num;

	ret = sched_setaffinity(getpid(), sizeof(set), &set);
	if (ret == -1) {
		PFM_LOG("cannot set affinity to CPU%d: %s", cpu, strerror(errno));
		goto error;
	}

	load.load_pid = cpu;
	ret = pfm_load_context(fd, &load);
	if (ret == -1) {
		PFM_LOG("load failed: %s", strerror(errno));
		goto error;
	}
	if (pfm_write_pmcs(fd, pc, outp.pfp_pmc_count)) {
		PFM_LOG("cannot write pmc: %s", strerror(errno));
		goto error;
	}


	if (pfm_write_pmds(fd, pd, outp.pfp_pmd_count)) {
		PFM_LOG("cannot write pmd: %s", strerror(errno));
		goto error;
	}

	if (pfm_start(fd, NULL)) {
		PFM_LOG("cannot start: %s", strerror(errno));
		goto error;
	}

	sleep(2);

	if (pfm_stop(fd)) {
		PFM_LOG("cannot stop: %s", strerror(errno));
		goto error;
	}

	if (pfm_read_pmds(fd, pd, outp.pfp_pmd_count)) {
		PFM_LOG("cannot write pmd: %s", strerror(errno));
		goto error;
	}
	if (pd[0].reg_value == 0) {
		PFM_LOG("PMD%u did not count anything on CPU%d", pd[0].reg_num, cpu);
		ret = -1;
	} else
		ret = 0;
error:
	close(fd);
	return ret;
}

/*
 * test creation, programming, attach, start,stop, read on all available processors
 */
static int
test1(void)
{
	long i, ncpus;
	pid_t *pids;
	int status, ret = 0;

	if (pfmlib_ok == 0)
		return -2;

	ncpus = sysconf(_SC_NPROCESSORS_ONLN);
	if (ncpus == -1) {
		PFM_LOG("cannot retrieve number of online cpus: %s", strerror(errno));
		return -1;
	}

	pids = calloc(ncpus, sizeof(pid_t));
	if (!pids) {
		PFM_LOG("cannot allocate pid array: %s", strerror(errno));
		return -1;
	}
	for(i=0; i < ncpus; i++) {
		pids[i] = fork();
		if (pids[i] == -1)
			goto undo;
		if (pids[i] == 0)
			exit(test1_child(i));	
	}
	while(waitpid(-1, &status, 0) > -1) {
		if (WEXITSTATUS(status) )
			ret = -1;
	}
	free(pids);
	return ret;
undo:
	ncpus = i;
	for(i=0; i < ncpus; i++) {
		kill(pids[i], SIGTERM);
	}
	while(waitpid(-1, &status, 0) > -1);
	free(pids);
	return -1;
}

/*
 * tests of pfm_write_pmcs() after pfm_load_context() are deferred to
 * a series after the tests for pfm_load_context
 */

static int (*ctx_tests[])(void) = {
	test1,
	NULL
};

int
main_system_wide(int argc, char **argv)
{
	int ret = -1, i;
	int (**pf)(void);

	for (pf = ctx_tests, i=0; *pf ; pf++, i++) {
		printf("test_syst.test%-3d", i); fflush(stdout);
		ret = (*pf)();
		printf(" [%s]\n", ret == -1 ? "FAIL" : ret == -2 ? "SKIP" : "PASS");
		/* -2 means skip test */
		if (ret == -1)
			break;
	}
	return ret != -1 ? 0 : 1;
}
