/*
 *  flail.c
 *  Copyright 2009 Ramon de Carvalho Valle <ramon@risesecurity.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/syscall.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>
#include <math.h>
#include "flail.h"

#if defined(__powerpc64__) || defined(__powerpc__)
#include "powerpc.h"
#elif defined(__s390x__) || defined(__s390__)
#include "s390.h"
#elif defined(__i386__)
#include "x86.h"
#elif defined(__x86_64__)
#include "x86_64.h"
#else
#error "Unsupported architecture."
#endif

int duration = 3600;
char *name = NULL;
unsigned int seed = 0;

char *initialized_data = "Hello world!\n";
char *uninitialized_data;
void *external_function = (void *)printf;

void
executable_code(void)
{
	printf("%s", initialized_data);
}

unsigned int
rand_seed(void)
{
	FILE *fp;
	unsigned int seed = 1;

	if ((fp = fopen("/dev/urandom", "r")) == NULL) {
		perror("fopen");
		exit(EXIT_FAILURE);
	}

	fread(&seed, sizeof(seed), 1, fp);

	fclose(fp);

	return seed;
}

int
rand_int(int rand_max)
{
	int i;

	i = 1 + (int)(rand_max * (rand() / (RAND_MAX + 1.0)));

	return i;
}

long int
rand_arg(void)
{
	int i;
	char *local_variable = NULL;
	long int arg = 0;

	i = rand_int(9);

	switch (i) {
	case 1:
		return (long int)initialized_data;
	case 2:
		return (long int)uninitialized_data;
	case 3:
		return (long int)&uninitialized_data;
	case 4:
		return (long int)executable_code;
	case 5:
		return (long int)external_function;
	case 6:
		arg = (long int)&local_variable;
		return arg;
	case 7:
		return 0;
	case 8:
		arg = (long int)pow((double)rand_int(4096), 2);
	}

	i = rand_int(4);

	switch (i) {
	case 1:
		return arg - 1;
	case 2:
		return arg - 2;
	case 3:
		return arg + 1;
	case 4:
		return arg + 2;
	}

	return 0;
}


void
initialize_num_syscalls(void)
{
	int i;

	for (i = 0; !(syscall_end(i)); i++)
		num_syscalls++;
}

void
initialize_data(void)
{
	if ((uninitialized_data = malloc(4096)) == NULL) {
		perror("malloc");
		exit(EXIT_FAILURE);
	}

	memset(uninitialized_data, 0, 4096);
}

void
syscall_number_by_name(char *name, int *number)
{
	int i;

	for (i = 0; i < num_syscalls; i++)
		if (!strcmp(syscall_name(i), name)) {
			*number = syscall_number(i);
			break;
		}
}

static inline void
syscall_index_by_number(int number, int *index)
{
	int i;

	for (i = 0; i < num_syscalls; i++)
		if (syscall_number(i) == number) {
			*index = i;
			break;
		}
}

static inline void
randomize_args(long int *args)
{
	int i;

	for (i = 0; i < 6; i++)
		args[i] = rand_arg();
}

static inline void
print_call(int index, long int *args)
{
	int i;

#ifndef __LP64__
	printf("%-16.16s ", syscall_name(index));

	for (i = 0; i < 6; i++)
		printf("%08x ", (int)args[i]);

	printf("\n");

#else
	printf("%-16.16s ", syscall_name(index));

	for (i = 0; i < 3; i++)
		printf("%016lx ", args[i]);

	printf("\n");

	printf("%-16.16s ", "");

	for (i = 3; i < 6; i++)
		printf("%016lx ", args[i]);

	printf("\n");

#endif

	fflush(stdout);
}

static inline void
call(int index, long int *args)
{
	int status;

	if (!fork()) {
		alarm(3);

		syscall(syscall_number(index), args[0], args[1], args[2],
			args[3], args[4], args[5]);

		exit(EXIT_SUCCESS);
	}

	waitpid(-1, &status, 0);
}

void
fuzz(void)
{
	int number = 0, random = 1, index = 0;
	struct timeval tv0, tv1;
	long int args[6];

	if (name) {
		syscall_number_by_name(name, &number);

		if (number)
			random = 0;
	}

	gettimeofday(&tv0, NULL);

	for (;;) {
		gettimeofday(&tv1, NULL);

		if ((tv1.tv_sec - tv0.tv_sec) >= duration)
			break;

		if (random)
			number = random_syscall_number();

		syscall_index_by_number(number, &index);

		if (syscall_ignore(index))
			continue;

		randomize_args(args);

		print_call(index, args);

		call(index, args);
	}
}

int
main(int argc, char **argv)
{
	int c;

	seed = rand_seed();

	while ((c = getopt(argc, argv, "d:n:s:")) != -1) {
		switch (c) {
		case 'd':
			duration = atoi(optarg);
			break;
		case 'n':
			name = optarg;
			break;
		case 's':
			seed = atoi(optarg);
		}
	}

	if (name)
		printf("name = %s, ", name);

	printf("duration = %d seconds, seed = %d\n", duration, seed);

	initialize_num_syscalls();

	printf("num_syscalls = %d\n", num_syscalls);

	fflush(stdout);

	sleep(3);

	initialize_data();

	srand(seed);

	fuzz();

	exit(EXIT_SUCCESS);
}
